"""This module writes a cmcards file for CMS-Flow."""

# 1. Standard Python modules
from contextlib import suppress
from datetime import datetime
from functools import cached_property
from io import StringIO
from itertools import count
import logging
import math
import os
import shutil
import sys
from typing import Optional
import urllib
import uuid
from zipfile import ZipFile

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QDateTime
import xarray as xr

# 3. Aquaveo modules
from xms.adcirc.file_io.fort14_writer import export_geometry_to_fort14
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util, TreeNode
from xms.constraint import QuadtreeGrid2d, read_grid_from_file
from xms.constraint.ugrid_builder import UGridBuilder
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Coverage, FilterLocation, Point, Projection, UGrid as DoUGrid
from xms.datasets.dataset_writer import DatasetWriter
from xms.gdal.utilities.gdal_utils import transform_points_from_wkt, wkt_from_epsg, wkt_to_sr
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.grid.ugrid import UGrid as XmUGrid
from xms.guipy.data.target_type import TargetType
from xms.guipy.time_format import ISO_DATETIME_FORMAT
from xms.interp.interpolate import InterpIdw
from xms.interp.interpolate.interp_linear import InterpLinear
from xms.tides.data import tidal_data as td
from xms.tides.data.tidal_data import STANDARD_CONSTITUENTS, TidalData, USER_DEFINED_INDEX
from xms.tides.data.tidal_extractor import TidalExtractor

# 4. Local modules
from xms.cmsflow.components.bc_component import BCComponent
from xms.cmsflow.components.rm_structures_component import RMStructuresComponent
from xms.cmsflow.components.save_points_component import SavePointsComponent
from xms.cmsflow.components.structures_component import StructuresComponent
from xms.cmsflow.data.rm_structures_data import CALCULATION_METHODS
from xms.cmsflow.data.simulation_data import SimulationData
from xms.cmsflow.file_io.culvert_structure_writer import write_culvert_structures
from xms.cmsflow.file_io.rubble_mound_structure_writer import write_rubble_mound_structures
from xms.cmsflow.file_io.tide_gate_structure_writer import write_tide_gate_structures
from xms.cmsflow.file_io.weir_structure_writer import write_weir_structures
from xms.cmsflow.mapping.coverage_mapper import CoverageMapper

# Mappings of stored data strings to card keys
WIND_TYPES = {
    'None': 'None',
    'Spatially constant': 'Constant',
    'Meteorological stations': 'Stations',
    'Temporally and spatially varying from file': 'File'
}
WIND_FILE_TYPES = {'Navy fleet numeric with pressure': 'Fleet', 'OWI/PBL': 'OWI', 'Single ASCII file': 'ASCII'}
WIND_GRID_TYPES = {'Parameters': 'Params', 'XY file': 'XYFile'}
WAVE_INFO_TYPES = {'None': 'None', 'Single wave condition': 'Single', 'Inline steering': 'Inline'}
FLOW_WAVE_TYPES = {'Automatic': 'Automatic', 'User specified': 'Specified'}
PREDICTOR_TYPES = {'Last time step': 'LAST', 'Tidal': 'TIDAL', 'Tidal plus variation': 'TIDAL_PLUS_VARIATION'}


def format_datetime(dt_str):
    """Converts a string representation of a datetime to a datetime object.

    Args:
        dt_str (str): The string representation of the datetime. Should be in the format 'YYYY-MM-DDTHH:MM:SS' or a
            valid ISO datetime format.

    Returns:
        datetime: The datetime object converted from the input string.
    """
    try:
        return datetime.strptime(dt_str, '%Y-%m-%dT%H:%M:%S')
    except ValueError:
        return datetime.strptime(dt_str, ISO_DATETIME_FORMAT)


class CMSFlowCmcardsExporter:
    """Exporter for the CMS-Flow cmcards file."""
    def __init__(self):
        """Constructor for cmcards exporter class."""
        self.ss = StringIO()
        self.proj_name = ''  # Name of the project. Used as base for several file names.
        self.datasets_file = ''
        self.dredge_datasets_file = ''
        self.placement_datasets_file = ''
        self.save_pts_file = ''
        self._logger = logging.getLogger('xms.cmsflow')
        self._save_pts_comp = None
        self._save_pts_name = None
        self._save_pts_cov = None
        self._bc_comp = None
        self._bc_cov = None
        self._pe_tree = None
        self._tidal_data = None
        self._is_harmonic = False
        self._cov_mapper = CoverageMapper(False)
        self._time_to_export = {
            'seconds': "'sec'",
            'minutes': "'min'",
            'hours': "'hrs'",
            'days': "'days'",
            'weeks': "'weeks'",
        }
        self._exported14 = {}  # {geom_uuid: filename}
        self._dataset_paths = {}  # {uuid: h5_path}
        self._dataset_counts = 0
        self._user_amps = None
        self._user_phases = None
        self._user_cons = None
        self._user_geoms = None
        self._get_amp_phase = None
        self._xm_ugrid: Optional[XmUGrid] = None
        self._cogrid: Optional[QuadtreeGrid2d] = None
        self._bc_arc_id_to_loc = {}
        self._extractor = None
        self._sms_version = ''
        self._geom_uuid = ''
        self._projection: Optional[Projection] = None
        self._rubble_mound_datasets = {}  # {(filename, path): DatasetReader}
        self.query: Optional[Query] = None

    @property
    def projection(self) -> Projection:
        """
        The current display projection.

        Result is a default-initialized projection if self.query is not initialized.
        """
        if not self._projection:
            self._projection = self.query.display_projection if self.query is not None else Projection()
        return self._projection

    @property
    def _grid_angle(self) -> float:
        """
        The angle of the grid.

        Result is 0.0 if no grid.
        """
        return self._cogrid.angle if self._cogrid is not None else 0.0

    @property
    def _grid_origin(self) -> tuple[float, float]:
        """
        The origin of the grid.

        A tuple of (x, y). Result is (0.0, 0.0) if no grid.
        """
        if self._cogrid is not None:
            x, y, _z = self._cogrid.origin
            return x, y
        return 0.0, 0.0

    @property
    def mp_file(self) -> str:
        """The name of the *_mp.h5 file."""
        return f'{self.proj_name}_mp.h5'

    @staticmethod
    def _get_number_string(number_value):
        """Format a floating point widget value using the 'g' specifier (scientific if necessary).

        Args:
            number_value (float): The number to format

        Returns:
            (str): The formatted string

        """
        try:
            return f"{number_value:g}"
        except Exception:
            return ""

    def _get_dataset_string(self, dataset, quote=''):
        """Build the string required for dataset cards.

        Args:
            dataset (:obj:`str`): Name of the dataset
            quote (:obj:`str`, optional): Character to enclose the dataset file name in. Sometimes this was quoted,
                and sometimes it wasn't. Should it always be?

        Returns:
            (str): The formatted dataset string

        """
        if dataset:
            return f'{quote}{self.datasets_file}{quote} "Datasets/{dataset}"'
        else:
            return '""'

    def _write_constant_line(self, constant_val, card_name, spacing="", card_space=36):
        """Write a dataset card line for a Model Control dataset selector.

        Args:
            constant_val (float): value of the constant to be written
            card_name (str): Name of the constant card
            spacing (:obj:`str`, optional): Text to prepend to the line
            card_space (int): The amount of total space for the card and whitespace that comes after it.
        """
        card_space = card_space - len(spacing)
        self.ss.write(f'{spacing}{card_name:<{card_space}}{constant_val}\n')

    def _write_dataset_line(self, dataset_uuid, card_name, spacing="", card_space=36, write_blank=True):
        """Write a dataset card line for a Model Control dataset selector.

        Args:
            dataset_uuid (str): UUID of the dataset.
            card_name (str): Name of the dataset card
            spacing (:obj:`str`, optional): Text to prepend to the line
            card_space (int): The amount of total space for the card and whitespace that comes after it.
            write_blank (bool): True if the line should be written if the dataset is not found.
        """
        dataset_node = tree_util.find_tree_node_by_uuid(self._pe_tree, dataset_uuid)
        if not dataset_node:
            if not write_blank:
                return
            dataset_name = ''
        else:
            dataset_name = dataset_node.name
        quote = '"'  # Makes it easier to use in function call inside f-string.
        card_space = card_space - len(spacing)
        self.ss.write(f'{spacing}{card_name:<{card_space}}'
                      f'{self._get_dataset_string(dataset_name, quote)}\n')

    @staticmethod
    def _get_on_off_string(int_as_bool):
        """Returns a string 'ON' or 'OFF' depending on the value.

        Args:
            int_as_bool (int): 1 or 0 value for ON or OFF respectively.

        Returns:
            String of 'ON' or 'OFF'
        """
        return 'ON' if int_as_bool != 0 else 'OFF'

    @staticmethod
    def _get_file_string(file_selected, report_error=True):
        """Returns a string for the file.

        Args:
            file_selected (str): The file absolute path, or '(none selected)'.
            report_error (bool): True if error should be logged to the XMS stderr file (simulation export)

        Returns:
            String of '' if given '(none selected)' or the file absolute path.
        """
        assert isinstance(report_error, bool)
        filename = file_selected.replace('(none selected)', '')
        if report_error and not os.path.exists(filename) and XmEnv.xms_environ_running_tests() != 'TRUE':
            XmEnv.report_error(f'Unable to find file {filename}')

        with suppress(ValueError):
            # In most cases the referenced files will be in the simulation directory, or next to the XMS project, so
            # we'd like to make them relative so users can move them around together. They won't always be there though.
            # Maybe there's a reference to something stored on a network drive, for example. If the referenced file is
            # on a different drive, then relpath raises ValueError on Windows. If that happens, we'll just suppress the
            # exception and continue using the absolute path, since the relative ones are just a convenience anyway.
            filename = os.path.relpath(filename)

        return filename

    @staticmethod
    def get_file_string_or_report_error(file_selected):
        """Returns a string for the file.

        Args:
            file_selected (str): The file absolute path, or '(none selected)'.

        Returns:
            String of '' if given '(none selected)' or the file absolute path.
        """
        return CMSFlowCmcardsExporter._get_file_string(file_selected, True)

    @staticmethod
    def get_file_string_without_report(file_selected):
        """Returns a string for the file.

        Args:
            file_selected (str): The file absolute path, or '(none selected)'.

        Returns:
            String of '' if given '(none selected)' or the file absolute path.
        """
        return CMSFlowCmcardsExporter._get_file_string(file_selected, False)

    @staticmethod
    def _copy_cms_wave_files(sim_file):
        """Copy all input files for a linked CMS-Wave simulation.

        Args:
            sim_file (str): Path to the CMS-Wave *.sim file

        Returns:
            bool: True if we copied any files, False on error
        """
        if os.path.isfile(sim_file):
            cmswave_folder = os.path.dirname(sim_file)
            cmsflow_folder = os.getcwd()
            # Read all the filenames from the sim file
            with open(sim_file, 'r') as f:
                lines = f.readlines()
            if not lines:
                return False  # Empty sim file
            new_sim_lines = []
            for sim_line in lines:
                if sim_line.startswith('CMS-WAVE'):
                    new_sim_lines.append(sim_line)
                    continue  # First line is the header
                split_line = sim_line.split(maxsplit=1)
                if split_line:
                    basename = split_line[1].strip().strip('"').strip("'").strip()
                    filename = io_util.resolve_relative_path(cmswave_folder, basename)
                    if os.path.isfile(filename):  # Found a referenced file that exists
                        io_util.copyfile(filename, os.path.join(cmsflow_folder, os.path.basename(filename)))
                    new_sim_lines.append(f'{split_line[0]:<10}{os.path.basename(filename)}\n')
            if len(new_sim_lines) < 2:
                return False  # Didn't find any files
            with open(os.path.join(cmsflow_folder, os.path.basename(sim_file)), 'w') as f:
                f.writelines(new_sim_lines)
            return True
        return False  # Couldn't find sim file

    def _write_grid_geometry(self):
        """Write the grid geometry definition."""
        grid_angle = self._grid_angle
        origin_x, origin_y = self._grid_origin
        self.ss.write(
            "!Grid Geometry\n"
            f"GRID_ANGLE                          {grid_angle}\n"
            f"GRID_ORIGIN_X                       {origin_x}\n"
            f"GRID_ORIGIN_Y                       {origin_y}\n"
        )

        self.ss.write(f"TELESCOPING                         \"{self.proj_name}.tel\"\n")
        coriolis = self._sim_data.flow.attrs['LATITUDE_CORIOLIS']
        if coriolis == 'From projection':
            self.ss.write(f'CELL_LATITUDES                      "{self.mp_file}" "PROPERTIES/Model Params/Lats"\n')
            self.ss.write(f'CELL_LONGITUDES                     "{self.mp_file}" "PROPERTIES/Model Params/Lons"\n')
        else:
            self.ss.write(f"AVERAGE_LATITUDE                    {self._sim_data.flow.attrs['DEGREES']}\n")
        self.ss.write('\n')

    def _write_general_parameters(self):
        """Write the general parameters cards.

        """
        fric_coefficient_to_output = {
            'Quadratic': 'QUAD',
            'Soulsby (1995) Data2': 'DATA2',
            'Soulsby (1995) Data13': 'DATA13',
            'Fredsoe (1984)': 'F84',
            'Huynh-Thanh and Termperville (1991)': 'HT91'
        }
        fric_coefficient = self._sim_data.flow.attrs['WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT']
        fric_coefficient = fric_coefficient_to_output[fric_coefficient]

        turb_model_to_output = {
            'Subgrid': 'SUBGRID',
            'Falconer': 'FALCONER',
            'Parabolic': 'PARABOLIC',
            'Mixing length': 'MIXING_LENGTH'
        }
        turb_model = self._sim_data.flow.attrs['TURBULENCE_MODEL']
        turb_model = turb_model_to_output[turb_model]
        self.ss.write(
            "!General Parameters\n"
            f"USE_WALL_FRICTION_TERMS             "
            f"{self._get_on_off_string(self._sim_data.flow.attrs['WALL_FRICTION'])}\n"
            f"WAVE_MASS_FLUX                      "
            f"{self._get_on_off_string(self._sim_data.flow.attrs['WAVE_FLUXES'])}\n"
            f"ROLLER_MASS_FLUX                    "
            f"{self._get_on_off_string(self._sim_data.flow.attrs['ROLLER_FLUXES'])}\n"
            f"DRYING_DEPTH                        "
            f"{self._get_number_string(self._sim_data.flow.attrs['WETTING_DEPTH'])}\n"
            f"BED_SLOPE_FRICTION_FACTOR           "
            f"{self._get_on_off_string(self._sim_data.flow.attrs['BED_SLOPE_FRIC_COEFFICIENT'])}\n"
            f"WAVE_CURRENT_MEAN_STRESS            {fric_coefficient}\n"
            f"WAVE_BOTTOM_FRICTION_COEFFICIENT    "
            f"{self._get_number_string(self._sim_data.flow.attrs['QUAD_WAVE_BOTTOM_COEFFICIENT'])}\n"
            f"TURBULENCE_MODEL                    {turb_model}\n"
        )
        if self._sim_data.flow.attrs['TURBULENCE_PARAMETERS'] != 0:
            self.ss.write(
                f"EDDY_VISCOSITY_BOTTOM               "
                f"{self._get_number_string(self._sim_data.flow.attrs['CURRENT_BOTTOM_COEFFICIENT'])}\n"
                f"EDDY_VISCOSITY_HORIZONTAL           "
                f"{self._get_number_string(self._sim_data.flow.attrs['CURRENT_HORIZONTAL_COEFFICIENT'])}\n"
                f"EDDY_VISCOSITY_CONSTANT             "
                f"{self._get_number_string(self._sim_data.flow.attrs['BASE_VALUE'])}\n"
                f"EDDY_VISCOSITY_WAVE                 "
                f"{self._get_number_string(self._sim_data.flow.attrs['WAVE_BOTTOM_COEFFICIENT'])}\n"
                f"EDDY_VISCOSITY_BREAKING             "
                f"{self._get_number_string(self._sim_data.flow.attrs['WAVE_BREAKING_COEFFICIENT'])}\n"
            )
        self.ss.write(
            f"SIMULATION_LABEL                    \"{self._sim_data.output.attrs['SIMULATION_LABEL']}\"\n"
            f"NUM_THREADS                         {int(self._sim_data.general.attrs['NUM_THREADS'])}\n"
            '\n'
        )

    def _write_timing_parameters(self):
        """Write the timing parameters cards.

        """
        self.ss.write("!Timing\n")
        self.ss.write(
            f"HYDRO_TIMESTEP                      "
            f"{self._sim_data.flow.attrs['HYDRO_TIME_STEP_VALUE']:g} "
            f"{self._time_to_export[self._sim_data.flow.attrs['HYDRO_TIME_STEP_UNITS']]}\n"
        )
        date_start = self._sim_data.general.attrs['DATE_START']
        if date_start:
            date_start_py = format_datetime(date_start)
        else:
            date_start_py = datetime(2000, 1, 1, 0, 0, 0)
        self.ss.write(
            f"STARTING_DATE_TIME                  {date_start_py.strftime('%Y-%m-%d %H:%M:%S UTC')}\n"
            f"DURATION_RUN                        "
            f"{self._sim_data.general.attrs['SIM_DURATION_VALUE']:g} "
            f"{self._time_to_export[self._sim_data.general.attrs['SIM_DURATION_UNITS']]}\n"
            f"DURATION_RAMP                       "
            f"{self._sim_data.general.attrs['RAMP_DURATION_VALUE']:g} "
            f"{self._time_to_export[self._sim_data.general.attrs['RAMP_DURATION_UNITS']]}\n"
            '\n'
        )

    def _write_hot_start_parameters(self):
        """Write the hot start parameters cards."""
        init_file = self._sim_data.general.attrs['USE_INIT_CONDITIONS_FILE']
        hot_start_file = self._sim_data.general.attrs['USE_HOT_START_OUTPUT_FILE']
        recurring_hot_start_file = self._sim_data.general.attrs['RECURRING_HOT_START_FILE']

        if not init_file and not hot_start_file and not recurring_hot_start_file:
            return

        self.ss.write("!Hot Start\n")
        if init_file == 1:
            filename = self.get_file_string_or_report_error(self._sim_data.general.attrs['INIT_CONDITIONS_FILE'])
            self.ss.write(f'INITIAL_STARTUP_FILE                "{filename}"\n')
        if hot_start_file == 1:
            self.ss.write(
                f"HOT_START_TIME                      "
                f"{self._sim_data.general.attrs['HOT_WRITE_OUT_DURATION_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.general.attrs['HOT_WRITE_OUT_DURATION_UNITS']]}\n"
            )
        if recurring_hot_start_file == 1:
            self.ss.write(
                f"AUTO_HOT_START_INTERVAL             "
                f"{self._sim_data.general.attrs['AUTO_HOT_DURATION_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.general.attrs['AUTO_HOT_DURATION_UNITS']]}\n"
            )
        self.ss.write('\n')

    def _write_transport_parameters(self):
        """Write the transport parameters cards."""
        concentration_export = {'Global concentration': 'Global', 'Spatially varied': 'Varied'}
        temperature_export = {'Constant water temperature': '', 'Spatially varied': 'Spatially varied'}
        self.ss.write("!Transport\n")
        salinity = self._get_on_off_string(self._sim_data.salinity.attrs['CALCULATE_SALINITY'])
        salinity_con = concentration_export[self._sim_data.salinity.attrs['SALINITY_CONCENTRATION']]
        self.ss.write(f"CALC_SALINITY                       {salinity}\n")
        if salinity == "ON":
            if salinity_con == "Global":
                self.ss.write(
                    f"  SALINITY_IC                       "
                    f"{self._get_number_string(self._sim_data.salinity.attrs['GLOBAL_CONCENTRATION'])} ppt\n"
                )
                sal_transport_units = self._sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_UNITS']
                self.ss.write(
                    f"  SALINITY_CALC_INTERVAL            "
                    f"{self._sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_VALUE']:g} "
                    f"{self._time_to_export[sal_transport_units]}\n"
                )
            else:
                dset = self._sim_data.salinity.attrs['SALINITY_INITIAL_CONCENTRATION']
                self._write_dataset_line(dset, "SALINITY_IC_DATASET", "  ")
        self.ss.write(
            f"WATER_DENSITY                       "
            f"{self._get_number_string(self._sim_data.salinity.attrs['WATER_DENSITY'])} 'kg/m^3'\n"
        )
        calc_temp = self._get_on_off_string(self._sim_data.salinity.attrs['CALCULATE_TEMPERATURE'])
        if calc_temp == "OFF":
            self.ss.write(
                f"WATER_TEMPERATURE                   "
                f"{self._get_number_string(self._sim_data.salinity.attrs['WATER_TEMP'])} 'deg C'\n"
            )
        self.ss.write(f"CALC_TEMPERATURE                    {calc_temp}\n")
        temp = temperature_export[self._sim_data.salinity.attrs['INITIAL_TEMPERATURE_TYPE']]
        if calc_temp == "ON":
            if temp == "Spatially varied":
                dset = self._sim_data.salinity.attrs['INITIAL_TEMPERATURE_DATASET']
                self._write_dataset_line(dset, "TEMPERATURE_IC_DATASET", "  ")
            else:
                self.ss.write(
                    f"  WATER_TEMPERATURE                 "
                    f"{self._get_number_string(self._sim_data.salinity.attrs['WATER_TEMP'])} 'deg C'\n"
                )
                temp_transport_units = self._sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_UNITS']
                self.ss.write(
                    f"  TEMPERATURE_CALC_INTERVAL         "
                    f"{self._sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_VALUE']:g} "
                    f"{self._time_to_export[temp_transport_units]}\n"
                )
        self.ss.write('\n')

    def _write_save_points(self):
        """Write the save points cards."""
        if not self._save_pts_name:
            return

        self.ss.write("!Save Points\n")
        self.ss.write(f"SAVE_POINT_LABEL                    \"{self._save_pts_name}\"\n")
        self.ss.write(
            f"HYDRO_OUTPUT_INTERVAL               "
            f"{self._save_pts_comp.data.general.attrs['HYDRO_VALUE']} "
            f"{self._time_to_export[self._save_pts_comp.data.general.attrs['HYDRO_UNITS']]}\n"
        )
        self.ss.write(
            f"SEDIMENT_OUTPUT_INTERVAL            "
            f"{self._save_pts_comp.data.general.attrs['SEDIMENT_VALUE']} "
            f"{self._time_to_export[self._save_pts_comp.data.general.attrs['SEDIMENT_UNITS']]}\n"
        )
        self.ss.write(
            f"SALINITY_OUTPUT_INTERVAL            "
            f"{self._save_pts_comp.data.general.attrs['SALINITY_VALUE']} "
            f"{self._time_to_export[self._save_pts_comp.data.general.attrs['SALINITY_UNITS']]}\n"
        )
        self.ss.write(
            f"WAVE_OUTPUT_INTERVAL                "
            f"{self._save_pts_comp.data.general.attrs['WAVES_VALUE']} "
            f"{self._time_to_export[self._save_pts_comp.data.general.attrs['WAVES_UNITS']]}\n"
        )

        save_pts = self._save_pts_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        pt_comp_ids = self._save_pts_comp.comp_to_xms.get(self._save_pts_comp.cov_uuid, {}).get(TargetType.point, {})
        att_id_to_comp_id = {}
        for comp_id, att_ids in pt_comp_ids.items():
            for att_id in att_ids:
                att_id_to_comp_id[att_id] = comp_id
        comp_ids = self._save_pts_comp.data.points['comp_id'].data.tolist()
        names = self._save_pts_comp.data.points['name'].data.tolist()
        all_hydro = self._save_pts_comp.data.points['hydro'].data.tolist()
        all_sediment = self._save_pts_comp.data.points['sediment'].data.tolist()
        all_salinity = self._save_pts_comp.data.points['salinity'].data.tolist()
        all_waves = self._save_pts_comp.data.points['waves'].data.tolist()
        snapper = self._cov_mapper.get_save_points_snapper()
        point_ids = [point.id for point in save_pts]  # xms.xnap call below changes from data_objects to (x,y,z)
        snap_output = snapper.get_snapped_points(save_pts)
        if snap_output:
            cell_ids = snap_output['id']
            if self._cov_mapper.small_id_to_original_id:
                cell_ids = [self._cov_mapper.small_id_to_original_id[small_cell_id] for small_cell_id in cell_ids]
        else:
            cell_ids = [-2] * len(save_pts)  # Initialize to -2 so it becomes -1 when converted to 1-base

        # write each of the "SAVE_POINT" lines to a secondary file when there are more than 10 of them.
        if len(save_pts) > 10:
            self.ss.write(f"SAVE_POINT_CARDS                    \"{self.save_pts_file}\"\n")
            ss2 = StringIO()  # If >10, create a new buffer
            spaces = '  '  # Remove unneeded spaces in the separate file
        else:
            ss2 = self.ss  # If <10, use the existing buffer
            spaces = '                          '

        for idx, pt in enumerate(save_pts):
            comp_id = att_id_to_comp_id.get(point_ids[idx], -1)
            cell_id = cell_ids[idx] + 1
            if isinstance(pt, Point):
                pt = (pt.x, pt.y, pt.z)
            if comp_id > -1:  # Attributes are not in "unassigned" state
                comp_idx = comp_ids.index(comp_id)
                name = names[comp_idx] if names[comp_idx] else str(cell_id)  # Stringified cell id if not defined
                hydro = all_hydro[comp_idx]
                sed = all_sediment[comp_idx]
                sal = all_salinity[comp_idx]
                waves = all_waves[comp_idx]
            else:
                name = str(cell_id)  # Stringified cell id if not defined
                hydro = 0
                sed = 0
                sal = 0
                waves = 0
            ss2.write(f"SAVE_POINT{spaces}\"{name}\" {cell_id} {pt[0]} {pt[1]} ")
            if hydro == 1:
                ss2.write("HYDRO ")
            if sed == 1:
                ss2.write("SEDIMENT ")
            if sal == 1:
                ss2.write("SALINITY ")
            if waves == 1:
                ss2.write("WAVE ")
            ss2.write('\n')

        if len(save_pts) > 10:  # If >10, go ahead and write the cards and close the extra file.
            out = open(self.save_pts_file, 'w', newline='')  # Remove carriage return for test compatibility.
            ss2.seek(0)
            shutil.copyfileobj(ss2, out)
            out.close()
        else:  # Otherwise remove the extra file if it exists from a previous run.
            if os.path.exists(self.save_pts_file):
                os.remove(self.save_pts_file)
        self.ss.write('\n')

    def _gather_block_cards(self):
        """Gets the advanced cards for a block.

        Returns:
            A list of strings where each string is a full line in the cmcards file.
        """
        cards_to_write = []
        blocks = self._sim_data.advanced_card_table.Block.data.tolist()
        cards = self._sim_data.advanced_card_table.Card.data.tolist()
        values = self._sim_data.advanced_card_table.Value.data.tolist()
        for idx in range(len(blocks)):
            cards_to_write.append(f"{cards[idx]:<35} {values[idx]}\n")  # Properly shift values over to the right.
        return cards_to_write

    def _write_advanced_cards(self):
        """Writes the advanced cards.

        Returns:
            True if any cards were written.
        """
        cards_to_write = self._gather_block_cards()
        if cards_to_write:
            self.ss.write('!Advanced\n')
        for card in cards_to_write:
            self.ss.write(card)
        if cards_to_write:
            self.ss.write('\n')

    def _write_output_times_lists(self):
        """Write the output times lists."""
        self.ss.write("!Output Times List\n")
        all_lists = [
            self._sim_data.list_1_table,
            self._sim_data.list_2_table,
            self._sim_data.list_3_table,
            self._sim_data.list_4_table,
        ]
        list_to_id = {'List 1': 1, 'List 2': 2, 'List 3': 3, 'List 4': 4}

        for list_idx, out_list in enumerate(all_lists):
            list_start = out_list.start_time.data.tolist()
            list_end = out_list.end_time.data.tolist()
            list_inc = out_list.increment.data.tolist()

            self.ss.write(f"TIME_LIST_{list_idx + 1}                         {len(list_start)}")
            for start, end, inc in zip(list_start, list_end, list_inc):
                self.ss.write(f" {start:g} {end:g} {inc:g}")
            self.ss.write("\n")

        morph_list = self._sim_data.output.attrs['USE_MORPHOLOGY']
        trans_list = self._sim_data.output.attrs['USE_TRANSPORT']
        wave_list = self._sim_data.output.attrs['USE_WAVE']
        wind_list = self._sim_data.output.attrs['USE_WIND']
        visc_list = self._sim_data.output.attrs['USE_EDDY_VISCOSITY']
        self.ss.write(
            f"WSE_OUT_TIMES_LIST                  {list_to_id[self._sim_data.output.attrs['WSE_LIST']]}\n"
            f"VEL_OUT_TIMES_LIST                  {list_to_id[self._sim_data.output.attrs['CURRENT_VELOCITY_LIST']]}\n"
        )
        if morph_list == 1:
            self.ss.write(
                f"MORPH_OUT_TIMES_LIST                "
                f"{list_to_id[self._sim_data.output.attrs['MORPHOLOGY_LIST']]}\n"
            )
        if trans_list == 1:
            self.ss.write(
                f"TRANS_OUT_TIMES_LIST                "
                f"{list_to_id[self._sim_data.output.attrs['TRANSPORT_LIST']]}\n"
            )
        if wave_list == 1:
            self.ss.write(
                f"WAVE_OUT_TIMES_LIST                 "
                f"{list_to_id[self._sim_data.output.attrs['WAVE_LIST']]}\n"
            )
        if wind_list == 1:
            self.ss.write(
                f"WIND_OUT_TIMES_LIST                 "
                f"{list_to_id[self._sim_data.output.attrs['WIND_LIST']]}\n"
            )
        if visc_list == 1:
            self.ss.write(
                f"VISC_OUT_TIMES_LIST                 "
                f"{list_to_id[self._sim_data.output.attrs['EDDY_VISCOSITY_LIST']]}\n"
            )
        self.ss.write('\n')

    def _write_output_parameters(self):
        """Write the output parameters cards.

        """
        self.ss.write(
            "!Output\n"
            f"VEL_MAG_OUTPUT                      "
            f"{self._get_on_off_string(self._sim_data.output.attrs['CURRENT_MAGNITUDE'])}\n"
        )
        morph_out = self._get_on_off_string(self._sim_data.output.attrs['MORPHOLOGY_CHANGE'])
        conc_out = self._get_on_off_string(self._sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CONCENTRATION'])
        capac_out = self._get_on_off_string(self._sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CAPACITY'])
        frac_susp_out = self._get_on_off_string(self._sim_data.output.attrs['FRACTION_SUSPENDED'])
        frac_bed_out = self._get_on_off_string(self._sim_data.output.attrs['FRACTION_BEDLOAD'])
        wave_diss_out = self._get_on_off_string(self._sim_data.output.attrs['WAVE_DISSIPATION'])
        wind_mag_out = self._get_on_off_string(self._sim_data.output.attrs['WIND_SPEED'])
        atm_pressure_out = self._get_on_off_string(self._sim_data.output.attrs['ATM_PRESSURE'])
        if morph_out == 'ON':
            self.ss.write(f"MORPH_OUTPUT                        {morph_out}\n")
        if conc_out == 'ON':
            self.ss.write(f"CONC_OUTPUT                         {conc_out}\n")
        if capac_out == 'ON':
            self.ss.write(f"CAPAC_OUTPUT                        {capac_out}\n")
        if frac_susp_out == 'ON':
            self.ss.write(f"FRAC_SUSP_OUTPUT                    {frac_susp_out}\n")
        if frac_bed_out == 'ON':
            self.ss.write(f"BED_FRAC_OUTPUT                     {frac_bed_out}\n")
        if wave_diss_out == 'ON':
            self.ss.write(f"WAVE_DISS_OUTPUT                    {wave_diss_out}\n")
        if wind_mag_out == 'ON':
            self.ss.write(f"WIND_MAG_OUTPUT                     {wind_mag_out}\n")
        if atm_pressure_out == 'ON':
            self.ss.write(f"ATM_PRESS_OUTPUT                    {atm_pressure_out}\n")

        # Default values if from an earlier case which didn't originally save wave statistics.
        if 'ENABLE_WAVE_STATISTICS' not in self._sim_data.output.attrs:
            self._sim_data.output.attrs['ENABLE_WAVE_STATISTICS'] = 0
            self._sim_data.output.attrs['WAVE_START_TIME'] = 0.0
            self._sim_data.output.attrs['WAVE_END_TIME'] = 720.0
            self._sim_data.output.attrs['WAVE_INCREMENT'] = 1.0

        stats = self._sim_data.output.attrs['ENABLE_STATISTICS'] != 0
        if stats:
            hydro_table = self._sim_data.output.attrs['ENABLE_HYDRO_STATISTICS']
            sed_table = self._sim_data.output.attrs['ENABLE_SEDIMENT_STATISTICS']
            salinity_table = self._sim_data.output.attrs['ENABLE_SALINITY_STATISTICS']
            wave_table = self._sim_data.output.attrs['ENABLE_WAVE_STATISTICS']

            if hydro_table == 1:
                hydro_start = self._sim_data.output.attrs['HYDRO_START_TIME']
                hydro_end = self._sim_data.output.attrs['HYDRO_END_TIME']
                hydro_inc = self._sim_data.output.attrs['HYDRO_INCREMENT']

                self.ss.write(
                    f"FLOW_STATISTICS                     "
                    f"{float(hydro_start):g} {float(hydro_end):g} {float(hydro_inc):g}\n"
                )

            if sed_table == 1:
                sed_start = self._sim_data.output.attrs['SEDIMENT_START_TIME']
                sed_end = self._sim_data.output.attrs['SEDIMENT_END_TIME']
                sed_inc = self._sim_data.output.attrs['SEDIMENT_INCREMENT']

                self.ss.write(
                    f"SEDIMENT_STATISTICS                 "
                    f"{float(sed_start):g} {float(sed_end):g} {float(sed_inc):g}\n"
                )

            if salinity_table == 1:
                sal_start = self._sim_data.output.attrs['SALINITY_START_TIME']
                sal_end = self._sim_data.output.attrs['SALINITY_END_TIME']
                sal_inc = self._sim_data.output.attrs['SALINITY_INCREMENT']

                self.ss.write(
                    f"SALINITY_STATISTICS                 "
                    f"{float(sal_start):g} {float(sal_end):g} {float(sal_inc):g}\n"
                )

            if wave_table == 1:
                wav_start = self._sim_data.output.attrs['WAVE_START_TIME']
                wav_end = self._sim_data.output.attrs['WAVE_END_TIME']
                wav_inc = self._sim_data.output.attrs['WAVE_INCREMENT']

                self.ss.write(
                    f"WAVE_STATISTICS                     "
                    f"{float(wav_start):g} {float(wav_end):g} {float(wav_inc):g}\n"
                )
        self.ss.write('\n')

    def _write_file_parameters(self):
        """Write the file parameters cards.

        """
        output_type = {'XMDF binary output': 'XMDF', 'ASCII output only': 'ASCII'}
        self.ss.write("!Files\n")
        single_sol = self._sim_data.output.attrs['SINGLE_SOLUTION']
        if single_sol == 1:
            self.ss.write("USE_COMMON_SOLUTION_FILE            ON\n")

        write_ascii = self._sim_data.output.attrs['WRITE_ASCII']
        if write_ascii == 1:
            self.ss.write("WRITE_ASCII_INPUT_FILES             ON\n")

        self.ss.write(
            f"OUTPUT_FILE_TYPE                    "
            f"{output_type[self._sim_data.output.attrs['SOLUTION_OUTPUT']]}\n"
        )
        tec_plot = self._sim_data.output.attrs['TECPLOT']
        if tec_plot == 1:
            self.ss.write("GLOBAL_TECPLOT_FILES                ON\n")

        self.ss.write(f"XMDF_COMPRESSION                    "
                      f"{self._sim_data.output.attrs['XMDF_COMPRESSION']}\n")
        self.ss.write('\n')

    def _write_implicit_explicit_parameters(self):
        """Write the implicit/explicit parameters cards.

        """
        self.ss.write("!Implicit/Explicit\n")
        sol_scheme = self._sim_data.general.attrs['SOLUTION_SCHEME'].upper()
        self.ss.write(f"SOLUTION_SCHEME                     {sol_scheme}\n")

        if sol_scheme == "IMPLICIT":
            self.ss.write(
                f"MATRIX_SOLVER                       {self._sim_data.general.attrs['MATRIX_SOLVER'].upper()}\n"
            )

        self.ss.write(
            f"SKEWNESS_CORRECTION                 "
            f"{self._get_on_off_string(self._sim_data.general.attrs['SKEW_CORRECT'])}\n"
            '\n'
        )

    def _write_wind_and_wave_parameters(self):
        """Write the wind and wave parameters cards.

        """
        wind_type = WIND_TYPES[self._sim_data.wind.attrs['WIND_TYPE']]
        self.ss.write("!Wind/Wave\n")

        # get the meteorological stations table.
        met_name = self._sim_data.meteorological_stations_table.name.data.tolist()
        met_x = self._sim_data.meteorological_stations_table.x.data.tolist()
        met_y = self._sim_data.meteorological_stations_table.y.data.tolist()
        met_height = self._sim_data.meteorological_stations_table.height.data.tolist()

        idx = 0
        for name, x, y, height in zip(met_name, met_x, met_y, met_height):
            idx += 1
            self.ss.write(
                "MET_STATION_BEGIN\n"
                f"  STATION_NAME                        {name}\n"
                f"  COORDINATES                         {x:g} {y:g}\n"
                f"  WIND_INPUT_CURVE                    \"{self.proj_name}_mp.h5\" "
                f"\"PROPERTIES/Model Params/Met Stations/Sta{str(idx)}\"\n"
                f"  ANEMOMETER_HEIGHT                   {height:g}\n"
            )
            self.ss.write("MET_STATION_END\n")

        file_wind = wind_type == 'File'
        if wind_type == 'Constant':
            self.ss.write(
                f'WIND_INPUT_CURVE                    "{self.proj_name}_mp.h5" "PROPERTIES/Model Params/WindCurve"\n'
                f"ANEMOMETER_HEIGHT                   "
                f"{self._get_number_string(self._sim_data.wind.attrs['ANEMOMETER'])}\n"
            )
        wind_file_type = WIND_FILE_TYPES[self._sim_data.wind.attrs['WIND_FILE_TYPE']]
        fleet_or_ascii = False
        if file_wind and wind_file_type == 'OWI':
            self.ss.write(
                f"OCEANWEATHER_WIND_FILE              "
                f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['OCEAN_WIND_FILE'])}\"\n"
                f"OCEANWEATHER_PRES_FILE              "
                f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['OCEAN_PRESSURE_FILE'])}\"\n"
                f"OCEANWEATHER_XY_FILE                "
                f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['OCEAN_XY_FILE'])}\"\n"
            )
            fleet_or_ascii = False
        elif file_wind and wind_file_type == 'ASCII':
            self.ss.write(
                f"WIND_PRESSURE_SINGLE_FILE           "
                f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['WIND_FILE'])}\"\n"
            )
            fleet_or_ascii = True
        elif file_wind and wind_file_type == 'Fleet':
            self.ss.write(
                f"WIND_FLEET_FILE                     "
                f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['WIND_FILE'])}\"\n"
            )
            fleet_or_ascii = True
        wind_grid_type = WIND_GRID_TYPES[self._sim_data.wind.attrs['WIND_GRID_TYPE']]
        if fleet_or_ascii:
            if wind_grid_type == 'XYFile':
                self.ss.write(
                    f"WIND_PRESSURE_XY_FILE               "
                    f"\"{self.get_file_string_or_report_error(self._sim_data.wind.attrs['WIND_GRID_FILE'])}\"\n"
                )
            elif wind_grid_type == 'Params':
                self.ss.write(
                    f"WIND_PRESSURE_GRID_PARAM            {self._sim_data.wind.attrs['WIND_GRID_NUM_Y_VALUES']} "
                    f"{self._sim_data.wind.attrs['WIND_GRID_NUM_X_VALUES']} "
                    f"{self._get_number_string(self._sim_data.wind.attrs['WIND_GRID_MAX_Y_LOCATION'])} "
                    f"{self._get_number_string(self._sim_data.wind.attrs['WIND_GRID_MIN_X_LOCATION'])} "
                    f"{self._get_number_string(self._sim_data.wind.attrs['WIND_GRID_Y_DISTANCE'])} "
                    f"{self._get_number_string(self._sim_data.wind.attrs['WIND_GRID_X_DISTANCE'])}\n"
                )
            self.ss.write(
                f"WIND_PRESSURE_TIME_INCREMENT        "
                f"{self._sim_data.wind.attrs['WIND_GRID_TIME_INCREMENT']}\n"
            )

        wave_info = WAVE_INFO_TYPES[self._sim_data.wave.attrs['WAVE_INFO']]
        if wave_info == 'Single':
            dset_height = self._sim_data.wave.attrs['WAVE_HEIGHT']
            dset_period = self._sim_data.wave.attrs['PEAK_PERIOD']
            dset_dir = self._sim_data.wave.attrs['MEAN_WAVE_DIR']
            dset_break = self._sim_data.wave.attrs['WAVE_BREAKING']
            dset_rad = self._sim_data.wave.attrs['WAVE_RADIATION']
            self._write_dataset_line(dset_rad, "WAVE_RSTRESS_DATASET")
            self._write_dataset_line(dset_height, "WAVE_HEIGHT_DATASET")
            self._write_dataset_line(dset_period, "WAVE_PERIOD_DATASET")
            self._write_dataset_line(dset_dir, "WAVE_DIRECTION_DATASET")
            self._write_dataset_line(dset_break, "WAVE_DISS_DATASET")
            # self.write_dataset_line("dsetSurfaceRoller", "ROLLER_STRESS_DATASET")  # experimental

        is_inline = False
        if wave_info == 'Inline':
            is_inline = True
            sim_file = self.get_file_string_or_report_error(self._sim_data.wave.attrs['FILE_WAVE_SIM'])
            if self._copy_cms_wave_files(sim_file):
                self.ss.write(f"CMS-WAVE_SIM_FILE                   \"{os.path.basename(sim_file)}\"\n")
            elif XmEnv.xms_environ_running_tests() != 'TRUE':
                XmEnv.report_error('Unknown error copying CMS-Wave input files.')

        if is_inline:
            self.ss.write(
                f"CMS-STEERING_INTERVAL               "
                f"{self._get_number_string(self._sim_data.wave.attrs['STEERING_INTERVAL_CONST'])}\n"
            )

            extrap_distance = self._sim_data.wave.attrs['EXTRAPOLATION_DISTANCE']
            flow_to_wave = FLOW_WAVE_TYPES[self._sim_data.wave.attrs['FLOW_TO_WAVE']]
            if extrap_distance == 1:
                if flow_to_wave == 'Automatic':
                    self.ss.write('FLOW_EXTRAPOLATION_DISTANCE         -999.0 m\n')
                elif flow_to_wave == 'Specified':
                    self.ss.write(
                        f"FLOW_EXTRAPOLATION_DISTANCE         "
                        f"{self._sim_data.wave.attrs['FLOW_TO_WAVE_USER_VALUE']:g} "
                        f"{self._sim_data.wave.attrs['FLOW_TO_WAVE_USER_UNITS']}\n"
                    )
                if flow_to_wave == 'Automatic':
                    self.ss.write('WAVE_EXTRAPOLATION_DISTANCE         -999.0 m\n')
                elif flow_to_wave == 'Specified':
                    self.ss.write(
                        f"WAVE_EXTRAPOLATION_DISTANCE         "
                        f"{self._sim_data.wave.attrs['WAVE_TO_FLOW_USER_VALUE']:g} "
                        f"{self._sim_data.wave.attrs['WAVE_TO_FLOW_USER_UNITS']}\n"
                    )
        if is_inline:
            self.ss.write(
                f"WAVE_WATER_LEVEL                    "
                f"{PREDICTOR_TYPES[self._sim_data.wave.attrs['WAVE_WATER_PREDICTOR']]}\n"
            )
        self.ss.write('\n')

    def _write_bottom_friction(self):
        """Write the bottom friction parameters cards.

        """
        roughness_types = {
            'Mannings N': 'Mannings',
            'Bottom friction coefficient': 'FrictionCoef',
            'Roughness height (m)': 'RoughnessHt'
        }
        self.ss.write('!Bottom Friction\n')
        rough_type = roughness_types[self._sim_data.flow.attrs['BOTTOM_ROUGHNESS']]
        dset = self._sim_data.flow.attrs['BOTTOM_ROUGHNESS_DSET']
        constant = self._sim_data.flow.attrs['ROUGHNESS_CONSTANT']
        source = self._sim_data.flow.attrs['ROUGHNESS_SOURCE']

        if source == 'Dataset':
            if rough_type == 'Mannings':
                self._write_dataset_line(dset, "MANNINGS_N_DATASET")
            elif rough_type == 'FrictionCoef':
                self._write_dataset_line(dset, "BOTTOM_FRICTION_COEF_DATASET")
            elif rough_type == 'RoughnessHt':
                self._write_dataset_line(dset, "ROUGHNESS_HEIGHT_DATASET")
        else:
            if rough_type == 'Mannings':
                self._write_constant_line(constant, "MANNINGS_N_CONSTANT")
            elif rough_type == 'FrictionCoef':
                self._write_constant_line(constant, "BOTTOM_FRICTION_COEF_CONSTANT")
            elif rough_type == 'RoughnessHt':
                self._write_constant_line(constant, "ROUGHNESS_HEIGHT_CONSTANT")

        self.ss.write('\n')

    def _write_sediment_transport(self):
        """Write the sediment transport parameters cards.

        """
        sediment_enabled = self._get_on_off_string(self._sim_data.sediment.attrs['CALCULATE_SEDIMENT'])
        self.ss.write('!Sediment Transport\n'
                      f"CALC_SEDIMENT_TRANSPORT             {sediment_enabled}\n")
        if sediment_enabled == 'ON':
            formulation_units = {
                'Equilibrium total load': 'EXNER',
                'Equilibrium bed load plus nonequilibrium susp. load': 'A-D',
                'Nonequilibrium total load': 'NET'
            }
            transport_formulas = {
                'Lund-CIRP': 'LUND-CIRP',
                'van Rijn': 'VAN_RIJN',
                'Soulsby-van Rijn': 'SOULSBY-VAN_RIJN',
                'Watanabe': 'WATANABE',
                'C2SHORE': 'C2SHORE'
            }
            concentration_profiles = {
                'Exponential': 'EXPONENTIAL',
                'Rouse': 'ROUSE',
                'Lund-CIRP': 'LUND-CIRP',
                'van Rijn': 'VAN_RIJN'
            }
            adaption_methods = {
                'Constant length': 'CONSTANT_LENGTH',
                'Constant time': 'CONSTANT_TIME',
                'Maximum of bed and suspended adaptation lengths': 'MAX_BED_SUSP_LENGTH',
                'Weighted average of bed and suspended adaptation lengths': 'WGHT_AVG_BED_SUSP_LENGTH'
            }
            bed_adaption_methods = {
                'Constant length': 'CONSTANT_LENGTH',
                'Constant time': 'CONSTANT_TIME',
                'Depth dependent': 'DEPTH_DEPENDENT'
            }
            susp_adaption_methods = {
                'Constant length': 'CONSTANT_LENGTH',
                'Constant time': 'CONSTANT_TIME',
                'Constant coefficient': 'CONSTANT_ALPHA',
                'Armanini and Di Silvio': 'ARMANNI_DISILVIO',
                'Lin': 'LIN',
                'Gallappatti': 'GALLAPPATTI'
            }
            self.ss.write(
                f"SED_TRAN_CALC_INTERVAL              "
                f"{self._sim_data.sediment.attrs['TRANSPORT_TIME_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.sediment.attrs['TRANSPORT_TIME_UNITS']]}\n"
                f"MORPH_UPDATE_INTERVAL               "
                f"{self._sim_data.sediment.attrs['MORPHOLOGIC_TIME_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.sediment.attrs['MORPHOLOGIC_TIME_UNITS']]}\n"
                f"MORPH_START_TIME                    "
                f"{self._sim_data.sediment.attrs['MORPHOLOGY_TIME_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.sediment.attrs['MORPHOLOGY_TIME_UNITS']]}\n"
                f"SED_TRANS_FORMULATION               "
                f"{formulation_units[self._sim_data.sediment.attrs['FORMULATION_UNITS']]}\n"
                f"TRANSPORT_FORMULA                   "
                f"{transport_formulas[self._sim_data.sediment.attrs['TRANSPORT_FORMULA']]}\n"
                f"CONCENTRATION_PROFILE               "
                f"{concentration_profiles[self._sim_data.sediment.attrs['CONCENTRATION_PROFILE']]}\n"
            )
            if transport_formulas[self._sim_data.sediment.attrs['TRANSPORT_FORMULA']] == 'Watanabe':
                self.ss.write(
                    f"A_COEFFICIENT_WATANABE              "
                    f"{self._get_number_string(self._sim_data.sediment.attrs['WATANABE_RATE'])}\n"
                )
            elif transport_formulas[self._sim_data.sediment.attrs['TRANSPORT_FORMULA']] == 'C2SHORE':
                self.ss.write(
                    f"C2SHORE_EFFICIENCY                   "
                    f"{self._get_number_string(self._sim_data.sediment.attrs['C2SHORE_EFFICIENCY'])}\n"
                    f"C2SHORE_BED_LOAD                     "
                    f"{self._get_number_string(self._sim_data.sediment.attrs['C2SHORE_BED_LOAD'])}\n"
                    f"C2SHORE_SUSP_LOAD                    "
                    f"{self._get_number_string(self._sim_data.sediment.attrs['C2SHORE_SUSP_LOAD'])}\n"
                )
            self.ss.write(
                f"SEDIMENT_DENSITY                    "
                f"{self._sim_data.sediment.attrs['SEDIMENT_DENSITY_VALUE']:g} "
                f"'{self._sim_data.sediment.attrs['SEDIMENT_DENSITY_UNITS']}'\n"
                f"SEDIMENT_POROSITY                   "
                f"{self._get_number_string(self._sim_data.sediment.attrs['SEDIMENT_POROSITY'])}\n"
                f"BED_LOAD_SCALE_FACTOR               "
                f"{self._get_number_string(self._sim_data.sediment.attrs['BED_LOAD_SCALING'])}\n"
                f"SUSP_LOAD_SCALE_FACTOR              "
                f"{self._get_number_string(self._sim_data.sediment.attrs['SUSPENDED_LOAD_SCALING'])}\n"
                f"MORPH_ACCEL_FACTOR                  "
                f"{self._get_number_string(self._sim_data.sediment.attrs['MORPHOLOGIC_ACCELERATION'])}\n"
                f"BEDSLOPE_COEFFICIENT                "
                f"{self._get_number_string(self._sim_data.sediment.attrs['BED_SLOPE_DIFFUSION'])}\n"
                f"HIDING_EXPOSURE_COEFFICIENT         "
                f"{self._get_number_string(self._sim_data.sediment.attrs['HIDING_AND_EXPOSURE'])}\n"
                f"ADAPTATION_METHOD_TOTAL             "
                f"{adaption_methods[self._sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD']]}\n"
            )
            if self._sim_data.sediment.attrs['FORMULATION_UNITS'] != 'Nonequilibrium total load':
                self.ss.write(
                    f"CONSTANT_GRAIN_SIZE                 "
                    f"{self._sim_data.sediment.attrs['GRAIN_SIZE_VALUE']:g}"
                    f" '{self._sim_data.sediment.attrs['GRAIN_SIZE_UNITS']}'\n"
                )
            if self._sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD'] == 'Constant length':
                self.ss.write(
                    f"ADAPTATION_LENGTH_TOTAL             "
                    f"{self._sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_VALUE']:g} "
                    f"'{self._sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_UNITS']}'\n"
                )
            elif self._sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD'] == 'Constant time':
                self.ss.write(
                    f"ADAPTATION_TIME_TOTAL               "
                    f"{self._sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_VALUE']:g} "
                    f"{self._time_to_export[self._sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_UNITS']]}\n"
                )
            else:
                self.ss.write(
                    f"ADAPTATION_METHOD_BED               "
                    f"{bed_adaption_methods[self._sim_data.sediment.attrs['BED_ADAPTATION_METHOD']]}\n"
                )
                if self._sim_data.sediment.attrs['BED_ADAPTATION_METHOD'] == 'Constant length':
                    self.ss.write(
                        f"ADAPTATION_LENGTH_BED               "
                        f"{self._sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_VALUE']:g} "
                        f"'{self._sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_UNITS']}'\n"
                    )
                elif self._sim_data.sediment.attrs['BED_ADAPTATION_METHOD'] == 'Constant time':
                    self.ss.write(
                        f"ADAPTATION_TIME_BED                 "
                        f"{self._sim_data.sediment.attrs['BED_ADAPTATION_TIME_VALUE']:g} "
                        f"{self._time_to_export[self._sim_data.sediment.attrs['BED_ADAPTATION_TIME_UNITS']]}\n"
                    )
                else:
                    self.ss.write(
                        f"ADAPTATION_DEPTH_FACTOR_BED         "
                        f"{self._get_number_string(self._sim_data.sediment.attrs['BED_ADAPTATION_DEPTH'])}\n"
                    )
                self.ss.write(
                    f"ADAPTATION_METHOD_SUSPENDED         "
                    f"{susp_adaption_methods[self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD']]}\n"
                )
                if self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] == 'Constant length':
                    self.ss.write(
                        f"ADAPTATION_LENGTH_SUSPENDED         "
                        f"{self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_VALUE']:g} "
                        f"'{self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_UNITS']}'\n"
                    )
                elif self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] == 'Constant time':
                    self.ss.write(
                        f"ADAPTATION_TIME_SUSPENDED           "
                        f"{self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_VALUE']:g} "
                        f"{self._time_to_export[self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_UNITS']]}\n"
                    )
                elif self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] == 'Constant coefficient':
                    coefficient = self._sim_data.sediment.attrs['SUSPENDED_ADAPTATION_COEFFICIENT']
                    self.ss.write(f"ADAPTATION_COEFFICIENT_SUSPENDED    "
                                  f"{self._get_number_string(coefficient)}\n")

            diameter = self._sim_data.advanced_sediment_diameters_table.diameter_value.data.tolist()
            diameter_units = self._sim_data.advanced_sediment_diameters_table.diameter_units.data.tolist()
            fall_velocity_formula = self._sim_data.advanced_sediment_diameters_table.fall_velocity_method.data.tolist()
            fall_velocity = self._sim_data.advanced_sediment_diameters_table.fall_velocity_value.data.tolist()
            fall_velocity_units = self._sim_data.advanced_sediment_diameters_table.fall_velocity_units.data.tolist()
            corey_shape = self._sim_data.advanced_sediment_diameters_table.corey_shape_factor.data.tolist()
            shear_formula = self._sim_data.advanced_sediment_diameters_table.critical_shear_method.data.tolist()
            critical_shear = self._sim_data.advanced_sediment_diameters_table.critical_shear_stress.data.tolist()
            advanced = self._sim_data.sediment.attrs['USE_ADVANCED_SIZE_CLASSES'] == 1

            fall_velocity_formulas = {
                'Soulsby (1997)': 'SOULSBY',
                'Wu and Wang (2006)': 'WU_WANG',
                'User specified': 'MANUAL'
            }
            critical_shear_formulas = {
                'Soulsby (1997)': 'SOULSBY',
                'van Rijn (2007)': 'VAN_RIJN',
                'Wu and Wang (1999)': 'WU_WANG',
                'User specified': 'MANUAL'
            }

            items = zip(
                diameter, diameter_units, fall_velocity_formula, fall_velocity, fall_velocity_units, corey_shape,
                shear_formula, critical_shear
            )
            for dia, dia_units, fvf, fv, fv_units, cs, sf, shear in items:
                self.ss.write(
                    f"SEDIMENT_SIZE_CLASS_BEGIN\n"
                    f"  DIAMETER                          {dia:g} '{dia_units}'\n"
                )
                if advanced:
                    fvf_export = fall_velocity_formulas[fvf]
                    self.ss.write(f"  FALL_VELOCITY_FORMULA             {fvf_export}\n")
                    if fvf == 'User specified':
                        self.ss.write(f"  FALL_VELOCITY                     {fv:g} '{fv_units}'\n")
                    if fvf == 'Wu and Wang (2006)':
                        self.ss.write(f"  COREY_SHAPE_FACTOR                {cs:g}\n")
                    sf_export = critical_shear_formulas[sf]
                    self.ss.write(f"  CRITICAL_SHEAR_FORMULA            {sf_export}\n")
                    if sf == 'User specified':
                        self.ss.write(f"  CRITICAL_SHEAR                    {shear:g}\n")
                self.ss.write("SEDIMENT_SIZE_CLASS_END\n")

            simple_multi_grain_size = self._sim_data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] == 1
            if simple_multi_grain_size:
                self.ss.write("BED_LAYER_BEGIN\n")
                if self._sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D50 Sigma':
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D50'], 'D50_DATASET', '  ')
                elif self._sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D16 D50 D84':
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D16'], 'D16_DATASET', '  ')
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D50'], 'D50_DATASET', '  ')
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D84'], 'D84_DATASET', '  ')
                else:
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D35'], 'D35_DATASET', '  ')
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D50'], 'D50_DATASET', '  ')
                    self._write_dataset_line(self._sim_data.sediment.attrs['MULTI_D90'], 'D90_DATASET', '  ')
                self.ss.write("BED_LAYER_END\n")
                if self._sim_data.sediment.attrs['MULTIPLE_GRAIN_SIZES'] == 'Specify number of size classes only':
                    self.ss.write(
                        f"SEDIMENT_SIZE_CLASS_NUMBER          "
                        f"{self._sim_data.sediment.attrs['SIMPLE_MULTI_SIZE']}\n"
                    )
                else:
                    grain_sizes = self._sim_data.simple_grain_sizes_table.grain_size.data.tolist()
                    self.ss.write(
                        f"MULTIPLE_GRAIN_SIZES                "
                        f"{len(grain_sizes)} {' '.join(format(size, 'g') for size in grain_sizes)}\n"
                    )
                # Change to only write out the SEDIMENT_STANDARD_DEVIATION when choosing the "D50 Sigma" option.
                if self._sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D50 Sigma':
                    self.ss.write(
                        f"SEDIMENT_STANDARD_DEVIATION         "
                        f"{self._sim_data.sediment.attrs['SEDIMENT_STANDARD_DEVIATION']:g}\n"
                    )
                self.ss.write(
                    f"BED_COMPOSITION_INPUT               "
                    f"{self._sim_data.sediment.attrs['BED_COMPOSITION_INPUT'].upper().replace(' ', '_')}\n"
                    f"MIXING_LAYER_CONSTANT_THICKNESS     "
                    f"{self._sim_data.sediment.attrs['THICKNESS_FOR_MIXING']:g}\n"
                    f"BED_LAYER_CONSTANT_THICKNESS        "
                    f"{self._sim_data.sediment.attrs['THICKNESS_FOR_BED']:g}\n"
                    f"NUMBER_BED_LAYERS                   "
                    f"{self._sim_data.sediment.attrs['NUMBER_BED_LAYERS']}\n"
                )

            else:
                self.ss.write(
                    f"BED_LAYERS_MAX_NUMBER               {self._sim_data.sediment.attrs['MAX_NUMBER_BED_LAYERS']}\n"
                    f"BED_LAYERS_MIN_THICKNESS            "
                    f"{self._sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_VALUE']} "
                    f"'{self._sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_UNITS']}'\n"
                    f"BED_LAYERS_MAX_THICKNESS            "
                    f"{self._sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_VALUE']} "
                    f"'{self._sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_UNITS']}'\n"
                )

                layer_id = self._sim_data.bed_layer_table.layer_id.data.tolist()
                thickness_type = self._sim_data.bed_layer_table.layer_thickness_type.data.tolist()
                thickness = self._sim_data.bed_layer_table.layer_thickness.data.tolist()
                thickness_const = self._sim_data.bed_layer_table.layer_thickness_const.data.tolist()
                c05 = self._sim_data.bed_layer_table.d05.data.tolist()
                c10 = self._sim_data.bed_layer_table.d10.data.tolist()
                c16 = self._sim_data.bed_layer_table.d16.data.tolist()
                c20 = self._sim_data.bed_layer_table.d20.data.tolist()
                c30 = self._sim_data.bed_layer_table.d30.data.tolist()
                c35 = self._sim_data.bed_layer_table.d35.data.tolist()
                c50 = self._sim_data.bed_layer_table.d50.data.tolist()
                c65 = self._sim_data.bed_layer_table.d65.data.tolist()
                c84 = self._sim_data.bed_layer_table.d84.data.tolist()
                c90 = self._sim_data.bed_layer_table.d90.data.tolist()
                c95 = self._sim_data.bed_layer_table.d95.data.tolist()

                # Normally, layer_id would already be integers, but some old tests have strings.
                try:
                    layer_id = [int(l_id) for l_id in layer_id]
                except ValueError:
                    raise Exception('Error converting string layer ids to int.')
                mix_layer_idx = layer_id.index(1)
                self.ss.write(f"MIXING_LAYER_FORMULATION            "
                              f"{thickness_type[mix_layer_idx].upper()}\n")
                if thickness_type[mix_layer_idx] == 'Constant':
                    self.ss.write(f"MIXING_LAYER_THICKNESS_CONSTANT     "
                                  f"{thickness_const[mix_layer_idx]} 'm'\n")

                write_thickness = len(layer_id) > 1

                dsets = zip(
                    layer_id, thickness_type, thickness, thickness_const, c05, c10, c16, c20, c30, c35, c50, c65, c84,
                    c90, c95
                )
                for lyr, thick_type, thick, thick_const, d05, d10, d16, d20, d30, d35, d50, d65, d84, d90, d95 in dsets:
                    self.ss.write(f"BED_LAYER_BEGIN\n"
                                  f"  LAYER                             {lyr}\n")
                    if write_thickness:
                        if thick_type == 'Constant':
                            self.ss.write(f"  THICKNESS_CONSTANT                {thick_const}\n")
                        elif thick_type == 'Dataset':
                            self._write_dataset_line(thick, "THICKNESS_DATASET", "  ")
                    self._write_dataset_line(d05, "D05_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d10, "D10_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d16, "D16_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d20, "D20_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d30, "D30_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d35, "D35_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d50, "D50_DATASET", "  ")
                    self._write_dataset_line(d65, "D65_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d84, "D84_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d90, "D90_DATASET", "  ", write_blank=False)
                    self._write_dataset_line(d95, "D95_DATASET", "  ", write_blank=False)
                    self.ss.write("BED_LAYER_END\n")

            self.ss.write(
                f"USE_AVALANCHING                     "
                f"{self._get_on_off_string(self._sim_data.sediment.attrs['CALCULATE_AVALANCHING'])}\n"
                f"AVALANCHE_CRITICAL_BEDSLOPE         "
                f"{self._get_number_string(self._sim_data.sediment.attrs['CRITICAL_BED_SLOPE'])}\n"
                f"AVALANCHE_MAX_ITERATIONS            {self._sim_data.sediment.attrs['MAX_NUMBER_ITERATIONS']}\n"
            )
            if self._sim_data.sediment.attrs['USE_HARD_BOTTOM'] == 1:
                self._write_dataset_line(self._sim_data.sediment.attrs['HARD_BOTTOM'], 'HARDBOTTOM_DATASET')
        self.ss.write('\n')

    def _write_projection(self):
        """Write the projection cards."""
        # Mapping of XMS projection names to CMS-Flow coordinate system cards
        xms_to_cms_systems = {
            'Geographic (Latitude/Longitude)': 'GEOGRAPHIC',
            'State Plane Coordinate System': 'STATE_PLANE',
            'STATEPLANE': 'STATE_PLANE',  # API keyword for State Plane
            'UTM': 'UTM',
        }
        xms_to_cms_units = {
            'ARC_DEGREES': 'DEGREES',
            'FEET (INTERNATIONAL)': 'INTL_FEET',
            'FEET (U.S. SURVEY)': 'FEET',
            'METERS': 'METERS',
            # TODO: Support Arc seconds and radians?
        }

        display_proj = self.projection

        # Use GDAL to parse the horizontal datum from the display projection's WKT.
        spatial_ref = wkt_to_sr(display_proj.well_known_text)
        horiz_datum = spatial_ref.GetAttrValue('DATUM')
        if horiz_datum == 'North_American_Datum_1983' or horiz_datum == 'D_NORTH_AMERICAN_1983':
            horiz_datum = 'NAD83'

        coord_system = display_proj.coordinate_system
        if coord_system in xms_to_cms_systems:
            coord_system = xms_to_cms_systems[coord_system]
        else:
            coord_system = coord_system.replace(" ", "_").upper()
        horiz_units = display_proj.horizontal_units
        if horiz_units in xms_to_cms_units:
            horiz_units = xms_to_cms_units[horiz_units]

        self.ss.write(
            '!Projection\n'
            'HORIZONTAL_PROJECTION_BEGIN\n'
            f"  DATUM                             {horiz_datum}\n"
            f"  COORDINATE_SYSTEM                 {coord_system}\n"
            f"  UNITS                             {horiz_units}\n"
            f"  ZONE                              {display_proj.coordinate_zone}\n"
        )
        self.ss.write('HORIZONTAL_PROJECTION_END\n')

        vert_units = display_proj.vertical_units
        if vert_units in xms_to_cms_units:
            vert_units = xms_to_cms_units[vert_units]
        # Set vertical datum to local if not defined.
        vert_datum = display_proj.vertical_datum if display_proj.vertical_datum else 'LOCAL'
        self.ss.write(
            'VERTICAL_PROJECTION_BEGIN\n'
            f"  DATUM                             {vert_datum}\n"
            f"  UNITS                             {vert_units}\n"
        )
        self.ss.write('VERTICAL_PROJECTION_END\n\n')

    def _write_xmdf_parent14(self, attributes: xr.Dataset):
        """
        Write a fort.14 file for parent ADCIRC BC boundaries.

        Args:
            attributes: Attributes for the arc.

        Returns:
            str: Filename of the BC's parent ADCIRC fort.14
        """
        geom_uuid = attributes['parent_adcirc_14_uuid'].item()
        filename = self._exported14.get(geom_uuid)
        if filename:
            return filename
        if not self.query:
            return ''
        do_ugrid = self.query.item_with_uuid(geom_uuid)
        return self._write_fort_14(do_ugrid, geom_uuid)

    def _write_fort_14(self, do_ugrid, geom_uuid):
        """Writes data to a fort.14 file.

        Args:
            do_ugrid (UGrid): The UGrid object containing the data to write.
            geom_uuid (str): The UUID of the geometry.

        Returns:
            str: The filename of the written fort.14 file.
        """
        # Reproject locations if not geographic
        if do_ugrid.projection.coordinate_system != 'GEOGRAPHIC':
            do_ugrid = self._transform_grid_to_geographic(do_ugrid)
        ss = StringIO()
        export_geometry_to_fort14(ss, do_ugrid.cogrid_file, do_ugrid.name)
        ss.write('0 = Number of open boundaries\n')
        ss.write('0 = Total number of open boundary nodes\n')
        ss.write('0 = Number of land boundaries\n')
        ss.write('0 = Total number of land boundary nodes\n')
        filename = io_util.make_filename_unique(os.path.join(os.getcwd(), f'{do_ugrid.name}.grd'))
        with open(filename, 'w') as f:
            ss.seek(0)
            shutil.copyfileobj(ss, f, 1000000)
        self._exported14[geom_uuid] = filename
        return filename

    def _transform_grid_to_geographic(self, do_ugrid):
        """Transform a grid into geographic space.

        Args:
            do_ugrid (DoUGrid): The data_objects UGrid to transform

        Returns:
            DoUGrid: The transformed data_objects UGrid
        """
        # Transform the points to geographic.
        self._logger.info('Transforming parent ADCIRC grid into geographic space.')
        cogrid = read_grid_from_file(do_ugrid.cogrid_file)
        ugrid = cogrid.ugrid
        _, geo_wkt = wkt_from_epsg(4326)
        from_wkt = do_ugrid.projection.well_known_text
        pts = ugrid.locations
        geo_pts = transform_points_from_wkt(pts, from_wkt, geo_wkt, False, True)
        # Write the new CoGrid file.
        xmugrid = XmUGrid(geo_pts, ugrid.cellstream)
        co_builder = UGridBuilder()
        co_builder.set_is_2d()
        co_builder.set_ugrid(xmugrid)
        cogrid = co_builder.build_grid()
        cogrid.write_to_file(do_ugrid.cogrid_file, True)
        # Create the new data_objects UGrid
        return DoUGrid(do_ugrid.cogrid_file, name=do_ugrid.name, uuid=str(uuid.uuid4()), projection=do_ugrid.projection)

    @staticmethod
    def _get_tidal_library_path():
        """
        Gets the path for the tidal libraries.

        Returns:
            str: Path of the tidal libraries
        """
        return os.path.join(os.getenv('APPDATA', os.path.dirname(os.path.dirname(__file__))), 'harmonica', 'data')

    def _write_boundary_conditions(self):
        """Write the boundary conditions cards."""
        self.ss.write('!Boundary Conditions\n')

        if not self._bc_comp:
            self.ss.write('\n')
            return

        arc_comp_ids = self._bc_comp.comp_to_xms.get(self._bc_comp.cov_uuid, {}).get(TargetType.arc, {})
        if not arc_comp_ids:
            self.ss.write('\n')
            return

        for comp_id in arc_comp_ids:
            attributes = self._bc_comp.data.arcs.loc[{'comp_id': comp_id}]
            arc_ids = arc_comp_ids[comp_id]

            for arc_id in arc_ids:
                self._write_boundary_condition(attributes, arc_id)

    def _write_boundary_condition(self, attributes: xr.Dataset, arc_id: int):
        """Write any blocks needed by a particular boundary condition."""
        elevation_forcing = 'WSE-forcing'
        velocity_forcing = 'Flow rate-forcing'

        bc_type = attributes['bc_type'].item()

        name = attributes['name'].item()

        self.ss.write(
            "BOUNDARY_BEGIN\n"
            f"  NAME \"{name}\"\n"
            f"  CELLSTRING \"{self.mp_file}\" \"PROPERTIES/Model Params/Boundary_#{arc_id}\"\n"
        )

        if bc_type == velocity_forcing:
            self._write_velocity_boundary_condition(attributes, arc_id)
        elif bc_type == elevation_forcing:
            self._write_elevation_boundary_condition(attributes, arc_id)
        self.ss.write("BOUNDARY_END\n\n")

    def _write_velocity_boundary_condition(self, attributes: xr.Dataset, arc_id: int):
        """Write the block for a velocity-forcing boundary condition."""
        flow_source = attributes['flow_source'].item()
        self.ss.write("  FLUX_BEGIN\n"
                      "    FLUX_UNITS      'm^3/s/boundary'\n")
        if flow_source == 'Constant':
            self.ss.write(f"    FLUX_VALUE {attributes['constant_flow'].item()}\n")
        elif flow_source == 'Curve':
            self.ss.write(
                f"    FLUX_CURVE \"{self.mp_file}\" \"PROPERTIES/Model Params/Boundary_#{arc_id}\"\n"
                # The following line looks wrong. It was in the 13.0 XML.
                f"    UNITS 'm^3/s'\n"
            )
        if attributes['specify_inflow_direction'].item() != 0:
            self.ss.write(f"    DIRECTION {attributes['flow_direction'].item()} 'deg'\n")
        self.ss.write(f"    CONVEYANCE {attributes['flow_conveyance'].item()}\n")
        self.ss.write("  FLUX_END\n")

    def _write_elevation_boundary_condition(self, attributes: xr.Dataset, arc_id: int):
        """Determine and write the appropriate block for a WSE-forcing boundary condition."""
        self._write_elevation_wse_block(attributes, arc_id)
        self._write_elevation_vel_block(attributes)

        wse_source = attributes['wse_source'].item()
        if wse_source == 'Parent CMS-Flow':
            self._write_elevation_cms(attributes)
        elif wse_source == 'Parent ADCIRC':
            self._write_elevation_adcirc(attributes)
        elif wse_source == 'Harmonic':
            self._write_elevation_harmonic(attributes)
        elif wse_source == 'Tidal constituent':
            self._write_elevation_constituent(attributes)
        elif wse_source == 'External tidal':
            self._write_external_tides(arc_id)

    def _write_elevation_wse_block(self, attributes: xr.Dataset, arc_id: int):
        """Write the WSE block for a WSE-forcing boundary condition."""
        wse_source = attributes['wse_source'].item()

        self.ss.write("  WSE_BEGIN\n")
        if wse_source == 'Constant':
            self.ss.write(f"    WSE_VALUE {attributes['wse_const'].item()}\n")
        elif wse_source == 'Curve':
            self.ss.write(f"    WSE_CURVE \"{self.mp_file}\" "
                          f"\"PROPERTIES/Model Params/Boundary_#{arc_id}\"\n")
        elif wse_source == 'Extracted':
            self.ss.write('    EXTRACTED_WSE\n')

        wse_offset_type = attributes['wse_offset_type'].item()
        if wse_offset_type == 'Constant':
            self.ss.write(f"    OFFSET_CONSTANT {attributes['wse_offset_const'].item()} m\n")
        elif wse_offset_type == 'Curve':
            self.ss.write(f"    OFFSET_CURVE \"{self.mp_file}\" "
                          f"\"PROPERTIES/Model Params/Boundary_#{arc_id}\"\n")
        self.ss.write("  WSE_END\n")

    def _write_elevation_vel_block(self, attributes: xr.Dataset):
        """Write the velocity block for elevation forcing."""
        wse_source = attributes['wse_source'].item()
        if wse_source != 'Extracted':
            return

        # Writing velocity information when we're doing elevation forcing seems weird, but it's what CMS expects.
        velocity_dataset = self._bc_comp.data.info.attrs.get('wse_forcing_velocity_source', '')
        if velocity_dataset:
            self.ss.write('  VEL_BEGIN\n'
                          '    EXTRACTED_VEL\n'
                          '  VEL_END\n')

    def _write_elevation_cms(self, attributes: xr.Dataset):
        """Write the parent-CMS block for a WSE-forcing boundary condition."""
        self.ss.write(
            "  PARENT_BEGIN\n"
            f'    CONTROL_FILE "{self.get_file_string_or_report_error(attributes["parent_cmsflow"].item())}"\n'
        )
        self.ss.write("  PARENT_END\n")

    def _write_elevation_adcirc(self, attributes: xr.Dataset):
        """Write the parent-ADCIRC block for a WSE-forcing boundary condition."""
        use_velocity = attributes['use_velocity'].item()
        if use_velocity != 0:
            self.ss.write("  VEL_BEGIN\n  VEL_END\n")
        is_ascii = attributes['parent_adcirc_solution_type'].item() == 'ASCII'
        if is_ascii:
            fort14_filename = attributes["parent_adcirc_14"].item()
        else:
            fort14_filename = self._write_xmdf_parent14(attributes)

        self.ss.write("  PARENT_BEGIN\n"
                      f'    GRID_FILE "{self.get_file_string_or_report_error(fort14_filename)}"\n')
        if is_ascii:
            self.ss.write(
                f'    WSE_FILE "{self.get_file_string_or_report_error(attributes["parent_adcirc_63"].item())}"\n'
            )
            if use_velocity:
                filename = self.get_file_string_or_report_error(attributes["parent_adcirc_64"].item())
                self.ss.write(f'    VEL_FILE "{filename}"\n')
        else:
            wse_dataset = self.query.item_with_uuid(attributes["parent_adcirc_solution_wse"].item())
            filename = wse_dataset.h5_filename if wse_dataset else ''
            group_path = wse_dataset.group_path if wse_dataset else ''
            self.ss.write(f'    WSE_FILE "{filename}" "{group_path}"\n')
            if use_velocity:
                vel_dataset = self.query.item_with_uuid(attributes["parent_adcirc_solution"].item())
                filename = vel_dataset.h5_filename if vel_dataset else ''
                group_path = vel_dataset.group_path if vel_dataset else ''
                self.ss.write(f'    VEL_FILE "{filename}" "{group_path}"\n')
        parent_start = attributes['parent_adcirc_start_time'].item()
        parent_start_py = format_datetime(parent_start)
        self.ss.write(
            f"    STARTING_DATE_TIME {parent_start_py.timetuple().tm_year:04}-"
            f"{parent_start_py.timetuple().tm_mon:02}"
            f"-{parent_start_py.timetuple().tm_mday:02} "
            f"{parent_start_py.timetuple().tm_hour:02}:{parent_start_py.timetuple().tm_min:02}:"
            f"{parent_start_py.timetuple().tm_sec:02}\n"
        )
        self.ss.write("  PARENT_END\n")

    def _write_elevation_harmonic(self, attributes: xr.Dataset):
        """Write the harmonic block for a WSE-forcing boundary condition."""
        self.ss.write("  HARMONIC_BEGIN\n")
        harmonic_id = attributes['harmonic_table'].item()
        table = self._bc_comp.data.harmonic_table_from_id(harmonic_id)
        for s, a, p in zip(table.speed, table.amplitude, table.phase):
            self.ss.write(f"    CONSTITUENT {float(s):g} {float(a):g} {float(p):g}\n")
        self.ss.write("  HARMONIC_END\n")

    def _write_elevation_constituent(self, attributes: xr.Dataset):
        """Write the constituent block for a WSE-forcing boundary condition."""
        self.ss.write("  TIDAL_BEGIN\n")
        tidal_id = attributes['tidal_table'].item()
        table = self._bc_comp.data.tidal_table_from_id(tidal_id)
        for c, a, p in zip(table.constituent, table.amplitude, table.phase):
            self.ss.write(f"    CONSTITUENT {c.item()} {float(a):g} {float(p):g}\n")
        self.ss.write("  TIDAL_END\n")

    def _write_external_tides(self, arc_id):
        """Write tidal database boundary conditions to the cmcards file.

        Args:
            arc_id (int): The ID of the arc.
        """
        tidal = self._tidal_data
        source = 5
        if tidal:
            source = tidal.info.attrs['source'].item()
        # handle the databases
        if source < 5:
            origin_x, origin_y, _ = self._cogrid.origin
            comment, model_txt, tide_dir, url = self._build_tidal_db_card(source)

            # Always write Velocity cards with ADCIRC tidal database.  These cards need to come just before the TIDAL
            # DATABASE block.
            if model_txt.lower() in ['ec2015', 'enpac2015']:
                self.ss.write("  VEL_BEGIN\n  VEL_END\n")

            self.ss.write('  TIDAL_DATABASE_BEGIN\n')
            # now write the tide block to the cmcards file
            self.ss.write(f"    NAME {model_txt}\n")

            # get the library path
            path = os.path.join(self._get_tidal_library_path(), tide_dir)
            # ADCIRC has two databases, so we need to add the model text to the path.
            if model_txt.lower() != 'leprovost':
                model_path = os.path.join(path, model_txt)
            else:
                model_path = path
            self.ss.write(f"    PATH \"{model_path}\"{comment}\n")

            # Write the constituents to extract.
            cons = tidal.cons.where(tidal.cons.enabled == 1, drop=True)['name'].data.tolist()
            self.ss.write(f"    CONSTITUENTS {len(cons)}")
            for con in cons:
                self.ss.write(f" {con}")
            self.ss.write("\n")
            self.ss.write("  TIDAL_DATABASE_END\n")
            self._download_legacy_tidal_db(model_path, url)
        else:
            if self._get_amp_phase and arc_id in self._bc_arc_id_to_loc:
                amp_phase = self._get_amp_phase(self._bc_arc_id_to_loc[arc_id])
                self.ss.write("  TIDAL_BEGIN\n")
                printed_error = False
                for c, a, p in zip(amp_phase.con, amp_phase.amplitude, amp_phase.phase):
                    if not printed_error and (math.isnan(float(a)) or math.isnan(float(p))):
                        sys.stderr.write(
                            f'Location {self._bc_arc_id_to_loc[arc_id]} outside of '
                            f'tidal database domain.\n'
                        )
                        printed_error = True
                    self.ss.write(f"    CONSTITUENT {c.item()} {float(a):g} {float(p):g}\n")
                self.ss.write("  TIDAL_END\n")

    def _build_tidal_db_card(self, source):
        """Build a tidal database card based on the specified source and parameters.

        Args:
            source (int): The index of the tidal database source.

        Returns:
            tuple: A tuple containing the following elements:
                - comment (str): A comment string for the tidal database card.
                - model_txt (str): The model text for the tidal database card.
                - tide_dir (str): The directory path for the tidal database.
                - url (str): The URL of the tidal database resource, if applicable.
        """
        display_proj = self.projection

        comment = model_txt = tide_dir = url = ''
        if source == td.LEPROVOST_INDEX:
            model_txt = 'LEPROVOST'
            tide_dir = 'leprovost_legacy'
            # save the url for the leprovost datafiles - for later check for download
            url = 'https://s3.amazonaws.com/sms.aquaveo.com/leprovost_legi_format.zip'
        elif source == td.FES2014_INDEX:
            model_txt = tide_dir = 'FES2014'
            comment = '     !This is in development.'
        elif source == td.TPX08_INDEX:
            model_txt = tide_dir = 'TPX08'
            comment = '     !This is in development.'
        elif source == td.TPX09_INDEX:
            model_txt = tide_dir = 'TPX09'
            comment = '     !This is in development.'
        elif source == td.ADCIRC_INDEX:
            tide_dir = 'adcirc2015_legacy'
            # see where this is located
            if display_proj is not None:
                _, geo_wkt = wkt_from_epsg(4326)
                cov_sys = display_proj.coordinate_system
                if cov_sys != '' and cov_sys != 'GEOGRAPHIC':
                    from_wkt = display_proj.well_known_text
                    origin_x, origin_y = self._grid_origin
                    p = transform_points_from_wkt([(origin_x, origin_y)], from_wkt, geo_wkt, False, False)[0]
                    if -60 > p[1] > -100:
                        model_txt = 'EC2015'
                        # save the url for the eastcoast datafiles - for later check for download
                        url = 'https://s3.amazonaws.com/sms.aquaveo.com/ADCIRC2015_EC_tidaldb.zip'
                    elif -110 > p[1] > -160:
                        model_txt = 'ENPAC2015'
                        # save the url for the eastcoast datafiles - for later check for download
                        url = 'https://s3.amazonaws.com/sms.aquaveo.com/ADCIRC2015_WC_tidaldb.zip'
                    else:
                        comment = '     !This simulation is out of range for ADCIRC.'
        return comment, model_txt, tide_dir, url

    def _download_legacy_tidal_db(self, path, url):
        """Download and extract a legacy tidal database if it is not already available.

        Args:
            path (str): The path where the database should be downloaded and extracted.
            url (str): The URL of the legacy tidal database.

        Raises:
            urllib.error.URLError: If there is an error while downloading the resource.
            zipfile.BadZipFile: If the downloaded file is not a valid zip file.
        """
        # if using a legacy tidal database, make sure it is available
        if url and not os.path.isdir(path):  # make the path if it is not there
            os.makedirs(path)

            # tell user we are downloading
            self._logger.info(f'Downloading resource: {url}')
            with urllib.request.urlopen(url) as response:
                # for the legacy tidal databases they are always zip files
                zip_file = os.path.join(path, os.path.basename(url))
                with open(zip_file, 'wb') as out_file:
                    shutil.copyfileobj(response, out_file)
                # Unzip the files
                self._logger.info(f'Unzipping files to: {path}')
                with ZipFile(zip_file, 'r') as unzipper:
                    # Extract all the files in the archive
                    unzipper.extractall(path=path)
                    self._logger.info(f'Deleting zip file: {zip_file}')
                # Move the files up a directory
                source = os.path.join(path, os.path.splitext(os.path.basename(zip_file))[0])
                dir_files = os.listdir(source)
                # Iterate on all files to move them to destination folder
                self._logger.info('Moving tidal database files.')
                for file in dir_files:
                    src_path = os.path.join(source, file)
                    dst_path = os.path.join(path, file)
                    shutil.move(src_path, dst_path)
                try:  # Delete the unzipped folder (should be empty now).
                    os.rmdir(source)
                except OSError:
                    pass
            self._logger.info(f'Deleting zip file: {zip_file}')
            io_util.removefile(zip_file)  # delete the zip file

    def _write_dredge_parameters(self):
        """Write the dredge cards.

        """
        # Only write out dredge information if it is enabled
        if self._sim_data.dredge.attrs['ENABLE_DREDGE'] == 0:
            return

        self.ss.write("!Dredge Module\n")

        if self._sim_data.general.attrs['SOLUTION_SCHEME'] == 'Explicit':
            self.ss.write(
                f"DREDGE_UPDATE_INTERVAL               "
                f"{self._sim_data.dredge.attrs['UPDATE_INTERVAL_VALUE']:g} "
                f"{self._time_to_export[self._sim_data.dredge.attrs['UPDATE_INTERVAL_UNITS']]}\n"
            )
        self.ss.write(
            f"OUTPUT_DREDGE_DIAGNOSTICS            "
            f"{self._get_on_off_string(self._sim_data.dredge.attrs['ENABLE_DIAGNOSTIC'])}\n"
            f"DREDGE_OPERATION_BEGIN\n"
            f"  NAME                               {self._sim_data.dredge.attrs['DREDGE_NAME']}\n"
            f"  DREDGE_BEGIN\n"
        )
        old_dataset_file = self.datasets_file
        self.datasets_file = self.dredge_datasets_file
        self._write_dataset_line(self._sim_data.dredge.attrs['DREDGE_DATASET'], 'DEPTH_DATASET', '    ', 37)
        self.datasets_file = old_dataset_file
        if self._sim_data.dredge.attrs['DREDGE_METHOD'] == "Shallowest cell":
            self.ss.write("    START_METHOD                     SHALLOW\n")
        else:
            self.ss.write(
                f"    START_METHOD                     \"SPECIFIED CELL\"\n"
                f"    START_CELL                       {self._sim_data.dredge.attrs['SPECIFIED_CELL']}\n"
            )
        self.ss.write(
            f"    DREDGE_RATE                      "
            f"{self._sim_data.dredge.attrs['DREDGE_RATE_VALUE']:g} "
            f"'{self._sim_data.dredge.attrs['DREDGE_RATE_UNITS']}'\n"
        )
        trigger_out = self._sim_data.dredge.attrs['TRIGGER_METHOD'].upper().replace(' ', '_')
        self.ss.write(f"    TRIGGER_METHOD                   {trigger_out}\n")
        if trigger_out == 'DEPTH':
            self.ss.write(
                f"    TRIGGER_DEPTH                    "
                f"{self._sim_data.dredge.attrs['TRIGGER_DEPTH_VALUE']:g} "
                f"'{self._sim_data.dredge.attrs['TRIGGER_DEPTH_UNITS']}'\n"
            )
        elif trigger_out == 'VOLUME':
            self.ss.write(
                f"    TRIGGER_VOLUME                   "
                f"{self._sim_data.dredge.attrs['TRIGGER_VOLUME_VALUE']:g} "
                f"'{self._sim_data.dredge.attrs['TRIGGER_VOLUME_UNITS']}'\n"
            )
        elif trigger_out == 'PERCENT':
            self.ss.write(
                f"    TRIGGER_PERCENT                  "
                f"{self._sim_data.dredge.attrs['TRIGGER_PERCENT']:g}\n"
                f"    TRIGGER_DEPTH                    "
                f"{self._sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_VALUE']:g} "
                f"'{self._sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_UNITS']}'\n"
            )
        elif trigger_out == 'TIME_PERIODS':
            starts = self._sim_data.dredge_time_periods.start.data.tolist()
            finishes = self._sim_data.dredge_time_periods.finish.data.tolist()
            self.ss.write(f"    TRIGGER_TIME_PERIODS             {len(starts)} ")
            for start, finish in zip(starts, finishes):
                self.ss.write(f"{start:g} {finish:g} ")
            self.ss.write("'hr'\n")
        self.ss.write(
            f"    DISTRIBUTION                     "
            f"{self._sim_data.dredge.attrs['DISTRIBUTION'].upper()}\n"
        )

        self.ss.write("  DREDGE_END\n\n")
        for placement in [1, 2, 3]:
            self.export_placement(placement)
        self.ss.write("DREDGE_OPERATION_END\n\n")

    def export_placement(self, placement):
        """Writes out the placement.

        Args:
            placement (int): The placement number to export.
        """
        if self._sim_data.dredge_placement.attrs[f'DEFINE_PLACEMENT_{placement}'] == 0:
            return

        percentage = f"PLACEMENT_{placement}_PERCENTAGE"
        dataset = f"PLACEMENT_{placement}_DATASET"
        method = f"PLACEMENT_{placement}_METHOD"
        method_cell = f"PLACEMENT_{placement}_METHOD_CELL"
        limit_method = f"PLACEMENT_{placement}_LIMIT_METHOD"
        limit_depth_value = f"PLACEMENT_{placement}_LIMIT_DEPTH_VALUE"
        limit_depth_units = f"PLACEMENT_{placement}_LIMIT_DEPTH_UNITS"
        limit_thickness_value = f"PLACEMENT_{placement}_LIMIT_THICKNESS_VALUE"
        limit_thickness_units = f"PLACEMENT_{placement}_LIMIT_THICKNESS_UNITS"
        # start writing out a placement
        self.ss.write("  PLACEMENT_BEGIN\n")
        if self._sim_data.dredge.attrs['DISTRIBUTION'] == 'Percent':
            self.ss.write(
                f"    DISTRIBUTION_PERCENTAGE          "
                f"{self._sim_data.dredge_placement.attrs[percentage]:g}\n"
            )
        old_dataset_file = self.datasets_file
        self.datasets_file = self.placement_datasets_file
        self._write_dataset_line(self._sim_data.dredge_placement.attrs[dataset], 'AREA_DATASET', '    ', 37)
        self.datasets_file = old_dataset_file

        if self._sim_data.dredge_placement.attrs[method] == 'Uniform':
            self.ss.write("    PLACEMENT_METHOD                 UNIFORM\n")
        else:
            self.ss.write(
                f"    PLACEMENT_METHOD                 \"SPECIFIED CELL\"\n"
                f"    START_CELL                       "
                f"{self._sim_data.dredge_placement.attrs[method_cell]}\n"
            )

        if self._sim_data.dredge_placement.attrs[limit_method] == 'Depth':
            self.ss.write(
                f"    DEPTH_LIMIT                      "
                f"{self._sim_data.dredge_placement.attrs[limit_depth_value]:g} "
                f"'{self._sim_data.dredge_placement.attrs[limit_depth_units]}'\n"
            )
        else:
            self.ss.write(
                f"    THICKNESS_LIMIT                  "
                f"{self._sim_data.dredge_placement.attrs[limit_thickness_value]:g} "
                f"'{self._sim_data.dredge_placement.attrs[limit_thickness_units]}'\n"
            )
        self.ss.write("  PLACEMENT_END\n\n")

    def _write_structures(self):
        """Write all the structures for a coverage."""
        coverage, data = self._structure_coverage
        wrote_header = False

        wrote_header = write_rubble_mound_structures(
            coverage=coverage,
            data=data,
            ugrid=self._cogrid,
            project_name=self.proj_name,
            query=self.query,
            logger=self._logger,
            cards=self.ss,
            wrote_header=wrote_header
        )

        wrote_header = write_weir_structures(
            coverage=coverage,
            data=data,
            ugrid=self._cogrid,
            cards=self.ss,
            logger=self._logger,
            wrote_header=wrote_header
        )

        wrote_header = write_culvert_structures(
            coverage=coverage,
            data=data,
            ugrid=self._cogrid,
            cards=self.ss,
            logger=self._logger,
            wrote_header=wrote_header
        )

        write_tide_gate_structures(
            coverage=coverage,
            data=data,
            ugrid=self._cogrid,
            cards=self.ss,
            logger=self._logger,
            wrote_header=wrote_header
        )

    def _write_rubble_mound_jetties(self):
        """Write out the rubble mound jetties."""
        coverage, component = self._rubble_mound_coverage
        if not component:
            return

        snap = self._cov_mapper.get_rm_poly_snapper()

        rm_table = {}
        for key in component.data.polygons.keys():
            rm_table[key] = component.data.polygons[key][:].data.tolist()
        rm_table['comp_id'] = component.data.polygons['comp_id'].data.tolist()
        poly_comp_ids = component.comp_to_xms[component.cov_uuid][TargetType.polygon]

        self.ss.write("!Rubble Mound Jetties\n")
        polys = coverage.polygons

        # This line has to come before all the rubble mound blocks.
        self.ss.write(f'RUBBLE_MOUND_ID_DATASET              "{self.proj_name}_RM.h5" "/Datasets/ID"\n')
        cell_ids = np.array([0] * self._xm_ugrid.cell_count, dtype=int)
        used_datasets = set()  # Used dataset UUIDs

        # Since we will be appending datasets to the rubble mound H5 file, make sure one doesn't exist already.
        rm_filename = f'{self.proj_name}_RM.h5'
        if os.path.isfile(rm_filename):
            io_util.removefile(rm_filename)

        # When we write the cell ID dataset below, CMS wants the first rubble mound's cells to have ID 1, the
        # second mound's cells to have ID 2, and so on. This used to use the polygon's ID, but it's possible to make
        # polygons come through here out of order (e.g. by assigning polygon 2, then polygon 1, so 1's component ID is
        # chosen before 2's). We renumber the polygons here to match the order they're written into the .cmcards file,
        # whatever that happens to be.
        renumbered_poly_ids = (i for i in count(1))
        snap.add_polygons(polys)
        for idx, comp_id in enumerate(rm_table['comp_id']):
            if comp_id in poly_comp_ids:
                poly_ids = poly_comp_ids[comp_id]
            else:
                continue

            method = rm_table['calculation_method'][idx]
            method_id = CALCULATION_METHODS.index(method) + 1

            for poly_id in poly_ids:
                renumbered_poly_id = next(renumbered_poly_ids)
                cells = np.array(snap.get_cells_in_polygon(poly_id), dtype=int)
                cell_ids[cells] = renumbered_poly_id
                rock_diam = self._build_rubble_mound_attribute_line(
                    idx=idx,
                    rm_table=rm_table,
                    type_str='rock_diameter_type',
                    att_name='ROCK_DIAMETER',
                    used_datasets=used_datasets,
                    spaces=14
                )
                porosity = self._build_rubble_mound_attribute_line(
                    idx=idx,
                    rm_table=rm_table,
                    type_str='porosity_type',
                    att_name='STRUCTURE_POROSITY',
                    used_datasets=used_datasets,
                    spaces=9
                )
                base_depth = self._build_rubble_mound_attribute_line(
                    idx=idx,
                    rm_table=rm_table,
                    type_str='base_depth_type',
                    att_name='STRUCTURE_BASE_DEPTH',
                    used_datasets=used_datasets,
                    spaces=7
                )
                self.ss.write(
                    f"RUBBLE_MOUND_BEGIN\n"
                    f"  NAME                               '{rm_table['name'][idx]}'\n"
                    f"{rock_diam}\n"
                    f"{porosity}\n"
                    f"{base_depth}\n"
                    f"  FORCHHEIMER_COEFF_METHOD           {method_id}     !{method}\n"
                )
                self.ss.write("RUBBLE_MOUND_END\n")
        self.ss.write('\n')
        # Now write the ID dataset
        writer = DatasetWriter(h5_filename=rm_filename, name='ID', location='cells', overwrite=False)
        writer.write_xmdf_dataset(times=[0.0], data=[cell_ids])

    def _build_rubble_mound_attribute_line(self, idx, rm_table, type_str, att_name, used_datasets, spaces):
        """Build a line for a rubble mound attribute.

        Args:
            idx (int): Polygon idx in the data dict
            rm_table (dict): The data dict
            type_str (str): Variable name of the attribute type flag
            att_name (str): Name of the attribute without the '_CONSTANT' or '_DATASET' suffixes
            used_datasets (set): UUIDs of datasets we have already exported
            spaces (int): Number of spaces between the card and the value
        """
        if rm_table[type_str][idx] == 1:
            card_str = f'{att_name}_DATASET'
            self._dataset_counts += 1
            dset_uuid = rm_table[card_str][idx]
            if dset_uuid not in used_datasets:  # We have not used this dataset before
                used_datasets.add(dset_uuid)
                path = self._write_rubble_mound_dataset(att_name, dset_uuid)
            else:
                path = self._dataset_paths[dset_uuid]
            column_spaces = [' '] * spaces
            column_spaces = ''.join(column_spaces)
            line = f'  {att_name}_DATASET{column_spaces}"{self.proj_name}_RM.h5" "{path}"'
        else:
            column_spaces = [' '] * (spaces - 1)  # Constant versions have one less space between card and value.
            column_spaces = ''.join(column_spaces)
            var_name = type_str.replace('_type', '')
            line = f"  {att_name}_CONSTANT{column_spaces}{rm_table[var_name][idx]:g}"
        return line

    def _write_rubble_mound_dataset(self, att_name, dset_uuid):
        """Append a rubble mound dataset to the H5 file.

        Args:
            att_name (str): Name of the attribute without the '_CONSTANT' or '_DATASET' suffixes
            dset_uuid (str): UUID of the dataset
        """
        # Write the H5 file
        reader = self._rubble_mound_datasets.get(dset_uuid)
        if reader is None:
            raise RuntimeError(f'Unable to find spatially varying {att_name} dataset.')
        path = os.path.join('/Datasets/', reader.name)
        self._dataset_paths[dset_uuid] = path
        writer = DatasetWriter(
            h5_filename=f'{self.proj_name}_RM.h5',
            name=f'{path}',
            dset_uuid=dset_uuid,
            null_value=0,
            time_units=reader.time_units,
            location='cells',
            overwrite=False
        )
        writer.write_xmdf_dataset(times=[0.0], data=[reader.values[0]])
        return path

    def export_cards(self):
        """Write the CMS-Flow cmcards file."""
        self.query = Query()
        self.proj_name = os.path.splitext(os.path.basename(self.query.xms_project_path))[0]
        # Build save points card file from project name for save points.
        self.save_pts_file = f"{self.proj_name}.spcards"

        # Build datasets file name from project name for references in cmcards file.
        self.datasets_file = f"{self.proj_name}_datasets.h5"
        self.dredge_datasets_file = f"{self.proj_name}_DredgeArea.h5"
        self.placement_datasets_file = f"{self.proj_name}_PlacementArea.h5"

        save_pts_cov_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Save Points',
            recurse=False,
            only_first=True
        )

        # Get the linked Save Points coverage and hidden component data.
        if save_pts_cov_item:
            self._save_pts_cov = self.query.item_with_uuid(save_pts_cov_item.uuid)
            self._save_pts_name = self._save_pts_cov.name
            save_point_do_comp = self.query.item_with_uuid(
                save_pts_cov_item.uuid, model_name='CMS-Flow', unique_name='Save_Points_Component'
            )
            self._save_pts_comp = SavePointsComponent(save_point_do_comp.main_file)
            if self._save_pts_comp.cov_uuid == '' and self._save_pts_cov:
                self._save_pts_comp.cov_uuid = self._save_pts_cov.uuid
            self._save_pts_comp.refresh_component_ids(self.query, points=True)
            self._cov_mapper.set_save_points(self._save_pts_cov, self._save_pts_comp)

        # Get the linked tidal constituent simulation and hidden component data.
        tidal_sim_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_DYN_SIM_PTR'],
            allow_pointers=True,
            model_name='Tidal Constituents',
            recurse=False,
            only_first=True
        )
        if tidal_sim_item:
            tidal_do_comp = self.query.item_with_uuid(
                tidal_sim_item.uuid, model_name='Tidal Constituents', unique_name='Tidal_Component'
            )
            self._tidal_data = TidalData(tidal_do_comp.main_file)
            self._tidal_data._info.attrs['reftime'] = QDateTime.fromString(self._sim_data.general.attrs['DATE_START'])

        # Get the linked Boundary Conditions coverage and hidden component data.
        bc_cov_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Boundary Conditions',
            recurse=False,
            only_first=True
        )
        if bc_cov_item:
            self._bc_cov = self.query.item_with_uuid(bc_cov_item.uuid)
            bc_do_comp = self.query.item_with_uuid(bc_cov_item.uuid, model_name='CMS-Flow', unique_name='BC_Component')
            self._bc_comp = BCComponent(bc_do_comp.main_file)
            if self._bc_comp.cov_uuid == '':
                self._bc_comp.cov_uuid = self._bc_cov.uuid
            self._bc_comp.refresh_component_ids(self.query, arcs=True)
            self._cov_mapper.set_boundary_conditions(self._bc_cov, self._bc_comp)

        if self._tidal_data:  # Setup tidal constituent extractor
            self._user_amps, self._user_phases, self._user_cons, self._user_geoms, is_user = \
                self._query_for_user_defined_datasets(self.query, self._tidal_data)
            if is_user:
                self._get_amp_phase = lambda cell_loc: self.get_linear_interp_with_idw_extrap_values(cell_loc)
            else:
                self._extractor = TidalExtractor(self._tidal_data)
                self._get_amp_phase = lambda cell_loc: self._extractor.get_amplitude_and_phase([cell_loc], [0])

        if self._rubble_mound_coverage != (None, None):
            rm_cov, rm_comp = self._rubble_mound_coverage
            self._cov_mapper.set_rubble_mound(rm_cov, rm_comp)
            self._store_rubble_mound_datasets('rock_diameter_type', 'ROCK_DIAMETER_DATASET')
            self._store_rubble_mound_datasets('porosity_type', 'STRUCTURE_POROSITY_DATASET')
            self._store_rubble_mound_datasets('base_depth_type', 'STRUCTURE_BASE_DEPTH_DATASET')

        self._pe_tree = self.query.project_tree

        # get the Quadtree geometry.
        quad_geom = None
        quad_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_UGRID_PTR', 'TI_CGRID2D_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True
        )
        if quad_item:
            # convert from SMS Cartesian convention to bearing convention
            quad_geom = self.query.item_with_uuid(quad_item.uuid)
            self._cogrid = read_grid_from_file(quad_geom.cogrid_file)
            self._xm_ugrid = self._cogrid.ugrid
            origin_x, origin_y, _ = self._cogrid.origin
            wkt = quad_geom.projection.well_known_text
            self._cov_mapper.set_quadtree(self._cogrid, wkt)

            # get the activity coverage.
            activity_item = tree_util.descendants_of_type(
                self._sim_item,
                xms_types=['TI_COVER_PTR'],
                recurse=False,
                allow_pointers=True,
                only_first=True,
                coverage_type='ACTIVITY_CLASSIFICATION'
            )
            activity_cov = None
            if activity_item:  # Activity defined in a coverage
                activity_cov = self.query.item_with_uuid(activity_item.uuid, generic_coverage=True)
            self._cov_mapper.set_activity(activity_cov)

        if self._tidal_data and quad_geom:
            self._get_for_tidal_locations()

        self._sms_version = os.environ['XMS_PYTHON_APP_VERSION']

        self._write_cmcards_file()

    def _store_rubble_mound_datasets(self, type_var, uuid_var):
        """Store a rubble mound spatially varying dataset for later.

        Args:
            type_var (str): The name of the variable that tells us if the dataset is enabled
            uuid_var (str): The name of the variable containing the dataset UUIDs
        """
        _, component = self._rubble_mound_coverage
        enabled = component.data.polygons[type_var].data
        uuids = component.data.polygons[uuid_var].data
        for i in range(len(enabled)):
            dset_uuid = uuids[i]
            if enabled[i] == 0 or dset_uuid in self._rubble_mound_datasets:
                continue
            dset = self.query.item_with_uuid(dset_uuid)
            self._rubble_mound_datasets[dset_uuid] = dset

    def _write_cmcards_file(self):
        """Writes the cmcards file."""
        self.ss.write(
            f"CMS_VERSION                         5.4\n"
            f"SMS_VERSION                         {self._sms_version}\n"
            '\n'
        )
        self._logger.info('Writing grid geometry')
        self._write_grid_geometry()
        self._logger.info('Writing general parameters')
        self._write_general_parameters()
        self._logger.info('Writing timing parameters')
        self._write_timing_parameters()
        self._logger.info('Writing hot start parameters')
        self._write_hot_start_parameters()
        self._logger.info('Writing transport parameters')
        self._write_transport_parameters()
        self._logger.info('Writing save_points')
        self._write_save_points()
        self._logger.info('Writing output times lists')
        self._write_output_times_lists()
        self._logger.info('Writing output parameters')
        self._write_output_parameters()
        self._logger.info('Writing file parameters')
        self._write_file_parameters()
        self._logger.info('Writing implicit/explicit parameters')
        self._write_implicit_explicit_parameters()
        self._logger.info('Writing wave and wind parameters')
        self._write_wind_and_wave_parameters()
        self._logger.info('Writing bottom friction')
        self._write_bottom_friction()
        self._logger.info('Writing sediment transport')
        self._write_sediment_transport()
        self._logger.info('Writing projection')
        self._write_projection()
        self._logger.info('Writing boundary conditions')
        self._write_boundary_conditions()
        self._logger.info('Writing dredge parameters')
        self._write_dredge_parameters()
        if self._structure_coverage != (None, None):
            if self._rubble_mound_coverage != (None, None):
                self._logger.warning(
                    'Exporting Structures and Rubble Mound Jetties coverages at the same time is unsupported. '
                    'Only the Structures coverage will be exported.'
                )
            self._write_structures()
        elif self._rubble_mound_coverage != (None, None):
            self._logger.info('Writing rubble mound jetties')
            self._write_rubble_mound_jetties()

        self._write_advanced_cards()
        self.ss.write('END_PARAMETERS\n')
        # Dump the in-memory stream to file on disk.
        self._logger.info('Flushing to disk')
        out = open(self.proj_name + '.cmcards', 'w', newline='')  # Remove carriage return for test compatibility.
        self.ss.seek(0)
        shutil.copyfileobj(self.ss, out)
        out.close()

    @staticmethod
    def _query_for_user_defined_datasets(query, tidal_data):
        """Get the dataset and geometry dumps of the selected user constituent amplitudes and phases.

        Args:
            query (Query): Query for getting data from XMS.
            tidal_data (TidalData): Data of the source tidal component

        Returns:
            (list, list, list, dict, bool): The amplitude datasets, the phase datasets, the constituent names,
                dict of geometries keyed by geometry UUID, and True if successful.
        """
        is_user = False
        amp_dsets = []
        phase_dsets = []
        cons = []
        geoms = {}
        try:
            if tidal_data.info.attrs['source'].item() == USER_DEFINED_INDEX:
                is_user = True
                if tidal_data.user.Amplitude.data.size != tidal_data.user.Phase.data.size:
                    raise RuntimeError('Number of amplitude datasets does not match the number of phase datasets.')

                for i in range(tidal_data.user.Amplitude.size):
                    # Get the dataset dumps from XMS.
                    # The label index of the source tidal Dataset is one-based (comes from the GUI table).
                    amp_uuid = tidal_data.user.Amplitude.loc[i + 1].item()
                    phase_uuid = tidal_data.user.Phase.loc[i + 1].item()

                    amp_dset = query.item_with_uuid(amp_uuid)
                    if not amp_dset:
                        raise RuntimeError('Error getting user defined amplitude dataset.')
                    phase_dset = query.item_with_uuid(phase_uuid)
                    if not phase_dset:
                        raise RuntimeError('Error getting user defined phase dataset.')
                    amp_dsets.append(amp_dset)
                    phase_dsets.append(phase_dset)

                    # Get the dataset geometry dumps from XMS if we haven't already.
                    amp_geom_uuid = amp_dsets[-1].geom_uuid
                    if amp_geom_uuid not in geoms:
                        amp_geom = query.item_with_uuid(amp_geom_uuid)
                        if not amp_geom:
                            raise RuntimeError('Error getting user defined amplitude dataset geometry.')
                        co_grid = read_grid_from_file(amp_geom.cogrid_file)
                        ugrid = co_grid.ugrid
                        cell_pts = [ugrid.get_cell_centroid(cell)[1] for cell in range(ugrid.cell_count)]
                        pts = ugrid.locations
                        if len(pts) < 3 or (amp_dsets[-1].location == 'cells' and len(cell_pts) < 3):
                            raise RuntimeError('User defined amplitude dataset must contain at least three points.')
                        geoms[amp_geom.uuid] = {'NODE': InterpLinear(points=pts)}
                        if len(cell_pts) >= 3:
                            geoms[amp_geom.uuid]['CELL'] = InterpLinear(points=cell_pts)

                    phase_geom_uuid = phase_dsets[-1].geom_uuid
                    if phase_geom_uuid not in geoms:
                        phase_geom = query.item_with_uuid(phase_geom_uuid)
                        if not phase_geom:
                            raise RuntimeError('Error getting user defined phase dataset geometry.')
                        co_grid = read_grid_from_file(phase_geom.cogrid_file)
                        ugrid = co_grid.ugrid
                        cell_pts = [ugrid.get_cell_centroid(cell)[1] for cell in range(ugrid.cell_count)]
                        pts = ugrid.locations
                        if len(pts) < 3 or (phase_dsets[-1].location == 'cells' and len(cell_pts) < 3):
                            raise RuntimeError('User defined phase dataset must contain at least three points.')
                        geoms[phase_geom.uuid] = {'NODE': InterpLinear(points=pts)}
                        if len(cell_pts) >= 3:
                            geoms[phase_geom.uuid]['CELL'] = InterpLinear(points=cell_pts)

                    con = STANDARD_CONSTITUENTS[tidal_data.user.Constituent.loc[i + 1].item()]
                    if con == 'User specified':
                        cons.append(tidal_data.user.Name.loc[i + 1].item())
                    else:
                        cons.append(con)
        except Exception as e:
            sys.stderr.write(f'{e}\n')
            return [], [], [], {}, False
        return amp_dsets, phase_dsets, cons, geoms, is_user

    @staticmethod
    def linear_interp_with_idw_extrap(linear_interper, to_points, do_dset, idw_interpers):
        """Interpolate a dataset to to_points using linear interpolation and use IDW for extrapolated points.

        Args:
            linear_interper (InterpLinear): Linear interpolator initially used
            to_points (list): List of x,y,z locations to interpolate to
            do_dset (data_objects.parameters.Dataset): The dataset to interpolate
            idw_interpers (dict): Dictionary of IDW interpolators keyed by geom UUID. If linear interpolation results
                in extrapolated values, this dictionary will be checked for an existing IDW interpolator for the
                geometry. If no IDW interpolator present, one will be created.

        Returns:
            list: interp_vals with extrapolated values replaced with IDW values
        """
        do_dset.ts_idx = 0  # Use the first timestep if transient
        linear_interper.scalars = do_dset.data
        interp_vals = linear_interper.interpolate_to_points(to_points)
        extrap_pt_idxs = linear_interper.extrapolation_point_indexes
        if len(extrap_pt_idxs) > 0:  # Have extrapolated values, use IDW for those points.
            interp_vals = list(interp_vals)  # Convert immutable tuple to mutable list
            geom_uuid = do_dset.geom_uuid
            if geom_uuid not in idw_interpers:
                # Set up an IDW interpolator if linear interpolation resulted in extrapolated points.
                idw_interpers[geom_uuid] = InterpIdw(
                    points=linear_interper.points, nodal_function='constant', number_nearest_points=2
                )
                idw_interpers[geom_uuid].set_search_options(2, False)

            idw_interp = idw_interpers[geom_uuid]
            idw_interp.scalars = linear_interper.scalars
            idw_interp_vals = idw_interp.interpolate_to_points([to_points[extrap_pt] for extrap_pt in extrap_pt_idxs])
            for extrap_idx, extrap_pt in enumerate(extrap_pt_idxs):
                interp_vals[extrap_pt] = idw_interp_vals[extrap_idx]
        return interp_vals

    def get_linear_interp_with_idw_extrap_values(self, cell_loc):
        """Gets the linear interpolation value for the cell location.

        If the value is outside of the domain of the amplitude or phase source, then IDW extrapolation will be used.

        Args:
            cell_loc (tuple): A cell centroid location (x, y)
        """
        idw_interpers = {}
        amps = []
        phases = []
        for amp, phase in zip(self._user_amps, self._user_phases):
            amp_uuid = amp.geom_uuid
            phase_uuid = phase.geom_uuid
            amp_interp = self._user_geoms[amp_uuid][amp.data_basis]
            phase_interp = self._user_geoms[phase_uuid][phase.data_basis]
            amps.append(self.linear_interp_with_idw_extrap(amp_interp, [cell_loc], amp, idw_interpers)[0])
            phases.append(self.linear_interp_with_idw_extrap(phase_interp, [cell_loc], phase, idw_interpers)[0])
        # raise Exception(f'{self._user_cons} {amps} {phases}')
        values_data = {
            'amplitude': ('con', np.array(amps, dtype=np.float64)),
            'phase': ('con', np.array(phases, dtype=np.float64)),
        }
        values_coords = {
            'con': self._user_cons,
            'node_id': [0 for _ in self._user_cons],
        }
        return xr.Dataset(data_vars=values_data, coords=values_coords)

    def _get_for_tidal_locations(self):
        """Sets up a single location for each arc that uses the tidal constituents component.

        These locations are used for interpolation or tidal database querying.
        """
        arc_ex_snapper = self._cov_mapper.get_bc_arc_snapper()

        bc_table = {}
        for key in self._bc_comp.data.arcs.keys():
            bc_table[key] = self._bc_comp.data.arcs[key][:].data.tolist()
        bc_table['comp_id'] = self._bc_comp.data.arcs['comp_id'].data.tolist()
        arc_comp_ids = self._bc_comp.comp_to_xms[self._bc_comp.cov_uuid][TargetType.arc]
        arcs = self._bc_cov.arcs
        arc_id_to_arc = {}
        for arc in arcs:
            arc_id_to_arc[arc.id] = arc
        for idx, comp_id in enumerate(bc_table['comp_id']):
            if comp_id in arc_comp_ids:
                arc_ids = arc_comp_ids[comp_id]
            else:
                continue
            bc_type = bc_table['bc_type'][idx]
            for arc_id in arc_ids:
                use_arc = bc_type == 'WSE-forcing' and bc_table['wse_source'][idx] == 'External tidal'
                if use_arc:
                    # get the cell ids of the snapped arc locations.
                    arc = arc_id_to_arc[arc_id]
                    snap_arc = arc_ex_snapper.get_snapped_points(arc)
                    if snap_arc:
                        cell_locs = snap_arc['location']
                        if cell_locs.size > 0:
                            self._bc_arc_id_to_loc[arc_id] = cell_locs[int(len(cell_locs) / 2)]

    @cached_property
    def _sim_item(self) -> Optional[TreeNode]:
        """
        Get the simulation's tree node.

        if self.query is None, returns None.
        """
        if not self.query:
            return  # Some tests trigger this even though it's probably unrealistic in practice.

        sim_uuid = self.query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, sim_uuid)
        return sim_item

    @cached_property
    def _sim_data(self) -> Optional[SimulationData]:
        """
        Get the simulation's data manager.

        if self.query is None, returns None.
        """
        if not self._sim_item:
            return

        sim_uuid = self._sim_item.uuid
        do_comp = self.query.item_with_uuid(sim_uuid, model_name='CMS-Flow', unique_name='Simulation_Component')
        main_file = do_comp.main_file
        return SimulationData(main_file)

    @cached_property
    def _rubble_mound_coverage(self) -> tuple[Optional[Coverage], Optional[RMStructuresComponent]]:
        """
        Get the rubble mound coverage and its data manager.

        If there is no linked structure coverage, returns (None, None).
        """
        if not self._sim_item:
            return None, None

        rm_cov_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Rubble Mound Jetties',
            recurse=False,
            only_first=True
        )
        if not rm_cov_item:
            return None, None

        coverage = self.query.item_with_uuid(rm_cov_item.uuid)
        do_component = self.query.item_with_uuid(
            rm_cov_item.uuid, model_name='CMS-Flow', unique_name='RM_Structures_Component'
        )
        main_file = do_component.main_file
        component = RMStructuresComponent(main_file)
        if component.cov_uuid == '':
            component.cov_uuid = coverage.uuid
        component.refresh_component_ids(self.query, polygons=True)
        return coverage, component

    @cached_property
    def _structure_coverage(self) -> tuple[Optional[Coverage], Optional[CoverageBaseData]]:
        """
        Get the structure coverage and its data manager.

        If there is no linked structure coverage, returns (None, None).
        """
        if not self._sim_item:
            return None, None

        rm_cov_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Structures',
            recurse=False,
            only_first=True
        )
        if not rm_cov_item:
            return None, None

        coverage = self.query.item_with_uuid(rm_cov_item.uuid)
        do_component = self.query.item_with_uuid(
            rm_cov_item.uuid, model_name='CMS-Flow', unique_name='StructuresComponent'
        )
        main_file = do_component.main_file
        component = StructuresComponent(main_file)
        self.query.load_component_ids(component, arcs=True, polygons=True)
        return coverage, component.data
