"""This module is for the base class of all other hidden components in cmsflow."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import enum
import os
from pathlib import Path
import shutil

# 2. Third party modules
from packaging import version

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem, XmsEnvironment as XmEnv
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.cmsflow.components.id_files import UNINITIALIZED_COMP_ID
from xms.cmsflow.file_io import io_util


class ColAttType(enum.IntEnum):
    """An enumeration of possible shapefile field data types.
    """
    COL_ATT_INT = 0
    COL_ATT_DBL = 1
    COL_ATT_STR = 2


class CmsflowComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the CMS-Flow model."""
    def __init__(self, main_file: str | Path):
        """
        Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        main_file = str(main_file)
        super().__init__(main_file.strip('"\''))
        self.data = None
        self.tree_commands = []  # [(menu_text, menu_method)...]
        self.polygon_commands = []  # [(menu_text, menu_method)...]
        self.arc_commands = []  # [(menu_text, menu_method)...]
        self.point_commands = []  # [(menu_text, menu_method)...]
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''
        self.uuid = os.path.basename(os.path.dirname(self.main_file))
        self.disp_opts_file = ''

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        if not self.copy_component_folder(new_path):  # Already in the same location (simple save)
            return self.main_file, [], []
        new_main_file = os.path.normpath(os.path.join(new_path, os.path.basename(self.main_file)))
        return new_main_file, [], []

    def project_open_event(self, new_path):
        """Called when XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            new_path (str): Path to the new save location.

        """
        if not self.disp_opts_file:  # Component has no display lists
            return

        # If we are not a legacy file from 13.1, we do not need to initialize our display because SMS already did.
        # Will be 0.0 if no currently saved project.
        proj_version = version.parse(XmEnv.xms_environ_project_version())
        if proj_version < version.parse('1.0') or proj_version >= version.parse('13.2'):
            return

        new_disp_opts = os.path.join(new_path, os.path.basename(self.disp_opts_file))
        if os.path.isfile(new_disp_opts):
            if self.cov_uuid:
                self.display_option_list.append(XmsDisplayMessage(file=new_disp_opts, edit_uuid=self.cov_uuid))
            else:  # This can happen for snap previews and other free draw components.
                self.display_option_list.append(
                    XmsDisplayMessage(file=new_disp_opts, draw_type=DrawType.draw_at_locations)
                )

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list` of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (:obj:`list` of :obj:`xmsapi.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.

        """
        if not main_file_list or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for command_text, command_method in self.tree_commands:
            action = ActionRequest(
                modality='MODAL',
                method_name=command_method,
                class_name=self.class_name,
                module_name=self.module_name,
                main_file=self.main_file
            )
            menu_list.append(MenuItem(text=command_text, action=action))
        return menu_list

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        if self.tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            action = ActionRequest(
                modality='MODAL',
                method_name=self.tree_commands[0][1],
                class_name=self.class_name,
                module_name=self.module_name,
                main_file=self.main_file
            )
            actions.append(action)
        return messages, actions

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list` of :obj:`xmsapi.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.

        """
        menu_list = [None]  # None == spacer
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                id_files[entity] = ('', '')
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        menu_list = self.add_menu_items_for_selection(menu_list, selection, unpacked_id_files)
        if not menu_list:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list

    def add_menu_items_for_selection(self, menu_list, selection, unpacked_id_files):
        """This adds menu items appropriate for the given selection.

        Args:
            menu_list (:obj:`list` of :obj:`xmsapi.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                    that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem.
                    "None" may be added to the list to indicate a separator.
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                    The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            unpacked_id_files (dict): A dictionary with the key being an entity type, and the value being a tuple.
                    The first value of the tuple is a temporary file of XMS ids. The second value of the tuple is
                    a temporary file of component ids.

        Returns:
            menu_list (:obj:`list` of :obj:`xmsapi.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.
        """
        if 'POLYGON' in selection and selection['POLYGON']:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for command_text, command_method in self.polygon_commands:
                parameters = {'id_files': poly_id_files, 'selection': selection['POLYGON']}
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                menu_list.append(MenuItem(text=command_text, action=action))
        if 'ARC' in selection and selection['ARC']:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for command_text, command_method in self.arc_commands:
                parameters = {'id_files': arc_id_files, 'selection': selection['ARC']}
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                menu_list.append(MenuItem(text=command_text, action=action))
        if ('POINT' in selection and selection['POINT']) or ('NODE' in selection and selection['NODE']):
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            node_id_files = unpacked_id_files['NODE'] if 'NODE' in unpacked_id_files else None
            for command_text, command_method in self.point_commands:
                parameters = {
                    'id_files': point_id_files,
                    'selection': selection['POINT'],
                    'node_id_files': node_id_files,
                    'node_selection': selection['NODE']
                }
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                menu_list.append(MenuItem(text=command_text, action=action))
        return menu_list

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)
        unpacked_id_files = {}

        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                id_files[entity] = ('', '')
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        actions = []
        if 'POLYGON' in selection:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for _, command_method in self.polygon_commands:
                parameters = {'id_files': poly_id_files, 'selection': selection['POLYGON']}
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                actions.append(action)
                break  # Only expecting one dialog ActionRequest on double-click
        elif 'ARC' in selection:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for _, command_method in self.arc_commands:
                parameters = {'id_files': arc_id_files, 'selection': selection['ARC']}
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                actions.append(action)
                break  # Only expecting one dialog ActionRequest on double-click
        elif 'POINT' in selection:
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
            for _, command_method in self.point_commands:
                parameters = {'id_files': point_id_files, 'selection': selection['POINT']}
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self.class_name,
                    module_name=self.module_name,
                    main_file=self.main_file,
                    parameters=parameters
                )
                actions.append(action)
                break  # Only expecting one dialog ActionRequest on double-click
        if not actions:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return [], actions

    def get_display_refresh_action(self):
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        return ActionRequest(
            modality='NO_DIALOG',
            method_name='get_initial_display_options',
            comp_uuid=self.uuid,
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file
        )

    def refresh_component_ids(self, query, points=False, arcs=False, polygons=False):
        """Load all the currently used component ids and clean out the unused ones.

        Args:
            query (Query): The xmsapi interprocess communicator
            points (bool): True to refresh point component ids (Save Points coverage).
            arcs (bool): True to refresh arc component ids (Boundary Conditions coverage).
            polygons (bool): True to refresh polygon component ids (Rubble Mound coverage).
        """
        query.load_component_ids(self, points=points, arcs=arcs, polygons=polygons)
        self._remove_unused_component_ids()

    def _remove_unused_component_ids(self):
        """Remove unused component ids from the data and display option id files.

        Notes:
            Coverage component ids must be loaded from the Query before calling this.
        """
        pass

    def _unpack_xms_data(self, param_map, target_type):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (dict): The ActionRequest parameter map
            target_type (TargetType): The coverage entity to unpack data for

        Returns:
            (int): component id of atts to display
        """
        # Get the XMS attribute ids of the selected arcs (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if len(self.selected_att_ids) == 0:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected (if any).
        comp_id = UNINITIALIZED_COMP_ID
        id_files = param_map.get('id_files', [])
        if id_files and id_files[0]:
            if target_type == TargetType.point:
                entity_key = 'POINT'
                multi_select = 'Multiple save points selected. Changes will apply to all selected points.'
            elif target_type == TargetType.arc:
                entity_key = 'ARC'
                multi_select = 'Multiple boundary condition arcs selected. Changes will apply to all selected arcs.'
            else:
                entity_key = 'POLYGON'
                multi_select = 'Multiple polygons selected. Changes will apply to all selected polygons.'
            self.load_coverage_component_id_map({entity_key: (id_files[0], id_files[1])})
            comp_id = self._check_selected_types(target_type, multi_select)
            shutil.rmtree(os.path.dirname(id_files[0]), ignore_errors=True)

        return comp_id

    def _check_selected_types(self, target_type, multi_select):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Args:
            target_type (TargetType): The coverage entity to unpack data for
            multi_select (str): If multiple entities selected with different component ids, will display this message
                in the dialog.

        Returns:
            (int): component id of attributes to display
        """
        num_features = len(self.selected_att_ids)
        # 1 entity selected, use those atts
        if num_features == 1:
            comp_id = self.get_comp_id(target_type, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one entity selected, check types
            # Get the component ids of all the selected entities
            try:
                self.selected_comp_ids = list(self.comp_to_xms[self.cov_uuid][target_type].keys())
            except KeyError:
                return UNINITIALIZED_COMP_ID  # No component ids assigned for any of the selected arcs

            # If there are multiple entities selected with the same component id, display those attributes. Otherwise
            # display an empty, default dialog.
            self.dlg_message = multi_select
            return UNINITIALIZED_COMP_ID if len(self.selected_comp_ids) != 1 else self.selected_comp_ids[0]

    @staticmethod
    def _update_display_uuids_for_duplicate(new_comp_uuid, new_display_options_file):
        """Updates the display options file for a new component created from a duplicate event.

        Args:
            new_comp_uuid (str): The UUID of the new component.
            new_display_options_file (str): The display options file to update.

        Returns:
            The new display uuid as a string.
        """
        json_dict = read_display_options_from_json(new_display_options_file)
        categories = CategoryDisplayOptionList()
        new_uuid = categories.uuid
        categories.from_dict(json_dict)
        categories.uuid = new_uuid
        categories.comp_uuid = new_comp_uuid
        write_display_options_to_json(new_display_options_file, categories)
        return new_uuid

    @staticmethod
    def _append_comp_id_to_file(id_file, new_comp_id):
        """Appends component id to the display id file.

        Args:
            id_file (str): Relative path to the id file.
            new_comp_id (int): The new component id to add to the id file.
        """
        updated_comp_ids = read_display_option_ids(id_file)
        updated_comp_ids.append(new_comp_id)
        write_display_option_ids(id_file, updated_comp_ids)
