"""Hard bottom class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'
__all__ = ['HardBottomCoverageComponent']

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.components2.gmi_coverage_component_base import GmiCoverageComponentBase
from xms.gmi.data.generic_model import GenericModel, Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules


class HardBottomCoverageComponent(GmiCoverageComponentBase):
    """A Hard bottom component for the CMS-Flow model."""
    @property
    def features(self) -> dict[TargetType, list[tuple[str, str]]]:
        """The features this coverage supports."""
        feature_dict = {}
        for target in [TargetType.point, TargetType.arc, TargetType.polygon]:
            section = self._get_section(target)
            feature_dict[target] = []
            for group_name in section.group_names:
                label = section.group(group_name).label
                feature_dict[target].append((group_name, label))

        return feature_dict

    def _get_section(self, target: TargetType) -> Section:
        return _hard_bottom_data_definition().section_from_target_type(target)

    def generic_model(self) -> GenericModel:
        """Get the generic model for the hard bottom data."""
        return _hard_bottom_data_definition()


def _hard_bottom_data_definition():
    """Create the hard bottom data definition."""
    gm = GenericModel(exclusive_point_conditions=True, exclusive_arc_conditions=True, exclusive_polygon_conditions=True)
    pp = gm.polygon_parameters
    pp.add_group('non_erodible_cell', 'Non-erodible cell')
    grp = pp.add_group('specified_distance_erodible_cell', 'Specified distance erodible cell')
    grp.add_float(
        name='distance_below_elevation', label='Specified distance below bathymetric cell elevation', default=0.0
    )
    grp = pp.add_group('specified_elevation_erodible_cell', 'Specified elevation erodible cell')
    grp.add_float(name='specified_elevation', label='Specified elevation (negative down)', default=0.0)
    # TODO: way to add a specific help page that isn't https://xmswiki.com/wiki/SMS:GenericModelDialog?
    return gm
