"""MappedSavePointsComponent class. Data mapped from a Save Points Coverage onto an CMS-Flow quadtree."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.cmsflow.components.cmsflow_component import CmsflowComponent


class MappedSavePointsComponent(CmsflowComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Flow model mapped Save Points coverage."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.class_name = 'MappedSavePointsComponent'
        self.module_name = 'xms.cmsflow.components.mapped_save_points_component'
        # [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), 'save_point_display_options.json')

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            fname = os.path.join(new_path, 'save_point_display_options.json')
            json_dict = read_display_options_from_json(fname)
            if 'uuid' in json_dict:
                json_dict['uuid'] = str(uuid.uuid4())
                json_dict['comp_uuid'] = os.path.basename(new_path)
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                categories.from_dict(json_dict)
                write_display_options_to_json(fname, categories)
            self.update_display_options(new_main_file, json_dict, action_requests)

        return new_main_file, messages, action_requests

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:'xmsapi.dmi.Query'): A way to communicate with XMS.
            params (:obj:'list' of :obj:'str'): Generic map of parameters.
            win_cont (:obj:'PySide2.QtWidgets.QWidget'): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.disp_opts_file)
        categories.from_dict(json_dict)
        categories_list = [categories]

        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont)
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                write_display_options_to_json(self.disp_opts_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=self.disp_opts_file, draw_type=DrawType.draw_at_locations)
                )
                break  # only one list
        return [], []

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        return [], [self.get_display_options_action()]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        # Send the default display options list to XMS.
        self.display_option_list.append(XmsDisplayMessage(file=self.main_file, draw_type=DrawType.draw_at_locations))
        return [], []

    def get_display_options_action(self):
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        return ActionRequest(
            modality='NO_DIALOG',
            method_name='get_initial_display_options',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file
        )

    def update_display_options(self, new_main_file, json_dict, action_requests):
        """Update the display options.

        Args:
            new_main_file (str): file name
            json_dict (dict): display options dictionary
            action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        if 'comp_uuid' in json_dict:
            action = self.get_display_options_action()
            action.main_file = new_main_file
            action.component_uuid = json_dict['comp_uuid']
            action_requests.append(action)
