"""This module is for the save points hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.cmsflow.components.cmsflow_component import CmsflowComponent, ColAttType
from xms.cmsflow.components.id_files import SAVE_INITIAL_ATT_ID_FILE, SAVE_INITIAL_COMP_ID_FILE
from xms.cmsflow.data.save_points_data import SavePointsData
from xms.cmsflow.file_io import io_util
from xms.cmsflow.gui.save_point_attributes_dlg import SavePointAttributesDlg
from xms.cmsflow.gui.save_points_properties_dlg import SavePointsPropertiesDlg
from xms.cmsflow.merge.merge_save_points import SavePointsMerger


class SavePointsComponent(CmsflowComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Flow model save points coverage."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Display Options...', 'open_display_options'),
            ('Save Points Properties...', 'open_save_points_properties'),
        ]
        self.point_commands = [('Assign Save Points...', 'open_point_attributes')]
        self.data = SavePointsData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        comp_dir = os.path.dirname(self.main_file)
        self.point_comp_id_file = os.path.join(comp_dir, 'save_point.display_ids')
        # Copy default display options if needed
        self.disp_opts_file = os.path.join(comp_dir, 'save_point_display_options.json')
        if not os.path.exists(self.disp_opts_file):
            # Read the default point display options, and save ourselves a copy with a randomized UUID.
            point_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            point_default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'save_point_default_display_options.json'
            )
            point_json_dict = read_display_options_from_json(point_default_file)
            point_categories.from_dict(point_json_dict)
            point_categories.comp_uuid = self.uuid
            write_display_options_to_json(self.disp_opts_file, point_categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['point_display_uuid'] = point_categories.uuid
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            # Change the display UUID to be something different.
            new_comp = SavePointsComponent(new_main_file)
            new_uuid = self._update_display_uuids_for_duplicate(new_comp.uuid, new_comp.disp_opts_file)
            new_comp.data.info.attrs['point_display_uuid'] = new_uuid
            new_comp.data.commit()
            action_requests.append(self.get_display_refresh_action())
        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        return [], [self.get_display_refresh_action()]

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list` of :obj:`tuple`): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        merger = SavePointsMerger(self)
        return merger.merge(merge_list)

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        self.cov_uuid = query.parent_item_uuid()
        if not self.cov_uuid:
            return [('ERROR', 'Could not get CMS-Flow save points coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.main_file), SAVE_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), SAVE_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            io_util.remove(initial_att_file)
            io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.point, att_id, comp_id)

        self.data.info.attrs['cov_uuid'] = self.cov_uuid
        self.data.commit()
        # # Send the default display list to XMS.
        self.display_option_list.append(
            XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        return [], []

    def open_save_points_properties(self, query, params, win_cont):
        """Opens the save point properties dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        dlg = SavePointsPropertiesDlg(self.data, win_cont)
        if dlg.exec_():
            self.data = dlg.save_points_data
            self.data.commit()
        return [], []

    def open_point_attributes(self, query, params, win_cont):
        """Opens the save point dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        comp_id = self._unpack_xms_data(params[0], TargetType.point)
        incremented_id = False
        if comp_id not in self.data.points.comp_id:  # Create default attributes if this point has never been assigned.
            comp_id = self.data.add_save_point_atts()
            incremented_id = True  # We just got a new component id, so don't duplicate on OK
        single_point = self.data.points.loc[dict(comp_id=[comp_id])]
        dlg = SavePointAttributesDlg(single_point, self.dlg_message, win_cont)
        if dlg.exec_():
            if incremented_id:  # Was previously unassigned, so we already got a new component id. Just update it.
                self.data.update_save_point(comp_id, dlg.save_point_data)
            else:  # Always generate a new component id, old one might be shared if assigned during a multi-select.
                comp_id = self.data.add_save_point_atts(dlg.save_point_data)
            self._append_comp_id_to_file(self.point_comp_id_file, comp_id)
            self.data.commit()
            for pt_id in self.selected_att_ids:
                self.update_component_id(TargetType.point, pt_id, comp_id)
            # Send back updated display lists to XMS after ActionRequest
            self.display_option_list.append(
                # Update the point display list only.
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid)
            )
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for save points.

        Args:
            query (:obj:'xmsapi.dmi.Query'): A way to communicate with XMS.
            params (:obj:'list' of :obj:'str'): Generic map of parameters.
            win_cont (:obj:'PySide2.QtWidgets.QWidget'): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Read the default point display options, and save ourselves a copy with a randomized UUID.
        point_categories = CategoryDisplayOptionList()
        point_json_dict = read_display_options_from_json(self.disp_opts_file)
        point_categories.from_dict(point_json_dict)

        dlg = CategoryDisplayOptionsDialog([point_categories], win_cont)
        if dlg.exec_():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    def get_table_def(self, target_type):
        """Get the shapefile attribute table definition for a feature object.

        Args:
            target_type (TargetType): The feature object type

        Returns:
            list: list of tuples where first element is column name and second element is ColAttType

        """
        if target_type == TargetType.point:
            return [
                ('ID', ColAttType.COL_ATT_INT),
                ('NAME', ColAttType.COL_ATT_STR),
                ('HYDRO', ColAttType.COL_ATT_INT),
                ('SEDIMENT', ColAttType.COL_ATT_INT),
                ('SALINITY', ColAttType.COL_ATT_INT),
                ('WAVES', ColAttType.COL_ATT_INT),
            ]
        return None

    def get_att_table(self, target_type):
        """Get the shapefile attribute table for a feature object.

        Args:
            target_type (TargetType): The feature object type

        Returns:
            pandas.DataFrame: The shapefile attributes to write

        """
        if target_type == TargetType.point:
            data = {'ID': [], 'NAME': [], 'HYDRO': [], 'SEDIMENT': [], 'SALINITY': [], 'WAVES': []}
            df = self.data.points.to_dataframe()
            cov_dict = self.comp_to_xms.get(self.cov_uuid, {})
            point_dict = cov_dict.get(TargetType.point, {})
            for comp_id, xms_id_list in point_dict.items():
                record = df.loc[float(comp_id)].to_dict()
                name = record['name']
                hydro = record['hydro']
                sediment = record['sediment']
                salinity = record['salinity']
                waves = record['waves']
                for xms_id in xms_id_list:
                    data['ID'].append(xms_id)
                    data['NAME'].append(name)
                    data['HYDRO'].append(hydro)
                    data['SEDIMENT'].append(sediment)
                    data['SALINITY'].append(salinity)
                    data['WAVES'].append(waves)
            return pd.DataFrame(data)
        return None

    def populate_from_att_tables(self, att_dfs):
        """Populate attributes from an attribute table written by XMS.

        Args:
            att_dfs (dict): Dictionary of attribute pandas.DataFrame. Key is TargetType enum, value is None if not
                applicable.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        df = att_dfs[TargetType.point]
        xms_ids = df['ID'].to_list()
        pt_names = df['NAME'].to_list()

        df = df.fillna(value={'ID': 0, 'NAME': '', 'HYDRO': 0, 'SEDIMENT': 0, 'SALINITY': 0, 'WAVES': 0})
        df.index = pd.RangeIndex(len(df.index))
        if len(df) == 1 and not df.loc[0, 'NAME']:  # Ensure save point name field was mapped
            msg = 'NAME must be mapped to shapefile attribute to import save point properties'
            return [('INFO', msg)], []

        temp_folder = os.path.join(os.getcwd(), str(uuid.uuid4()))
        os.makedirs(temp_folder)
        mainfile = os.path.join(temp_folder, str(uuid.uuid4()))
        save_pt_comp = SavePointsComponent(mainfile)
        pt_df = save_pt_comp.data.points.to_dataframe()
        pt_name_comp_id = {'': 0}
        for i in range(len(df)):
            pt_name = df.loc[i, 'NAME']
            row = i
            pt_name_comp_id[pt_name] = i + 1
            pt_df.loc[row, 'comp_id'] = i + 1
            pt_df.loc[row, 'name'] = pt_name
            pt_df.loc[row, 'hydro'] = int(df.loc[i, 'HYDRO'])
            pt_df.loc[row, 'sediment'] = int(df.loc[i, 'SEDIMENT'])
            pt_df.loc[row, 'salinity'] = int(df.loc[i, 'SALINITY'])
            pt_df.loc[row, 'waves'] = int(df.loc[i, 'WAVES'])
        save_pt_comp.data.points = pt_df.to_xarray()
        save_pt_comp.data.info.attrs['next_comp_id'] = len(pt_df)
        save_pt_comp.data.commit()

        comp_ids = [pt_name_comp_id[pt_name] for pt_name in pt_names]
        id_file = os.path.join(temp_folder, str(uuid.uuid4()))
        comp_id_file = os.path.join(temp_folder, str(uuid.uuid4()))
        write_display_option_ids(id_file, xms_ids)
        write_display_option_ids(comp_id_file, comp_ids)
        shapefile_comp = [(mainfile, {'POINT': (id_file, comp_id_file)})]
        msgs, actions = self.handle_merge(shapefile_comp)
        save_pt_comp.data.close()
        shutil.rmtree(temp_folder, ignore_errors=True)
        return msgs, actions

    # def _remove_unused_component_ids(self):
    #     """Remove unused component ids from the data and display option id files.
    #
    #     Notes:
    #         Coverage component ids must be loaded from the Query before calling this.
    #     """
    #     # Drop unused component ids from the datasets
    #     comp_ids = list(self.comp_to_xms.get(self.cov_uuid, {}).get(TargetType.point, {}).keys())
    #     mask = self.data.points.comp_id.isin(comp_ids)
    #     self.data.points = self.data.points.where(mask, drop=True)
    #     # Rewrite the display option id file now to clear out non-existent component ids.
    #     if len(self.data.points.comp_id) > 0:
    #         write_display_option_ids(self.point_comp_id_file, self.data.points.comp_id.data)
    #     else:  # If we don't have any points, just delete the id file.
    #         io_util.remove(self.point_comp_id_file)
    #     self.data.vacuum()  # Reclaim disk space in the NetCDF file
