"""This module defines data for the simulation hidden component."""

# 1. Standard Python modules
from datetime import datetime, timedelta
import os
from pathlib import Path
from typing import Optional

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase

# 4. Local modules


class SimulationData(XarrayBase):
    """Manages data file for the hidden save points component."""
    def __init__(self, data_file: str | Path):
        """
        Initializes the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        data_file = str(data_file)
        self._filename = data_file
        self._info = None
        self._general = None
        self._flow = None
        self._sediment = None
        self._salinity = None
        self._wave = None
        self._wind = None
        self._output = None
        self._advanced_sediment_diameters_table = None
        self._simple_grain_sizes_table = None
        self._bed_layer_table = None
        self._atmospheric_table = None
        self._meteorological_stations_table = None
        self._meteorological_stations_direction_curves = dict()
        self._wind_from_table = None
        self._list_1_table = None
        self._list_2_table = None
        self._list_3_table = None
        self._list_4_table = None
        self._dredge = None
        self._dredge_placement = None
        self._dredge_time_periods = None
        self._advanced_card_table = None
        self._advanced_block_table = None
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets(data_file)

        super().__init__(data_file)

        self._check_for_simple_migrations()

    @property
    def main_file(self) -> str:
        """The data manager's main-file."""
        return self._filename

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (str): Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'CMSFLOW_SIM',
                # 'VERSION': pkg_resources.get_distribution('cmsflow').version,
                'domain_uuid': '',  # Unused. Didn't want to update a million test files.
            }
            self._info = xr.Dataset(attrs=info)

            general = {
                'DATE_START': '',
                'SIM_DURATION_VALUE': 48.0,
                'SIM_DURATION_UNITS': 'hours',
                'RAMP_DURATION_VALUE': 24.0,
                'RAMP_DURATION_UNITS': 'hours',
                'SKEW_CORRECT': 1,
                'USE_INIT_CONDITIONS_FILE': 0,
                'INIT_CONDITIONS_FILE': '(none selected)',
                'USE_HOT_START_OUTPUT_FILE': 1,
                'HOT_WRITE_OUT_DURATION_VALUE': 48.0,
                'HOT_WRITE_OUT_DURATION_UNITS': 'hours',
                'RECURRING_HOT_START_FILE': 1,
                'AUTO_HOT_DURATION_VALUE': 0.5,
                'AUTO_HOT_DURATION_UNITS': 'hours',
                'SOLUTION_SCHEME': 'Implicit',
                'MATRIX_SOLVER': 'GMRES',
                'NUM_THREADS': 1
            }
            self._general = xr.Dataset(attrs=general)

            flow = {
                'HYDRO_TIME_STEP_VALUE': 600.0,
                'HYDRO_TIME_STEP_UNITS': 'seconds',
                'WETTING_DEPTH': 0.05,
                'WAVE_FLUXES': 0,
                'ROLLER_FLUXES': 0,
                'LATITUDE_CORIOLIS': 'From projection',
                'DEGREES': 0.0,
                'TURBULENCE_MODEL': 'Subgrid',
                'TURBULENCE_PARAMETERS': 0,
                'BASE_VALUE': 0.000001,
                'CURRENT_BOTTOM_COEFFICIENT': 0.067,
                'CURRENT_HORIZONTAL_COEFFICIENT': 0.2,
                'WAVE_BOTTOM_COEFFICIENT': 0.5,
                'WAVE_BREAKING_COEFFICIENT': 0.1,
                'WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT': 'Quadratic',
                'QUAD_WAVE_BOTTOM_COEFFICIENT': 0.65,
                'BED_SLOPE_FRIC_COEFFICIENT': 0,
                'WALL_FRICTION': 1,
                'BOTTOM_ROUGHNESS': 'Mannings N',
                'ROUGHNESS_SOURCE': 'Dataset',
                'BOTTOM_ROUGHNESS_DSET': '',
                'ROUGHNESS_CONSTANT': 0.025
            }
            self._flow = xr.Dataset(attrs=flow)

            sediment = {
                'CALCULATE_SEDIMENT': 0,
                'TRANSPORT_TIME_VALUE': 600.0,
                'TRANSPORT_TIME_UNITS': 'seconds',
                'MORPHOLOGIC_TIME_VALUE': 600.0,
                'MORPHOLOGIC_TIME_UNITS': 'seconds',
                'MORPHOLOGY_TIME_VALUE': 0.0,
                'MORPHOLOGY_TIME_UNITS': 'hours',
                'FORMULATION_UNITS': 'Nonequilibrium total load',
                'TRANSPORT_FORMULA': 'Lund-CIRP',
                'CONCENTRATION_PROFILE': 'Exponential',
                'WATANABE_RATE': 1.0,
                'C2SHORE_EFFICIENCY': 0.003,
                'C2SHORE_BED_LOAD': 0.002,
                'C2SHORE_SUSP_LOAD': 0.3,
                'SEDIMENT_DENSITY_VALUE': 2650.0,
                'SEDIMENT_DENSITY_UNITS': 'kg/m^3',
                'SEDIMENT_POROSITY': 0.4,
                'GRAIN_SIZE_VALUE': 0.2,
                'GRAIN_SIZE_UNITS': 'mm',
                'BED_LOAD_SCALING': 1.0,
                'SUSPENDED_LOAD_SCALING': 1.0,
                'MORPHOLOGIC_ACCELERATION': 1.0,
                'BED_SLOPE_DIFFUSION': 0.1,
                'HIDING_AND_EXPOSURE': 1.0,
                'TOTAL_ADAPTATION_METHOD': 'Constant length',
                'TOTAL_ADAPTATION_LENGTH_VALUE': 10.0,
                'TOTAL_ADAPTATION_LENGTH_UNITS': 'm',
                'TOTAL_ADAPTATION_TIME_VALUE': 5.0,
                'TOTAL_ADAPTATION_TIME_UNITS': 'seconds',
                'BED_ADAPTATION_METHOD': 'Constant length',
                'BED_ADAPTATION_LENGTH_VALUE': 10.0,
                'BED_ADAPTATION_LENGTH_UNITS': 'm',
                'BED_ADAPTATION_TIME_VALUE': 5.0,
                'BED_ADAPTATION_TIME_UNITS': 'seconds',
                'BED_ADAPTATION_DEPTH': 10.0,
                'SUSPENDED_ADAPTATION_METHOD': 'Constant length',
                'SUSPENDED_ADAPTATION_LENGTH_VALUE': 10.0,
                'SUSPENDED_ADAPTATION_LENGTH_UNITS': 'm',
                'SUSPENDED_ADAPTATION_TIME_VALUE': 5.0,
                'SUSPENDED_ADAPTATION_TIME_UNITS': 'seconds',
                'SUSPENDED_ADAPTATION_COEFFICIENT': 2.0,
                'USE_ADVANCED_SIZE_CLASSES': 0,
                'ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE': 0,
                'MULTIPLE_GRAIN_SIZES': 'Specify number of size classes only',
                'SIMPLE_MULTI_SIZE': 1,
                'SEDIMENT_STANDARD_DEVIATION': 1.5,
                'BED_COMPOSITION_INPUT': 'D50 Sigma',
                'MULTI_D16': '',
                'MULTI_D50': '',
                'MULTI_D84': '',
                'MULTI_D35': '',
                'MULTI_D90': '',
                'NUMBER_BED_LAYERS': 5,
                'THICKNESS_FOR_MIXING': 0.0,
                'THICKNESS_FOR_BED': 0.0,
                'MAX_NUMBER_BED_LAYERS': 10,
                'MIN_BED_LAYER_THICKNESS_VALUE': 0.05,
                'MIN_BED_LAYER_THICKNESS_UNITS': 'm',
                'MAX_BED_LAYER_THICKNESS_VALUE': 0.5,
                'MAX_BED_LAYER_THICKNESS_UNITS': 'm',
                'CALCULATE_AVALANCHING': 0,
                'CRITICAL_BED_SLOPE': 32.0,
                'MAX_NUMBER_ITERATIONS': 100,
                'USE_HARD_BOTTOM': 0,
                'HARD_BOTTOM': ''
            }
            self._sediment = xr.Dataset(attrs=sediment)

            advanced_sediment_diameters_table = {
                'diameter_value': xr.DataArray(data=np.array([], dtype=float)),
                'diameter_units': xr.DataArray(data=np.array([], dtype=object)),
                'fall_velocity_method': xr.DataArray(data=np.array([], dtype=object)),
                'fall_velocity_value': xr.DataArray(data=np.array([], dtype=float)),
                'fall_velocity_units': xr.DataArray(data=np.array([], dtype=object)),
                'corey_shape_factor': xr.DataArray(data=np.array([], dtype=float)),
                'critical_shear_method': xr.DataArray(data=np.array([], dtype=object)),
                'critical_shear_stress': xr.DataArray(data=np.array([], dtype=float))
            }
            self._advanced_sediment_diameters_table = xr.Dataset(data_vars=advanced_sediment_diameters_table)

            simple_grain_sizes_table = {'grain_size': xr.DataArray(data=np.array([], dtype=float))}
            self._simple_grain_sizes_table = xr.Dataset(data_vars=simple_grain_sizes_table)

            bed_layer_table = {
                'layer_id': xr.DataArray(data=np.array([1], dtype=int)),
                'layer_thickness_type': xr.DataArray(data=np.array(['Automatic'], dtype=object)),
                'layer_thickness': xr.DataArray(data=np.array([''], dtype=object)),
                'layer_thickness_const': xr.DataArray(data=np.array([0.5], dtype=float)),
                'd05': xr.DataArray(data=np.array([''], dtype=object)),
                'd10': xr.DataArray(data=np.array([''], dtype=object)),
                'd16': xr.DataArray(data=np.array([''], dtype=object)),
                'd20': xr.DataArray(data=np.array([''], dtype=object)),
                'd30': xr.DataArray(data=np.array([''], dtype=object)),
                'd35': xr.DataArray(data=np.array([''], dtype=object)),
                'd50': xr.DataArray(data=np.array([''], dtype=object)),
                'd65': xr.DataArray(data=np.array([''], dtype=object)),
                'd84': xr.DataArray(data=np.array([''], dtype=object)),
                'd90': xr.DataArray(data=np.array([''], dtype=object)),
                'd95': xr.DataArray(data=np.array([''], dtype=object))
            }
            self._bed_layer_table = xr.Dataset(data_vars=bed_layer_table)

            salinity = {
                'WATER_DENSITY': 1025.0,
                'WATER_TEMP': 15.0,
                'CALCULATE_SALINITY': 0,
                'SALINITY_TRANSPORT_RATE_VALUE': 60.0,
                'SALINITY_TRANSPORT_RATE_UNITS': 'seconds',
                'SALINITY_CONCENTRATION': 'Global concentration',
                'GLOBAL_CONCENTRATION': 0.0,
                'SALINITY_INITIAL_CONCENTRATION': '',
                'CALCULATE_TEMPERATURE': 0,
                'TEMPERATURE_TRANSPORT_RATE_VALUE': 60.0,
                'TEMPERATURE_TRANSPORT_RATE_UNITS': 'seconds',
                'INITIAL_TEMPERATURE_TYPE': 'Constant water temperature',
                'INITIAL_TEMPERATURE_DATASET': ''
            }
            self._salinity = xr.Dataset(attrs=salinity)

            atmospheric_table = {
                'time': xr.DataArray(data=np.array([], dtype=float)),
                'air_temp': xr.DataArray(data=np.array([], dtype=float)),
                'dewpoint': xr.DataArray(data=np.array([], dtype=float)),
                'cloud_cover': xr.DataArray(data=np.array([], dtype=float)),
                'solar_radiation': xr.DataArray(data=np.array([], dtype=float))
            }
            self._atmospheric_table = xr.Dataset(data_vars=atmospheric_table)

            wave = {
                'WAVE_INFO': 'None',
                'WAVE_HEIGHT': '',
                'PEAK_PERIOD': '',
                'MEAN_WAVE_DIR': '',
                'WAVE_BREAKING': '',
                'WAVE_RADIATION': '',
                'FILE_WAVE_SIM': '(none selected)',
                'STEERING_INTERVAL_TYPE': 'Constant',
                'STEERING_INTERVAL_CONST': 1.0,
                'WAVE_WATER_PREDICTOR': 'Tidal plus variation',
                'EXTRAPOLATION_DISTANCE': 1,
                'FLOW_TO_WAVE': 'Automatic',
                'FLOW_TO_WAVE_USER_VALUE': 0.0,
                'FLOW_TO_WAVE_USER_UNITS': 'm',
                'WAVE_TO_FLOW': 'Automatic',
                'WAVE_TO_FLOW_USER_VALUE': 0.0,
                'WAVE_TO_FLOW_USER_UNITS': 'm',
            }
            self._wave = xr.Dataset(attrs=wave)

            wind = {
                'WIND_TYPE': 'None',
                'ANEMOMETER': 10.0,
                'WIND_FILE_TYPE': 'Navy fleet numeric with pressure',
                'WIND_FILE': '(none selected)',
                'WIND_GRID_TYPE': 'Parameters',
                'WIND_GRID_FILE': '(none selected)',
                'WIND_GRID_NUM_X_VALUES': 1,
                'WIND_GRID_NUM_Y_VALUES': 1,
                'WIND_GRID_MIN_X_LOCATION': 0.0,
                'WIND_GRID_MAX_Y_LOCATION': 0.0,
                'WIND_GRID_TIME_INCREMENT': 0.0,
                'WIND_GRID_X_DISTANCE': 0.0,
                'WIND_GRID_Y_DISTANCE': 0.0,
                'OCEAN_WIND_FILE': '(none selected)',
                'OCEAN_PRESSURE_FILE': '(none selected)',
                'OCEAN_XY_FILE': '(none selected)'
            }
            self._wind = xr.Dataset(attrs=wind)

            meteorological_stations = {'NEXT_CURVE_ID': 0}

            meteorological_stations_table = {
                'name': xr.DataArray(data=np.array([], dtype=object)),
                'x': xr.DataArray(data=np.array([], dtype=float)),
                'y': xr.DataArray(data=np.array([], dtype=float)),
                'height': xr.DataArray(data=np.array([], dtype=float)),
                'direction': xr.DataArray(data=np.array([], dtype=int))
            }
            self._meteorological_stations_table = xr.Dataset(
                attrs=meteorological_stations, data_vars=meteorological_stations_table
            )

            wind_from_table = {
                'time': xr.DataArray(data=np.array([], dtype=float)),
                'direction': xr.DataArray(data=np.array([], dtype=float)),
                'velocity': xr.DataArray(data=np.array([], dtype=float))
            }
            self._wind_from_table = xr.Dataset(data_vars=wind_from_table)

            output = {
                'SIMULATION_LABEL': 'Simulation',
                'WSE_LIST': 'List 1',
                'CURRENT_VELOCITY_LIST': 'List 1',
                'CURRENT_MAGNITUDE': 0,
                'USE_MORPHOLOGY': 0,
                'MORPHOLOGY_LIST': 'List 1',
                'MORPHOLOGY_CHANGE': 0,
                'USE_TRANSPORT': 0,
                'TRANSPORT_LIST': 'List 1',
                'SEDIMENT_TOTAL_LOAD_CAPACITY': 0,
                'SEDIMENT_TOTAL_LOAD_CONCENTRATION': 0,
                'FRACTION_SUSPENDED': 0,
                'FRACTION_BEDLOAD': 0,  # Added new option to output fraction of bedload as well.
                'USE_WAVE': 0,
                'WAVE_LIST': 'List 1',
                'WAVE_DISSIPATION': 0,
                'USE_WIND': 0,
                'WIND_LIST': 'List 1',
                'WIND_SPEED': 0,
                'ATM_PRESSURE': 0,
                'USE_EDDY_VISCOSITY': 0,
                'EDDY_VISCOSITY_LIST': 'List 1',
                'ENABLE_STATISTICS': 0,
                'ENABLE_HYDRO_STATISTICS': 0,
                'HYDRO_START_TIME': 0.0,
                'HYDRO_INCREMENT': 1.0,
                'HYDRO_END_TIME': 720.0,
                'ENABLE_SEDIMENT_STATISTICS': 0,
                'SEDIMENT_START_TIME': 0.0,
                'SEDIMENT_INCREMENT': 1.0,
                'SEDIMENT_END_TIME': 720.0,
                'ENABLE_SALINITY_STATISTICS': 0,
                'SALINITY_START_TIME': 0.0,
                'SALINITY_INCREMENT': 1.0,
                'SALINITY_END_TIME': 720.0,
                'ENABLE_WAVE_STATISTICS': 0,
                'WAVE_START_TIME': 0.0,
                'WAVE_INCREMENT': 1.0,
                'WAVE_END_TIME': 720.0,
                'SOLUTION_OUTPUT': 'XMDF binary output',
                'XMDF_COMPRESSION': 0,
                'SINGLE_SOLUTION': 0,
                'WRITE_ASCII': 0,
                'TECPLOT': 0,
            }
            self._output = xr.Dataset(attrs=output)

            list_1_table = {
                'start_time': xr.DataArray(data=np.array([0.0], dtype=float)),
                'increment': xr.DataArray(data=np.array([1.0], dtype=float)),
                'end_time': xr.DataArray(data=np.array([720.0], dtype=float))
            }
            self._list_1_table = xr.Dataset(data_vars=list_1_table)

            list_table = {
                'start_time': xr.DataArray(data=np.array([], dtype=float)),
                'increment': xr.DataArray(data=np.array([], dtype=float)),
                'end_time': xr.DataArray(data=np.array([], dtype=float))
            }
            self._list_2_table = xr.Dataset(data_vars=list_table)
            self._list_3_table = xr.Dataset(data_vars=list_table)
            self._list_4_table = xr.Dataset(data_vars=list_table)

            dredge = {
                'ENABLE_DREDGE': 0,
                'DREDGE_NAME': '',
                'UPDATE_INTERVAL_VALUE': 0.0,
                'UPDATE_INTERVAL_UNITS': 'seconds',
                'DREDGE_DATASET': '',
                'DREDGE_METHOD': 'Shallowest cell',
                'SPECIFIED_CELL': 0,
                'DREDGE_RATE_VALUE': 0.0,
                'DREDGE_RATE_UNITS': 'm^3/day',
                'TRIGGER_METHOD': 'Depth',
                'TRIGGER_DEPTH_VALUE': 0.0,
                'TRIGGER_DEPTH_UNITS': 'm',
                'TRIGGER_VOLUME_VALUE': 0.0,
                'TRIGGER_VOLUME_UNITS': 'm^3',
                'TRIGGER_PERCENT': 0.0,
                'TRIGGER_PERCENT_DEPTH_VALUE': 0.0,
                'TRIGGER_PERCENT_DEPTH_UNITS': 'm',
                'DISTRIBUTION': 'Sequential',
                'ENABLE_DIAGNOSTIC': 0
            }
            self._dredge = xr.Dataset(attrs=dredge)
            self.set_dredge_time_periods([0.0], [0.0])

            placement = {
                'DEFINE_PLACEMENT_1': 1,
                'PLACEMENT_1_DATASET': '',
                'PLACEMENT_1_METHOD': 'Uniform',
                'PLACEMENT_1_METHOD_CELL': 0,
                'PLACEMENT_1_PERCENTAGE': 100.0,
                'PLACEMENT_1_LIMIT_METHOD': 'Depth',
                'PLACEMENT_1_LIMIT_DEPTH_VALUE': 0.0,
                'PLACEMENT_1_LIMIT_DEPTH_UNITS': 'm',
                'PLACEMENT_1_LIMIT_THICKNESS_VALUE': 0.0,
                'PLACEMENT_1_LIMIT_THICKNESS_UNITS': 'm',
                'DEFINE_PLACEMENT_2': 0,
                'PLACEMENT_2_DATASET': '',
                'PLACEMENT_2_METHOD': 'Uniform',
                'PLACEMENT_2_METHOD_CELL': 0,
                'PLACEMENT_2_PERCENTAGE': 0.0,
                'PLACEMENT_2_LIMIT_METHOD': 'Depth',
                'PLACEMENT_2_LIMIT_DEPTH_VALUE': 0.0,
                'PLACEMENT_2_LIMIT_DEPTH_UNITS': 'm',
                'PLACEMENT_2_LIMIT_THICKNESS_VALUE': 0.0,
                'PLACEMENT_2_LIMIT_THICKNESS_UNITS': 'm',
                'DEFINE_PLACEMENT_3': 0,
                'PLACEMENT_3_DATASET': '',
                'PLACEMENT_3_METHOD': 'Uniform',
                'PLACEMENT_3_METHOD_CELL': 0,
                'PLACEMENT_3_PERCENTAGE': 0.0,
                'PLACEMENT_3_LIMIT_METHOD': 'Depth',
                'PLACEMENT_3_LIMIT_DEPTH_VALUE': 0.0,
                'PLACEMENT_3_LIMIT_DEPTH_UNITS': 'm',
                'PLACEMENT_3_LIMIT_THICKNESS_VALUE': 0.0,
                'PLACEMENT_3_LIMIT_THICKNESS_UNITS': 'm',
            }
            self._dredge_placement = xr.Dataset(attrs=placement)

            self.set_advanced_card_table([], [], [])
            self._get_default_advanced_block_table()

            if not os.path.exists(data_file):
                self.commit()

    def _check_for_simple_migrations(self):
        """Check for easy fixes we can make to the file to avoid doing it during DMI project migration."""
        commit = False

        version = self.info.attrs.get('adcirc_file_version', 0)
        self.info.attrs['adcirc_file_version'] = 1
        if version == 0:
            proj_dir = self.info.attrs['proj_dir']
            _make_absolute(proj_dir, self.general.attrs, 'INIT_CONDITIONS_FILE')
            _make_absolute(proj_dir, self.wave.attrs, 'FILE_WAVE_SIM')
            _make_absolute(proj_dir, self.wind.attrs, 'WIND_FILE')
            _make_absolute(proj_dir, self.wind.attrs, 'WIND_GRID_FILE')
            _make_absolute(proj_dir, self.wind.attrs, 'OCEAN_WIND_FILE')
            _make_absolute(proj_dir, self.wind.attrs, 'OCEAN_PRESSURE_FILE')
            _make_absolute(proj_dir, self.wind.attrs, 'OCEAN_XY_FILE')
            commit = True

        if 'ATM_PRESSURE' not in self.output.attrs:
            self.output.attrs['ATM_PRESSURE'] = 0
            commit = True
        if 'C2SHORE_EFFICIENCY' not in self.sediment.attrs:
            # If efficiency is not there, then the other two will not be either. MEB  05/03/2022
            self.sediment.attrs['C2SHORE_EFFICIENCY'] = 0.003
            self.sediment.attrs['C2SHORE_BED_LOAD'] = 0.002
            self.sediment.attrs['C2SHORE_SUSP_LOAD'] = 0.3
            commit = True

        if 'MULTI_D35' not in self.sediment.attrs:
            # If MULTI_D35 is not there, then the other one will not be either. MEB  10/04/2022
            self.sediment.attrs['MULTI_D35'] = ''
            self.sediment.attrs['MULTI_D90'] = ''
            commit = True

        # No more D50_SIGMA dataset. Use as D50 dataset if we have a D50_SIGMA dataset but no D50 dataset.
        if not self.sediment.attrs['MULTI_D50'] and self.sediment.attrs.get('D50_SIGMA'):
            self.sediment.attrs['MULTI_D50'] = self.sediment.attrs['D50_SIGMA']
            commit = True

        # Default old projects that do not have this attribute.
        if 'FRACTION_BEDLOAD' not in self.output.attrs:
            self.output.attrs['FRACTION_BEDLOAD'] = 0
            commit = True

        if 'ROUGHNESS_SOURCE' not in self.flow.attrs:
            self.flow.attrs['ROUGHNESS_SOURCE'] = 'Dataset'
            self.flow.attrs['ROUGHNESS_CONSTANT'] = 0.025
            commit = True

        if commit:
            self.commit()

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._general is not None:
            self._general.close()
            self._general.to_netcdf(self._filename, group='general', mode='a')
        if self._flow is not None:
            self._flow.close()
            self._flow.to_netcdf(self._filename, group='flow', mode='a')
        if self._sediment is not None:
            self._sediment.close()
            self._sediment.to_netcdf(self._filename, group='sediment', mode='a')
        if self._salinity is not None:
            self._salinity.close()
            self._salinity.to_netcdf(self._filename, group='salinity', mode='a')
        if self._wave is not None:
            self._wave.close()
            self._wave.to_netcdf(self._filename, group='wave', mode='a')
        if self._wind is not None:
            self._wind.close()
            self._wind.to_netcdf(self._filename, group='wind', mode='a')
        if self._output is not None:
            self._output.close()
            self._output.to_netcdf(self._filename, group='output', mode='a')
        if self._advanced_sediment_diameters_table is not None:
            self._advanced_sediment_diameters_table.close()
            self._drop_h5_groups(['advanced_sediment_diameters_table'])
            self._advanced_sediment_diameters_table.to_netcdf(
                self._filename, group='advanced_sediment_diameters_table', mode='a'
            )
        if self._simple_grain_sizes_table is not None:
            self._simple_grain_sizes_table.close()
            self._drop_h5_groups(['simple_grain_sizes_table'])
            self._simple_grain_sizes_table.to_netcdf(self._filename, group='simple_grain_sizes_table', mode='a')
        if self._bed_layer_table is not None:
            self._bed_layer_table.close()
            self._drop_h5_groups(['bed_layer_table'])
            self._bed_layer_table.to_netcdf(self._filename, group='bed_layer_table', mode='a')
        if self._atmospheric_table is not None:
            self._atmospheric_table.close()
            self._drop_h5_groups(['atmospheric_table'])
            self._atmospheric_table.to_netcdf(self._filename, group='atmospheric_table', mode='a')
        if self._meteorological_stations_table is not None:
            self._meteorological_stations_table.close()
            self._drop_h5_groups(['meteorological_stations_table'])
            self._meteorological_stations_table.to_netcdf(
                self._filename, group='meteorological_stations_table', mode='a'
            )
        # write the meteorological station direction curves
        for curve_id, data in self._meteorological_stations_direction_curves.items():
            grp = self._meteorological_stations_direction_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        if self._wind_from_table is not None:
            self._wind_from_table.close()
            self._drop_h5_groups(['wind_from_table'])
            self._wind_from_table.to_netcdf(self._filename, group='wind_from_table', mode='a')
        if self._list_1_table is not None:
            self._list_1_table.close()
            self._drop_h5_groups(['list_1_table'])
            self._list_1_table.to_netcdf(self._filename, group='list_1_table', mode='a')
        if self._list_2_table is not None:
            self._list_2_table.close()
            self._drop_h5_groups(['list_2_table'])
            self._list_2_table.to_netcdf(self._filename, group='list_2_table', mode='a')
        if self._list_3_table is not None:
            self._list_3_table.close()
            self._drop_h5_groups(['list_3_table'])
            self._list_3_table.to_netcdf(self._filename, group='list_3_table', mode='a')
        if self._list_4_table is not None:
            self._list_4_table.close()
            self._drop_h5_groups(['list_4_table'])
            self._list_4_table.to_netcdf(self._filename, group='list_4_table', mode='a')
        if self._dredge is not None:
            self._dredge.close()
            self._drop_h5_groups(['dredge'])
            if self._dredge.DREDGE_DATASET is not None:
                self._dredge.to_netcdf(self._filename, group='dredge', mode='a')
        if self._dredge_placement is not None:
            self._dredge_placement.close()
            self._drop_h5_groups(['dredge_placement'])
            self._dredge_placement.to_netcdf(self._filename, group='dredge_placement', mode='a')
        if self._dredge_time_periods is not None:
            self._dredge_time_periods.close()
            self._drop_h5_groups(['dredge_time_periods'])
            self._dredge_time_periods.to_netcdf(self._filename, group='dredge_time_periods', mode='a')
        if self._advanced_card_table is not None:
            self._advanced_card_table.close()
            self._drop_h5_groups(['advanced_card_table'])
            self._advanced_card_table.to_netcdf(self._filename, group='advanced_card_table', mode='a')
        if self._advanced_block_table is not None:
            self._advanced_block_table.close()
            self._drop_h5_groups(['advanced_block_table'])
            self._advanced_block_table.to_netcdf(self._filename, group='advanced_block_table', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All sim datasets that need to be written to the file must be loaded into memory before calling this method.

        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._general is None:
            self._general = self.get_dataset('general', False)
        if self._flow is None:
            self._flow = self.get_dataset('flow', False)
        if self._sediment is None:
            self._sediment = self.get_dataset('sediment', False)
        if self._salinity is None:
            self._salinity = self.get_dataset('salinity', False)
        if self._wave is None:
            self._wave = self.get_dataset('wave', False)
        if self._wind is None:
            self._wind = self.get_dataset('wind', False)
        if self._output is None:
            self._output = self.get_dataset('output', False)
        if self._advanced_sediment_diameters_table is None:
            self._advanced_sediment_diameters_table = self.get_dataset('advanced_sediment_diameters_table', False)
        if self._simple_grain_sizes_table is None:
            self._simple_grain_sizes_table = self.get_dataset('simple_grain_sizes_table', False)
        if self._bed_layer_table is None:
            self._bed_layer_table = self.get_dataset('bed_layer_table', False)
        if self._atmospheric_table is None:
            self._atmospheric_table = self.get_dataset('atmospheric_table', False)
        if self._meteorological_stations_table is None:
            self._meteorological_stations_table = self.get_dataset('meteorological_stations_table', False)
        if self._meteorological_stations_table is not None:
            for direction in self._meteorological_stations_table['direction']:
                direction = int(direction)
                grp = self._meteorological_stations_direction_curve_group_name(direction)
                if direction not in self._meteorological_stations_direction_curves or\
                        self._meteorological_stations_direction_curves[direction] is None:
                    self._meteorological_stations_direction_curves[direction] = self.get_dataset(grp, False)
        if self._wind_from_table is None:
            self._wind_from_table = self.get_dataset('wind_from_table', False)
        if self._list_1_table is None:
            self._list_1_table = self.get_dataset('list_1_table', False)
        if self._list_2_table is None:
            self._list_2_table = self.get_dataset('list_2_table', False)
        if self._list_3_table is None:
            self._list_3_table = self.get_dataset('list_3_table', False)
        if self._list_4_table is None:
            self._list_4_table = self.get_dataset('list_4_table', False)
        if self._dredge is None:
            self._dredge = self.get_dataset('dredge', False)
        if self._dredge_placement is None:
            self._dredge_placement = self.get_dataset('dredge_placement', False)
        if self._dredge_time_periods is None:
            self._dredge_time_periods = self.get_dataset('dredge_time_periods', False)
        if self._advanced_card_table is None:
            self._advanced_card_table = self.get_dataset('advanced_card_table', False)
        if self._advanced_block_table is None:
            self._advanced_block_table = self.get_dataset('advanced_block_table', False)
        try:
            os.remove(self._filename)
        except Exception:
            pass
        self.commit()  # Rewrite all datasets

    @property
    def simulation_start(self) -> Optional[datetime]:
        """The date and time when the simulation starts, or None if it was not set."""
        try:
            start_time = datetime.fromisoformat(self.general.attrs['DATE_START'])
            return start_time
        except ValueError:
            return None

    @property
    def simulation_end(self) -> Optional[datetime]:
        """The date and time when the simulation ends, or None if the start time was not set."""
        start = self.simulation_start
        if start is None:
            return None

        try:
            duration = self.general.attrs['SIM_DURATION_VALUE']
            units = self.general.attrs['SIM_DURATION_UNITS']
            kwargs = {units: duration}
            delta = timedelta(**kwargs)
            end_time = start + delta
            return end_time
        except ValueError:
            return None

    @property
    def general(self):
        """Load the general dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the general datasets in the main file

        """
        if self._general is None:
            self._general = self.get_dataset('general', False)
        return self._general

    @general.setter
    def general(self, dset):
        """Setter for the general attribute."""
        if dset:
            self._general = dset

    @property
    def flow(self):
        """Load the flow dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the flow datasets in the main file

        """
        if self._flow is None:
            self._flow = self.get_dataset('flow', False)
        return self._flow

    @flow.setter
    def flow(self, dset):
        """Setter for the flow attribute."""
        if dset:
            self._flow = dset

    @property
    def sediment(self):
        """Load the sediment dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the sediment datasets in the main file

        """
        if self._sediment is None:
            self._sediment = self.get_dataset('sediment', False)
        return self._sediment

    @sediment.setter
    def sediment(self, dset):
        """Setter for the sediment attribute."""
        if dset:
            self._sediment = dset

    @property
    def salinity(self):
        """Load the salinity dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the salinity datasets in the main file

        """
        if self._salinity is None:
            self._salinity = self.get_dataset('salinity', False)
        return self._salinity

    @salinity.setter
    def salinity(self, dset):
        """Setter for the salinity attribute."""
        if dset:
            self._salinity = dset

    @property
    def wave(self):
        """Load the wave dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the wave datasets in the main file

        """
        if self._wave is None:
            self._wave = self.get_dataset('wave', False)
        return self._wave

    @wave.setter
    def wave(self, dset):
        """Setter for the wave attribute."""
        if dset:
            self._wave = dset

    @property
    def wind(self):
        """Load the wind dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the wind datasets in the main file

        """
        if self._wind is None:
            self._wind = self.get_dataset('wind', False)
        return self._wind

    @wind.setter
    def wind(self, dset):
        """Setter for the wind attribute."""
        if dset:
            self._wind = dset

    @property
    def output(self):
        """Load the output dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the output datasets in the main file

        """
        if self._output is None:
            self._output = self.get_dataset('output', False)
        return self._output

    @output.setter
    def output(self, dset):
        """Setter for the output attribute."""
        if dset:
            self._output = dset

    @property
    def advanced_sediment_diameters_table(self):
        """Load the advanced_sediment_diameters_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the advanced_sediment_diameters_table datasets in the main file

        """
        if self._advanced_sediment_diameters_table is None:
            self._advanced_sediment_diameters_table = self.get_dataset('advanced_sediment_diameters_table', False)
        return self._advanced_sediment_diameters_table

    @advanced_sediment_diameters_table.setter
    def advanced_sediment_diameters_table(self, dset):
        """Setter for the advanced_sediment_diameters_table attribute."""
        if dset:
            self._advanced_sediment_diameters_table = dset

    @property
    def simple_grain_sizes_table(self):
        """Load the simple_grain_sizes_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the simple_grain_sizes_table datasets in the main file

        """
        if self._simple_grain_sizes_table is None:
            self._simple_grain_sizes_table = self.get_dataset('simple_grain_sizes_table', False)
        return self._simple_grain_sizes_table

    @simple_grain_sizes_table.setter
    def simple_grain_sizes_table(self, dset):
        """Setter for the simple_grain_sizes_table attribute."""
        if dset:
            self._simple_grain_sizes_table = dset

    @property
    def bed_layer_table(self):
        """Load the bed_layer_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the bed_layer_table datasets in the main file

        """
        if self._bed_layer_table is None:
            self._bed_layer_table = self.get_dataset('bed_layer_table', False)
        return self._bed_layer_table

    @bed_layer_table.setter
    def bed_layer_table(self, dset):
        """Setter for the bed_layer_table attribute."""
        if dset:
            self._bed_layer_table = dset

    @property
    def atmospheric_table(self):
        """Load the atmospheric_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the atmospheric_table datasets in the main file

        """
        if self._atmospheric_table is None:
            self._atmospheric_table = self.get_dataset('atmospheric_table', False)
        return self._atmospheric_table

    @atmospheric_table.setter
    def atmospheric_table(self, dset):
        """Setter for the atmospheric_table attribute."""
        if dset:
            self._atmospheric_table = dset

    @property
    def meteorological_stations_table(self):
        """Load the meteorological_stations_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the meteorological_stations_table datasets in the main file

        """
        if self._meteorological_stations_table is None:
            self._meteorological_stations_table = self.get_dataset('meteorological_stations_table', False)
        return self._meteorological_stations_table

    @meteorological_stations_table.setter
    def meteorological_stations_table(self, dset):
        """Setter for the meteorological_stations_table attribute."""
        if dset:
            self._meteorological_stations_table = dset

    @property
    def wind_from_table(self):
        """Load the wind_from_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the wind_from_table datasets in the main file

        """
        if self._wind_from_table is None:
            self._wind_from_table = self.get_dataset('wind_from_table', False)
        return self._wind_from_table

    @wind_from_table.setter
    def wind_from_table(self, dset):
        """Setter for the wind_from_table attribute."""
        if dset:
            self._wind_from_table = dset

    @property
    def list_1_table(self):
        """Load the list_1_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the list_1_table datasets in the main file

        """
        if self._list_1_table is None:
            self._list_1_table = self.get_dataset('list_1_table', False)
        return self._list_1_table

    @list_1_table.setter
    def list_1_table(self, dset):
        """Setter for the list_1_table attribute."""
        if dset:
            self._list_1_table = dset

    @property
    def list_2_table(self):
        """Load the list_2_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the list_2_table datasets in the main file

        """
        if self._list_2_table is None:
            self._list_2_table = self.get_dataset('list_2_table', False)
        return self._list_2_table

    @list_2_table.setter
    def list_2_table(self, dset):
        """Setter for the list_2_table attribute."""
        if dset:
            self._list_2_table = dset

    @property
    def list_3_table(self):
        """Load the list_3_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the list_3_table datasets in the main file

        """
        if self._list_3_table is None:
            self._list_3_table = self.get_dataset('list_3_table', False)
        return self._list_3_table

    @list_3_table.setter
    def list_3_table(self, dset):
        """Setter for the list_3_table attribute."""
        if dset:
            self._list_3_table = dset

    @property
    def list_4_table(self):
        """Load the list_4_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the list_4_table datasets in the main file

        """
        if self._list_4_table is None:
            self._list_4_table = self.get_dataset('list_4_table', False)
        return self._list_4_table

    @list_4_table.setter
    def list_4_table(self, dset):
        """Setter for the list_4_table attribute."""
        if dset:
            self._list_4_table = dset

    @property
    def dredge(self):
        """Load the dredge dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the dredge datasets in the main file

        """
        if self._dredge is None:
            self._dredge = self.get_dataset('dredge', False)
        return self._dredge

    @dredge.setter
    def dredge(self, dset):
        """Setter for the dredge attribute."""
        if dset:
            self._dredge = dset

    @property
    def dredge_placement(self):
        """Load the dredge_placement dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the dredge_placement datasets in the main file

        """
        if self._dredge_placement is None:
            self._dredge_placement = self.get_dataset('dredge_placement', False)
        return self._dredge_placement

    @dredge_placement.setter
    def dredge_placement(self, dset):
        """Setter for the dredge_placement attribute."""
        if dset:
            self._dredge_placement = dset

    @property
    def dredge_time_periods(self):
        """Load the dredge_time_periods dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the dredge_time_periods datasets in the main file

        """
        if self._dredge_time_periods is None:
            self._dredge_time_periods = self.get_dataset('dredge_time_periods', False)
        return self._dredge_time_periods

    def set_dredge_time_periods(self, start, finish):
        """Sets the dredge time period dataset.

        Args:
            start (list): A list of floating point numbers representing dredge start times.
            finish (list): A list of floating point numbers representing dredge end times.
        """
        dredge_time_periods_table = {
            'start': xr.DataArray(data=np.array(start, dtype=float)),
            'finish': xr.DataArray(data=np.array(finish, dtype=float))
        }
        self._dredge_time_periods = xr.Dataset(data_vars=dredge_time_periods_table)

    @dredge_time_periods.setter
    def dredge_time_periods(self, dset):
        """Setter for the dredge_time_periods attribute."""
        if dset:
            self._dredge_time_periods = dset

    @property
    def advanced_card_table(self):
        """Load the advanced_card_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the advanced_card_table datasets in the main file

        """
        if self._advanced_card_table is None:
            self._advanced_card_table = self.get_dataset('advanced_card_table', False)
        if self._advanced_card_table is None:
            self.set_advanced_card_table([], [], [])
        return self._advanced_card_table

    @advanced_card_table.setter
    def advanced_card_table(self, dset):
        """Setter for the advanced_card_table attribute."""
        if dset:
            self._advanced_card_table = dset

    def set_advanced_card_table(self, block, card, value):
        """Sets the advanced card table.

        Args:
            block (list): A list of strings designating the card block the card belongs to.
            card (list): A list of strings of cards.
            value (kist): A list of string values for the rest of the card.
        """
        advanced_card_table = {
            'Block': xr.DataArray(data=np.array(block, dtype=str)),
            'Card': xr.DataArray(data=np.array(card, dtype=str)),
            'Value': xr.DataArray(data=np.array(value, dtype=str))
        }
        self._advanced_card_table = xr.Dataset(data_vars=advanced_card_table)

    @property
    def advanced_block_table(self):
        """Load the advanced_block_table dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the advanced_block_table datasets in the main file

        """
        if self._advanced_block_table is None:
            self._advanced_block_table = self.get_dataset('advanced_block_table', False)
        if self._advanced_block_table is None:
            self._get_default_advanced_block_table()
        return self._advanced_block_table

    @advanced_block_table.setter
    def advanced_block_table(self, dset):
        """Setter for the advanced_block_table attribute."""
        if dset:
            self._advanced_block_table = dset

    def set_direction_curve_from_meteorological_station(self, curve_id, curve):
        """Sets the depth curve by material id.

        Args:
            curve_id (int): The curve id.
            curve (xarray.Dataset): The direction curve
        """
        self._meteorological_stations_direction_curves[curve_id] = curve

    def direction_curve_from_meteorological_station(self, curve_id):
        """Gets the meteorological station direction curve from the curve id.

        Args:
            curve_id (int): curve id

        Returns:
            xarray.Dataset: The meteorological station direction list dataset
        """
        if curve_id not in self._meteorological_stations_direction_curves:
            # load from the file if the curve exists
            grp = self._meteorological_stations_direction_curve_group_name(curve_id)
            self._meteorological_stations_direction_curves[curve_id] = self.get_dataset(grp, False)
            if self._meteorological_stations_direction_curves[curve_id] is None:
                # create a default curve
                self._meteorological_stations_direction_curves[curve_id] =\
                    self._default_meteorological_stations_direction_curve()
        return self._meteorological_stations_direction_curves[curve_id]

    def _get_default_advanced_block_table(self):
        """Initialize the advanced block table with defaults.
        """
        advanced_block_table = {
            'Block Name': xr.DataArray(data=np.array([], dtype=str)),
            'Block Start': xr.DataArray(data=np.array([], dtype=str)),
            'Block End': xr.DataArray(data=np.array([], dtype=str))
        }
        self._advanced_block_table = xr.Dataset(data_vars=advanced_block_table)

    @staticmethod
    def _meteorological_stations_direction_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'met_station_direction_curves/{curve_id}'

    @staticmethod
    def _default_meteorological_stations_direction_curve():
        """Creates a xarray.Dataset for a meteorological station direction curve."""
        default_data = {
            'time': xr.DataArray(data=np.array([0.0], dtype=float)),
            'direction': xr.DataArray(data=np.array([0.0], dtype=float)),
            'velocity': xr.DataArray(data=np.array([0.0], dtype=float)),
        }
        return xr.Dataset(data_vars=default_data)


def _make_absolute(base, container, key):
    value = container[key]
    if value == '' or value == '(none selected)' or not base:
        return
    if not os.path.isabs(value):
        value = os.path.join(base, value)

    value = os.path.normpath(value)
    container[key] = value
