"""This module creates a thread for work to be done on and a progress dialog."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.cmsflow._cmsflow_cmcards_import import CmCardsImporter
from xms.cmsflow.feedback.xmlog import XmLog


class CmcardsImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing CMS-Flow cmcard file."""
    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.reader = None

    def _do_work(self):
        """Thread runner that imports the cmcards."""
        try:
            self.reader = CmCardsImporter()
            self.reader.read()
        except Exception:
            XmLog().instance.exception('Errors encountered importing the CMS-Flow simulation.')


def import_simulation_with_feedback():
    """Driver for the cmcards import script with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    # Don't want debug info about Python invocation in feedback dialog
    XmLog().instance

    worker = CmcardsImportWorkerThread(parent)
    display_text = {
        'title': 'Reading CMS-Flow Simulation',
        'working_prompt': 'Reading CMS-Flow cmcards file. Please wait...',
        'error_prompt': 'Error(s) encountered reading cmcards file. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading cmcards file. Review log output for more details.',
        'success_prompt': 'Successfully read CMS-Flow simulation.',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    if feedback_dlg.exec():
        try:
            worker.reader.send_sms_data()  # Send data back to XMS
        except Exception:
            msg = 'Unknown error sending imported data. The imported simulation and data will not be loaded into SMS.'
            message_with_ok(
                parent=parent, message=msg, app_name='SMS', icon='Critical', win_icon=QIcon(xms_dlg.get_xms_icon())
            )
            XmLog().instance.exception(msg)
    if parent is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
