"""This is a dialog for specifying boundary conditions."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import uuid
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel, QWidget
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment
from xms.api.tree import TreeNode
from xms.api.tree.tree_util import build_tree_path, find_tree_node_by_uuid, ProjectExplorerTreeCreator
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.time_format import datetime_to_qdatetime, datetime_to_string, ISO_DATETIME_FORMAT, qdatetime_to_datetime
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.cmsflow.data.bc_data import BCData
from xms.cmsflow.dmi.xms_data import XmsData
from xms.cmsflow.gui.boundary_condition_dlg_ui import Ui_BoundaryConditionDlg
from xms.cmsflow.gui.file_selector import FileSelector
from xms.cmsflow.gui.harmonic_table_widget import HarmonicTableWidget
from xms.cmsflow.gui.text_converter import TextConverter
from xms.cmsflow.gui.tidal_table_widget import TidalTableWidget


class BoundaryConditionDlg(XmsDlg):
    """A dialog for viewing boundary conditions data for an arc."""
    def __init__(
        self,
        bc_data: xr.Dataset,
        all_data: BCData,
        comp_id: int,
        pe_tree: TreeNode,
        xms_data: XmsData,
        multi_msg: str,
        parent: QWidget = None
    ):
        """Initializes the class, sets up the ui.

        Args:
            bc_data: The BC data for the arc to view
            all_data: The BC data for all arcs. This includes all curves.
            comp_id: The component id for this arc.
            pe_tree: The project explorer for selecting datasets
            xms_data: Interprocess communication object.
            multi_msg: Warning/multi-select message, if any
            parent: The parent window
        """
        super().__init__(parent, 'xms.cmsflow.gui.boundary_condition_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/CMS-Flow/Boundary_Conditions'
        self.ui = Ui_BoundaryConditionDlg()
        self.ui.setupUi(self)
        self.bc_data = bc_data
        self.all_data = all_data
        self.pe_tree = pe_tree
        self.petc = ProjectExplorerTreeCreator()
        self.multi_msg = multi_msg

        self.bc_types = ['Unassigned', 'Flow rate-forcing', 'WSE-forcing']
        self.flow_sources = ['Constant', 'Curve']
        self.wse_sources = [
            'Constant', 'Curve', 'Parent CMS-Flow', 'Parent ADCIRC', 'Extracted', 'Harmonic', 'Tidal constituent',
            'External tidal'
        ]
        self.adcirc_sol_types = ['ASCII', 'XMDF']
        self.wse_offset_types = ['Constant', 'Curve']

        self.flow_forcing_curve_id = 0
        self.wse_forcing_curve_id = 0
        self.wse_offset_curve_id = 0
        self.salinity_curve_id = 0
        self.temperature_curve_id = 0
        self.comp_id = comp_id
        self._xms_data = xms_data

        self.dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_dir_validator = QxDoubleValidator(parent=self)
        self.dbl_conveyance_validator = QxDoubleValidator(parent=self)

        # Set the project directory (if there is one) for resolving relative paths
        FileSelector.project_folder = XmsEnvironment.xms_environ_project_path()

        self._add_multi_select_warning()
        self._add_tables()
        self._setup_combo_boxes()
        self._setup_edit_fields()
        self._connect_widgets()
        self._load_data()
        self.adjustSize()

    def _make_file_selector_slot(self, label_widget: QWidget, caption: str, file_filter: str):
        """Make a file selector slot."""
        def helper():
            FileSelector.select_file(self, label_widget, caption, file_filter)

        return helper

    def _add_multi_select_warning(self):
        """Add a label for the multi-select warning, if applicable."""
        if self.multi_msg:
            label = QLabel(self.multi_msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.layout().insertWidget(0, label)

    def _add_tables(self):
        """Adds tables to dialog for harmonic and tidal WSE types."""
        if not self.bc_data.name.size == 0:
            table_id = int(self.bc_data['harmonic_table'].item())
            if table_id <= 0:
                table_id = self.comp_id
                self.bc_data['harmonic_table'] = table_id
            harmonic_table = self.all_data.harmonic_table_from_id(table_id)
            self.harmonic_table = HarmonicTableWidget(self.ui.harmonic_page, harmonic_table.to_dataframe())
            self.harmonic_table.setObjectName('harmonic_table')
            self.harmonic_table.setMinimumHeight(205)  # min height from the XML
            self.ui.harmonic_page.layout().insertWidget(0, self.harmonic_table)

            table_id = int(self.bc_data['tidal_table'].item())
            if table_id <= 0:
                table_id = self.comp_id
                self.bc_data['tidal_table'] = table_id
            tidal_table = self.all_data.tidal_table_from_id(table_id)
            self.tidal_table = TidalTableWidget(self.ui.tidal_page, tidal_table.to_dataframe())
            self.tidal_table.setObjectName('tidal_table')
            self.tidal_table.setMinimumHeight(205)  # min height from the XML
            self.ui.tidal_page.layout().insertWidget(0, self.tidal_table)

    def _connect_widgets(self):
        """Sets up the signal/slot connections for the dialog."""
        self.ui.flow_forcing.pressed.connect(self._edit_flow_forcing_curve)
        self.ui.wse_forcing.pressed.connect(self._edit_wse_forcing_curve)
        self.ui.wse_offset_curve.pressed.connect(self._edit_wse_offset_curve)
        self.ui.btn_salinity_curve.pressed.connect(self._edit_salinity_curve)
        self.ui.btn_temperature_curve.pressed.connect(self._edit_temperature_curve)
        slot = self._make_file_selector_slot(
            self.ui.cmcards_file_path_label, 'Select parent .cmcards file',
            'CMS-Flow Parent (*.cmcards);;All Files (*.*)'
        )
        self.ui.parent_cmsflow.pressed.connect(slot)
        slot = self._make_file_selector_slot(
            self.ui.fort_14_path_label, 'Select parent grid(fort.14) file', 'ADCIRC Parent (*.14);;All Files (*.*)'
        )
        self.ui.parent_adcirc_14.pressed.connect(slot)
        slot = self._make_file_selector_slot(
            self.ui.fort_63_path_label, 'Select parent grid(fort.63) file', 'Parent Solution (*.63);;All Files (*.*)'
        )
        self.ui.adcirc_63.pressed.connect(slot)
        slot = self._make_file_selector_slot(
            self.ui.fort_64_path_label, 'Select parent grid(fort.64) file', 'Parent Solution (*.64);;All Files (*.*)'
        )
        self.ui.adcirc_64.pressed.connect(slot)
        self.ui.parent_adcirc_14_xmdf.pressed.connect(self._on_adcirc_parent_14_xmdf_clicked)
        self.ui.adcirc_solution_wse.pressed.connect(self._on_adcirc_solution_clicked)
        self.ui.adcirc_solution.pressed.connect(self._on_adcirc_velocity_solution_clicked)
        self.ui.bc_type.currentIndexChanged[int].connect(self.ui.bc_type_stack.setCurrentIndex)
        self.ui.flow_source.currentIndexChanged[int].connect(self.ui.flow_type_stack.setCurrentIndex)
        self.ui.wse_source.currentIndexChanged[int].connect(self.ui.wse_source_stack.setCurrentIndex)
        self.ui.adcirc_solution_type.currentIndexChanged[int].connect(
            self.ui.adcirc_solution_type_stack.setCurrentIndex
        )
        self.ui.wse_offset_type.currentIndexChanged[int].connect(self.ui.wse_offset_stack.setCurrentIndex)
        self.ui.tog_salinity.stateChanged[int].connect(self.ui.btn_salinity_curve.setEnabled)
        self.ui.tog_temperature.stateChanged[int].connect(self.ui.btn_temperature_curve.setEnabled)
        self.ui.tog_flux_direction.stateChanged[int].connect(self.ui.flow_direction.setEnabled)
        self.ui.tog_flux_direction.stateChanged[int].connect(self.ui.flow_direction_label.setEnabled)
        self.ui.use_velocity.stateChanged[int].connect(self._on_use_velocity_changed)
        self.ui.btn_box.helpRequested.connect(self.help_requested)
        self.ui.extracted_dataset_parent_button.clicked.connect(self._on_extracted_dataset_parent_clicked)
        self.ui.extracted_dataset_elevation_button.clicked.connect(self._extracted_wse_dataset_button_clicked)
        self.ui.extracted_dataset_velocity_button.clicked.connect(self._extracted_velocity_dataset_button_clicked)
        self.ui.extracted_dataset_velocity_checkbox.stateChanged.connect(self._on_extracted_velocity_checkbox_changed)

    def _on_extracted_velocity_checkbox_changed(self):
        """Slot for when the extracted velocity checkbox is checked or unchecked."""
        enabled = self.ui.extracted_dataset_velocity_checkbox.isChecked()
        self.ui.extracted_dataset_velocity_button.setEnabled(enabled)

    def _extracted_wse_dataset_button_clicked(self):
        """Slot for when the extracted WSE dataset button is clicked."""
        mesh_uuid = self.all_data.wse_forcing_geometry
        mesh_node = find_tree_node_by_uuid(self.pe_tree, mesh_uuid)
        if not mesh_node:
            message_with_ok(
                self, 'No parent mesh selected. Select a mesh before selecting datasets.',
                XmsEnvironment.xms_environ_app_name()
            )
            return

        d = {'value': self.all_data.wse_forcing_wse_source}
        condition = DatasetSelector.is_scalar_if_dset

        ok = DatasetSelector.select_dataset(
            self,
            None,
            'Select WSE dataset',
            self.petc.copy(mesh_node),
            condition,
            d,
            'value',
            None,
        )

        if not ok:
            return

        self.all_data.wse_forcing_wse_source = d['value'] or ''
        self._update_extracted_dataset_labels()

    def _extracted_velocity_dataset_button_clicked(self):
        """Slot for when the extracted velocity dataset button is clicked."""
        mesh_uuid = self.all_data.wse_forcing_geometry
        mesh_node = find_tree_node_by_uuid(self.pe_tree, mesh_uuid)
        if not mesh_node:
            message_with_ok(
                self, 'No parent mesh selected. Select a mesh before selecting datasets.',
                XmsEnvironment.xms_environ_app_name()
            )
            return

        d = {'value': self.all_data.wse_forcing_velocity_source}
        condition = DatasetSelector.is_vector_if_dset

        ok = DatasetSelector.select_dataset(
            self,
            None,
            'Select velocity dataset',
            self.petc.copy(mesh_node),
            condition,
            d,
            'value',
            None,
        )

        if not ok:
            return

        self.all_data.wse_forcing_velocity_source = d['value'] or ''
        self._update_extracted_dataset_labels()

    def _update_extracted_dataset_labels(self):
        """Update the labels for both extracted datasets."""
        self._update_extracted_dataset_label(
            self.ui.extracted_dataset_elevation_label,
            self.all_data.wse_forcing_wse_source,
        )
        self._update_extracted_dataset_label(
            self.ui.extracted_dataset_velocity_label,
            self.all_data.wse_forcing_velocity_source,
        )

    def _update_extracted_dataset_label(self, label: QLabel, item_uuid: str):
        """
        Update a label for an extracted dataset.

        Args:
            label: The label widget to update.
            item_uuid: UUID of the dataset the widget should describe. An empty string is interpreted as no dataset.
        """
        if not item_uuid:
            text = """(none selected)<br><span style="color:red;">(no ref time)</span>"""
            label.setText(text)
            return

        path = build_tree_path(self.pe_tree, item_uuid)
        try:
            dataset = self._xms_data.get_dataset(item_uuid)
            if not dataset or not dataset.ref_time:
                time = """<span style="color:red;">(no ref time)</span>"""
            else:
                time_string = datetime_to_string(dataset.ref_time)
                time = f'Starts {time_string}'
        except ValueError:
            time = """<span style="color:red;">(ref time appears to be BCE; see Help page to fix</span>"""

        text = f'{path}<br>{time}'
        label.setText(text)

    def _on_use_velocity_changed(self, state):
        """Called when the use parent ADCIRC velocity solution toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        use_velocity = state == Qt.Checked
        self.ui.adcirc_64.setEnabled(use_velocity)
        self.ui.adcirc_64_label.setEnabled(use_velocity)
        self.ui.fort_64_path_label.setEnabled(use_velocity)
        self.ui.adcirc_solution.setEnabled(use_velocity)
        self.ui.adcirc_solution_label.setEnabled(use_velocity)
        self.ui.adcirc_solution_path_label.setEnabled(use_velocity)

    def _on_adcirc_solution_clicked(self):
        """Handler for ADCIRC WSE solution button click."""
        # Find the parent grid
        grid_item = find_tree_node_by_uuid(self.pe_tree, self.bc_data['parent_adcirc_14_uuid'])
        if not grid_item:
            return
        attr = {'parent_adcirc_solution_wse': ''}
        DatasetSelector.select_dataset(
            self,
            label_widget=self.ui.adcirc_solution_path_label_wse,
            caption='Select parent WSE data set',
            pe_tree=grid_item,
            condition=DatasetSelector.is_scalar_if_dset,
            storage_dict=attr,
            key='parent_adcirc_solution_wse',
            icon_method=None
        )
        self.bc_data['parent_adcirc_solution_wse'] = attr['parent_adcirc_solution_wse']

    def _on_adcirc_velocity_solution_clicked(self):
        """Handler for ADCIRC WSE solution button click."""
        # Find the parent grid
        grid_item = find_tree_node_by_uuid(self.pe_tree, self.bc_data['parent_adcirc_14_uuid'])
        if not grid_item:
            return
        attr = {'parent_adcirc_solution': ''}
        DatasetSelector.select_dataset(
            self,
            label_widget=self.ui.adcirc_solution_path_label,
            caption='Select parent velocity data set',
            pe_tree=grid_item,
            condition=DatasetSelector.is_vector_if_dset,
            storage_dict=attr,
            key='parent_adcirc_solution',
            icon_method=None
        )
        self.bc_data['parent_adcirc_solution'] = attr['parent_adcirc_solution']

    def _on_adcirc_parent_14_xmdf_clicked(self):
        """Handler for parent ADCIRC grid selector button clicked."""
        dlg = TreeItemSelectorDlg(
            title='Select Parent ADCIRC mesh',
            target_type=type(None),
            previous_selection=self.bc_data['parent_adcirc_14_uuid'],
            pe_tree=self.petc.copy(self.pe_tree),
            show_root=False,
            parent=self,
            selectable_xms_types=['TI_MESH2D']
        )
        if not dlg.has_selectable_items():
            msg = 'No Mesh2D items exist. Load the parent ADCIRC mesh into SMS to select it.'
            message_with_ok(parent=self, message=msg, app_name='SMS', win_icon=self.windowIcon())
            self.bc_data['parent_adcirc_14_uuid'] = ''  # Clear any previous selection, it doesn't exist anymore
            self.bc_data['parent_adcirc_solution'] = ''
            self.bc_data['parent_adcirc_solution_wse'] = ''
        elif dlg.exec():
            item_uuid = dlg.get_selected_item_uuid()
            # Set to empty string if user accepts dialog but cleared selection. get_selected_item_uuid will return None.
            self.bc_data['parent_adcirc_14_uuid'] = item_uuid if item_uuid else ''
            if not item_uuid:  # If no valid grid selected, clear the solution pickers. They must be children.
                self.bc_data['parent_adcirc_solution'] = ''
                self.bc_data['parent_adcirc_solution_wse'] = ''
            else:  # Valid grid selected, make sure any selected solution datasets are children
                grid_item = find_tree_node_by_uuid(self.pe_tree, item_uuid)
                wse_uuid = self.bc_data['parent_adcirc_solution_wse']
                if wse_uuid and not find_tree_node_by_uuid(grid_item, wse_uuid):
                    self.bc_data['parent_adcirc_solution_wse'] = ''
                vel_uuid = self.bc_data['parent_adcirc_solution']
                if vel_uuid and not find_tree_node_by_uuid(grid_item, vel_uuid):
                    self.bc_data['parent_adcirc_solution'] = ''
        else:
            return  # User cancelled
        self._set_selected_dataset_path_for_widget(self.ui.fort_14_path_label_xmdf, 'parent_adcirc_14_uuid')
        self._set_selected_dataset_path_for_widget(self.ui.adcirc_solution_path_label_wse, 'parent_adcirc_solution_wse')
        self._set_selected_dataset_path_for_widget(self.ui.adcirc_solution_path_label, 'parent_adcirc_solution')

    def _on_extracted_dataset_parent_clicked(self):
        """Slot for when the button to select a parent for extracted datasets is clicked."""
        dlg = TreeItemSelectorDlg(
            title='Select Parent geometry',
            target_type=type(None),
            previous_selection=self.bc_data['extracted_grid_uuid'],
            pe_tree=self.petc.copy(self.pe_tree),
            show_root=False,
            parent=self,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS', 'TI_CGRID2D']
        )
        if not dlg.has_selectable_items():
            msg = 'Unable to select parent. No supported geometries exist.'
            message_with_ok(parent=self, message=msg, app_name='SMS', win_icon=self.windowIcon())
        elif dlg.exec():
            item_uuid = dlg.get_selected_item_uuid()
            # Set to empty string if user accepts dialog but cleared selection. get_selected_item_uuid will return None.
            if item_uuid != self.all_data.wse_forcing_geometry:
                self.all_data.wse_forcing_wse_source = ''
                self.all_data.wse_forcing_velocity_source = ''

            self.all_data.wse_forcing_geometry = item_uuid or ''

        self._set_selected_dataset_path_for_widget(
            self.ui.extracted_dataset_parent_label, item_uuid=self.all_data.wse_forcing_geometry
        )
        self._update_extracted_dataset_labels()

    def _setup_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.ui.bc_type.addItems(self.bc_types)
        self.ui.flow_source.addItems(self.flow_sources)
        self.ui.wse_source.addItems(self.wse_sources)
        self.ui.adcirc_solution_type.addItems(self.adcirc_sol_types)
        self.ui.wse_offset_type.addItems(self.wse_offset_types)

    def _setup_edit_fields(self):
        """Sets up the validators and number correctors on edit fields."""
        self.dbl_validator.setDecimals(10)
        self.dbl_dir_validator.setDecimals(10)
        self.dbl_conveyance_validator.setDecimals(10)

        self.dbl_dir_validator.setBottom(-360.0)
        self.dbl_dir_validator.setTop(360.0)
        self.dbl_conveyance_validator.setBottom(0.3)
        self.dbl_conveyance_validator.setTop(0.8)

        self.ui.flow_constant.setValidator(self.dbl_validator)
        self.ui.flow_direction.setValidator(self.dbl_dir_validator)
        self.ui.flow_conveyance.setValidator(self.dbl_conveyance_validator)
        self.ui.wse_const.setValidator(self.dbl_validator)
        self.ui.wse_offset_const.setValidator(self.dbl_validator)

    def _load_data(self):
        """Loads the data from self.bc_data into the appropriate widgets."""
        if not self.bc_data.name.size == 0:
            self.ui.name_edit.setText(TextConverter.get_text_for_field(self.bc_data['name'].item()))
            self.ui.bc_type.setCurrentIndex(self.bc_types.index(self.bc_data['bc_type'].item()))
            self.ui.flow_source.setCurrentIndex(self.flow_sources.index(self.bc_data['flow_source'].item()))
            self.ui.flow_constant.setText(str(self.bc_data['constant_flow'].item()))
            self.ui.tog_flux_direction.setChecked(self.bc_data['specify_inflow_direction'].item() != 0)
            self.ui.flow_direction.setText(str(self.bc_data['flow_direction'].item()))
            self.ui.flow_conveyance.setText(str(self.bc_data['flow_conveyance'].item()))
            self.ui.wse_source.setCurrentIndex(self.wse_sources.index(self.bc_data['wse_source'].item()))
            self.ui.wse_const.setText(str(self.bc_data['wse_const'].item()))
            self.ui.use_velocity.setChecked(self.bc_data['use_velocity'].item() != 0)
            # For some reason the slot connection was working on initial load
            self._on_use_velocity_changed(self.ui.use_velocity.checkState())
            self.ui.adcirc_solution_type.setCurrentIndex(
                self.adcirc_sol_types.index(self.bc_data['parent_adcirc_solution_type'].item())
            )
            date_time_text = self.bc_data['parent_adcirc_start_time'].item()
            if date_time_text:
                date_time = datetime.datetime.strptime(date_time_text, ISO_DATETIME_FORMAT)
                self.ui.adcirc_start.setDateTime(datetime_to_qdatetime(date_time))
            self.ui.wse_offset_type.setCurrentIndex(self.wse_offset_types.index(self.bc_data['wse_offset_type'].item()))
            self.ui.wse_offset_const.setText(str(self.bc_data['wse_offset_const'].item()))
            self.ui.tog_salinity.setChecked(self.bc_data['use_salinity_curve'].item() != 0)
            self.ui.tog_temperature.setChecked(self.bc_data['use_temperature_curve'].item() != 0)
            self._set_selected_file_for_widget(self.ui.cmcards_file_path_label, 'parent_cmsflow')
            self._set_selected_file_for_widget(self.ui.fort_63_path_label, 'parent_adcirc_63')
            self._set_selected_file_for_widget(self.ui.fort_64_path_label, 'parent_adcirc_64')
            self._set_selected_file_for_widget(self.ui.fort_14_path_label, 'parent_adcirc_14')
            self._set_selected_dataset_path_for_widget(self.ui.fort_14_path_label_xmdf, 'parent_adcirc_14_uuid')
            self._set_selected_dataset_path_for_widget(
                self.ui.adcirc_solution_path_label_wse, 'parent_adcirc_solution_wse'
            )
            self._set_selected_dataset_path_for_widget(self.ui.adcirc_solution_path_label, 'parent_adcirc_solution')
            self._set_selected_dataset_path_for_widget(
                self.ui.extracted_dataset_parent_label, item_uuid=self.all_data.wse_forcing_geometry
            )
            self._update_extracted_dataset_labels()
            self.flow_forcing_curve_id = int(self.bc_data['flow_curve'].item())
            self.wse_forcing_curve_id = int(self.bc_data['wse_forcing_curve'].item())
            self.wse_offset_curve_id = int(self.bc_data['wse_offset_curve'].item())
            self.salinity_curve_id = int(self.bc_data['salinity_curve'].item())
            self.temperature_curve_id = int(self.bc_data['temperature_curve'].item())
        self.ui.bc_type_stack.setCurrentIndex(self.ui.bc_type.currentIndex())
        self.ui.flow_type_stack.setCurrentIndex(self.ui.flow_source.currentIndex())
        self.ui.wse_source_stack.setCurrentIndex(self.ui.wse_source.currentIndex())
        self.ui.adcirc_solution_type_stack.setCurrentIndex(self.ui.adcirc_solution_type.currentIndex())
        self.ui.wse_offset_stack.setCurrentIndex(self.ui.wse_offset_type.currentIndex())
        self.ui.btn_salinity_curve.setEnabled(self.ui.tog_salinity.checkState())
        self.ui.btn_temperature_curve.setEnabled(self.ui.tog_temperature.checkState())
        self.ui.flow_direction_label.setEnabled(self.ui.tog_flux_direction.checkState())
        self.ui.flow_direction.setEnabled(self.ui.tog_flux_direction.checkState())
        self.ui.adcirc_64.setEnabled(self.ui.use_velocity.checkState())
        self.ui.adcirc_64_label.setEnabled(self.ui.use_velocity.checkState())
        self.ui.fort_64_path_label.setEnabled(self.ui.use_velocity.checkState())
        enabled = self.all_data.wse_forcing_geometry != '' and self.all_data.wse_forcing_velocity_source != ''
        self.ui.extracted_dataset_velocity_checkbox.setChecked(enabled)
        self._on_extracted_velocity_checkbox_changed()

    def _save_bc_data(self):
        """Save data from the save point attributes."""
        self.bc_data['name'] = TextConverter.get_text_from_field(self.ui.name_edit.text())
        self.bc_data['bc_type'] = self.ui.bc_type.currentText()
        self.bc_data['flow_source'] = self.ui.flow_source.currentText()
        self.bc_data['constant_flow'] = float(self.ui.flow_constant.text())
        self.bc_data['flow_curve'] = int(self.flow_forcing_curve_id)
        self.bc_data['specify_inflow_direction'] = 1 if self.ui.tog_flux_direction.checkState() == Qt.Checked else 0
        self.bc_data['flow_direction'] = float(self.ui.flow_direction.text())
        self.bc_data['flow_conveyance'] = float(self.ui.flow_conveyance.text())
        self.bc_data['wse_source'] = self.ui.wse_source.currentText()
        self.bc_data['wse_const'] = float(self.ui.wse_const.text())
        self.bc_data['wse_forcing_curve'] = int(self.wse_forcing_curve_id)
        self.bc_data['use_velocity'] = 1 if self.ui.use_velocity.checkState() == Qt.Checked else 0
        self.bc_data['parent_cmsflow'] = self._get_selected_file(self.ui.cmcards_file_path_label)
        self.bc_data['parent_adcirc_14'] = self._get_selected_file(self.ui.fort_14_path_label)
        self.bc_data['parent_adcirc_solution_type'] = self.ui.adcirc_solution_type.currentText()
        self.bc_data['parent_adcirc_63'] = self._get_selected_file(self.ui.fort_63_path_label)
        self.bc_data['parent_adcirc_64'] = self._get_selected_file(self.ui.fort_64_path_label)
        # self.bc_data['parent_adcirc_solution_wse'] = self._get_selected_file(self.ui.adcirc_solution_path_label_wse)
        # self.bc_data['parent_adcirc_solution'] = self._get_selected_file(self.ui.adcirc_solution_path_label)
        start_time = qdatetime_to_datetime(self.ui.adcirc_start.dateTime()).strftime(ISO_DATETIME_FORMAT)
        self.bc_data['parent_adcirc_start_time'] = start_time
        self.bc_data['wse_offset_type'] = self.ui.wse_offset_type.currentText()
        self.bc_data['wse_offset_const'] = float(self.ui.wse_offset_const.text())
        self.bc_data['wse_offset_curve'] = int(self.wse_offset_curve_id)
        self.bc_data['use_salinity_curve'] = 1 if self.ui.tog_salinity.checkState() == Qt.Checked else 0
        self.bc_data['salinity_curve'] = int(self.salinity_curve_id)
        self.bc_data['use_temperature_curve'] = 1 if self.ui.tog_temperature.checkState() == Qt.Checked else 0
        self.bc_data['temperature_curve'] = int(self.temperature_curve_id)
        # Ensure the index is int dtype. pandas <-> xarray conversions and operations like to coerce it to a float.
        self.all_data.set_harmonic_table(
            int(self.bc_data['harmonic_table'].item()), self.harmonic_table.model.data_frame.to_xarray()
        )
        self.all_data.set_tidal_table(
            int(self.bc_data['tidal_table'].item()), self.tidal_table.model.data_frame.to_xarray()
        )
        if not self.ui.extracted_dataset_velocity_checkbox.isChecked():
            self.all_data.wse_forcing_velocity_source = ''

    def _get_selected_file(self, widget):
        """Gets the selected file text.

        Args:
            widget (QLabel): The label widget that shows the file path.

        Returns:
            Returns the path of the selected file. The file path is an empty string if nothing is selected.
        """
        path = widget.text()
        if path == '(none_selected)':
            return ''
        else:
            return path

    def _set_selected_file_for_widget(self, widget, key):
        """Sets the selected file text for the label widget.

        Args:
            widget (QLabel): The label widget that shows the file path.
            key (str): The key in self.bc_data for the widget.
        """
        path = self.bc_data[key].item()
        if path:
            widget.setText(path)

    def _set_selected_dataset_path_for_widget(self, widget: QLabel, key: str = '', item_uuid: str = ''):
        """Sets the selected dataset path for the label widget.

        Args:
            widget: The label widget that shows the dataset path.
            key: If provided, the value of self.bc_data[key] is taken to be the item's UUID.
            item_uuid: If provided, this is taken to be the item's UUID rather than looking it up in self.bc_data.
        """
        if key:
            dset_uuid = self.bc_data[key].item()
        elif item_uuid:
            dset_uuid = item_uuid
        else:
            dset_uuid = ''
        try:
            _ = uuid.UUID(dset_uuid, version=4)  # throw an exception if an invalid UUID
            dset_path = build_tree_path(self.pe_tree, dset_uuid)
        except ValueError:
            dset_path = '(none selected)'
        widget.setText(dset_path if dset_path else '(none selected)')

    def _edit_curve(self, curve_id, curve_function, series_name):
        """Brings up the XY series editor for a curve.

        Args:
            curve_id (int): The id of the curve.
            curve_function (function): The method to call to get the curve.
            series_name (str): The name of the series.
        """
        if curve_id != self.comp_id and curve_id > 0:
            curve = curve_function(curve_id).to_dataframe()
        else:
            curve = curve_function(self.comp_id).to_dataframe()
        default_curve = curve_function(-1).to_dataframe()
        curve.columns = default_curve.columns
        # Row index needs to be 1-based and sequential for the XySeriesEditor.
        if len(curve.index) > 0:
            curve.index = [i + 1 for i in range(len(curve.index))]
        dlg = XySeriesEditor(curve, series_name, dialog_title='XY Series Editor', icon=self.windowIcon(), parent=self)
        if dlg.exec():
            dlg.model.data_frame.columns = default_curve.columns
            return dlg.model.data_frame.to_xarray()
        return xr.Dataset()

    def _edit_flow_forcing_curve(self):
        """Brings up the XY series editor for the flow forcing curve."""
        df = self._edit_curve(self.flow_forcing_curve_id, self.all_data.flow_curve_from_id, 'Flow Forcing Over Time')
        if df:
            self.all_data.set_flow_forcing_curve(self.comp_id, df)
            self.flow_forcing_curve_id = self.comp_id

    def _edit_wse_forcing_curve(self):
        """Brings up the XY series editor for the WSE forcing curve."""
        df = self._edit_curve(
            self.wse_forcing_curve_id, self.all_data.wse_forcing_curve_from_id, 'WSE Forcing Over Time'
        )
        if df:
            self.all_data.set_wse_forcing_curve(self.comp_id, df)
            self.wse_forcing_curve_id = self.comp_id

    def _edit_wse_offset_curve(self):
        """Brings up the XY series editor for the WSE offset curve."""
        df = self._edit_curve(self.wse_offset_curve_id, self.all_data.wse_offset_curve_from_id, 'WSE Offset Over Time')
        if df:
            self.all_data.set_wse_offset_curve(self.comp_id, df)
            self.wse_offset_curve_id = self.comp_id

    def _edit_salinity_curve(self):
        """Brings up the XY series editor for the salinity curve."""
        df = self._edit_curve(self.salinity_curve_id, self.all_data.salinity_curve_from_id, 'Salinity Over Time')
        if df:
            self.all_data.set_salinity_curve(self.comp_id, df)
            self.salinity_curve_id = self.comp_id

    def _edit_temperature_curve(self):
        """Brings up the XY series editor for the temperature curve."""
        df = self._edit_curve(
            self.temperature_curve_id, self.all_data.temperature_curve_from_id, 'Temperature Over Time'
        )
        if df:
            self.all_data.set_temperature_curve(self.comp_id, df)
            self.temperature_curve_id = self.comp_id

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_bc_data()
        super().accept()
