"""This is a dialog for specifying a dredge event."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QGuiApplication
from PySide2.QtWidgets import QApplication, QDialogButtonBox, QScrollArea, QSizePolicy, QTabWidget, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources import resources_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.cmsflow.gui.dredge_placement_tab_ui import Ui_DredgePlacementTab
from xms.cmsflow.gui.dredge_tab_ui import Ui_DredgeTab
from xms.cmsflow.gui.dredge_trigger_time_table_widget import DredgeTriggerTimeTableWidget
from xms.cmsflow.gui.gui_util import find_domain_uuid
from xms.cmsflow.gui.text_converter import TextConverter
from xms.cmsflow.gui.unit_converter import UnitConverter


class DredgeDlg(XmsDlg):
    """A dialog for editing dredge data."""
    def __init__(self, sim_data, pe_tree, sim_uuid, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            sim_data (SimulationData): The dredge data to view
            pe_tree (TreeNode): Project explorer dump from Query at root
            sim_uuid (str): UUID of the simulation
            parent (Something derived from :obj:`QWidget`): The parent window

        """
        super().__init__(parent, 'xms.cmsflow.gui.dredge_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/CMS/DredgeModule'
        self.sim_data = sim_data
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowTitle('CMS-Flow Dredge Module')
        self.pe_tree = pe_tree  # For unrestricted tree item pickers
        domain_uuid = find_domain_uuid(pe_tree, sim_uuid)
        self.domain_pe_tree = tree_util.trim_project_explorer(pe_tree, domain_uuid)

        s_min = 60.0  # seconds per minute
        min_hr = 60.0  # minutes per hour
        hr_day = 24.0  # hours per day

        ft_m = 3.048  # feet per meter
        ft_yd = 3.0  # feet per yard
        in_ft = 12.0  # inches per foot
        cm_m = 100.0  # centimeters per meter

        self.unit_dict = {'dredge_rate': -1, 'trigger_depth': -1, 'trigger_volume': -1}

        self.s_min_hr_names = ['seconds', 'minutes', 'hours']
        self.s_min_hr_nums = [1.0, s_min, s_min * min_hr]

        self.m_ft_yd_names = ['m', 'ft', 'yd']
        self.m_ft_yd_nums = [1.0, 1.0 / ft_m, ft_yd / ft_m]

        self.m_ft_names = ['m', 'ft']
        self.m_ft_nums = [1.0, 1.0 / ft_m]

        self.m_cm_ft_in_names = ['m', 'cm', 'ft', 'in']
        self.m_cm_ft_in_nums = [1.0, 1.0 / cm_m, 1.0 / ft_m, 1.0 / ft_m / in_ft]

        self.m_ft_yd_per_day_hr_names = ['m^3/day', 'm^3/hr', 'ft^3/day', 'ft^3/hr', 'yd^3/day', 'yd^3/hr']
        self.m_ft_yd_per_day_hr_nums = [
            1.0, hr_day, 1.0 / ft_m**3, hr_day / ft_m**3, 1.0 / (ft_m / ft_yd)**3, hr_day / (ft_m / ft_yd)**3
        ]

        self.m_ft_yd_qubed_names = ['m^3', 'ft^3', 'yd^3']
        self.m_ft_yd_qubed_nums = [1.0, 1.0 / ft_m**3, 1.0 / (ft_m / ft_yd)**3]

        self.dredge_methods = ['Shallowest cell', 'Specified cell']
        self.trigger_methods = ['Depth', 'Volume', 'Percent', 'Time periods']
        self.distributions = ['Percent', 'Sequential']

        self.placement_methods = ['Uniform', 'Specified cell']
        self.limits_methods = ['Depth', 'Thickness']

        # validators
        self.int_valid = QxIntValidator(parent=self, bottom=1)

        self.positive_dbl_valid = QxDoubleValidator(parent=self)
        self.positive_dbl_valid.setBottom(0.0)
        self.positive_dbl_valid.setDecimals(10)

        self.dbl_valid = QxDoubleValidator(parent=self)
        self.dbl_valid.setDecimals(10)

        # layout dialog with tabs

        # Dredge tab
        self.dredge_scroll_area = QScrollArea()
        self.dredge_scroll_area.setWidgetResizable(True)
        self.dredge_scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

        self.dredge_widget = QWidget()

        self.dredge_tab = Ui_DredgeTab()
        self.dredge_tab.setupUi(self.dredge_widget)
        self.dredge_tab.trigger_time_periods_group.setLayout(QVBoxLayout())

        self.trigger_time_table = DredgeTriggerTimeTableWidget(
            self.dredge_tab.trigger_time_periods_group, self.sim_data.dredge_time_periods.to_dataframe()
        )
        self.dredge_tab.trigger_time_periods_group.layout().insertWidget(1, self.trigger_time_table)
        self.trigger_time_table.setObjectName('trigger_time_table')
        self.trigger_time_table.setMinimumHeight(205)  # min height from the XML
        self.trigger_time_table.table_view.verticalHeader().hide()

        # Placement tab
        self.placement_tab = Ui_DredgePlacementTab()

        self.placement_widget = QWidget()  # ties the tab and scroll area together
        self.placement_tab.setupUi(self.placement_widget)
        # self.placement_widget.setMinimumSize()
        size = self.placement_widget.sizeHint()  # ties the tab and scroll area together

        self.placement_scroll_area = QScrollArea()
        self.placement_scroll_area.setWidgetResizable(True)
        self.placement_scroll_area.setWidget(self.placement_widget)
        self.placement_scroll_area.resize(size.width(), size.height())

        self.placement_widget.setMinimumSize(size)
        self.adjustSize()
        self.placement_widget.setMinimumSize(self.minimumSizeHint())
        size = self.placement_widget.sizeHint()  #

        # Pass the widget size past the scroll area up to the parent dialog
        geom = self.geometry()
        geom.setSize(size)
        self.setGeometry(geom)

        # center dialog in screen containing mouse, was at top left of all screens
        frame_gm = self.frameGeometry()
        screen = QGuiApplication.screenAt(QApplication.desktop().cursor().pos())
        if screen:
            center_point = screen.geometry().center()
            frame_gm.moveCenter(center_point)
            self.move(frame_gm.topLeft())

        self.tab_widget = QTabWidget(self)
        self.tab_widget.setObjectName(u"tabs")

        self.dredge_scroll_area.setWidget(self.dredge_widget)
        self.tab_widget.addTab(self.dredge_scroll_area, 'Dredge Definition')

        size = self.dredge_tab.use_dredge_module.sizeHint()
        # self.tab_widget.setMinimumSize(size.width(),size.height()) # fix auto size and remove this line
        self.tab_widget.setBaseSize(size.width(), size.height())  # fix auto size and remove this line
        self.tab_widget.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)
        self.tab_widget.addTab(self.placement_scroll_area, 'Placement Definition')

        self.btn_box = QDialogButtonBox(self)
        self.btn_box.setObjectName(u"btn_box")
        self.btn_box.setOrientation(Qt.Horizontal)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Help | QDialogButtonBox.Ok)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)

        self.vertical_layout = QVBoxLayout(self)
        self.vertical_layout.addWidget(self.tab_widget)
        self.vertical_layout.addWidget(self.btn_box)

        self._setup_tabs()

    def _make_dataset_selector_slot(self, label_widget: QWidget, caption: str, storage_dict, key):
        """Make a dataset selector slot."""
        def helper():
            DatasetSelector.select_dataset(
                self, label_widget, caption, self.domain_pe_tree, DatasetSelector.is_scalar_if_dset, storage_dict, key,
                resources_util.get_quadtree_icon
            )

        return helper

    def _setup_tabs(self):
        """This does the setup for the different tabs."""
        self._setup_dredge_combo_boxes()
        self._setup_dredge_edit_fields()
        self._load_dredge_data()  # load data before making connections
        self._setup_dredge_connections()

        self._setup_placement_combo_boxes()
        self._setup_placement_edit_fields()
        self._load_placement_data()  # load data before making connections
        self._setup_placement_connections()

    def _setup_dredge_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.dredge_tab.update_interval_units.addItems(self.s_min_hr_names)
        self.dredge_tab.dredge_method.addItems(self.dredge_methods)
        self.dredge_tab.dredge_rate_units.addItems(self.m_ft_yd_per_day_hr_names)
        self.dredge_tab.trigger_method.addItems(self.trigger_methods)
        self.dredge_tab.trigger_depth_units.addItems(self.m_ft_yd_names)
        self.dredge_tab.trigger_volume_units.addItems(self.m_ft_yd_qubed_names)
        self.dredge_tab.trigger_percent_depth_units.addItems(self.m_ft_yd_names)
        self.dredge_tab.distribution.addItems(self.distributions)

    def _setup_dredge_edit_fields(self):
        """Sets up validators for edit fields."""
        self.dredge_tab.update_interval_value.setValidator(self.positive_dbl_valid)
        self.dredge_tab.specified_cell.setValidator(self.int_valid)
        self.dredge_tab.dredge_rate_value.setValidator(self.positive_dbl_valid)
        self.dredge_tab.trigger_depth_value.setValidator(self.positive_dbl_valid)
        self.dredge_tab.trigger_volume_value.setValidator(self.positive_dbl_valid)
        self.dredge_tab.trigger_percent.setValidator(self.positive_dbl_valid)
        self.dredge_tab.trigger_percent_depth_value.setValidator(self.positive_dbl_valid)

    def _setup_dredge_connections(self):
        """Set up signal/slot connections in the dredge tab."""
        slot = UnitConverter.make_change_units(
            self.dredge_tab.update_interval_value, self.s_min_hr_nums, 'update_interval', self.unit_dict
        )
        self.dredge_tab.update_interval_units.currentIndexChanged[int].connect(slot)
        slot = self._make_dataset_selector_slot(
            self.dredge_tab.dredge_dataset_path, 'Select dredge dataset', self.sim_data.dredge.attrs, 'DREDGE_DATASET'
        )
        self.dredge_tab.dredge_dataset.pressed.connect(slot)
        self.dredge_tab.dredge_method.currentIndexChanged[int].connect(self._dredge_method_changed)
        slot = UnitConverter.make_change_units(
            self.dredge_tab.dredge_rate_value, self.m_ft_yd_per_day_hr_nums, 'dredge_rate', self.unit_dict
        )
        self.dredge_tab.dredge_rate_units.currentIndexChanged[int].connect(slot)
        self.dredge_tab.trigger_method.currentIndexChanged[int].connect(
            self.dredge_tab.trigger_method_stack.setCurrentIndex
        )
        slot = UnitConverter.make_change_units(
            self.dredge_tab.trigger_depth_value, self.m_ft_yd_nums, 'trigger_depth', self.unit_dict
        )
        self.dredge_tab.trigger_depth_units.currentIndexChanged[int].connect(slot)
        slot = UnitConverter.make_change_units(
            self.dredge_tab.trigger_volume_value, self.m_ft_yd_qubed_nums, 'trigger_volume', self.unit_dict
        )
        self.dredge_tab.trigger_volume_units.currentIndexChanged[int].connect(slot)
        slot = UnitConverter.make_change_units(
            self.dredge_tab.trigger_percent_depth_value, self.m_ft_yd_nums, 'trigger_percent_depth', self.unit_dict
        )
        self.dredge_tab.trigger_percent_depth_units.currentIndexChanged[int].connect(slot)
        self.dredge_tab.distribution.currentIndexChanged[int].connect(self._distribution_changed)

    def _load_dredge_data(self):
        """Load the data for the dredge tab."""
        # make sure we have dredge data
        if self.sim_data.dredge is None:
            return ''
        self.dredge_tab.use_dredge_module.setChecked(self.sim_data.dredge.attrs['ENABLE_DREDGE'] != 0)
        self.placement_widget.setEnabled(self.sim_data.dredge.attrs['ENABLE_DREDGE'] != 0)

        self.dredge_tab.dredge_name_value.setText(self.sim_data.dredge.attrs['DREDGE_NAME'])

        is_explicit = self.sim_data.general.attrs['SOLUTION_SCHEME'] == 'Explicit'
        self.dredge_tab.update_interval_label.setEnabled(is_explicit)
        self.dredge_tab.update_interval_value.setEnabled(is_explicit)
        self.dredge_tab.update_interval_units.setEnabled(is_explicit)

        self.dredge_tab.update_interval_value.setText(str(self.sim_data.dredge.attrs['UPDATE_INTERVAL_VALUE']))
        self.dredge_tab.update_interval_units.setCurrentIndex(
            self.s_min_hr_names.index(self.sim_data.dredge.attrs['UPDATE_INTERVAL_UNITS'])
        )
        self.unit_dict['update_interval'] = self.dredge_tab.update_interval_units.currentIndex()

        tree_path = tree_util.build_tree_path(self.domain_pe_tree, self.sim_data.dredge.attrs['DREDGE_DATASET'])
        text = TextConverter.get_text_from_field(tree_path)
        self.dredge_tab.dredge_dataset_path.setText(text)

        self.dredge_tab.dredge_method.setCurrentIndex(
            self.dredge_methods.index(self.sim_data.dredge.attrs['DREDGE_METHOD'])
        )
        self.dredge_tab.specified_cell.setText(str(self.sim_data.dredge.attrs['SPECIFIED_CELL']))
        self._dredge_method_changed(self.dredge_tab.dredge_method.currentIndex())

        self.dredge_tab.dredge_rate_value.setText(str(self.sim_data.dredge.attrs['DREDGE_RATE_VALUE']))
        self.dredge_tab.dredge_rate_units.setCurrentIndex(
            self.m_ft_yd_per_day_hr_names.index(self.sim_data.dredge.attrs['DREDGE_RATE_UNITS'])
        )
        self.unit_dict['dredge_rate'] = self.dredge_tab.dredge_rate_units.currentIndex()

        self.dredge_tab.trigger_method.setCurrentIndex(
            self.trigger_methods.index(self.sim_data.dredge.attrs['TRIGGER_METHOD'])
        )
        self.dredge_tab.trigger_method_stack.setCurrentIndex(self.dredge_tab.trigger_method.currentIndex())

        self.dredge_tab.trigger_depth_value.setText(str(self.sim_data.dredge.attrs['TRIGGER_DEPTH_VALUE']))
        self.dredge_tab.trigger_depth_units.setCurrentIndex(
            self.m_ft_yd_names.index(self.sim_data.dredge.attrs['TRIGGER_DEPTH_UNITS'])
        )
        self.unit_dict['trigger_depth'] = self.dredge_tab.trigger_depth_units.currentIndex()

        self.dredge_tab.trigger_volume_value.setText(str(self.sim_data.dredge.attrs['TRIGGER_VOLUME_VALUE']))
        self.dredge_tab.trigger_volume_units.setCurrentIndex(
            self.m_ft_yd_qubed_names.index(self.sim_data.dredge.attrs['TRIGGER_VOLUME_UNITS'])
        )
        self.unit_dict['trigger_volume'] = self.dredge_tab.trigger_volume_units.currentIndex()

        self.dredge_tab.trigger_percent.setText(str(self.sim_data.dredge.attrs['TRIGGER_PERCENT']))
        self.dredge_tab.trigger_percent_depth_value.setText(
            str(self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_VALUE'])
        )
        self.dredge_tab.trigger_percent_depth_units.setCurrentIndex(
            self.m_ft_yd_names.index(self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_UNITS'])
        )
        self.unit_dict['trigger_percent_depth'] = self.dredge_tab.trigger_percent_depth_units.currentIndex()

        self.dredge_tab.distribution.setCurrentIndex(
            self.distributions.index(self.sim_data.dredge.attrs['DISTRIBUTION'])
        )
        self._distribution_changed(self.dredge_tab.distribution.currentIndex())

        self.dredge_tab.enable_diagnostic.setChecked(self.sim_data.dredge.attrs['ENABLE_DIAGNOSTIC'] != 0)

    def _save_sim_data(self):
        """Save data from the dredge attributes."""
        self.sim_data.dredge.attrs['ENABLE_DREDGE'] =\
            1 if self.dredge_tab.use_dredge_module.isChecked() else 0
        self.sim_data.dredge.attrs['DREDGE_NAME'] = self.dredge_tab.dredge_name_value.text()

        self.sim_data.dredge.attrs['UPDATE_INTERVAL_VALUE'] = float(self.dredge_tab.update_interval_value.text())
        self.sim_data.dredge.attrs['UPDATE_INTERVAL_UNITS'] = self.dredge_tab.update_interval_units.currentText()

        self.sim_data.dredge.attrs['DREDGE_METHOD'] = self.dredge_tab.dredge_method.currentText()
        self.sim_data.dredge.attrs['SPECIFIED_CELL'] = int(self.dredge_tab.specified_cell.text())

        self.sim_data.dredge.attrs['DREDGE_RATE_VALUE'] = float(self.dredge_tab.dredge_rate_value.text())
        self.sim_data.dredge.attrs['DREDGE_RATE_UNITS'] = self.dredge_tab.dredge_rate_units.currentText()

        self.sim_data.dredge.attrs['TRIGGER_METHOD'] = self.dredge_tab.trigger_method.currentText()

        self.sim_data.dredge.attrs['TRIGGER_DEPTH_VALUE'] = float(self.dredge_tab.trigger_depth_value.text())
        self.sim_data.dredge.attrs['TRIGGER_DEPTH_UNITS'] = self.dredge_tab.trigger_depth_units.currentText()

        self.sim_data.dredge.attrs['TRIGGER_VOLUME_VALUE'] = float(self.dredge_tab.trigger_volume_value.text())
        self.sim_data.dredge.attrs['TRIGGER_VOLUME_UNITS'] = self.dredge_tab.trigger_volume_units.currentText()

        self.sim_data.dredge.attrs['TRIGGER_PERCENT'] = float(self.dredge_tab.trigger_percent.text())
        trigger_percent_depth_value = float(self.dredge_tab.trigger_percent_depth_value.text())
        self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_VALUE'] = trigger_percent_depth_value
        trigger_percent_depth_units = self.dredge_tab.trigger_percent_depth_units.currentText()
        self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_UNITS'] = trigger_percent_depth_units

        self.sim_data.dredge_time_periods = self.trigger_time_table.model.data_frame.to_xarray()

        self.sim_data.dredge.attrs['DISTRIBUTION'] = self.dredge_tab.distribution.currentText()
        self.sim_data.dredge.attrs['ENABLE_DIAGNOSTIC'] =\
            1 if self.dredge_tab.enable_diagnostic.checkState() == Qt.Checked else 0

        self._save_placement_data()

    def _dredge_method_changed(self, idx):
        """Sets the Shallowest or specified cell method.

        Args:
            idx (int): The index in the dredge_method combobox.
        """
        # 0 is the index of the 'Shallowest cell' entry
        is_shallowest = idx == 0
        self.dredge_tab.specified_cell_label.setVisible(not is_shallowest)
        self.dredge_tab.specified_cell.setVisible(not is_shallowest)
        method_note = 'Note: The volume dredged initiates from the shallowest point in the source area.'\
            if is_shallowest else '''Note: A dredging starting point is defined by a specified cell id and progresses
             to cells farther away from the starting point.'''
        self.dredge_tab.method_note.setText(method_note)

    def _distribution_changed(self, idx):
        """Sets distribution to percent or sequential.

        Args:
            idx (int): The index in the distribution combobox.
        """
        # 0 is the index of the 'percent' entry
        is_percent = idx == 0
        self.dredge_tab.distribution_percent_note1.setVisible(is_percent)
        self.dredge_tab.distribution_percent_note2.setVisible(is_percent)
        self.dredge_tab.distribution_sequential_note.setVisible(not is_percent)

    # Placement

    def _setup_placement_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.placement_tab.placement_method_1.addItems(self.placement_methods)
        self.placement_tab.limits_method_1.addItems(self.limits_methods)
        self.placement_tab.limits_method_depth_units_1.addItems(self.m_ft_names)
        self.placement_tab.limits_method_thickness_units_1.addItems(self.m_cm_ft_in_names)

        self.placement_tab.placement_method_2.addItems(self.placement_methods)
        self.placement_tab.limits_method_2.addItems(self.limits_methods)
        self.placement_tab.limits_method_depth_units_2.addItems(self.m_ft_names)
        self.placement_tab.limits_method_thickness_units_2.addItems(self.m_cm_ft_in_names)

        self.placement_tab.placement_method_3.addItems(self.placement_methods)
        self.placement_tab.limits_method_3.addItems(self.limits_methods)
        self.placement_tab.limits_method_depth_units_3.addItems(self.m_ft_names)
        self.placement_tab.limits_method_thickness_units_3.addItems(self.m_cm_ft_in_names)

    def _setup_placement_edit_fields(self):
        """Sets up validators for edit fields."""
        self.placement_tab.placement_method_id_1.setValidator(self.int_valid)
        self.placement_tab.distribution_1.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_depth_value_1.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_thickness_value_1.setValidator(self.positive_dbl_valid)

        self.placement_tab.placement_method_id_2.setValidator(self.int_valid)
        self.placement_tab.distribution_2.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_depth_value_2.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_thickness_value_2.setValidator(self.positive_dbl_valid)

        self.placement_tab.placement_method_id_3.setValidator(self.int_valid)
        self.placement_tab.distribution_3.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_depth_value_3.setValidator(self.positive_dbl_valid)
        self.placement_tab.limits_method_thickness_value_3.setValidator(self.positive_dbl_valid)

    def _setup_placement_connections(self):
        """Set up signal/slot connections in the placement tab."""
        self.dredge_tab.use_dredge_module.toggled.connect(self.placement_widget.setEnabled)

        self.placement_tab.use_area_1.stateChanged[int].connect(self.placement_tab.area_box_1.setVisible)
        slot = self._make_dataset_selector_slot(
            self.placement_tab.dataset_path_1, 'Select placement dataset', self.sim_data.dredge_placement.attrs,
            'PLACEMENT_1_DATASET'
        )
        self.placement_tab.dataset_1.pressed.connect(slot)
        self.placement_tab.placement_method_1.currentIndexChanged[int].connect(self._placement_method_changed_1)
        self.placement_tab.limits_method_1.currentIndexChanged[int].connect(self._placement_limits_method_changed_1)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_depth_value_1, self.m_ft_nums, 'limits_method_depth_1', self.unit_dict
        )
        self.placement_tab.limits_method_depth_units_1.currentIndexChanged[int].connect(slot)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_thickness_value_1, self.m_cm_ft_in_nums, 'limits_method_thickness_1',
            self.unit_dict
        )
        self.placement_tab.limits_method_thickness_units_1.currentIndexChanged[int].connect(slot)

        self.placement_tab.use_area_2.stateChanged[int].connect(self.placement_tab.area_box_2.setVisible)
        slot = self._make_dataset_selector_slot(
            self.placement_tab.dataset_path_2,
            'Select placement dataset',
            self.sim_data.dredge_placement.attrs,
            'PLACEMENT_2_DATASET',
        )
        self.placement_tab.dataset_2.pressed.connect(slot)
        self.placement_tab.placement_method_2.currentIndexChanged[int].connect(self._placement_method_changed_2)
        self.placement_tab.limits_method_2.currentIndexChanged[int].connect(self._placement_limits_method_changed_2)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_depth_value_2, self.m_ft_nums, 'limits_method_depth_2', self.unit_dict
        )
        self.placement_tab.limits_method_depth_units_2.currentIndexChanged[int].connect(slot)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_thickness_value_2, self.m_cm_ft_in_nums, 'limits_method_thickness_2',
            self.unit_dict
        )
        self.placement_tab.limits_method_thickness_units_2.currentIndexChanged[int].connect(slot)

        self.placement_tab.use_area_3.stateChanged[int].connect(self.placement_tab.area_box_3.setVisible)
        slot = self._make_dataset_selector_slot(
            self.placement_tab.dataset_path_3,
            'Select placement dataset',
            self.sim_data.dredge_placement.attrs,
            'PLACEMENT_3_DATASET',
        )
        self.placement_tab.dataset_3.pressed.connect(slot)
        self.placement_tab.placement_method_3.currentIndexChanged[int].connect(self._placement_method_changed_3)
        self.placement_tab.limits_method_3.currentIndexChanged[int].connect(self._placement_limits_method_changed_3)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_depth_value_3, self.m_ft_nums, 'limits_method_depth_3', self.unit_dict
        )
        self.placement_tab.limits_method_depth_units_3.currentIndexChanged[int].connect(slot)
        slot = UnitConverter.make_change_units(
            self.placement_tab.limits_method_thickness_value_3, self.m_cm_ft_in_nums, 'limits_method_thickness_3',
            self.unit_dict
        )
        self.placement_tab.limits_method_thickness_units_3.currentIndexChanged[int].connect(slot)

    def _load_placement_data(self):
        """Load the data for the placement tab."""
        # Area
        self.placement_tab.use_area_1.setChecked(self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_1'] != 0)
        self.placement_tab.area_box_1.setVisible(self.placement_tab.use_area_1.checkState())

        tree_path = tree_util.build_tree_path(
            self.domain_pe_tree, self.sim_data.dredge_placement.attrs['PLACEMENT_1_DATASET']
        )
        text = TextConverter.get_text_from_field(tree_path)
        self.placement_tab.dataset_path_1.setText(text)

        self.placement_tab.placement_method_1.setCurrentIndex(
            self.placement_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_1_METHOD'])
        )
        self._placement_method_changed_1(self.placement_tab.placement_method_1.currentIndex())

        self.placement_tab.placement_method_id_1.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_1_METHOD_CELL'])
        )
        self.placement_tab.distribution_1.setText(str(self.sim_data.dredge_placement.attrs['PLACEMENT_1_PERCENTAGE']))

        self.placement_tab.limits_method_1.setCurrentIndex(
            self.limits_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_METHOD'])
        )
        self._placement_limits_method_changed_1(self.placement_tab.limits_method_1.currentIndex())

        self.placement_tab.limits_method_depth_value_1.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_DEPTH_VALUE'])
        )
        self.placement_tab.limits_method_depth_units_1.setCurrentIndex(
            self.m_ft_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_DEPTH_UNITS'])
        )
        self.unit_dict['limits_method_depth_1'] = self.placement_tab.limits_method_depth_units_1.currentIndex()

        self.placement_tab.limits_method_thickness_value_1.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_THICKNESS_VALUE'])
        )
        self.placement_tab.limits_method_thickness_units_1.setCurrentIndex(
            self.m_cm_ft_in_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_THICKNESS_UNITS'])
        )
        self.unit_dict['limits_method_thickness_1'] = \
            self.placement_tab.limits_method_thickness_units_1.currentIndex()

        # Area
        self.placement_tab.use_area_2.setChecked(self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_2'] != 0)
        self.placement_tab.area_box_2.setVisible(self.placement_tab.use_area_2.checkState())

        tree_path = tree_util.build_tree_path(
            self.domain_pe_tree, self.sim_data.dredge_placement.attrs['PLACEMENT_2_DATASET']
        )
        text = TextConverter.get_text_from_field(tree_path)
        self.placement_tab.dataset_path_2.setText(text)

        self.placement_tab.placement_method_2.setCurrentIndex(
            self.placement_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_2_METHOD'])
        )
        self._placement_method_changed_2(self.placement_tab.placement_method_2.currentIndex())

        self.placement_tab.placement_method_id_2.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_2_METHOD_CELL'])
        )
        self.placement_tab.distribution_2.setText(str(self.sim_data.dredge_placement.attrs['PLACEMENT_2_PERCENTAGE']))

        self.placement_tab.limits_method_2.setCurrentIndex(
            self.limits_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_METHOD'])
        )
        self._placement_limits_method_changed_2(self.placement_tab.limits_method_2.currentIndex())

        self.placement_tab.limits_method_depth_value_2.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_DEPTH_VALUE'])
        )
        self.placement_tab.limits_method_depth_units_2.setCurrentIndex(
            self.m_ft_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_DEPTH_UNITS'])
        )
        self.unit_dict['limits_method_depth_2'] = self.placement_tab.limits_method_depth_units_2.currentIndex()

        self.placement_tab.limits_method_thickness_value_2.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_THICKNESS_VALUE'])
        )
        self.placement_tab.limits_method_thickness_units_2.setCurrentIndex(
            self.m_cm_ft_in_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_THICKNESS_UNITS'])
        )
        self.unit_dict['limits_method_thickness_2'] = \
            self.placement_tab.limits_method_thickness_units_2.currentIndex()

        # Area
        self.placement_tab.use_area_3.setChecked(self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_3'] != 0)
        self.placement_tab.area_box_3.setVisible(self.placement_tab.use_area_3.checkState())

        tree_path = tree_util.build_tree_path(
            self.domain_pe_tree, self.sim_data.dredge_placement.attrs['PLACEMENT_3_DATASET']
        )
        text = TextConverter.get_text_from_field(tree_path)
        self.placement_tab.dataset_path_3.setText(text)

        self.placement_tab.placement_method_3.setCurrentIndex(
            self.placement_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_3_METHOD'])
        )
        self._placement_method_changed_3(self.placement_tab.placement_method_3.currentIndex())

        self.placement_tab.placement_method_id_3.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_3_METHOD_CELL'])
        )
        self.placement_tab.distribution_3.setText(str(self.sim_data.dredge_placement.attrs['PLACEMENT_3_PERCENTAGE']))

        self.placement_tab.limits_method_3.setCurrentIndex(
            self.limits_methods.index(self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_METHOD'])
        )
        self._placement_limits_method_changed_3(self.placement_tab.limits_method_3.currentIndex())

        self.placement_tab.limits_method_depth_value_3.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_DEPTH_VALUE'])
        )
        self.placement_tab.limits_method_depth_units_3.setCurrentIndex(
            self.m_ft_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_DEPTH_UNITS'])
        )
        self.unit_dict['limits_method_depth_3'] = self.placement_tab.limits_method_depth_units_3.currentIndex()

        self.placement_tab.limits_method_thickness_value_3.setText(
            str(self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_THICKNESS_VALUE'])
        )
        self.placement_tab.limits_method_thickness_units_3.setCurrentIndex(
            self.m_cm_ft_in_names.index(self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_THICKNESS_UNITS'])
        )
        self.unit_dict['limits_method_thickness_3'] = \
            self.placement_tab.limits_method_thickness_units_3.currentIndex()

    def _save_placement_data(self):
        """Save data from the dredge attributes."""
        # Area
        self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_1'] =\
            1 if self.placement_tab.use_area_1.isChecked() else 0
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_METHOD'] = self.placement_tab.placement_method_1.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_METHOD_CELL'] = \
            int(self.placement_tab.placement_method_id_1.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_PERCENTAGE'] = float(self.placement_tab.distribution_1.text())
        limit_method_1 = self.placement_tab.limits_method_1.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_METHOD'] = limit_method_1

        self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_DEPTH_VALUE'] =\
            float(self.placement_tab.limits_method_depth_value_1.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_DEPTH_UNITS'] = \
            self.placement_tab.limits_method_depth_units_1.currentText()

        self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_THICKNESS_VALUE'] =\
            float(self.placement_tab.limits_method_thickness_value_1.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_1_LIMIT_THICKNESS_UNITS'] = \
            self.placement_tab.limits_method_thickness_units_1.currentText()

        # Area
        self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_2'] =\
            1 if self.placement_tab.use_area_2.isChecked() else 0
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_METHOD'] = self.placement_tab.placement_method_2.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_METHOD_CELL'] = \
            int(self.placement_tab.placement_method_id_2.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_PERCENTAGE'] = float(self.placement_tab.distribution_2.text())
        limit_method_2 = self.placement_tab.limits_method_2.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_METHOD'] = limit_method_2

        self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_DEPTH_VALUE'] =\
            float(self.placement_tab.limits_method_depth_value_2.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_DEPTH_UNITS'] = \
            self.placement_tab.limits_method_depth_units_2.currentText()

        self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_THICKNESS_VALUE'] =\
            float(self.placement_tab.limits_method_thickness_value_2.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_2_LIMIT_THICKNESS_UNITS'] = \
            self.placement_tab.limits_method_thickness_units_2.currentText()

        # Area
        self.sim_data.dredge_placement.attrs['DEFINE_PLACEMENT_3'] =\
            1 if self.placement_tab.use_area_3.isChecked() else 0
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_METHOD'] = self.placement_tab.placement_method_3.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_METHOD_CELL'] = \
            int(self.placement_tab.placement_method_id_3.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_PERCENTAGE'] = float(self.placement_tab.distribution_3.text())
        limit_method_3 = self.placement_tab.limits_method_3.currentText()
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_METHOD'] = limit_method_3

        self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_DEPTH_VALUE'] =\
            float(self.placement_tab.limits_method_depth_value_3.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_DEPTH_UNITS'] = \
            self.placement_tab.limits_method_depth_units_3.currentText()

        self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_THICKNESS_VALUE'] =\
            float(self.placement_tab.limits_method_thickness_value_3.text())
        self.sim_data.dredge_placement.attrs['PLACEMENT_3_LIMIT_THICKNESS_UNITS'] = \
            self.placement_tab.limits_method_thickness_units_3.currentText()

    def _placement_method_changed_1(self, idx):
        """Sets placement method to Uniform or Specified Cell.

        Args:
            idx (int): The index in the placement_method_1 combobox.
        """
        is_uniform = idx == 0  # 0 is the index of the 'Uniform' entry
        self.placement_tab.placement_method_id_label_1.setVisible(not is_uniform)
        self.placement_tab.placement_method_id_1.setVisible(not is_uniform)
        method_note = 'Note: The dredge material is placed uniformly over the placement area.'\
            if is_uniform else 'Note: The dredge material is placed starting at the user-specified point.'
        self.placement_tab.placement_method_note_1.setText(method_note)

    def _placement_method_changed_2(self, idx):
        """Sets placement method to Uniform or Specified Cell.

        Args:
            idx (int): The index in the placement_method_2 combobox.
        """
        is_uniform = idx == 0  # 0 is the index of the 'Uniform' entry
        self.placement_tab.placement_method_id_label_2.setVisible(not is_uniform)
        self.placement_tab.placement_method_id_2.setVisible(not is_uniform)
        method_note = 'Note: The dredge material is placed uniformly over the placement area.'\
            if is_uniform else 'Note: The dredge material is placed starting at the user-specified point.'
        self.placement_tab.placement_method_note_2.setText(method_note)

    def _placement_method_changed_3(self, idx):
        """Sets placement method to Uniform or Specified Cell.

        Args:
            idx (int): The index in the placement_method_3 combobox.
        """
        is_uniform = idx == 0  # 0 is the index of the 'Uniform' entry
        self.placement_tab.placement_method_id_label_3.setVisible(not is_uniform)
        self.placement_tab.placement_method_id_3.setVisible(not is_uniform)
        method_note = 'Note: The dredge material is placed uniformly over the placement area.'\
            if is_uniform else 'Note: The dredge material is placed starting at the user-specified point.'
        self.placement_tab.placement_method_note_3.setText(method_note)

    def _placement_limits_method_changed_1(self, idx):
        """Sets placement limits method to Depth or Thickness.

        Args:
            idx (int): The index in the _placement_limits_method_1 combobox.
        """
        is_depth = idx == 0  # 0 is the index of the 'Depth' entry

        method_label = 'Depth below water:' if is_depth else 'Maximum thickness:'
        self.placement_tab.limits_method_label_1.setText(method_label)

        self.placement_tab.limits_method_depth_value_1.setVisible(is_depth)
        self.placement_tab.limits_method_depth_units_1.setVisible(is_depth)

        self.placement_tab.limits_method_thickness_value_1.setVisible(not is_depth)
        self.placement_tab.limits_method_thickness_units_1.setVisible(not is_depth)

        method_note = 'Enter the depth below water surface that material placement cannot exceed.'\
            if is_depth else 'Enter the maximum thickness above initial bed layer that placed material cannot exceed.'
        self.placement_tab.limits_method_note_1.setText(method_note)

    def _placement_limits_method_changed_2(self, idx):
        """Sets placement limits method to Depth or Thickness.

        Args:
            idx (int): The index in the _placement_limits_method_2 combobox.
        """
        is_depth = idx == 0  # 0 is the index of the 'Depth' entry

        method_label = 'Depth below water:' if is_depth else 'Maximum thickness:'
        self.placement_tab.limits_method_label_2.setText(method_label)

        self.placement_tab.limits_method_depth_value_2.setVisible(is_depth)
        self.placement_tab.limits_method_depth_units_2.setVisible(is_depth)

        self.placement_tab.limits_method_thickness_value_2.setVisible(not is_depth)
        self.placement_tab.limits_method_thickness_units_2.setVisible(not is_depth)

        method_note = 'Enter the depth below water surface that material placement cannot exceed.'\
            if is_depth else 'Enter the maximum thickness above initial bed layer that placed material cannot exceed.'
        self.placement_tab.limits_method_note_2.setText(method_note)

    def _placement_limits_method_changed_3(self, idx):
        """Sets placement limits method to Depth or Thickness.

        Args:
            idx (int): The index in the _placement_limits_method_3 combobox.
        """
        is_depth = idx == 0  # 0 is the index of the 'Depth' entry

        method_label = 'Depth below water:' if is_depth else 'Maximum thickness:'
        self.placement_tab.limits_method_label_3.setText(method_label)

        self.placement_tab.limits_method_depth_value_3.setVisible(is_depth)
        self.placement_tab.limits_method_depth_units_3.setVisible(is_depth)

        self.placement_tab.limits_method_thickness_value_3.setVisible(not is_depth)
        self.placement_tab.limits_method_thickness_units_3.setVisible(not is_depth)

        method_note = 'Enter the depth below water surface that material placement cannot exceed.'\
            if is_depth else 'Enter the maximum thickness above initial bed layer that placed material cannot exceed.'
        self.placement_tab.limits_method_note_3.setText(method_note)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_sim_data()
        super().accept()
