"""Qt delegate for displaying a file selector button."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.cmsflow.gui import meteorological_wind_dialog


class MeteorologicalCurveButtonDelegate(QStyledItemDelegate):
    """Delegate for the wind direction and velocity curve button column."""
    def __init__(self, get_dataframe_method, set_dataframe_method, parent=None):
        """Initializes the class.

        Args:
            get_dataframe_method (obj): The method that returns a dataframe, and new curve id, given a curve id.
            set_dataframe_method (obj): The method that sets a dataframe with a curve id.
            parent (QObject): The parent object.

        """
        super().__init__(parent)
        self.parent_dlg = parent
        self.get_dataframe_method = get_dataframe_method
        self.set_dataframe_method = set_dataframe_method

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Edit...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                curve_id = int(index.data())
                df, curve_id = self.get_dataframe_method(curve_id)
                # wind curve dialog
                dlg = meteorological_wind_dialog.MeteorologicalWindDialog(df, self.parent_dlg)
                if dlg.exec_():
                    df = dlg.wind_table.model.data_frame
                    self.set_dataframe_method(curve_id, df)
                    index.model().setData(index, curve_id)
        return super().editorEvent(event, model, option, index)
