"""This is a dialog for specifying non-spatially varying simulation model values."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QDateTime, QModelIndex, QSortFilterProxyModel, Qt
from PySide2.QtGui import QGuiApplication, QStandardItem, QStandardItemModel
from PySide2.QtWidgets import (
    QApplication, QDialogButtonBox, QHeaderView, QScrollArea, QSpacerItem, QTabWidget, QVBoxLayout, QWidget
)
import xarray as xr

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.dmi import XmsEnvironment
from xms.api.tree import tree_util
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources import resources_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.cmsflow.gui.atmospheric_table_widget import AtmosphericTableWidget
from xms.cmsflow.gui.bed_layers_table_widget import BedLayersTableWidget
from xms.cmsflow.gui.file_selector import FileSelector
from xms.cmsflow.gui.flow_tab_ui import Ui_FlowTab
from xms.cmsflow.gui.general_tab_ui import Ui_GeneralTab
from xms.cmsflow.gui.grain_sizes_table_widget import GrainSizesTableWidget
from xms.cmsflow.gui.gui_util import find_domain_uuid
from xms.cmsflow.gui.meteorological_stations_table_widget import MeteorologicalStationsTableWidget
from xms.cmsflow.gui.output_list_table_widget import OutputListTableWidget
from xms.cmsflow.gui.output_tab_ui import Ui_OutputTab
from xms.cmsflow.gui.salinity_temperature_tab_ui import Ui_SalinityTemperatureTab
from xms.cmsflow.gui.sediment_diameters_table_widget import SedimentDiametersTableWidget
from xms.cmsflow.gui.sediment_tab_ui import Ui_SedimentTab
from xms.cmsflow.gui.text_converter import TextConverter
from xms.cmsflow.gui.unit_converter import UnitConverter
from xms.cmsflow.gui.wave_tab_ui import Ui_WaveTab
from xms.cmsflow.gui.wind_parameters_table_widget import WindParametersTableWidget
from xms.cmsflow.gui.wind_tab_ui import Ui_WindTab


class ComboBoxFilterModel(QSortFilterProxyModel):
    """A filter model for comboboxes to allow options to be hidden."""
    def __init__(self, parent, remove_list):
        """Initializes the class, sets the parent.

        Args:
            parent (QObject): The parent object for this model.
            remove_list (list): A list of indexes to remove if hiding rows.
        """
        self.hide_rows = False
        super().__init__(parent)
        self.setDynamicSortFilter(True)
        self.remove_list = remove_list

    def filterAcceptsRow(self, source_row: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Hides rows if the UserRole allows it.

        Args:
            source_row (int): The row of the source model in question.
            source_parent (QModelIndex): The index of the row in question.

        Returns:
            True if the row should be visible.
        """
        if self.hide_rows:
            # hide = self.sourceModel().data(source_parent, Qt.UserRole)
            hide = self.remove_list[source_row]
            return not hide
        else:
            return True


CBX_OPT_C2SHORE_TRANSPORT = 4  # Index of the C2Shore sediment transport option
CBX_OPT_WAVE_FORCING_NONE = 0  # Index of the "None" wave forcing option


def scalar_cell_filter(item):
    """Check if a tree item is a cell-based scalar but only if it is a dataset.

    Args:
        item (TreeNode): The item to check

    Returns:
        (bool): True if the tree item is scalar and cell-based or is not a dataset.
    """
    if type(item.data) is xmd.DatasetItem:
        if item.num_components == 1 and item.data_location == 'CELL':
            return True
        return False
    return True


def vector_cell_filter(item):
    """Check if a tree item is a cell-based vector but only if it is a dataset.

    Args:
        item (TreeNode): The item to check

    Returns:
        (bool): True if the tree item is vector and cell-based or is not a dataset.
    """
    if type(item.data) is xmd.DatasetItem:
        if item.num_components == 2 and item.data_location == 'CELL':
            return True
        return False
    return True


class ModelControlDlg(XmsDlg):
    """A dialog for editing model control data."""
    def __init__(self, sim_data, pe_tree, sim_uuid, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            sim_data (SimulationData): The simulation data to view
            pe_tree (TreeNode): Project explorer dump from Query at root
            sim_uuid (str): UUID of the simulation
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.cmsflow.gui.model_control_dlg')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:CMS-Flow_Model_Control'
        self.sim_data = sim_data
        self.setWindowTitle('CMS-Flow Model Control')
        tree_copier = tree_util.ProjectExplorerTreeCreator()
        domain_uuid = find_domain_uuid(pe_tree, sim_uuid)
        st = tree_copier.copy(pe_tree)
        vt = tree_copier.copy(pe_tree)
        self.scalar_tree = tree_util.trim_project_explorer(st, domain_uuid)
        self.vector_tree = tree_util.trim_project_explorer(vt, domain_uuid)

        self.sim_duration_units = ['hours', 'days', 'weeks']
        self.ramp_duration_units = ['seconds', 'minutes', 'hours', 'days']
        self.hot_start_units = ['minutes', 'hours', 'days', 'weeks']
        self.solution_schemes = ['Implicit', 'Explicit']
        self.matrix_solvers = ['Gauss-Seidel', 'Gauss-Seidel-SOR', 'BiCGSTab', 'GMRES']
        self.hydro_time_duration_units = ['seconds', 'minutes', 'hours']
        self.latitude_types = ['From projection', 'User specified']
        self.turbulence_models = ['Subgrid', 'Falconer', 'Parabolic', 'Mixing length']
        self.bottom_friction = [
            'Quadratic', 'Soulsby (1995) Data2', 'Soulsby (1995) Data13', 'Fredsoe (1984)',
            'Huynh-Thanh and Termperville (1991)'
        ]
        self.bottom_roughness = ['Mannings N', 'Bottom friction coefficient', 'Roughness height (m)']
        self.roughness_source = ['Dataset', 'Constant']
        self.concentration_type = ['Global concentration', 'Spatially varied']
        self.temperature_type = ['Constant water temperature', 'Spatially varied']
        self.wave_types = ['None', 'Single wave condition', 'Inline steering']
        self.steering_types = ['Constant', 'Automatic']
        self.wave_water_predictors = ['Last time step', 'Tidal', 'Tidal plus variation']
        self.flow_wave = ['Automatic', 'User specified']
        self.flow_wave_units = ['m', 'ft']
        self.wind_types = [
            'None', 'Spatially constant', 'Meteorological stations', 'Temporally and spatially varying from file'
        ]
        self.wind_file_types = ['Navy fleet numeric with pressure', 'OWI/PBL', 'Single ASCII file']
        self.wind_grid_types = ['Parameters', 'XY file']
        self.solution_types = ['XMDF binary output', 'ASCII output only']
        self.transport_times = ['seconds', 'minutes']
        self.morphology_change_times = ['hours', 'minutes']
        self.formulation_units = [
            'Equilibrium total load', 'Equilibrium bed load plus nonequilibrium susp. load', 'Nonequilibrium total load'
        ]
        self.transport_formulas = ['Lund-CIRP', 'van Rijn', 'Soulsby-van Rijn', 'Watanabe', 'C2SHORE']
        self.concentration_profiles = ['Exponential', 'Rouse', 'Lund-CIRP', 'van Rijn']
        self.sediment_densities = ['kg/m^3', 'gr/cm^3', 'lb/ft^3']
        self.grain_sizes = ['m', 'cm', 'mm', 'um', 'ft', 'in']
        self.adaptation_methods = [
            'Constant length', 'Constant time', 'Maximum of bed and suspended adaptation lengths',
            'Weighted average of bed and suspended adaptation lengths'
        ]
        self.bed_adaptation_methods = ['Constant length', 'Constant time', 'Depth dependent']
        self.adaptation_length = ['m', 'cm', 'ft', 'in']
        self.suspended_adaptation_methods = [
            'Constant length', 'Constant time', 'Constant coefficient', 'Armanini and Di Silvio', 'Lin', 'Gallappatti'
        ]
        self.mulitple_grain_types = ['Specify number of size classes only', 'Enter grain size values to use']
        self.bed_compositions = ['D50 Sigma', 'D16 D50 D84', 'D35 D50 D90']

        self.unit_dict = {
            'sim': -1,
            'ramp': -1,
            'single': -1,
            'auto': -1,
            'hydro': -1,
            'salinity': -1,
            'temperature': -1,
            'flow_wave': -1,
            'wave_flow': -1,
            'transport_time': -1,
            'morphologic_time': -1,
            'morphology_change': -1,
            'sediment_density': -1,
            'grain_size': -1,
            'total_load_adaptation_length': -1,
            'total_load_adaptation_time': -1,
            'bed_load_adaptation_length': -1,
            'bed_load_adaptation_time': -1,
            'suspended_load_adaptation_length': -1,
            'suspended_load_adaptation_time': -1,
            'min_bed_layer_thickness': -1,
            'max_bed_layer_thickness': -1,
            'mixing_layer_thickness': -1
        }
        self.sim_units = [1.0, 24.0, 168.0]  # hours, days, weeks
        self.ramp_units = [1.0, 60.0, 3600.0, 86400.0]  # seconds, minutes, hours, days
        self.hot_units = [1.0, 60.0, 1440.0, 10080.0]  # minutes, hours, days, weeks
        self.hydro_time_units = [1.0, 60.0, 3600.0]  # seconds, minutes, hours
        self.flow_wave_length_units = [10.0, 3.048]  # meters and US survey feet
        self.transport_time_units = [1.0, 60.0]  # seconds and minutes
        self.morphology_change_time_units = [60.0, 1.0]  # hours and minutes
        self.sediment_density_units = [1000.0, 1.0, 62.4279606]  # kg/m^3, gr/cm^3, and lb/ft^3
        # meters, centimeters, millimeters, micrometers, US survey feet and US survey inches
        self.grain_size_units = [1000000.0, 10000.0, 1000.0, 1.0, 304800.0, 25400.0]
        # meters, centimeters, US survey feet and US survey inches
        self.adaptation_units = [100.0, 1.0, 30.48, 2.54]

        # Set the project directory (if there is one) for resolving relative paths
        FileSelector.project_folder = XmsEnvironment.xms_environ_project_path()

        self.int_valid = QxIntValidator()
        self.positive_dbl_valid = QxDoubleValidator(parent=self)
        self.dbl_valid = QxDoubleValidator(parent=self)
        self.positive_dbl_valid.setDecimals(10)
        self.dbl_valid.setDecimals(10)

        self.int_valid.setBottom(1)
        self.positive_dbl_valid.setBottom(0.0)

        self.steering_interval_validator = QxDoubleValidator(parent=self)
        self.steering_interval_validator.setDecimals(10)
        self.steering_interval_validator.setBottom(0.25)

        self.vertical_layout = QVBoxLayout(self)
        self.vertical_layout.setObjectName(u"vertical_layout")

        self.tab_widget = QTabWidget(self)
        self.tab_widget.setObjectName(u"tabs")
        self.general_scroll_area = QScrollArea()
        self.general_scroll_area.setWidgetResizable(True)
        self.general_widget = QWidget()
        self.general_tab = Ui_GeneralTab()
        self.general_tab.setupUi(self.general_widget)
        self.general_scroll_area.setWidget(self.general_widget)
        self.tab_widget.addTab(self.general_scroll_area, 'General')

        self.flow_scroll_area = QScrollArea()
        self.flow_scroll_area.setWidgetResizable(True)
        self.flow_widget = QWidget()
        self.flow_tab = Ui_FlowTab()
        self.flow_tab.setupUi(self.flow_widget)
        self.flow_scroll_area.setWidget(self.flow_widget)
        self.tab_widget.addTab(self.flow_scroll_area, 'Flow')

        self.sediment_scroll_area = QScrollArea()
        self.sediment_scroll_area.setWidgetResizable(True)
        self.sediment_widget = QWidget()
        self.sediment_tab = Ui_SedimentTab()
        self.sediment_tab.setupUi(self.sediment_widget)

        self.grain_table = GrainSizesTableWidget(
            self.sediment_tab.simplified_bed_page, self.sim_data.simple_grain_sizes_table.to_dataframe()
        )
        self.grain_table.setObjectName('grain_sizes_table')
        self.grain_table.setMinimumHeight(150)  # min height from the XML
        self.sediment_tab.simplified_bed_page.layout().insertWidget(3, self.grain_table)

        self.bed_layers_table = BedLayersTableWidget(
            self.sediment_tab.advanced_bed_page, self.sim_data.bed_layer_table.to_dataframe(), self.scalar_tree
        )
        self.bed_layers_table.setObjectName('bed_layers_table')
        self.bed_layers_table.setMinimumHeight(215)  # min height from the XML
        self.sediment_tab.advanced_bed_page.layout().insertWidget(5, self.bed_layers_table)

        self.diameters_table = \
            SedimentDiametersTableWidget(self.sediment_tab.size_classes_group,
                                         self.sim_data.advanced_sediment_diameters_table.to_dataframe())
        self.diameters_table.setObjectName('sediment_diameters_table')
        self.diameters_table.setMinimumHeight(215)  # min height from the XML
        self.sediment_tab.size_classes_group.layout().insertWidget(2, self.diameters_table)

        self._set_sediment_note_colors()
        self._set_size_and_layer_labels()
        self.sediment_tab.non_const_adaptation_group.setStyleSheet('#non_const_adaptation_group {border: none}')
        self.sediment_scroll_area.setWidget(self.sediment_widget)
        self.tab_widget.addTab(self.sediment_scroll_area, 'Sediment Transport')

        self.sal_temp_scroll_area = QScrollArea()
        self.sal_temp_scroll_area.setWidgetResizable(True)
        self.sal_temp_widget = QWidget()
        self.sal_temp_tab = Ui_SalinityTemperatureTab()
        self.sal_temp_tab.setupUi(self.sal_temp_widget)
        self.atmos_table = AtmosphericTableWidget(
            self.sal_temp_tab.atmospheric_group, self.sim_data.atmospheric_table.to_dataframe()
        )
        self.atmos_table.setObjectName('atmospheric_table')
        self.atmos_table.setMinimumHeight(325)  # min height from the XML
        self.sal_temp_tab.atmospheric_group.layout().addWidget(self.atmos_table)
        self.sal_temp_scroll_area.setWidget(self.sal_temp_widget)
        self.tab_widget.addTab(self.sal_temp_scroll_area, 'Salinity/Temperature')

        self.wave_scroll_area = QScrollArea()
        self.wave_scroll_area.setWidgetResizable(True)
        self.wave_widget = QWidget()
        self.wave_tab = Ui_WaveTab()
        self.wave_tab.setupUi(self.wave_widget)
        self.wave_scroll_area.setWidget(self.wave_widget)
        self.tab_widget.addTab(self.wave_scroll_area, 'Wave')

        self.wind_scroll_area = QScrollArea()
        self.wind_scroll_area.setWidgetResizable(True)
        self.wind_widget = QWidget()
        self.wind_tab = Ui_WindTab()
        self.wind_tab.setupUi(self.wind_widget)

        self.wind_table = WindParametersTableWidget(
            self.wind_tab.parameters_group, self.sim_data.wind_from_table.to_dataframe()
        )
        self.wind_table.setObjectName('wind_table')
        self.wind_table.setMinimumHeight(205)  # min height from the XML
        self.wind_tab.parameters_group.layout().insertWidget(3, self.wind_table)  # 3 is after the labels

        self.met_stations_table = MeteorologicalStationsTableWidget(
            self.wind_tab.meteorological_group, self.sim_data.meteorological_stations_table.to_dataframe(),
            self.sim_data
        )
        self.met_stations_table.setObjectName('meteorological_stations_table')
        self.met_stations_table.setMinimumHeight(205)  # min height from the XML
        self.wind_tab.meteorological_group.setLayout(QVBoxLayout())
        self.wind_tab.meteorological_group.layout().addWidget(self.met_stations_table)

        self.wind_scroll_area.setWidget(self.wind_widget)
        self.tab_widget.addTab(self.wind_scroll_area, 'Wind')

        self.output_scroll_area = QScrollArea()
        self.output_scroll_area.setWidgetResizable(True)
        self.output_widget = QWidget()
        self.output_tab = Ui_OutputTab()
        self.output_tab.setupUi(self.output_widget)

        self.list_1_table = self._add_output_list(
            self.output_tab.list_1_tab, self.sim_data.list_1_table.to_dataframe(), 'list_1_table'
        )
        self.list_2_table = self._add_output_list(
            self.output_tab.list_2_tab, self.sim_data.list_2_table.to_dataframe(), 'list_2_table'
        )
        self.list_3_table = self._add_output_list(
            self.output_tab.list_3_tab, self.sim_data.list_3_table.to_dataframe(), 'list_3_table'
        )
        self.list_4_table = self._add_output_list(
            self.output_tab.list_4_tab, self.sim_data.list_4_table.to_dataframe(), 'list_4_table'
        )
        self.hydro_table = self._add_output_stat_list(self.output_tab.statistical_output_group, 1, 'hydro_table')
        self.sediment_table = self._add_output_stat_list(self.output_tab.statistical_output_group, 3, 'sediment_table')
        self.salinity_table = self._add_output_stat_list(self.output_tab.statistical_output_group, 5, 'salinity_table')
        self.wave_table = self._add_output_stat_list(self.output_tab.statistical_output_group, 7, 'wave_table')
        self.output_option_model = QStandardItemModel()
        self._build_output_model()
        self.output_tab.dataset_options_tree.setModel(self.output_option_model)
        self.output_tab.dataset_options_tree.header().setVisible(False)
        self.output_tab.dataset_options_tree.header().resizeSection(1, 20)
        self.output_tab.dataset_options_tree.header().resizeSection(2, 30)
        self.output_tab.dataset_options_tree.header().setSectionResizeMode(0, QHeaderView.ResizeToContents)
        self.output_tab.dataset_options_tree.header().setSectionResizeMode(1, QHeaderView.Fixed)
        self.output_tab.dataset_options_tree.header().setSectionResizeMode(2, QHeaderView.Fixed)
        self.output_tab.dataset_options_tree.expandAll()
        self.check_delegate = CheckBoxNoTextDelegate()
        self.list_delegate = QxCbxDelegate()
        self.list_delegate.set_strings(['List 1', 'List 2', 'List 3', 'List 4'])
        self.output_tab.dataset_options_tree.setItemDelegateForColumn(1, self.check_delegate)
        self.output_tab.dataset_options_tree.setItemDelegateForColumn(2, self.list_delegate)

        # Test for added checkbox in sim_data
        if self.sim_data.output.attrs.get('ENABLE_WAVE_STATISTICS') is None:
            self.sim_data.output.attrs.update({'ENABLE_WAVE_STATISTICS': 0})
            self.sim_data.output.attrs.update({'WAVE_START_TIME': 0.0})
            self.sim_data.output.attrs.update({'WAVE_INCREMENT': 1.0})
            self.sim_data.output.attrs.update({'WAVE_END_TIME': 720.0})

        # Test for added bottom roughness cards in sim_data
        if self.sim_data.flow.attrs.get('ROUGHNESS_CONSTANT') is None:
            self.sim_data.flow.attrs.update({'ROUGHNESS_CONSTANT': 0.025})
            self.sim_data.flow.attrs.update({'ROUGHNESS_SOURCE': 'Dataset'})

        self.output_scroll_area.setWidget(self.output_widget)
        self.tab_widget.addTab(self.output_scroll_area, 'Output')

        self.btn_box = QDialogButtonBox(self)
        self.btn_box.setObjectName(u"btn_box")
        self.btn_box.setOrientation(Qt.Horizontal)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Help | QDialogButtonBox.Ok)

        self._setup_tabs()

        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)

        self.vertical_layout.addWidget(self.tab_widget)
        self.vertical_layout.addSpacerItem(QSpacerItem(0, 0))
        self.vertical_layout.addWidget(self.btn_box)

        # Pass the widget size past the scroll area up to the parent dialog
        size = self.flow_widget.sizeHint()
        geom = self.geometry()
        geom.setSize(size)
        self.setGeometry(geom)

        # center dialog in screen containing mouse, was at top left of all screens
        frame_gm = self.frameGeometry()
        screen = QGuiApplication.screenAt(QApplication.desktop().cursor().pos())
        if screen:
            center_point = screen.geometry().center()
            frame_gm.moveCenter(center_point)
            self.move(frame_gm.topLeft())

    def _get_selected_file(self, widget):
        """Gets the selected file text.

        Args:
            widget (QLabel): The label widget that shows the file path.

        Returns:
            Returns the path of the selected file. The file path is an empty string if nothing is selected.
        """
        path = widget.text()
        if path == '(none_selected)':
            return ''
        else:
            return path

    @staticmethod
    def _add_output_list(tab, df, name):
        """Creates an output time list.

        Args:
            tab (QTabWidget): The tab to add the table to.
            df (pandas.Dataframe): The dataframe of the table data.
            name (str): The object name for the new table.
        """
        table = OutputListTableWidget(tab, df)
        table.setObjectName(name)
        table.setMinimumHeight(125)
        tab.setLayout(QVBoxLayout())
        tab.layout().addWidget(table)
        return table

    @staticmethod
    def _add_output_stat_list(parent, pos, name):
        """Creates an output time list.

        Args:
            parent (QWidget): The parent widget.
            pos (int): Widget layout position.
            name (str): The object name for the new table.
        """
        stat_output_table = {
            'start_time': xr.DataArray(data=np.array([0.0], dtype=float)),
            'increment': xr.DataArray(data=np.array([1.0], dtype=float)),
            'end_time': xr.DataArray(data=np.array([720.0], dtype=float))
        }
        table = OutputListTableWidget(parent, xr.Dataset(data_vars=stat_output_table).to_dataframe(), True)
        table.setObjectName(name)
        table.table_view.verticalHeader().hide()
        table.setMinimumHeight(60)
        parent.layout().insertWidget(pos, table)
        return table

    def _build_output_model(self):
        """Build a model for the output options."""
        wse_group = self._build_list_output_group('Water surface elevation (m)', True)
        self.output_option_model.appendRow(wse_group)

        velocity_group = self._build_list_output_group('Current velocity', True)
        self.output_option_model.appendRow(velocity_group)
        velocity_group[0].appendRow(self._build_output_option('Current magnitude (m/s)', False))
        velocity_group[0].appendRow(self._build_output_option('Current velocity (m/s)', True))

        morphology_group = self._build_list_output_group('Morphology', False)
        self.output_option_model.appendRow(morphology_group)
        morphology_group[0].appendRow(self._build_output_option('Depth (m, through time)', True))
        morphology_group[0].appendRow(self._build_output_option('Morphology (m)', False))

        transport_group = self._build_list_output_group('Transport', False)
        self.output_option_model.appendRow(transport_group)
        transport_group[0].appendRow(self._build_output_option('Sediment total-load capacity (kg/m^3)', False))
        transport_group[0].appendRow(self._build_output_option('Sediment total-load concentration (kg/m/s)', False))
        transport_group[0].appendRow(self._build_output_option('Fraction suspended', False))
        transport_group[0].appendRow(self._build_output_option('Fraction bedload', False))
        transport_group[0].appendRow(self._build_output_option('Total sediment transport (kg/m^3)', True))
        transport_group[0].appendRow(self._build_output_option('Salinity concentration (ppt)', True))

        wave_group = self._build_list_output_group('Wave', False)
        self.output_option_model.appendRow(wave_group)
        wave_group[0].appendRow(self._build_output_option('Wave height (m)', True))
        wave_group[0].appendRow(self._build_output_option('Wave period (s)', True))
        wave_group[0].appendRow(self._build_output_option('Wave height vector (m)', True))
        wave_group[0].appendRow(self._build_output_option('Wave dissipation (m)', False))

        wind_group = self._build_list_output_group('Wind', False)
        self.output_option_model.appendRow(wind_group)
        wind_group[0].appendRow(self._build_output_option('Wind speed (m/s)', False))
        wind_group[0].appendRow(self._build_output_option('Wind speed vector (m/s)', True))
        wind_group[0].appendRow(self._build_output_option('Atmospheric pressure (Pa)', False))

        eddy_group = self._build_list_output_group('Eddy viscosity (m^2/s)', False)
        self.output_option_model.appendRow(eddy_group)

    @staticmethod
    def _build_list_output_group(group_name, is_always_checked):
        """Builds a single output list group.

        Args:
            group_name (str): Name of the output option group.
            is_always_checked (bool): True if the group is always on.
        """
        group = QStandardItem(group_name)
        check = QStandardItem('')
        if is_always_checked:
            check.setData(True, Qt.DisplayRole)
            # check.setEditable(False)
            check.setEnabled(False)
        else:
            check.setData(False, Qt.DisplayRole)
        list_select = QStandardItem('List 1')
        return [group, check, list_select]

    @staticmethod
    def _build_output_option(option_text, is_always_checked):
        """Builds a single output option.

        Args:
            option_text (str): Name of the output option.
            is_always_checked (bool): True if the option is always on.
        """
        option = QStandardItem(option_text)
        check = QStandardItem('')
        if is_always_checked:
            check.setData(True, Qt.DisplayRole)
            check.setData(True, Qt.UserRole)
            check.setEditable(False)
            check.setEnabled(False)
        else:
            check.setData(False, Qt.DisplayRole)
            check.setData(False, Qt.UserRole)
        return [option, check]

    def _setup_tabs(self):
        """This does the setup for the different tabs."""
        self._setup_general_combo_boxes()
        self._setup_general_edit_fields()
        self._load_general_data()  # load data before making connections
        self._setup_general_connections()
        self.general_tab.hot_start_note_label.setStyleSheet('color: red;')

        self._setup_flow_combo_boxes()
        self._setup_flow_edit_fields()
        self._load_flow_data()  # load data before making connections
        self._setup_flow_connections()

        self._setup_sediment_combo_boxes()
        self._setup_sediment_edit_fields()
        self._load_sediment_data()  # load data before making connections
        self._setup_sediment_connections()
        self._setup_c2shore_warning()  # Do this before loading wave data because it has a dependency on wave forcing

        self._setup_sal_temp_combo_boxes()
        self._setup_sal_temp_edit_fields()
        self._load_sal_temp_data()  # load data before making connections
        self._setup_sal_temp_connections()

        self._setup_wave_combo_boxes()
        self._setup_wave_edit_fields()
        self._load_wave_data()  # load data before making connections
        self._setup_wave_connections()

        self._setup_wind_combo_boxes()
        self._setup_wind_edit_fields()
        self._load_wind_data()  # load data before making connections
        self._setup_wind_connections()

        self._setup_output_combo_boxes()
        self._load_output_data()  # load data before making connections
        self._setup_output_connections()

    def _setup_general_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.general_tab.simulation_duration_units.addItems(self.sim_duration_units)
        self.general_tab.ramp_duration_units.addItems(self.ramp_duration_units)
        self.general_tab.single_hot_start_units.addItems(self.hot_start_units)
        self.general_tab.auto_hot_start_interval_units.addItems(self.hot_start_units)
        self.general_tab.solution_scheme.addItems(self.solution_schemes)
        self.general_tab.matrix_solver.addItems(self.matrix_solvers)

    def _setup_general_edit_fields(self):
        """Sets up validators for edit fields."""
        self.general_tab.number_threads.setValidator(self.int_valid)
        self.general_tab.simulation_duration_value.setValidator(self.positive_dbl_valid)
        self.general_tab.ramp_duration_value.setValidator(self.positive_dbl_valid)
        self.general_tab.single_hot_start_value.setValidator(self.positive_dbl_valid)
        self.general_tab.auto_hot_start_interval_value.setValidator(self.positive_dbl_valid)

    def _setup_general_connections(self):
        """Set up signal/slot connections in the general tab."""
        self.general_tab.simulation_duration_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.general_tab.simulation_duration_value, self.sim_units, 'sim', self.unit_dict
            )
        )
        self.general_tab.ramp_duration_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.general_tab.ramp_duration_value, self.ramp_units, 'ramp', self.unit_dict
            )
        )
        self.general_tab.single_hot_start_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.general_tab.single_hot_start_value, self.hot_units, 'single', self.unit_dict
            )
        )
        self.general_tab.auto_hot_start_interval_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.general_tab.auto_hot_start_interval_value, self.hot_units, 'auto', self.unit_dict
            )
        )
        self.general_tab.use_initial_conditions_file.stateChanged[int].connect(
            self.general_tab.initial_conditions_file.setEnabled
        )
        self.general_tab.use_initial_conditions_file.stateChanged[int].connect(
            self.general_tab.initial_conditions_file_label.setEnabled
        )

        self.general_tab.write_single_hot_start_file.stateChanged[int].connect(self.single_hot_start_visible)
        self.general_tab.write_recurring_hot_start_file.stateChanged[int].connect(self.auto_hot_start_visible)

        self.general_tab.write_single_hot_start_file.stateChanged[int].connect(self.hot_start_note_visible)
        self.general_tab.write_recurring_hot_start_file.stateChanged[int].connect(self.hot_start_note_visible)

        self.general_tab.use_initial_conditions_file.stateChanged[int].connect(self.initial_condition_enabled)
        self.general_tab.initial_conditions_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.general_tab.initial_conditions_file_label, 'Select initial conditions file',
                'Initial conditions file (*.h5 *.sup);;All Files (*.*)'
            )
        )
        self.general_tab.solution_scheme.currentIndexChanged[int].connect(self._set_general_implicit_explict)

    def _set_general_implicit_explict(self, idx):
        """Set the text the user sees warning them about the solution scheme used."""
        explicit_scheme_idx = 1  # Index in the combobox
        is_implicit = idx != explicit_scheme_idx
        self.general_tab.matrix_solver.setEnabled(is_implicit)
        self.general_tab.matrix_solver_label.setEnabled(is_implicit)

    def hot_start_note_visible(self):
        """Sets the visible state of the note in the general area."""
        visible = self.general_tab.write_single_hot_start_file.checkState() == Qt.Checked and\
            self.general_tab.write_recurring_hot_start_file.checkState() == Qt.Checked
        self.general_tab.hot_start_note_label.setVisible(visible)

    def single_hot_start_visible(self, check_state):
        """Sets the visible state of the single hot start."""
        self.general_tab.single_hot_start_label.setVisible(check_state)
        self.general_tab.single_hot_start_value.setVisible(check_state)
        self.general_tab.single_hot_start_units.setVisible(check_state)

    def auto_hot_start_visible(self, check_state):
        """Sets the visible state of the auto recurring hot start."""
        self.general_tab.auto_hot_start_interval_label.setVisible(check_state)
        self.general_tab.auto_hot_start_interval_value.setVisible(check_state)
        self.general_tab.auto_hot_start_interval_units.setVisible(check_state)

    def initial_condition_enabled(self, check_state):
        """Sets the enabled state of the initial condition file button and text."""
        self.general_tab.initial_conditions_file.setEnabled(check_state)
        self.general_tab.initial_conditions_file_label.setEnabled(check_state)

    def _load_general_data(self):
        """Load the data for the general tab."""
        date_time_text = self.sim_data.general.attrs['DATE_START']
        if date_time_text:
            date_time = QDateTime.fromString(date_time_text, Qt.ISODate)
            self.general_tab.start_date_time.setDateTime(date_time)
        self.general_tab.simulation_duration_value.setText(str(self.sim_data.general.attrs['SIM_DURATION_VALUE']))
        self.general_tab.simulation_duration_units.setCurrentIndex(
            self.sim_duration_units.index(self.sim_data.general.attrs['SIM_DURATION_UNITS'])
        )
        self.unit_dict['sim'] = self.general_tab.simulation_duration_units.currentIndex()
        self.general_tab.ramp_duration_value.setText(str(self.sim_data.general.attrs['RAMP_DURATION_VALUE']))
        self.general_tab.ramp_duration_units.setCurrentIndex(
            self.ramp_duration_units.index(self.sim_data.general.attrs['RAMP_DURATION_UNITS'])
        )
        self.unit_dict['ramp'] = self.general_tab.ramp_duration_units.currentIndex()
        self.general_tab.use_second_order_skewness.setChecked(self.sim_data.general.attrs['SKEW_CORRECT'] != 0)
        self.general_tab.use_initial_conditions_file.setChecked(
            self.sim_data.general.attrs['USE_INIT_CONDITIONS_FILE'] != 0
        )
        self.general_tab.initial_conditions_file_label.setText(self.sim_data.general.attrs['INIT_CONDITIONS_FILE'])
        self.general_tab.write_single_hot_start_file.setChecked(
            self.sim_data.general.attrs['USE_HOT_START_OUTPUT_FILE'] != 0
        )
        self.general_tab.single_hot_start_value.setText(
            str(self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_VALUE'])
        )
        self.general_tab.single_hot_start_units.setCurrentIndex(
            self.hot_start_units.index(self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_UNITS'])
        )
        self.unit_dict['single'] = self.general_tab.single_hot_start_units.currentIndex()
        self.general_tab.write_recurring_hot_start_file.setChecked(
            self.sim_data.general.attrs['RECURRING_HOT_START_FILE'] != 0
        )
        self.general_tab.auto_hot_start_interval_value.setText(
            str(self.sim_data.general.attrs['AUTO_HOT_DURATION_VALUE'])
        )
        self.general_tab.auto_hot_start_interval_units.setCurrentIndex(
            self.hot_start_units.index(self.sim_data.general.attrs['AUTO_HOT_DURATION_UNITS'])
        )
        self.unit_dict['auto'] = self.general_tab.auto_hot_start_interval_units.currentIndex()
        self.general_tab.solution_scheme.setCurrentIndex(
            self.solution_schemes.index(self.sim_data.general.attrs['SOLUTION_SCHEME'])
        )
        self.general_tab.matrix_solver.setCurrentIndex(
            self.matrix_solvers.index(self.sim_data.general.attrs['MATRIX_SOLVER'])
        )
        self.general_tab.number_threads.setText(str(self.sim_data.general.attrs['NUM_THREADS']))
        self.hot_start_note_visible()
        self._set_general_implicit_explict(self.general_tab.solution_scheme.currentIndex())
        self.single_hot_start_visible(self.general_tab.write_single_hot_start_file.checkState())
        self.auto_hot_start_visible(self.general_tab.write_recurring_hot_start_file.checkState())
        self.initial_condition_enabled(self.general_tab.use_initial_conditions_file.checkState())

    def _save_general_data(self):
        """Save data from the general attributes."""
        self.sim_data.general.attrs['DATE_START'] = self.general_tab.start_date_time.dateTime().toString(Qt.ISODate)
        self.sim_data.general.attrs['SIM_DURATION_VALUE'] = float(self.general_tab.simulation_duration_value.text())
        self.sim_data.general.attrs['SIM_DURATION_UNITS'] = self.general_tab.simulation_duration_units.currentText()
        self.sim_data.general.attrs['RAMP_DURATION_VALUE'] = float(self.general_tab.ramp_duration_value.text())
        self.sim_data.general.attrs['RAMP_DURATION_UNITS'] = self.general_tab.ramp_duration_units.currentText()
        self.sim_data.general.attrs['SKEW_CORRECT'] =\
            1 if self.general_tab.use_second_order_skewness.checkState() == Qt.Checked else 0
        self.sim_data.general.attrs['USE_INIT_CONDITIONS_FILE'] =\
            1 if self.general_tab.use_initial_conditions_file.checkState() == Qt.Checked else 0
        self.sim_data.general.attrs['INIT_CONDITIONS_FILE'] = self._get_selected_file(
            self.general_tab.initial_conditions_file_label
        )
        self.sim_data.general.attrs['USE_HOT_START_OUTPUT_FILE'] =\
            1 if self.general_tab.write_single_hot_start_file.checkState() == Qt.Checked else 0
        self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_VALUE'] =\
            float(self.general_tab.single_hot_start_value.text())
        self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_UNITS'] =\
            self.general_tab.single_hot_start_units.currentText()
        self.sim_data.general.attrs['RECURRING_HOT_START_FILE'] =\
            1 if self.general_tab.write_recurring_hot_start_file.checkState() == Qt.Checked else 0
        self.sim_data.general.attrs['AUTO_HOT_DURATION_VALUE'] =\
            float(self.general_tab.auto_hot_start_interval_value.text())
        self.sim_data.general.attrs['AUTO_HOT_DURATION_UNITS'] =\
            self.general_tab.auto_hot_start_interval_units.currentText()
        self.sim_data.general.attrs['SOLUTION_SCHEME'] = self.general_tab.solution_scheme.currentText()
        self.sim_data.general.attrs['MATRIX_SOLVER'] = self.general_tab.matrix_solver.currentText()
        self.sim_data.general.attrs['NUM_THREADS'] = int(self.general_tab.number_threads.text())

    def _setup_flow_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.flow_tab.hydro_time_step_units.addItems(self.hydro_time_duration_units)
        self.flow_tab.latitude_coriolis_type.addItems(self.latitude_types)
        self.flow_tab.turbulence_model.addItems(self.turbulence_models)
        self.flow_tab.wave_friction_type.addItems(self.bottom_friction)
        self.flow_tab.bottom_roughness_type.addItems(self.bottom_roughness)
        self.flow_tab.roughness_source.addItems(self.roughness_source)

    def _setup_flow_edit_fields(self):
        """Sets up validators for edit fields."""
        self.flow_tab.hydro_time_step_value.setValidator(self.positive_dbl_valid)
        self.flow_tab.wetting_depth.setValidator(self.dbl_valid)
        self.flow_tab.coriolis_degrees.setValidator(self.dbl_valid)
        self.flow_tab.base_value.setValidator(self.dbl_valid)
        self.flow_tab.current_bottom.setValidator(self.dbl_valid)
        self.flow_tab.current_horizontal.setValidator(self.dbl_valid)
        self.flow_tab.wave_bottom.setValidator(self.dbl_valid)
        self.flow_tab.wave_breaking.setValidator(self.dbl_valid)
        self.flow_tab.wave_bottom_friction.setValidator(self.dbl_valid)
        self.flow_tab.roughness_constant.setValidator(self.positive_dbl_valid)

    def _setup_flow_connections(self):
        """Set up signal/slot connections in the flow tab."""
        self.flow_tab.hydro_time_step_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.flow_tab.hydro_time_step_value, self.hydro_time_units, 'hydro', self.unit_dict
            )
        )

        self.general_tab.solution_scheme.currentIndexChanged[int].connect(self._set_flow_hydro_implicit_explict)

        self.flow_tab.latitude_coriolis_type.currentIndexChanged[int].connect(self.flow_tab.coriolis_degrees.setEnabled)
        self.flow_tab.wave_friction_type.currentIndexChanged[int].connect(
            self.flow_tab.wave_bottom_friction.setDisabled
        )
        self.flow_tab.wave_friction_type.currentIndexChanged[int].connect(
            self.flow_tab.wave_bottom_friction_label.setDisabled
        )

        self.flow_tab.bottom_roughness_dset.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.flow_tab.bottom_roughness_dset_path, 'Select bottom roughness dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.flow.attrs, 'BOTTOM_ROUGHNESS_DSET', resources_util.get_quadtree_icon
            )
        )

        self.flow_tab.roughness_source.currentIndexChanged[int].connect(
            self.flow_tab.roughness_type_stack.setCurrentIndex
        )

    def _set_flow_hydro_implicit_explict(self, idx):
        """Set the text the user sees warning them about the solution scheme used."""
        explicit_scheme_idx = 1  # Index in the combobox
        self.flow_tab.implicit_explicit_note_label.setText(
            f'Note: {self.general_tab.solution_scheme.currentText().upper()} solution scheme is being used.'
        )
        if idx == explicit_scheme_idx:
            self.flow_tab.implicit_explicit_note_label.setStyleSheet('color: red;')
        else:
            self.flow_tab.implicit_explicit_note_label.setStyleSheet('color: black;')

    def _load_flow_data(self):
        """Load the data for the flow tab."""
        self.flow_tab.hydro_time_step_value.setText(str(self.sim_data.flow.attrs['HYDRO_TIME_STEP_VALUE']))
        self.flow_tab.hydro_time_step_units.setCurrentIndex(
            self.hydro_time_duration_units.index(self.sim_data.flow.attrs['HYDRO_TIME_STEP_UNITS'])
        )
        self.unit_dict['hydro'] = self.flow_tab.hydro_time_step_units.currentIndex()
        self.flow_tab.wetting_depth.setText(str(self.sim_data.flow.attrs['WETTING_DEPTH']))
        self.flow_tab.use_wave_fluxes.setChecked(self.sim_data.flow.attrs['WAVE_FLUXES'] != 0)
        self.flow_tab.use_roller_fluxes.setChecked(self.sim_data.flow.attrs['ROLLER_FLUXES'] != 0)
        self.flow_tab.latitude_coriolis_type.setCurrentIndex(
            self.latitude_types.index(self.sim_data.flow.attrs['LATITUDE_CORIOLIS'])
        )
        self.flow_tab.coriolis_degrees.setText(str(self.sim_data.flow.attrs['DEGREES']))
        self.flow_tab.turbulence_model.setCurrentIndex(
            self.turbulence_models.index(self.sim_data.flow.attrs['TURBULENCE_MODEL'])
        )
        self.flow_tab.edit_turbulance_group.setChecked(self.sim_data.flow.attrs['TURBULENCE_PARAMETERS'] != 0)
        self.flow_tab.base_value.setText(str(self.sim_data.flow.attrs['BASE_VALUE']))
        self.flow_tab.current_bottom.setText(str(self.sim_data.flow.attrs['CURRENT_BOTTOM_COEFFICIENT']))
        self.flow_tab.current_horizontal.setText(str(self.sim_data.flow.attrs['CURRENT_HORIZONTAL_COEFFICIENT']))
        self.flow_tab.wave_bottom.setText(str(self.sim_data.flow.attrs['WAVE_BOTTOM_COEFFICIENT']))
        self.flow_tab.wave_breaking.setText(str(self.sim_data.flow.attrs['WAVE_BREAKING_COEFFICIENT']))
        self.flow_tab.wave_friction_type.setCurrentIndex(
            self.bottom_friction.index(self.sim_data.flow.attrs['WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT'])
        )
        self.flow_tab.wave_bottom_friction.setText(str(self.sim_data.flow.attrs['QUAD_WAVE_BOTTOM_COEFFICIENT']))
        self.flow_tab.use_bed_slope_friction.setChecked(self.sim_data.flow.attrs['BED_SLOPE_FRIC_COEFFICIENT'] != 0)
        self.flow_tab.use_wall_friction.setChecked(self.sim_data.flow.attrs['WALL_FRICTION'] != 0)
        self.flow_tab.bottom_roughness_type.setCurrentIndex(
            self.bottom_roughness.index(self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'])
        )
        self.flow_tab.roughness_source.setCurrentIndex(
            self.roughness_source.index(self.sim_data.flow.attrs['ROUGHNESS_SOURCE'])
        )
        self.flow_tab.roughness_constant.setText(str(self.sim_data.flow.attrs['ROUGHNESS_CONSTANT']))
        self.flow_tab.roughness_type_stack.setCurrentIndex(self.flow_tab.roughness_source.currentIndex())
        self.flow_tab.bottom_roughness_dset_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.flow.attrs['BOTTOM_ROUGHNESS_DSET'])
            )
        )
        self._set_flow_hydro_implicit_explict(self.general_tab.solution_scheme.currentIndex())
        self.flow_tab.coriolis_degrees.setEnabled(self.flow_tab.latitude_coriolis_type.currentIndex() != 0)
        self.flow_tab.wave_bottom_friction.setDisabled(self.flow_tab.wave_friction_type.currentIndex() != 0)
        self.flow_tab.wave_bottom_friction_label.setDisabled(self.flow_tab.wave_friction_type.currentIndex() != 0)

    def _save_flow_data(self):
        """Save data from the flow attributes."""
        self.sim_data.flow.attrs['HYDRO_TIME_STEP_VALUE'] = float(self.flow_tab.hydro_time_step_value.text())
        self.sim_data.flow.attrs['HYDRO_TIME_STEP_UNITS'] = self.flow_tab.hydro_time_step_units.currentText()
        self.sim_data.flow.attrs['WETTING_DEPTH'] = float(self.flow_tab.wetting_depth.text())
        self.sim_data.flow.attrs['WAVE_FLUXES'] = 1 if self.flow_tab.use_wave_fluxes.checkState() == Qt.Checked else 0
        self.sim_data.flow.attrs['ROLLER_FLUXES'] =\
            1 if self.flow_tab.use_roller_fluxes.checkState() == Qt.Checked else 0
        self.sim_data.flow.attrs['LATITUDE_CORIOLIS'] = self.flow_tab.latitude_coriolis_type.currentText()
        self.sim_data.flow.attrs['DEGREES'] = float(self.flow_tab.coriolis_degrees.text())
        self.sim_data.flow.attrs['TURBULENCE_MODEL'] = self.flow_tab.turbulence_model.currentText()
        self.sim_data.flow.attrs['TURBULENCE_PARAMETERS'] = 1 if self.flow_tab.edit_turbulance_group.isChecked() else 0
        self.sim_data.flow.attrs['BASE_VALUE'] = float(self.flow_tab.base_value.text())
        self.sim_data.flow.attrs['CURRENT_BOTTOM_COEFFICIENT'] = float(self.flow_tab.current_bottom.text())
        self.sim_data.flow.attrs['CURRENT_HORIZONTAL_COEFFICIENT'] = float(self.flow_tab.current_horizontal.text())
        self.sim_data.flow.attrs['WAVE_BOTTOM_COEFFICIENT'] = float(self.flow_tab.wave_bottom.text())
        self.sim_data.flow.attrs['WAVE_BREAKING_COEFFICIENT'] = float(self.flow_tab.wave_breaking.text())
        self.sim_data.flow.attrs['WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT'] =\
            self.flow_tab.wave_friction_type.currentText()
        self.sim_data.flow.attrs['QUAD_WAVE_BOTTOM_COEFFICIENT'] = float(self.flow_tab.wave_bottom_friction.text())
        self.sim_data.flow.attrs['BED_SLOPE_FRIC_COEFFICIENT'] =\
            1 if self.flow_tab.use_bed_slope_friction.checkState() == Qt.Checked else 0
        self.sim_data.flow.attrs['WALL_FRICTION'] =\
            1 if self.flow_tab.use_wall_friction.checkState() == Qt.Checked else 0
        self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = self.flow_tab.bottom_roughness_type.currentText()
        self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = self.flow_tab.roughness_source.currentText()
        self.sim_data.flow.attrs['ROUGHNESS_CONSTANT'] = float(self.flow_tab.roughness_constant.text())

    def _setup_sal_temp_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.sal_temp_tab.sal_transport_rate_units.addItems(self.ramp_duration_units)
        self.sal_temp_tab.salinity_concentration.addItems(self.concentration_type)
        self.sal_temp_tab.temp_transport_rate_units.addItems(self.ramp_duration_units)
        self.sal_temp_tab.temperature_type.addItems(self.temperature_type)

    def _setup_sal_temp_edit_fields(self):
        """Sets up validators for edit fields."""
        self.sal_temp_tab.water_density.setValidator(self.dbl_valid)
        self.sal_temp_tab.const_water_temp.setValidator(self.dbl_valid)
        self.sal_temp_tab.sal_transport_rate_value.setValidator(self.positive_dbl_valid)
        self.sal_temp_tab.global_concentration.setValidator(self.dbl_valid)
        self.sal_temp_tab.temp_transport_rate_value.setValidator(self.positive_dbl_valid)

    def _setup_sal_temp_connections(self):
        """Set up signal/slot connections in the salinity/temperature tab."""
        self.sal_temp_tab.sal_transport_rate_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sal_temp_tab.sal_transport_rate_value, self.ramp_units, 'salinity', self.unit_dict
            )
        )
        self.sal_temp_tab.temp_transport_rate_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sal_temp_tab.temp_transport_rate_value, self.ramp_units, 'temperature', self.unit_dict
            )
        )
        self.sal_temp_tab.calculate_salinity.stateChanged[int].connect(
            self.sal_temp_tab.sal_initial_condition_group.setVisible
        )
        self.sal_temp_tab.calculate_temperature.stateChanged[int].connect(
            self.sal_temp_tab.temp_initial_condition_group.setVisible
        )
        self.sal_temp_tab.calculate_temperature.stateChanged[int].connect(
            self.sal_temp_tab.atmospheric_group.setVisible
        )

        self.general_tab.solution_scheme.currentIndexChanged[int].connect(
            self._set_salinity_temperature_explicit_time_steps
        )

        self.sal_temp_tab.salinity_concentration.currentIndexChanged[int].connect(
            self._set_const_or_varied_salinity_visible
        )
        self.sal_temp_tab.temperature_type.currentIndexChanged[int].connect(self._set_varied_temperature_visible)

        self.sal_temp_tab.temperature_type.currentIndexChanged[int].connect(self._set_water_temperature_visible)
        self.sal_temp_tab.calculate_temperature.stateChanged[int].connect(self._set_water_temperature_visible)

        self.sal_temp_tab.varied_concentration.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sal_temp_tab.varied_concentration_label, 'Select bottom roughness dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.salinity.attrs, 'SALINITY_INITIAL_CONCENTRATION', resources_util.
                get_quadtree_icon
            )
        )
        self.sal_temp_tab.varied_temperature.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sal_temp_tab.varied_temperature_label, 'Select bottom roughness dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.salinity.attrs, 'INITIAL_TEMPERATURE_DATASET', resources_util.
                get_quadtree_icon
            )
        )

    def _set_water_temperature_visible(self):
        """Set the constant water temperature visible."""
        visible = False
        if self.sal_temp_tab.calculate_temperature.checkState() == Qt.Unchecked or\
                self.sal_temp_tab.temperature_type.currentIndex() == 0:
            visible = True
        self.sal_temp_tab.const_water_temp.setVisible(visible)
        self.sal_temp_tab.const_water_temp_label.setVisible(visible)

    def _set_const_or_varied_salinity_visible(self, idx):
        """Sets the constant or varied salinity visible.

        Args:
            idx (int): The index in the salinity type combobox.
        """
        # 0 is the index of the 'Global concentration' entry
        is_const = idx == 0
        self.sal_temp_tab.global_concentration.setVisible(is_const)
        self.sal_temp_tab.global_concentration_label.setVisible(is_const)
        self.sal_temp_tab.varied_concentration.setVisible(not is_const)
        self.sal_temp_tab.varied_concentration_label.setVisible(not is_const)

    def _set_varied_temperature_visible(self, idx):
        """Sets the varied temperature visible.

        Args:
            idx (int): The index in the temperature type combobox.
        """
        # 0 is the index of the 'Constant water temperature' entry
        visible = idx != 0
        self.sal_temp_tab.varied_temperature.setVisible(visible)
        self.sal_temp_tab.varied_temperature_label.setVisible(visible)

    def _set_salinity_temperature_explicit_time_steps(self, idx):
        """Sets time steps for salinity and temperature visible if explicit."""
        # 0 is Implicit, 1 is Explicit
        visible = idx != 0
        self.sal_temp_tab.sal_time_steps_group.setVisible(visible)
        self.sal_temp_tab.temp_time_steps_group.setVisible(visible)

    def _load_sal_temp_data(self):
        """Load the data for the salinity/temperature tab."""
        self.sal_temp_tab.water_density.setText(str(self.sim_data.salinity.attrs['WATER_DENSITY']))
        self.sal_temp_tab.const_water_temp.setText(str(self.sim_data.salinity.attrs['WATER_TEMP']))
        self.sal_temp_tab.calculate_salinity.setChecked(self.sim_data.salinity.attrs['CALCULATE_SALINITY'] != 0)
        self.sal_temp_tab.sal_transport_rate_value.setText(
            str(self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_VALUE'])
        )
        self.sal_temp_tab.sal_transport_rate_units.setCurrentIndex(
            self.ramp_duration_units.index(self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_UNITS'])
        )
        self.unit_dict['salinity'] = self.sal_temp_tab.sal_transport_rate_units.currentIndex()
        self.sal_temp_tab.salinity_concentration.setCurrentIndex(
            self.concentration_type.index(self.sim_data.salinity.attrs['SALINITY_CONCENTRATION'])
        )
        self.sal_temp_tab.global_concentration.setText(str(self.sim_data.salinity.attrs['GLOBAL_CONCENTRATION']))
        self.sal_temp_tab.varied_concentration_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(
                    self.scalar_tree, self.sim_data.salinity.attrs['SALINITY_INITIAL_CONCENTRATION']
                )
            )
        )
        self.sal_temp_tab.calculate_temperature.setChecked(self.sim_data.salinity.attrs['CALCULATE_TEMPERATURE'] != 0)
        self.sal_temp_tab.temp_transport_rate_value.setText(
            str(self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_VALUE'])
        )
        self.sal_temp_tab.temp_transport_rate_units.setCurrentIndex(
            self.ramp_duration_units.index(self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_UNITS'])
        )
        self.unit_dict['temperature'] = self.sal_temp_tab.temp_transport_rate_units.currentIndex()
        self.sal_temp_tab.temperature_type.setCurrentIndex(
            self.temperature_type.index(self.sim_data.salinity.attrs['INITIAL_TEMPERATURE_TYPE'])
        )
        self.sal_temp_tab.varied_temperature_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(
                    self.scalar_tree, self.sim_data.salinity.attrs['INITIAL_TEMPERATURE_DATASET']
                )
            )
        )
        self._set_water_temperature_visible()
        self.sal_temp_tab.atmospheric_group.setVisible(
            self.sal_temp_tab.calculate_temperature.checkState() == Qt.Checked
        )
        self.sal_temp_tab.temp_initial_condition_group.setVisible(
            self.sal_temp_tab.calculate_temperature.checkState() == Qt.Checked
        )
        self.sal_temp_tab.sal_initial_condition_group.setVisible(
            self.sal_temp_tab.calculate_salinity.checkState() == Qt.Checked
        )
        self._set_const_or_varied_salinity_visible(self.sal_temp_tab.salinity_concentration.currentIndex())
        self._set_varied_temperature_visible(self.sal_temp_tab.temperature_type.currentIndex())
        self._set_salinity_temperature_explicit_time_steps(self.general_tab.solution_scheme.currentIndex())

    def _save_sal_temp_data(self):
        """Save data from the salinity/temperature attributes."""
        self.sim_data.salinity.attrs['WATER_DENSITY'] = float(self.sal_temp_tab.water_density.text())
        self.sim_data.salinity.attrs['WATER_TEMP'] = float(self.sal_temp_tab.const_water_temp.text())
        self.sim_data.salinity.attrs['CALCULATE_SALINITY'] =\
            1 if self.sal_temp_tab.calculate_salinity.checkState() == Qt.Checked else 0
        self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_VALUE'] =\
            float(self.sal_temp_tab.sal_transport_rate_value.text())
        self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_UNITS'] =\
            self.sal_temp_tab.sal_transport_rate_units.currentText()
        self.sim_data.salinity.attrs['SALINITY_CONCENTRATION'] = self.sal_temp_tab.salinity_concentration.currentText()
        self.sim_data.salinity.attrs['GLOBAL_CONCENTRATION'] = float(self.sal_temp_tab.global_concentration.text())
        self.sim_data.salinity.attrs['CALCULATE_TEMPERATURE'] = \
            1 if self.sal_temp_tab.calculate_temperature.checkState() == Qt.Checked else 0
        self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_VALUE'] =\
            float(self.sal_temp_tab.temp_transport_rate_value.text())
        self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_UNITS'] =\
            self.sal_temp_tab.temp_transport_rate_units.currentText()
        self.sim_data.salinity.attrs['INITIAL_TEMPERATURE_TYPE'] = self.sal_temp_tab.temperature_type.currentText()
        self.sim_data.atmospheric_table = self.atmos_table.model.data_frame.to_xarray()

    def _setup_wave_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.wave_tab.wave_info.addItems(self.wave_types)
        self.wave_tab.wave_water_type.addItems(self.wave_water_predictors)
        self.wave_tab.flow_to_wave_type.addItems(self.flow_wave)
        self.wave_tab.wave_to_flow_type.addItems(self.flow_wave)
        self.wave_tab.flow_to_wave_units.addItems(self.flow_wave_units)
        self.wave_tab.wave_to_flow_units.addItems(self.flow_wave_units)

    def _setup_wave_edit_fields(self):
        """Sets up validators for edit fields."""
        self.wave_tab.const_steering_interval.setValidator(self.steering_interval_validator)
        self.wave_tab.flow_to_wave_value.setValidator(self.positive_dbl_valid)
        self.wave_tab.wave_to_flow_value.setValidator(self.positive_dbl_valid)

    def _load_wave_data(self):
        """Load the data for the wave tab."""
        self.wave_tab.wave_info.setCurrentIndex(self.wave_types.index(self.sim_data.wave.attrs['WAVE_INFO']))
        self.wave_tab.significant_wave_dset_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.wave.attrs['WAVE_HEIGHT'])
            )
        )
        self.wave_tab.peak_wave_period_dset_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.wave.attrs['PEAK_PERIOD'])
            )
        )
        self.wave_tab.mean_wave_dir_dset_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.wave.attrs['MEAN_WAVE_DIR'])
            )
        )
        self.wave_tab.wave_breaking_dset_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.wave.attrs['WAVE_BREAKING'])
            )
        )
        self.wave_tab.wave_radiation_dset_label.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.vector_tree, self.sim_data.wave.attrs['WAVE_RADIATION'])
            )
        )
        self.wave_tab.cms_wave_file_path_label.setText(self.sim_data.wave.attrs['FILE_WAVE_SIM'])
        self.wave_tab.const_steering_interval.setText(str(self.sim_data.wave.attrs['STEERING_INTERVAL_CONST']))
        self.wave_tab.wave_water_type.setCurrentIndex(
            self.wave_water_predictors.index(self.sim_data.wave.attrs['WAVE_WATER_PREDICTOR'])
        )
        self.wave_tab.extrapolation_group.setChecked(self.sim_data.wave.attrs['EXTRAPOLATION_DISTANCE'] != 0)
        self.wave_tab.flow_to_wave_type.setCurrentIndex(self.flow_wave.index(self.sim_data.wave.attrs['FLOW_TO_WAVE']))
        self.wave_tab.flow_to_wave_value.setText(str(self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_VALUE']))
        self.wave_tab.flow_to_wave_units.setCurrentIndex(
            self.flow_wave_units.index(self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_UNITS'])
        )
        self.unit_dict['flow_wave'] = self.wave_tab.flow_to_wave_units.currentIndex()
        self.wave_tab.wave_to_flow_type.setCurrentIndex(self.flow_wave.index(self.sim_data.wave.attrs['WAVE_TO_FLOW']))
        self.wave_tab.wave_to_flow_value.setText(str(self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_VALUE']))
        self.wave_tab.wave_to_flow_units.setCurrentIndex(
            self.flow_wave_units.index(self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_UNITS'])
        )
        self.unit_dict['wave_flow'] = self.wave_tab.wave_to_flow_units.currentIndex()
        # set current page on the stack, enabled/disabled states, etc...
        self.wave_tab.wave_stack.setCurrentIndex(self.wave_tab.wave_info.currentIndex())
        self._set_all_flow_wave_enabled(self.wave_tab.extrapolation_group.isChecked())

    def _save_wave_data(self):
        """Save data from the wave attributes."""
        self.sim_data.wave.attrs['WAVE_INFO'] = self.wave_tab.wave_info.currentText()
        self.sim_data.wave.attrs['FILE_WAVE_SIM'] = self._get_selected_file(self.wave_tab.cms_wave_file_path_label)
        self.sim_data.wave.attrs['STEERING_INTERVAL_CONST'] = float(self.wave_tab.const_steering_interval.text())
        self.sim_data.wave.attrs['WAVE_WATER_PREDICTOR'] = self.wave_tab.wave_water_type.currentText()
        self.sim_data.wave.attrs['EXTRAPOLATION_DISTANCE'] = 1 if self.wave_tab.extrapolation_group.isChecked() else 0
        self.sim_data.wave.attrs['FLOW_TO_WAVE'] = self.wave_tab.flow_to_wave_type.currentText()
        self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_VALUE'] = float(self.wave_tab.flow_to_wave_value.text())
        self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_UNITS'] = self.wave_tab.flow_to_wave_units.currentText()
        self.sim_data.wave.attrs['WAVE_TO_FLOW'] = self.wave_tab.wave_to_flow_type.currentText()
        self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_VALUE'] = float(self.wave_tab.wave_to_flow_value.text())
        self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_UNITS'] = self.wave_tab.wave_to_flow_units.currentText()

    def _setup_wave_connections(self):
        """Set up signal/slot connections in the wave tab."""
        self.wave_tab.wave_info.currentIndexChanged[int].connect(self.wave_tab.wave_stack.setCurrentIndex)
        self.wave_tab.extrapolation_group.toggled[bool].connect(self._set_all_flow_wave_enabled)
        self.wave_tab.flow_to_wave_type.currentIndexChanged[int].connect(self._set_flow_wave_enabled)
        self.wave_tab.wave_to_flow_type.currentIndexChanged[int].connect(self._set_wave_flow_enabled)
        self.wave_tab.flow_to_wave_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.wave_tab.flow_to_wave_value, self.flow_wave_length_units, 'flow_wave', self.unit_dict
            )
        )
        self.wave_tab.wave_to_flow_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.wave_tab.wave_to_flow_value, self.flow_wave_length_units, 'wave_flow', self.unit_dict
            )
        )
        self.wave_tab.significant_wave.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.wave_tab.significant_wave_dset_label, 'Select significant wave height dataset', self.
                scalar_tree, scalar_cell_filter, self.sim_data.wave.attrs, 'WAVE_HEIGHT', resources_util.
                get_quadtree_icon
            )
        )
        self.wave_tab.peak_wave_period.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.wave_tab.peak_wave_period_dset_label, 'Select peak wave period dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.wave.attrs, 'PEAK_PERIOD', resources_util.get_quadtree_icon
            )
        )
        self.wave_tab.mean_wave_dir.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.wave_tab.mean_wave_dir_dset_label, 'Select mean wave direction dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.wave.attrs, 'MEAN_WAVE_DIR', resources_util.get_quadtree_icon
            )
        )
        self.wave_tab.wave_breaking.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.wave_tab.wave_breaking_dset_label, 'Select wave breaking dissipation dataset', self.
                scalar_tree, scalar_cell_filter, self.sim_data.wave.attrs, 'WAVE_BREAKING', resources_util.
                get_quadtree_icon
            )
        )
        self.wave_tab.wave_radiation.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.wave_tab.wave_radiation_dset_label, 'Select radiation stress gradients dataset', self.
                vector_tree, vector_cell_filter, self.sim_data.wave.attrs, 'WAVE_RADIATION', resources_util.
                get_quadtree_icon
            )
        )
        self.wave_tab.cms_wave_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wave_tab.cms_wave_file_path_label, 'Select CMS-Wave solution file',
                'Simulation file (*.sim);;All Files (*.*)'
            )
        )

    def _set_all_flow_wave_enabled(self, state):
        """Set the state of all the dependent flow to wave and wave to flow widgets."""
        if state:
            self._set_flow_wave_enabled(self.wave_tab.flow_to_wave_type.currentIndex())
            self._set_wave_flow_enabled(self.wave_tab.wave_to_flow_type.currentIndex())

    def _set_flow_wave_enabled(self, idx):
        """Set the state of dependent flow to wave widgets."""
        # 1 is the index of 'User specified'
        enable = idx == 1
        self.wave_tab.flow_to_wave_value.setEnabled(enable)
        self.wave_tab.flow_to_wave_units.setEnabled(enable)

    def _set_wave_flow_enabled(self, idx):
        """Set the state of dependent wave to flow widgets."""
        # 1 is the index of 'User specified'
        enable = idx == 1
        self.wave_tab.wave_to_flow_value.setEnabled(enable)
        self.wave_tab.wave_to_flow_units.setEnabled(enable)

    def _setup_wind_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.wind_tab.wind_type.addItems(self.wind_types)
        self.wind_tab.file_type.addItems(self.wind_file_types)
        self.wind_tab.wind_grid_type.addItems(self.wind_grid_types)

    def _setup_wind_edit_fields(self):
        """Sets up validators for edit fields."""
        self.wind_tab.anemometer.setValidator(self.dbl_valid)
        self.wind_tab.number_x.setValidator(self.int_valid)
        self.wind_tab.number_y.setValidator(self.int_valid)
        self.wind_tab.min_x.setValidator(self.dbl_valid)
        self.wind_tab.max_y.setValidator(self.dbl_valid)
        self.wind_tab.time_increment.setValidator(self.dbl_valid)
        self.wind_tab.distance_x.setValidator(self.dbl_valid)
        self.wind_tab.distance_y.setValidator(self.dbl_valid)

    def _load_wind_data(self):
        """Load the data for the wind tab."""
        self.wind_tab.wind_type.setCurrentIndex(self.wind_types.index(self.sim_data.wind.attrs['WIND_TYPE']))
        self.wind_tab.anemometer.setText(str(self.sim_data.wind.attrs['ANEMOMETER']))
        self.wind_tab.file_type.setCurrentIndex(self.wind_file_types.index(self.sim_data.wind.attrs['WIND_FILE_TYPE']))
        # 0 is Navy fleet file
        if self.wind_tab.file_type.currentIndex() == 0:
            self.wind_tab.wind_navy_file_path.setText(self.sim_data.wind.attrs['WIND_FILE'])
        else:
            self.wind_tab.wind_ascii_file_path.setText(self.sim_data.wind.attrs['WIND_FILE'])
        self.wind_tab.wind_grid_type.setCurrentIndex(
            self.wind_grid_types.index(self.sim_data.wind.attrs['WIND_GRID_TYPE'])
        )
        self.wind_tab.wind_grid_file_path.setText(self.sim_data.wind.attrs['WIND_GRID_FILE'])
        self.wind_tab.number_x.setText(str(self.sim_data.wind.attrs['WIND_GRID_NUM_X_VALUES']))
        self.wind_tab.number_y.setText(str(self.sim_data.wind.attrs['WIND_GRID_NUM_Y_VALUES']))
        self.wind_tab.min_x.setText(str(self.sim_data.wind.attrs['WIND_GRID_MIN_X_LOCATION']))
        self.wind_tab.max_y.setText(str(self.sim_data.wind.attrs['WIND_GRID_MAX_Y_LOCATION']))
        self.wind_tab.time_increment.setText(str(self.sim_data.wind.attrs['WIND_GRID_TIME_INCREMENT']))
        self.wind_tab.distance_x.setText(str(self.sim_data.wind.attrs['WIND_GRID_X_DISTANCE']))
        self.wind_tab.distance_y.setText(str(self.sim_data.wind.attrs['WIND_GRID_Y_DISTANCE']))
        self.wind_tab.oceanweather_wind_file_path.setText(self.sim_data.wind.attrs['OCEAN_WIND_FILE'])
        self.wind_tab.oceanweather_pressure_file_path.setText(self.sim_data.wind.attrs['OCEAN_PRESSURE_FILE'])
        self.wind_tab.oceanweather_xy_file_path.setText(self.sim_data.wind.attrs['OCEAN_XY_FILE'])
        # set current page on the stack, enabled/disabled states, etc...
        self._set_widgets_by_wind_type(self.wind_tab.wind_type.currentIndex())
        self._set_wind_file_visible(self.wind_tab.file_type.currentIndex())
        self.wind_tab.wind_grid_stack.setCurrentIndex(self.wind_tab.wind_grid_type.currentIndex())

    def _save_wind_data(self):
        """Save data from the wind attributes."""
        self.sim_data.wind.attrs['WIND_TYPE'] = self.wind_tab.wind_type.currentText()
        self.sim_data.wind.attrs['ANEMOMETER'] = float(self.wind_tab.anemometer.text())
        self.sim_data.wind.attrs['WIND_FILE_TYPE'] = self.wind_tab.file_type.currentText()
        if self.wind_tab.file_type.currentIndex() == 0:
            self.sim_data.wind.attrs['WIND_FILE'] = self._get_selected_file(self.wind_tab.wind_navy_file_path)
        else:
            self.sim_data.wind.attrs['WIND_FILE'] = self._get_selected_file(self.wind_tab.wind_ascii_file_path)
        self.sim_data.wind.attrs['WIND_GRID_TYPE'] = self.wind_tab.wind_grid_type.currentText()
        self.sim_data.wind.attrs['WIND_GRID_FILE'] = self._get_selected_file(self.wind_tab.wind_grid_file_path)
        self.sim_data.wind.attrs['WIND_GRID_NUM_X_VALUES'] = int(self.wind_tab.number_x.text())
        self.sim_data.wind.attrs['WIND_GRID_NUM_Y_VALUES'] = int(self.wind_tab.number_y.text())
        self.sim_data.wind.attrs['WIND_GRID_MIN_X_LOCATION'] = float(self.wind_tab.min_x.text())
        self.sim_data.wind.attrs['WIND_GRID_MAX_Y_LOCATION'] = float(self.wind_tab.max_y.text())
        self.sim_data.wind.attrs['WIND_GRID_TIME_INCREMENT'] = float(self.wind_tab.time_increment.text())
        self.sim_data.wind.attrs['WIND_GRID_X_DISTANCE'] = float(self.wind_tab.distance_x.text())
        self.sim_data.wind.attrs['WIND_GRID_Y_DISTANCE'] = float(self.wind_tab.distance_y.text())
        self.sim_data.wind.attrs['OCEAN_WIND_FILE'] = self._get_selected_file(self.wind_tab.oceanweather_wind_file_path)
        self.sim_data.wind.attrs['OCEAN_PRESSURE_FILE'] = self._get_selected_file(
            self.wind_tab.oceanweather_pressure_file_path
        )
        self.sim_data.wind.attrs['OCEAN_XY_FILE'] = self._get_selected_file(self.wind_tab.oceanweather_xy_file_path)
        self.sim_data.wind_from_table = self.wind_table.model.data_frame.to_xarray()
        next_curve_id = self.sim_data.meteorological_stations_table.attrs['NEXT_CURVE_ID']
        self.sim_data.meteorological_stations_table = self.met_stations_table.model.data_frame.to_xarray()
        self.sim_data.meteorological_stations_table.attrs['NEXT_CURVE_ID'] = next_curve_id

    def _setup_wind_connections(self):
        """Set up signal/slot connections in the wind tab."""
        self.wind_tab.wind_type.currentIndexChanged[int].connect(self._set_widgets_by_wind_type)
        self.wind_tab.file_type.currentIndexChanged[int].connect(self._set_wind_file_visible)
        self.wind_tab.wind_grid_type.currentIndexChanged[int].connect(self.wind_tab.wind_grid_stack.setCurrentIndex)
        self.wind_tab.wind_navy_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wind_tab.wind_navy_file_path, 'Select Navy fleet file',
                'Navy Fleet File (*.wnd);;All Files (*.*)'
            )
        )
        self.wind_tab.wind_ascii_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wind_tab.wind_ascii_file_path, 'Select ASCII wind file', 'Ascii File (*.22);;All Files (*.*)'
            )
        )
        self.wind_tab.wind_grid_file.pressed.connect(
            lambda: FileSelector.
            select_file(self, self.wind_tab.wind_grid_file_path, 'Select XY file', 'XY File (*.xy);;All Files (*.*)')
        )
        self.wind_tab.oceanweather_wind_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wind_tab.oceanweather_wind_file_path, 'Select Oceanweather Wind file',
                'Oceanweather Wind Input File (*.win);;All Files (*.*)'
            )
        )
        self.wind_tab.oceanweather_pressure_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wind_tab.oceanweather_pressure_file_path, 'Select Oceanweather Pressure file',
                'Oceanweather Pressure Input File (*.pre);;All Files (*.*)'
            )
        )
        self.wind_tab.oceanweather_xy_file.pressed.connect(
            lambda: FileSelector.select_file(
                self, self.wind_tab.oceanweather_xy_file_path, 'Select Oceanweather XY file',
                'Oceanweather XY Input File (*.xy);;All Files (*.*)'
            )
        )

    def _set_widgets_by_wind_type(self, idx):
        """Sets widgets visible by the wind type."""
        # none_idx = self.wind_types.index('None')
        spatial_idx = self.wind_types.index('Spatially constant')
        met_idx = self.wind_types.index('Meteorological stations')
        temp_idx = self.wind_types.index('Temporally and spatially varying from file')
        self.wind_tab.meteorological_group.setVisible(idx == met_idx)
        self.wind_tab.parameters_group.setVisible(idx == spatial_idx)
        self.wind_tab.anemometer.setVisible(idx == spatial_idx)
        self.wind_tab.anemometer_label.setVisible(idx == spatial_idx)
        self.wind_tab.anemometer_unit_label.setVisible(idx == spatial_idx)
        self.wind_tab.wind_file_group.setVisible(idx == temp_idx)

    def _set_wind_file_visible(self, idx):
        """Sets the visible file widgets."""
        # 0 is Navy fleet file
        # 1 is OWI file
        is_navy = idx == 0
        is_owi = idx == 1
        self.wind_tab.wind_navy_file.setVisible(is_navy)
        self.wind_tab.wind_navy_file_label.setVisible(is_navy)
        self.wind_tab.wind_navy_file_path.setVisible(is_navy)

        self.wind_tab.wind_ascii_file.setVisible(not is_navy)
        self.wind_tab.wind_ascii_file_label.setVisible(not is_navy)
        self.wind_tab.wind_ascii_file_path.setVisible(not is_navy)

        self.wind_tab.oceanweather_group.setVisible(is_owi)
        self.wind_tab.file_data_group.setVisible(not is_owi)

    def _setup_output_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.output_tab.solution_type.addItems(self.solution_types)

    def _load_output_data(self):
        """Load the data for the output tab."""
        sim_label = self.sim_data.output.attrs['SIMULATION_LABEL']
        if not sim_label:
            sim_label = 'Simulation'
        self.output_tab.simulation_label.setText(sim_label)
        name_col = 0
        check_col = 1
        list_col = 2
        group_row = 0
        self.output_option_model.setData(
            self.output_option_model.index(group_row, list_col), self.sim_data.output.attrs['WSE_LIST']
        )
        group_row += 1
        current_velocity_idx = self.output_option_model.index(group_row, list_col)
        current_velocity_parent_idx = self.output_option_model.index(group_row, name_col)
        self.output_option_model.setData(current_velocity_idx, self.sim_data.output.attrs['CURRENT_VELOCITY_LIST'])
        self.output_option_model.setData(
            self.output_option_model.index(0, check_col, current_velocity_parent_idx),
            self.sim_data.output.attrs['CURRENT_MAGNITUDE']
        )
        group_row += 1

        morphology_idx = self.output_option_model.index(group_row, list_col)
        morphology_check_idx = self.output_option_model.index(group_row, check_col)
        morphology_parent_idx = self.output_option_model.index(group_row, name_col)
        self.output_option_model.setData(morphology_check_idx, self.sim_data.output.attrs['USE_MORPHOLOGY'])
        self.output_option_model.setData(morphology_idx, self.sim_data.output.attrs['MORPHOLOGY_LIST'])
        self.output_option_model.setData(
            self.output_option_model.index(1, check_col, morphology_parent_idx),
            self.sim_data.output.attrs['MORPHOLOGY_CHANGE']
        )
        group_row += 1
        if 'FRACTION_BEDLOAD' not in self.sim_data.output.attrs:
            self.sim_data.output.attrs['FRACTION_BEDLOAD'] = 0

        transport_idx = self.output_option_model.index(group_row, list_col)
        transport_check_idx = self.output_option_model.index(group_row, check_col)
        transport_parent_idx = self.output_option_model.index(group_row, name_col)
        self.output_option_model.setData(transport_check_idx, self.sim_data.output.attrs['USE_TRANSPORT'])
        self.output_option_model.setData(transport_idx, self.sim_data.output.attrs['TRANSPORT_LIST'])
        self.output_option_model.setData(
            self.output_option_model.index(0, check_col, transport_parent_idx),
            self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CAPACITY']
        )
        self.output_option_model.setData(
            self.output_option_model.index(1, check_col, transport_parent_idx),
            self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CONCENTRATION']
        )
        self.output_option_model.setData(
            self.output_option_model.index(2, check_col, transport_parent_idx),
            self.sim_data.output.attrs['FRACTION_SUSPENDED']
        )
        self.output_option_model.setData(
            self.output_option_model.index(3, check_col, transport_parent_idx),
            self.sim_data.output.attrs['FRACTION_BEDLOAD']
        )
        group_row += 1

        wave_idx = self.output_option_model.index(group_row, list_col)
        wave_check_idx = self.output_option_model.index(group_row, check_col)
        wave_parent_idx = self.output_option_model.index(group_row, name_col)
        self.output_option_model.setData(wave_check_idx, self.sim_data.output.attrs['USE_WAVE'])
        self.output_option_model.setData(wave_idx, self.sim_data.output.attrs['WAVE_LIST'])
        self.output_option_model.setData(
            self.output_option_model.index(3, check_col, wave_parent_idx),
            self.sim_data.output.attrs['WAVE_DISSIPATION']
        )
        group_row += 1

        wind_idx = self.output_option_model.index(group_row, list_col)
        wind_check_idx = self.output_option_model.index(group_row, check_col)
        wind_parent_idx = self.output_option_model.index(group_row, name_col)
        self.output_option_model.setData(wind_check_idx, self.sim_data.output.attrs['USE_WIND'])
        self.output_option_model.setData(wind_idx, self.sim_data.output.attrs['WIND_LIST'])
        self.output_option_model.setData(
            self.output_option_model.index(0, check_col, wind_parent_idx), self.sim_data.output.attrs['WIND_SPEED']
        )
        self.output_option_model.setData(
            self.output_option_model.index(2, check_col, wind_parent_idx), self.sim_data.output.attrs['ATM_PRESSURE']
        )
        group_row += 1

        eddy_idx = self.output_option_model.index(group_row, list_col)
        eddy_check_idx = self.output_option_model.index(group_row, check_col)
        self.output_option_model.setData(eddy_check_idx, self.sim_data.output.attrs['USE_EDDY_VISCOSITY'])
        self.output_option_model.setData(eddy_idx, self.sim_data.output.attrs['EDDY_VISCOSITY_LIST'])
        group_row += 1
        self.output_tab.statistical_output_group.setChecked(self.sim_data.output.attrs['ENABLE_STATISTICS'] != 0)
        self.output_tab.use_hydro_table.setChecked(self.sim_data.output.attrs['ENABLE_HYDRO_STATISTICS'] != 0)
        self.hydro_table.model.setData(
            self.hydro_table.model.index(0, 0), self.sim_data.output.attrs['HYDRO_START_TIME']
        )
        self.hydro_table.model.setData(
            self.hydro_table.model.index(0, 1), self.sim_data.output.attrs['HYDRO_INCREMENT']
        )
        self.hydro_table.model.setData(self.hydro_table.model.index(0, 2), self.sim_data.output.attrs['HYDRO_END_TIME'])
        self.output_tab.use_sediment_table.setChecked(self.sim_data.output.attrs['ENABLE_SEDIMENT_STATISTICS'] != 0)
        self.sediment_table.model.setData(
            self.sediment_table.model.index(0, 0), self.sim_data.output.attrs['SEDIMENT_START_TIME']
        )
        self.sediment_table.model.setData(
            self.sediment_table.model.index(0, 1), self.sim_data.output.attrs['SEDIMENT_INCREMENT']
        )
        self.sediment_table.model.setData(
            self.sediment_table.model.index(0, 2), self.sim_data.output.attrs['SEDIMENT_END_TIME']
        )
        self.output_tab.use_salinity_table.setChecked(self.sim_data.output.attrs['ENABLE_SALINITY_STATISTICS'] != 0)
        self.salinity_table.model.setData(
            self.salinity_table.model.index(0, 0), self.sim_data.output.attrs['SALINITY_START_TIME']
        )
        self.salinity_table.model.setData(
            self.salinity_table.model.index(0, 1), self.sim_data.output.attrs['SALINITY_INCREMENT']
        )
        self.salinity_table.model.setData(
            self.salinity_table.model.index(0, 2), self.sim_data.output.attrs['SALINITY_END_TIME']
        )
        self.output_tab.use_wave_table.setChecked(self.sim_data.output.attrs['ENABLE_WAVE_STATISTICS'] != 0)
        self.wave_table.model.setData(self.wave_table.model.index(0, 0), self.sim_data.output.attrs['WAVE_START_TIME'])
        self.wave_table.model.setData(self.wave_table.model.index(0, 1), self.sim_data.output.attrs['WAVE_INCREMENT'])
        self.wave_table.model.setData(self.wave_table.model.index(0, 2), self.sim_data.output.attrs['WAVE_END_TIME'])
        self.output_tab.solution_type.setCurrentIndex(
            self.solution_types.index(self.sim_data.output.attrs['SOLUTION_OUTPUT'])
        )
        self.output_tab.use_xmdf_file_compression.setChecked(self.sim_data.output.attrs['XMDF_COMPRESSION'] != 0)
        self.output_tab.use_single_solution.setChecked(self.sim_data.output.attrs['SINGLE_SOLUTION'] != 0)
        self.output_tab.write_ascii_output.setChecked(self.sim_data.output.attrs['WRITE_ASCII'] != 0)
        self.output_tab.write_tecplot.setChecked(self.sim_data.output.attrs['TECPLOT'] != 0)
        # set current enabled state
        for row in range(self.output_option_model.rowCount()):
            self._enable_output_options(self.output_option_model.item(row, check_col))
        self.hydro_table.setVisible(self.output_tab.use_hydro_table.isChecked())
        self.sediment_table.setVisible(self.output_tab.use_sediment_table.isChecked())
        self.salinity_table.setVisible(self.output_tab.use_salinity_table.isChecked())
        self.wave_table.setVisible(self.output_tab.use_wave_table.isChecked())
        self.output_tab.use_xmdf_file_compression.setHidden(self.output_tab.solution_type.currentIndex())
        self.output_tab.use_single_solution.setHidden(self.output_tab.solution_type.currentIndex())

    def _save_output_data(self):
        """Save data from the output attributes."""
        name_col = 0
        check_col = 1
        list_col = 2
        group_row = 0
        sim_label = self.output_tab.simulation_label.text()
        if not sim_label:
            sim_label = 'Simulation'
        self.sim_data.output.attrs['SIMULATION_LABEL'] = sim_label
        self.sim_data.list_1_table = self.list_1_table.model.data_frame.to_xarray()
        self.sim_data.list_2_table = self.list_2_table.model.data_frame.to_xarray()
        self.sim_data.list_3_table = self.list_3_table.model.data_frame.to_xarray()
        self.sim_data.list_4_table = self.list_4_table.model.data_frame.to_xarray()
        self.sim_data.output.attrs['WSE_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        group_row += 1
        parent_idx = self.output_option_model.index(group_row, name_col)
        self.sim_data.output.attrs['CURRENT_VELOCITY_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        self.sim_data.output.attrs['CURRENT_MAGNITUDE'] =\
            1 if self.output_option_model.data(self.output_option_model.index(0, check_col, parent_idx)) else 0
        group_row += 1
        parent_idx = self.output_option_model.index(group_row, name_col)
        self.sim_data.output.attrs['USE_MORPHOLOGY'] =\
            1 if self.output_option_model.data(self.output_option_model.index(group_row, check_col)) else 0
        self.sim_data.output.attrs['MORPHOLOGY_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        self.sim_data.output.attrs['MORPHOLOGY_CHANGE'] =\
            1 if self.output_option_model.data(self.output_option_model.index(1, check_col, parent_idx)) else 0
        group_row += 1
        parent_idx = self.output_option_model.index(group_row, name_col)
        self.sim_data.output.attrs['USE_TRANSPORT'] =\
            1 if self.output_option_model.data(self.output_option_model.index(group_row, check_col)) else 0
        self.sim_data.output.attrs['TRANSPORT_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CAPACITY'] =\
            1 if self.output_option_model.data(self.output_option_model.index(0, check_col, parent_idx)) else 0
        self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CONCENTRATION'] =\
            1 if self.output_option_model.data(self.output_option_model.index(1, check_col, parent_idx)) else 0
        self.sim_data.output.attrs['FRACTION_SUSPENDED'] =\
            1 if self.output_option_model.data(self.output_option_model.index(2, check_col, parent_idx)) else 0
        self.sim_data.output.attrs['FRACTION_BEDLOAD'] =\
            1 if self.output_option_model.data(self.output_option_model.index(3, check_col, parent_idx)) else 0
        group_row += 1
        parent_idx = self.output_option_model.index(group_row, name_col)
        self.sim_data.output.attrs['USE_WAVE'] =\
            1 if self.output_option_model.data(self.output_option_model.index(group_row, check_col)) else 0
        self.sim_data.output.attrs['WAVE_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        self.sim_data.output.attrs['WAVE_DISSIPATION'] =\
            1 if self.output_option_model.data(self.output_option_model.index(3, check_col, parent_idx)) else 0
        group_row += 1
        parent_idx = self.output_option_model.index(group_row, name_col)
        self.sim_data.output.attrs['USE_WIND'] =\
            1 if self.output_option_model.data(self.output_option_model.index(group_row, check_col)) else 0
        self.sim_data.output.attrs['WIND_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        self.sim_data.output.attrs['WIND_SPEED'] =\
            1 if self.output_option_model.data(self.output_option_model.index(0, check_col, parent_idx)) else 0
        self.sim_data.output.attrs['ATM_PRESSURE'] = \
            1 if self.output_option_model.data(self.output_option_model.index(2, check_col, parent_idx)) else 0
        group_row += 1
        self.sim_data.output.attrs['USE_EDDY_VISCOSITY'] =\
            1 if self.output_option_model.data(self.output_option_model.index(group_row, check_col)) else 0
        self.sim_data.output.attrs['EDDY_VISCOSITY_LIST'] =\
            self.output_option_model.data(self.output_option_model.index(group_row, list_col))
        group_row += 1
        self.sim_data.output.attrs['ENABLE_STATISTICS'] =\
            1 if self.output_tab.statistical_output_group.isChecked() else 0
        self.sim_data.output.attrs['ENABLE_HYDRO_STATISTICS'] =\
            1 if self.output_tab.use_hydro_table.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['HYDRO_START_TIME'] =\
            float(self.hydro_table.model.data(self.hydro_table.model.index(0, 0)))
        self.sim_data.output.attrs['HYDRO_INCREMENT'] =\
            float(self.hydro_table.model.data(self.hydro_table.model.index(0, 1)))
        self.sim_data.output.attrs['HYDRO_END_TIME'] = \
            float(self.hydro_table.model.data(self.hydro_table.model.index(0, 2)))
        self.sim_data.output.attrs['ENABLE_SEDIMENT_STATISTICS'] =\
            1 if self.output_tab.use_sediment_table.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['SEDIMENT_START_TIME'] =\
            float(self.sediment_table.model.data(self.sediment_table.model.index(0, 0)))
        self.sim_data.output.attrs['SEDIMENT_INCREMENT'] =\
            float(self.sediment_table.model.data(self.sediment_table.model.index(0, 1)))
        self.sim_data.output.attrs['SEDIMENT_END_TIME'] =\
            float(self.sediment_table.model.data(self.sediment_table.model.index(0, 2)))
        self.sim_data.output.attrs['ENABLE_SALINITY_STATISTICS'] =\
            1 if self.output_tab.use_salinity_table.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['SALINITY_START_TIME'] =\
            float(self.salinity_table.model.data(self.salinity_table.model.index(0, 0)))
        self.sim_data.output.attrs['SALINITY_INCREMENT'] =\
            float(self.salinity_table.model.data(self.salinity_table.model.index(0, 1)))
        self.sim_data.output.attrs['SALINITY_END_TIME'] =\
            float(self.salinity_table.model.data(self.salinity_table.model.index(0, 2)))
        self.sim_data.output.attrs['ENABLE_WAVE_STATISTICS'] =\
            1 if self.output_tab.use_wave_table.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['WAVE_START_TIME'] =\
            float(self.wave_table.model.data(self.wave_table.model.index(0, 0)))
        self.sim_data.output.attrs['WAVE_INCREMENT'] =\
            float(self.wave_table.model.data(self.wave_table.model.index(0, 1)))
        self.sim_data.output.attrs['WAVE_END_TIME'] =\
            float(self.wave_table.model.data(self.wave_table.model.index(0, 2)))
        self.sim_data.output.attrs['SOLUTION_OUTPUT'] = self.output_tab.solution_type.currentText()
        self.sim_data.output.attrs['XMDF_COMPRESSION'] =\
            1 if self.output_tab.use_xmdf_file_compression.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['SINGLE_SOLUTION'] =\
            1 if self.output_tab.use_single_solution.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['WRITE_ASCII'] =\
            1 if self.output_tab.write_ascii_output.checkState() == Qt.Checked else 0
        self.sim_data.output.attrs['TECPLOT'] = 1 if self.output_tab.write_tecplot.checkState() == Qt.Checked else 0

    def _setup_output_connections(self):
        """Set up signal/slot connections in the output tab."""
        self.output_option_model.itemChanged[QStandardItem].connect(self._enable_output_options)
        self.output_tab.use_hydro_table.stateChanged[int].connect(self.hydro_table.setVisible)
        self.output_tab.use_sediment_table.stateChanged[int].connect(self.sediment_table.setVisible)
        self.output_tab.use_salinity_table.stateChanged[int].connect(self.salinity_table.setVisible)
        self.output_tab.use_wave_table.stateChanged[int].connect(self.wave_table.setVisible)
        self.output_tab.solution_type.currentIndexChanged[int].connect(
            self.output_tab.use_xmdf_file_compression.setHidden
        )
        self.output_tab.solution_type.currentIndexChanged[int].connect(self.output_tab.use_single_solution.setHidden)

    def _enable_output_options(self, item):
        """Set the output options enabled/disabled state.

        Args:
            item (QStandardItem): The item that changed it's data.
        """
        # only concerned about check box column of top tree level
        name_col = 0
        check_box_col = 1
        list_col = 2
        # Mantis issue 14985. I removed the check for parent being None. Seems to work. Not sure why we had the check.
        # if item.index().column() == check_box_col and item.parent() is None:
        if item.index().column() == check_box_col:
            name_item = self.output_option_model.item(item.index().row(), name_col)
            enable = item.data(Qt.DisplayRole)
            self.output_option_model.item(item.index().row(), list_col).setEnabled(enable)
            for i in range(name_item.rowCount()):
                child_item = name_item.child(i, check_box_col)
                name_child_item = name_item.child(i, name_col)
                # only enable/disable children that are not always disabled
                if not child_item.data(Qt.UserRole):
                    child_item.setEnabled(enable)
                name_child_item.setEnabled(enable)

    def _create_adjustable_combo_box_model(self, text_list, removable_list):
        """Creates a QStandardModel for a combobox with user data for a filter model.

        Args:
            text_list (list): A list of all the possible strings shown in the combobox.
            removable_list (list): A list of whether the item can be removed for all the possible items.
        """
        list_model = QStandardItemModel(self)
        for opt, removable in zip(text_list, removable_list):
            opt_item = QStandardItem(opt)
            opt_item.setData(removable, Qt.UserRole)
            list_model.appendRow(opt_item)
        filter_model = ComboBoxFilterModel(self, removable_list)
        filter_model.setSourceModel(list_model)
        return filter_model

    def _setup_sediment_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.sediment_tab.transport_time_units.addItems(self.transport_times)
        self.sediment_tab.morphologic_time_units.addItems(self.transport_times)
        self.sediment_tab.morphology_change_time_units.addItems(self.morphology_change_times)
        self.sediment_tab.formulation_units.setModel(
            self._create_adjustable_combo_box_model(self.formulation_units, [True, True, False])
        )
        self.sediment_tab.transport_formula.setModel(
            self._create_adjustable_combo_box_model(self.transport_formulas, [False, False, True, False, False])
        )
        self.sediment_tab.concentration_profile.setModel(
            self._create_adjustable_combo_box_model(self.concentration_profiles, [False, False, True, True])
        )
        self.sediment_tab.sediment_density_units.addItems(self.sediment_densities)
        self.sediment_tab.grain_size_units.addItems(self.grain_sizes)
        self.sediment_tab.total_load_adaptation.addItems(self.adaptation_methods)
        self.sediment_tab.total_load_adaptation_length_units.addItems(self.grain_sizes)
        self.sediment_tab.total_load_adaptation_time_units.addItems(self.transport_times)
        self.sediment_tab.bed_load_adaptation.addItems(self.bed_adaptation_methods)
        self.sediment_tab.bed_load_adaptation_length_units.addItems(self.adaptation_length)
        self.sediment_tab.bed_load_adaptation_time_units.addItems(self.transport_times)
        self.sediment_tab.suspended_load_adaptation.addItems(self.suspended_adaptation_methods)
        self.sediment_tab.suspended_load_adaptation_length_units.addItems(self.adaptation_length)
        self.sediment_tab.suspended_load_adaptation_time_units.addItems(self.transport_times)
        self.sediment_tab.multiple_grain_sizes.addItems(self.mulitple_grain_types)
        self.sediment_tab.bed_composition_input.addItems(self.bed_compositions)
        self.sediment_tab.min_bed_layer_thickness_units.addItems(self.grain_sizes)
        self.sediment_tab.max_bed_layer_thickness_units.addItems(self.grain_sizes)

    def _setup_sediment_edit_fields(self):
        """Sets up validators for edit fields."""
        self.sediment_tab.transport_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.morphologic_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.morphology_change_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.watanabe_transport.setValidator(self.dbl_valid)
        self.sediment_tab.c2shore_efficiency.setValidator(self.dbl_valid)
        self.sediment_tab.c2shore_bedload.setValidator(self.dbl_valid)
        self.sediment_tab.c2shore_suspload.setValidator(self.dbl_valid)
        self.sediment_tab.sediment_density_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.sediment_porosity.setValidator(self.dbl_valid)
        self.sediment_tab.grain_size_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.bed_load_scaling.setValidator(self.dbl_valid)
        self.sediment_tab.suspended_load_scaling.setValidator(self.dbl_valid)
        self.sediment_tab.morphologic_acceleration.setValidator(self.dbl_valid)
        self.sediment_tab.bed_slope_diffusion.setValidator(self.dbl_valid)
        self.sediment_tab.hiding_exposure.setValidator(self.dbl_valid)
        self.sediment_tab.total_load_adaptation_length_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.total_load_adaptation_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.bed_load_adaptation_length_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.bed_load_adaptation_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.bed_load_adaptation_depth.setValidator(self.dbl_valid)
        self.sediment_tab.suspended_load_adaptation_length_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.suspended_load_adaptation_time_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.suspended_load_adaptation_coefficient.setValidator(self.dbl_valid)
        self.sediment_tab.number_of_size_classes.setValidator(self.int_valid)
        self.sediment_tab.sediment_standard_deviation.setValidator(self.dbl_valid)
        self.sediment_tab.number_bed_layers.setValidator(self.int_valid)
        self.sediment_tab.const_thickness_mixing.setValidator(self.dbl_valid)
        self.sediment_tab.const_thickness_bed.setValidator(self.dbl_valid)
        self.sediment_tab.max_number_bed_layers.setValidator(self.int_valid)
        self.sediment_tab.min_bed_layer_thickness_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.max_bed_layer_thickness_value.setValidator(self.positive_dbl_valid)
        self.sediment_tab.critical_bed_slope.setValidator(self.dbl_valid)
        self.sediment_tab.max_number_iterations.setValidator(self.int_valid)

    def _load_sediment_data(self):
        """Load the data for the sediment tab."""
        self.sediment_tab.calculate_sediment.setChecked(self.sim_data.sediment.attrs['CALCULATE_SEDIMENT'] != 0)
        self.sediment_tab.transport_time_value.setText(str(self.sim_data.sediment.attrs['TRANSPORT_TIME_VALUE']))
        self.sediment_tab.transport_time_units.setCurrentIndex(
            self.transport_times.index(self.sim_data.sediment.attrs['TRANSPORT_TIME_UNITS'])
        )
        self.sediment_tab.morphologic_time_value.setText(str(self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_VALUE']))
        self.sediment_tab.morphologic_time_units.setCurrentIndex(
            self.transport_times.index(self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_UNITS'])
        )
        self.sediment_tab.morphology_change_time_value.setText(
            str(self.sim_data.sediment.attrs['MORPHOLOGY_TIME_VALUE'])
        )
        self.sediment_tab.morphology_change_time_units.setCurrentIndex(
            self.morphology_change_times.index(self.sim_data.sediment.attrs['MORPHOLOGY_TIME_UNITS'])
        )
        self.sediment_tab.formulation_units.setCurrentIndex(
            self.formulation_units.index(self.sim_data.sediment.attrs['FORMULATION_UNITS'])
        )
        val = self.sim_data.sediment.attrs['TRANSPORT_FORMULA']
        if val == 'CSHORE':
            self.sediment_tab.transport_formula.setCurrentIndex(self.transport_formulas.index('C2SHORE'))
        else:
            self.sediment_tab.transport_formula.setCurrentIndex(
                self.transport_formulas.index(self.sim_data.sediment.attrs['TRANSPORT_FORMULA'])
            )
        self.sediment_tab.concentration_profile.setCurrentIndex(
            self.concentration_profiles.index(self.sim_data.sediment.attrs['CONCENTRATION_PROFILE'])
        )
        self.sediment_tab.watanabe_transport.setText(str(self.sim_data.sediment.attrs['WATANABE_RATE']))
        self.sediment_tab.c2shore_efficiency.setText(str(self.sim_data.sediment.attrs['C2SHORE_EFFICIENCY']))
        self.sediment_tab.c2shore_bedload.setText(str(self.sim_data.sediment.attrs['C2SHORE_BED_LOAD']))
        self.sediment_tab.c2shore_suspload.setText(str(self.sim_data.sediment.attrs['C2SHORE_SUSP_LOAD']))
        self.sediment_tab.sediment_density_value.setText(str(self.sim_data.sediment.attrs['SEDIMENT_DENSITY_VALUE']))
        self.sediment_tab.sediment_density_units.setCurrentIndex(
            self.sediment_densities.index(self.sim_data.sediment.attrs['SEDIMENT_DENSITY_UNITS'])
        )
        self.sediment_tab.sediment_porosity.setText(str(self.sim_data.sediment.attrs['SEDIMENT_POROSITY']))
        self.sediment_tab.grain_size_value.setText(str(self.sim_data.sediment.attrs['GRAIN_SIZE_VALUE']))
        self.sediment_tab.grain_size_units.setCurrentIndex(
            self.grain_sizes.index(self.sim_data.sediment.attrs['GRAIN_SIZE_UNITS'])
        )
        self.sediment_tab.bed_load_scaling.setText(str(self.sim_data.sediment.attrs['BED_LOAD_SCALING']))
        self.sediment_tab.suspended_load_scaling.setText(str(self.sim_data.sediment.attrs['SUSPENDED_LOAD_SCALING']))
        self.sediment_tab.morphologic_acceleration.setText(
            str(self.sim_data.sediment.attrs['MORPHOLOGIC_ACCELERATION'])
        )
        self.sediment_tab.bed_slope_diffusion.setText(str(self.sim_data.sediment.attrs['BED_SLOPE_DIFFUSION']))
        self.sediment_tab.hiding_exposure.setText(str(self.sim_data.sediment.attrs['HIDING_AND_EXPOSURE']))
        self.sediment_tab.total_load_adaptation.setCurrentIndex(
            self.adaptation_methods.index(self.sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD'])
        )
        self.sediment_tab.total_load_adaptation_length_value.setText(
            str(self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_VALUE'])
        )
        self.sediment_tab.total_load_adaptation_length_units.setCurrentIndex(
            self.grain_sizes.index(self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_UNITS'])
        )
        self.sediment_tab.total_load_adaptation_time_value.setText(
            str(self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_VALUE'])
        )
        self.sediment_tab.total_load_adaptation_time_units.setCurrentIndex(
            self.transport_times.index(self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_UNITS'])
        )
        self.sediment_tab.bed_load_adaptation.setCurrentIndex(
            self.bed_adaptation_methods.index(self.sim_data.sediment.attrs['BED_ADAPTATION_METHOD'])
        )
        self.sediment_tab.bed_load_adaptation_length_value.setText(
            str(self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_VALUE'])
        )
        self.sediment_tab.bed_load_adaptation_length_units.setCurrentIndex(
            self.adaptation_length.index(self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_UNITS'])
        )
        self.sediment_tab.bed_load_adaptation_time_value.setText(
            str(self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_VALUE'])
        )
        self.sediment_tab.bed_load_adaptation_time_units.setCurrentIndex(
            self.transport_times.index(self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_UNITS'])
        )
        self.sediment_tab.bed_load_adaptation_depth.setText(str(self.sim_data.sediment.attrs['BED_ADAPTATION_DEPTH']))
        self.sediment_tab.suspended_load_adaptation.setCurrentIndex(
            self.suspended_adaptation_methods.index(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'])
        )
        self.sediment_tab.suspended_load_adaptation_length_value.setText(
            str(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_VALUE'])
        )
        self.sediment_tab.suspended_load_adaptation_length_units.setCurrentIndex(
            self.adaptation_length.index(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_UNITS'])
        )
        self.sediment_tab.suspended_load_adaptation_time_value.setText(
            str(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_VALUE'])
        )
        self.sediment_tab.suspended_load_adaptation_time_units.setCurrentIndex(
            self.transport_times.index(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_UNITS'])
        )
        self.sediment_tab.suspended_load_adaptation_coefficient.setText(
            str(self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_COEFFICIENT'])
        )
        self.sediment_tab.use_advanced_size_classes.setChecked(
            self.sim_data.sediment.attrs['USE_ADVANCED_SIZE_CLASSES'] != 0
        )
        self.sediment_tab.use_simplified_multiple_grain_size.setChecked(
            self.sim_data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] != 0
        )
        self.sediment_tab.multiple_grain_sizes.setCurrentIndex(
            self.mulitple_grain_types.index(self.sim_data.sediment.attrs['MULTIPLE_GRAIN_SIZES'])
        )
        self.sediment_tab.number_of_size_classes.setText(str(self.sim_data.sediment.attrs['SIMPLE_MULTI_SIZE']))
        self.sediment_tab.bed_composition_input.setCurrentIndex(
            self.bed_compositions.index(self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'])
        )
        self.sediment_tab.d16_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['MULTI_D16'])
            )
        )
        self.sediment_tab.d50_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['MULTI_D50'])
            )
        )
        self.sediment_tab.d84_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['MULTI_D84'])
            )
        )
        self.sediment_tab.d35_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['MULTI_D35'])
            )
        )
        self.sediment_tab.d90_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['MULTI_D90'])
            )
        )
        self.sediment_tab.sediment_standard_deviation.setText(
            str(self.sim_data.sediment.attrs['SEDIMENT_STANDARD_DEVIATION'])
        )
        self.sediment_tab.number_bed_layers.setText(str(self.sim_data.sediment.attrs['NUMBER_BED_LAYERS']))
        self.sediment_tab.const_thickness_mixing.setText(str(self.sim_data.sediment.attrs['THICKNESS_FOR_MIXING']))
        self.sediment_tab.const_thickness_bed.setText(str(self.sim_data.sediment.attrs['THICKNESS_FOR_BED']))
        self.sediment_tab.max_number_bed_layers.setText(str(self.sim_data.sediment.attrs['MAX_NUMBER_BED_LAYERS']))
        self.sediment_tab.min_bed_layer_thickness_value.setText(
            str(self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_VALUE'])
        )
        self.sediment_tab.min_bed_layer_thickness_units.setCurrentIndex(
            self.grain_sizes.index(self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_UNITS'])
        )
        self.sediment_tab.max_bed_layer_thickness_value.setText(
            str(self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_VALUE'])
        )
        self.sediment_tab.max_bed_layer_thickness_units.setCurrentIndex(
            self.grain_sizes.index(self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_UNITS'])
        )
        self.sediment_tab.avalanching_group.setChecked(self.sim_data.sediment.attrs['CALCULATE_AVALANCHING'] != 0)
        self.sediment_tab.critical_bed_slope.setText(str(self.sim_data.sediment.attrs['CRITICAL_BED_SLOPE']))
        self.sediment_tab.max_number_iterations.setText(str(self.sim_data.sediment.attrs['MAX_NUMBER_ITERATIONS']))
        self.sediment_tab.hardbottom_group.setChecked(self.sim_data.sediment.attrs['USE_HARD_BOTTOM'] != 0)
        self.sediment_tab.hardbottom_path.setText(
            TextConverter.get_text_from_field(
                tree_util.build_tree_path(self.scalar_tree, self.sim_data.sediment.attrs['HARD_BOTTOM'])
            )
        )

        self.unit_dict['transport_time'] = self.sediment_tab.transport_time_units.currentIndex()
        self.unit_dict['morphologic_time'] = self.sediment_tab.morphologic_time_units.currentIndex()
        self.unit_dict['morphology_change'] = self.sediment_tab.morphology_change_time_units.currentIndex()
        self.unit_dict['sediment_density'] = self.sediment_tab.sediment_density_units.currentIndex()
        self.unit_dict['grain_size'] = self.sediment_tab.grain_size_units.currentIndex()
        self.unit_dict['total_load_adaptation_length'] =\
            self.sediment_tab.total_load_adaptation_length_units.currentIndex()
        self.unit_dict['total_load_adaptation_time'] = self.sediment_tab.total_load_adaptation_time_units.currentIndex()
        self.unit_dict['bed_load_adaptation_length'] = self.sediment_tab.bed_load_adaptation_length_units.currentIndex()
        self.unit_dict['bed_load_adaptation_time'] = self.sediment_tab.bed_load_adaptation_time_units.currentIndex()
        self.unit_dict['suspended_load_adaptation_length'] =\
            self.sediment_tab.suspended_load_adaptation_length_units.currentIndex()
        self.unit_dict['suspended_load_adaptation_time'] =\
            self.sediment_tab.suspended_load_adaptation_time_units.currentIndex()
        self.unit_dict['min_bed_layer_thickness'] = self.sediment_tab.min_bed_layer_thickness_units.currentIndex()
        self.unit_dict['max_bed_layer_thickness'] = self.sediment_tab.max_bed_layer_thickness_units.currentIndex()

        self._set_suspended_load_method_visible(self.sediment_tab.suspended_load_adaptation.currentIndex())
        self._set_bed_load_method_visible(self.sediment_tab.bed_load_adaptation.currentIndex())
        self._set_total_adaptation_visible(self.sediment_tab.total_load_adaptation.currentIndex())
        self._set_bed_stack(self.sediment_tab.use_simplified_multiple_grain_size.checkState())
        self._set_transport_formula_visible(self.sediment_tab.transport_formula.currentIndex())
        self._set_formulation_units_visible(self.sediment_tab.formulation_units.currentIndex())
        self._set_multiple_grain_sizes_visible(self.sediment_tab.multiple_grain_sizes.currentIndex())
        self._set_bed_composition_visible(self.sediment_tab.bed_composition_input.currentIndex())
        self.sediment_tab.option_group.setVisible(self.sediment_tab.calculate_sediment.isChecked())
        self.diameters_table.set_columns_visible(self.sediment_tab.use_advanced_size_classes.checkState())

        # do this last as it can change combo box indexes
        self._set_sediment_visible_for_explicit()

    def _save_sediment_data(self):
        """Save data from the sediment attributes."""
        self.sim_data.sediment.attrs['CALCULATE_SEDIMENT'] =\
            1 if self.sediment_tab.calculate_sediment.isChecked() else 0
        self.sim_data.sediment.attrs['TRANSPORT_TIME_VALUE'] = float(self.sediment_tab.transport_time_value.text())
        self.sim_data.sediment.attrs['TRANSPORT_TIME_UNITS'] = self.sediment_tab.transport_time_units.currentText()
        self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_VALUE'] = float(self.sediment_tab.morphologic_time_value.text())
        self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_UNITS'] = self.sediment_tab.morphologic_time_units.currentText()
        self.sim_data.sediment.attrs['MORPHOLOGY_TIME_VALUE'] =\
            float(self.sediment_tab.morphology_change_time_value.text())
        self.sim_data.sediment.attrs['MORPHOLOGY_TIME_UNITS'] =\
            self.sediment_tab.morphology_change_time_units.currentText()
        self.sim_data.sediment.attrs['FORMULATION_UNITS'] = self.sediment_tab.formulation_units.currentText()
        self.sim_data.sediment.attrs['TRANSPORT_FORMULA'] = self.sediment_tab.transport_formula.currentText()
        self.sim_data.sediment.attrs['CONCENTRATION_PROFILE'] = self.sediment_tab.concentration_profile.currentText()
        self.sim_data.sediment.attrs['WATANABE_RATE'] = float(self.sediment_tab.watanabe_transport.text())
        self.sim_data.sediment.attrs['C2SHORE_EFFICIENCY'] = float(self.sediment_tab.c2shore_efficiency.text())
        self.sim_data.sediment.attrs['C2SHORE_BED_LOAD'] = float(self.sediment_tab.c2shore_bedload.text())
        self.sim_data.sediment.attrs['C2SHORE_SUSP_LOAD'] = float(self.sediment_tab.c2shore_suspload.text())
        self.sim_data.sediment.attrs['SEDIMENT_DENSITY_VALUE'] = float(self.sediment_tab.sediment_density_value.text())
        self.sim_data.sediment.attrs['SEDIMENT_DENSITY_UNITS'] = self.sediment_tab.sediment_density_units.currentText()
        self.sim_data.sediment.attrs['SEDIMENT_POROSITY'] = float(self.sediment_tab.sediment_porosity.text())
        self.sim_data.sediment.attrs['GRAIN_SIZE_VALUE'] = float(self.sediment_tab.grain_size_value.text())
        self.sim_data.sediment.attrs['GRAIN_SIZE_UNITS'] = self.sediment_tab.grain_size_units.currentText()
        self.sim_data.sediment.attrs['BED_LOAD_SCALING'] = float(self.sediment_tab.bed_load_scaling.text())
        self.sim_data.sediment.attrs['SUSPENDED_LOAD_SCALING'] = float(self.sediment_tab.suspended_load_scaling.text())
        self.sim_data.sediment.attrs['MORPHOLOGIC_ACCELERATION'] =\
            float(self.sediment_tab.morphologic_acceleration.text())
        self.sim_data.sediment.attrs['BED_SLOPE_DIFFUSION'] = float(self.sediment_tab.bed_slope_diffusion.text())
        self.sim_data.sediment.attrs['HIDING_AND_EXPOSURE'] = float(self.sediment_tab.hiding_exposure.text())
        self.sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD'] = self.sediment_tab.total_load_adaptation.currentText()
        self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_VALUE'] =\
            float(self.sediment_tab.total_load_adaptation_length_value.text())
        self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_UNITS'] =\
            self.sediment_tab.total_load_adaptation_length_units.currentText()
        self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_VALUE'] =\
            float(self.sediment_tab.total_load_adaptation_time_value.text())
        self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_UNITS'] =\
            self.sediment_tab.total_load_adaptation_time_units.currentText()
        self.sim_data.sediment.attrs['BED_ADAPTATION_METHOD'] = self.sediment_tab.bed_load_adaptation.currentText()
        self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_VALUE'] =\
            float(self.sediment_tab.bed_load_adaptation_length_value.text())
        self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_UNITS'] =\
            self.sediment_tab.bed_load_adaptation_length_units.currentText()
        self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_VALUE'] =\
            float(self.sediment_tab.bed_load_adaptation_time_value.text())
        self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_UNITS'] =\
            self.sediment_tab.bed_load_adaptation_time_units.currentText()
        self.sim_data.sediment.attrs['BED_ADAPTATION_DEPTH'] = float(self.sediment_tab.bed_load_adaptation_depth.text())
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] =\
            self.sediment_tab.suspended_load_adaptation.currentText()
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_VALUE'] =\
            float(self.sediment_tab.suspended_load_adaptation_length_value.text())
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_UNITS'] =\
            self.sediment_tab.suspended_load_adaptation_length_units.currentText()
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_VALUE'] =\
            float(self.sediment_tab.suspended_load_adaptation_time_value.text())
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_UNITS'] =\
            self.sediment_tab.suspended_load_adaptation_time_units.currentText()
        self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_COEFFICIENT'] =\
            float(self.sediment_tab.suspended_load_adaptation_coefficient.text())
        self.sim_data.sediment.attrs['USE_ADVANCED_SIZE_CLASSES'] =\
            1 if self.sediment_tab.use_advanced_size_classes.isChecked() else 0
        self.sim_data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] =\
            1 if self.sediment_tab.use_simplified_multiple_grain_size.isChecked() else 0
        self.sim_data.sediment.attrs['MULTIPLE_GRAIN_SIZES'] = self.sediment_tab.multiple_grain_sizes.currentText()
        self.sim_data.sediment.attrs['SIMPLE_MULTI_SIZE'] = int(self.sediment_tab.number_of_size_classes.text())
        self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] = self.sediment_tab.bed_composition_input.currentText()
        self.sim_data.sediment.attrs['SEDIMENT_STANDARD_DEVIATION'] =\
            float(self.sediment_tab.sediment_standard_deviation.text())
        self.sim_data.sediment.attrs['NUMBER_BED_LAYERS'] = int(self.sediment_tab.number_bed_layers.text())
        self.sim_data.sediment.attrs['THICKNESS_FOR_MIXING'] = float(self.sediment_tab.const_thickness_mixing.text())
        self.sim_data.sediment.attrs['THICKNESS_FOR_BED'] = float(self.sediment_tab.const_thickness_bed.text())
        self.sim_data.sediment.attrs['MAX_NUMBER_BED_LAYERS'] = int(self.sediment_tab.max_number_bed_layers.text())
        self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_VALUE'] =\
            float(self.sediment_tab.min_bed_layer_thickness_value.text())
        self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_UNITS'] =\
            self.sediment_tab.min_bed_layer_thickness_units.currentText()
        self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_VALUE'] =\
            float(self.sediment_tab.max_bed_layer_thickness_value.text())
        self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_UNITS'] =\
            self.sediment_tab.max_bed_layer_thickness_units.currentText()
        self.sim_data.sediment.attrs['CALCULATE_AVALANCHING'] =\
            1 if self.sediment_tab.avalanching_group.isChecked() else 0
        self.sim_data.sediment.attrs['CRITICAL_BED_SLOPE'] = float(self.sediment_tab.critical_bed_slope.text())
        self.sim_data.sediment.attrs['MAX_NUMBER_ITERATIONS'] = int(self.sediment_tab.max_number_iterations.text())
        self.sim_data.sediment.attrs['USE_HARD_BOTTOM'] = 1 if self.sediment_tab.hardbottom_group.isChecked() else 0
        self.sim_data.simple_grain_sizes_table = self.grain_table.model.data_frame.to_xarray()
        self.sim_data.bed_layer_table = self.bed_layers_table.model.data_frame.to_xarray()
        self.sim_data.advanced_sediment_diameters_table = self.diameters_table.model.data_frame.to_xarray()

    def _setup_sediment_connections(self):
        """Set up signal/slot connections in the sediment tab."""
        self.sediment_tab.transport_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.transport_time_value, self.transport_time_units, 'transport_time', self.unit_dict
            )
        )
        self.sediment_tab.morphologic_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.morphologic_time_value, self.transport_time_units, 'morphologic_time', self.unit_dict
            )
        )
        self.sediment_tab.morphology_change_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.morphology_change_time_value, self.morphology_change_time_units, 'morphology_change',
                self.unit_dict
            )
        )
        self.sediment_tab.sediment_density_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.sediment_density_value, self.sediment_density_units, 'sediment_density',
                self.unit_dict
            )
        )
        self.sediment_tab.grain_size_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.grain_size_value, self.grain_size_units, 'grain_size', self.unit_dict
            )
        )
        self.sediment_tab.total_load_adaptation_length_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.total_load_adaptation_length_value, self.grain_size_units,
                'total_load_adaptation_length', self.unit_dict
            )
        )
        self.sediment_tab.total_load_adaptation_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.total_load_adaptation_time_value, self.transport_time_units,
                'total_load_adaptation_time', self.unit_dict
            )
        )
        self.sediment_tab.bed_load_adaptation_length_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.bed_load_adaptation_length_value, self.adaptation_units, 'bed_load_adaptation_length',
                self.unit_dict
            )
        )
        self.sediment_tab.bed_load_adaptation_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.bed_load_adaptation_time_value, self.transport_time_units, 'bed_load_adaptation_time',
                self.unit_dict
            )
        )
        self.sediment_tab.suspended_load_adaptation_length_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.suspended_load_adaptation_length_value, self.adaptation_units,
                'suspended_load_adaptation_length', self.unit_dict
            )
        )
        self.sediment_tab.suspended_load_adaptation_time_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.suspended_load_adaptation_time_value, self.transport_time_units,
                'suspended_load_adaptation_time', self.unit_dict
            )
        )
        self.sediment_tab.min_bed_layer_thickness_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.min_bed_layer_thickness_value, self.grain_size_units, 'min_bed_layer_thickness',
                self.unit_dict
            )
        )
        self.sediment_tab.max_bed_layer_thickness_units.currentIndexChanged[int].connect(
            UnitConverter.make_change_units(
                self.sediment_tab.max_bed_layer_thickness_value, self.grain_size_units, 'max_bed_layer_thickness',
                self.unit_dict
            )
        )

        self.sediment_tab.d16.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.d16_path, 'Select D16 dataset', self.scalar_tree, scalar_cell_filter, self.
                sim_data.sediment.attrs, 'MULTI_D16', resources_util.get_quadtree_icon
            )
        )
        self.sediment_tab.d50.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.d50_path, 'Select D50 dataset', self.scalar_tree, scalar_cell_filter, self.
                sim_data.sediment.attrs, 'MULTI_D50', resources_util.get_quadtree_icon
            )
        )
        self.sediment_tab.d84.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.d84_path, 'Select D84 dataset', self.scalar_tree, scalar_cell_filter, self.
                sim_data.sediment.attrs, 'MULTI_D84', resources_util.get_quadtree_icon
            )
        )
        self.sediment_tab.d35.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.d35_path, 'Select D35 dataset', self.scalar_tree, scalar_cell_filter, self.
                sim_data.sediment.attrs, 'MULTI_D35', resources_util.get_quadtree_icon
            )
        )
        self.sediment_tab.d90.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.d90_path, 'Select D90 dataset', self.scalar_tree, scalar_cell_filter, self.
                sim_data.sediment.attrs, 'MULTI_D90', resources_util.get_quadtree_icon
            )
        )

        self.sediment_tab.hardbottom.pressed.connect(
            lambda: DatasetSelector.select_dataset(
                self, self.sediment_tab.hardbottom_path, 'Select hardbottom dataset', self.scalar_tree,
                scalar_cell_filter, self.sim_data.sediment.attrs, 'HARD_BOTTOM', resources_util.get_quadtree_icon
            )
        )

        self.sediment_tab.use_simplified_multiple_grain_size.stateChanged[int].connect(self._set_bed_stack)
        self.sediment_tab.calculate_sediment.stateChanged[int].connect(self.sediment_tab.option_group.setVisible)
        self.general_tab.solution_scheme.currentIndexChanged[int].connect(self._set_sediment_visible_for_explicit)
        self.sediment_tab.total_load_adaptation.currentIndexChanged[int].connect(self._set_total_adaptation_visible)
        self.sediment_tab.formulation_units.currentIndexChanged[int].connect(self._set_formulation_units_visible)
        self.sediment_tab.transport_formula.currentIndexChanged[int].connect(self._set_transport_formula_visible)
        self.sediment_tab.suspended_load_adaptation.currentIndexChanged[int].connect(
            self._set_suspended_load_method_visible
        )
        self.sediment_tab.bed_load_adaptation.currentIndexChanged[int].connect(self._set_bed_load_method_visible)
        self.sediment_tab.multiple_grain_sizes.currentIndexChanged[int].connect(self._set_multiple_grain_sizes_visible)
        self.sediment_tab.bed_composition_input.currentIndexChanged[int].connect(self._set_bed_composition_visible)
        self.sediment_tab.use_advanced_size_classes.stateChanged[int].connect(self.diameters_table.set_columns_visible)

        # Add connections for events that could change the sediment labels.
        self.sediment_tab.use_simplified_multiple_grain_size.stateChanged[int].connect(self._set_size_and_layer_labels)
        self.bed_layers_table.model.rowsInserted[QModelIndex, int, int].connect(self._set_size_and_layer_labels)
        self.bed_layers_table.model.rowsRemoved[QModelIndex, int, int].connect(self._set_size_and_layer_labels)
        self.diameters_table.model.rowsInserted[QModelIndex, int, int].connect(self._set_size_and_layer_labels)
        self.diameters_table.model.rowsRemoved[QModelIndex, int, int].connect(self._set_size_and_layer_labels)

    def _setup_c2shore_warning(self):
        """Setup the C2Shore transport formula warning label."""
        self.sediment_tab.lbl_c2shore_warning.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
        self.sediment_tab.transport_formula.currentIndexChanged.connect(self._warn_if_c2shore_invalid)
        self.wave_tab.wave_info.currentIndexChanged.connect(self._warn_if_c2shore_invalid)

    def _warn_if_c2shore_invalid(self):
        """Show/hide the C2Shore transport formula warning based on the current wave forcing option."""
        # Can only use the C2Shore sediment transport formula if wave forcing is enabled.
        c2shore_transport = self.sediment_tab.transport_formula.currentIndex() == CBX_OPT_C2SHORE_TRANSPORT
        no_waves = self.wave_tab.wave_info.currentIndex() == CBX_OPT_WAVE_FORCING_NONE
        self.sediment_tab.lbl_c2shore_warning.setVisible(c2shore_transport and no_waves)

    def _set_sediment_visible_for_explicit(self):
        """Sets sediment tab widgets visible/hidden based on the solution scheme."""
        visible = self.general_tab.solution_scheme.currentIndex() == 1
        self.sediment_tab.transport_time_label.setVisible(visible)
        self.sediment_tab.transport_time_value.setVisible(visible)
        self.sediment_tab.transport_time_units.setVisible(visible)
        self.sediment_tab.morphologic_time_label.setVisible(visible)
        self.sediment_tab.morphologic_time_value.setVisible(visible)
        self.sediment_tab.morphologic_time_units.setVisible(visible)
        self.sediment_tab.formulation_units.model().hide_rows = not visible
        self.sediment_tab.formulation_units.model().invalidate()

    def _set_sediment_note_colors(self):
        """Sets the colors of some notes in the sediment tab."""
        self.sediment_tab.note_on_number_of_size_classes_label.setStyleSheet('color: green;')
        self.sediment_tab.layer_1_note_label.setStyleSheet('color: red;')

    def _set_size_and_layer_labels(self):
        """Sets labels warning the user about potentially bad combinations of bed layers and sediment diameters."""
        label_visible = False
        if not self.sediment_tab.use_simplified_multiple_grain_size.isChecked():
            num_bed_layers = self.bed_layers_table.model.rowCount()
            num_diameters = self.diameters_table.model.rowCount()
            if num_bed_layers == 1 and num_diameters > 1:
                label_visible = True
                text = 'Note: Multiple grain sizes requires at least 2 bed layers.'
                self.sediment_tab.grain_sizes_bed_layers_label.setText(text)
                self.sediment_tab.bed_layers_grain_sizes_label.setText(text)
            elif num_diameters == 1 and num_bed_layers > 1:
                label_visible = True
                text = 'Note: Multiple bed layers requires at least 2 grain sizes.'
                self.sediment_tab.grain_sizes_bed_layers_label.setText(text)
                self.sediment_tab.bed_layers_grain_sizes_label.setText(text)
        self.sediment_tab.grain_sizes_bed_layers_label.setVisible(label_visible)
        self.sediment_tab.bed_layers_grain_sizes_label.setVisible(label_visible)
        self.sediment_tab.grain_sizes_bed_layers_label.setStyleSheet('color: red;')
        self.sediment_tab.bed_layers_grain_sizes_label.setStyleSheet('color: red;')

    def _set_total_adaptation_visible(self, idx):
        """Sets the edit fields visible for the total adaptation method.

        Args:
            idx (int): The current index in the total load adaptation method combobox.
        """
        # 0 is Constant length, 1 is Constant time
        self.sediment_tab.total_load_adaptation_length_label.setVisible(idx == 0)
        self.sediment_tab.total_load_adaptation_length_value.setVisible(idx == 0)
        self.sediment_tab.total_load_adaptation_length_units.setVisible(idx == 0)
        self.sediment_tab.total_load_adaptation_time_label.setVisible(idx == 1)
        self.sediment_tab.total_load_adaptation_time_value.setVisible(idx == 1)
        self.sediment_tab.total_load_adaptation_time_units.setVisible(idx == 1)
        self.sediment_tab.non_const_adaptation_group.setVisible(idx > 1)

    def _set_bed_stack(self, check_state):
        """Sets the bed layer stack.

        Args:
            check_state(int): The state of the enabled simple multigrain checkbox.
        """
        if check_state == Qt.Checked:
            self.sediment_tab.bed_composition_stack.setCurrentIndex(0)
        else:
            self.sediment_tab.bed_composition_stack.setCurrentIndex(1)

    def _set_transport_formula_visible(self, idx):
        """Sets options appropriate for the given transport formula.

        Args:
            idx (int): The current index in the transport formula combobox.
        """
        # 3 = Watanabe
        # 4 = C2SHORE
        self.sediment_tab.watanabe_transport.setVisible(idx == 3)
        self.sediment_tab.watanabe_transport_label.setVisible(idx == 3)
        self.sediment_tab.c2shore_efficiency.setVisible(idx == 4)
        self.sediment_tab.c2shore_efficiency_label.setVisible(idx == 4)
        self.sediment_tab.c2shore_bedload.setVisible(idx == 4)
        self.sediment_tab.c2shore_bedload_label.setVisible(idx == 4)
        self.sediment_tab.c2shore_suspload.setVisible(idx == 4)
        self.sediment_tab.c2shore_suspload_label.setVisible(idx == 4)
        self.diameters_table.set_critical_shear_methods(self.sediment_tab.transport_formula.currentText())

    def _set_formulation_units_visible(self, idx):
        """Sets options appropriate for the given formulation units.

        Args:
            idx (int): The current index in the formulation units combobox.
        """
        # 0 = Equilibrium total load
        # 1 = Equilibrium bed load plus nonequilibrium susp load
        # 2 = Nonequilibrium total load
        self.sediment_tab.grain_size_label.setVisible(idx != 2)
        self.sediment_tab.grain_size_value.setVisible(idx != 2)
        self.sediment_tab.grain_size_units.setVisible(idx != 2)
        self.sediment_tab.adaptation_group.setVisible(idx == 2)
        self.sediment_tab.size_classes_group.setVisible(idx == 2)
        self.sediment_tab.bed_composition_group.setVisible(idx == 2)
        self.sediment_tab.transport_formula.model().hide_rows = idx != 2
        self.sediment_tab.transport_formula.model().invalidate()

    def _set_bed_load_method_visible(self, idx):
        """Sets options appropriate for the given bed load adaptation method.

        Args:
            idx (int): The current index in the bed load adaptation method combobox.
        """
        # 0 = Constant length
        # 1 = Constant time
        # 2 = Depth dependent
        self.sediment_tab.bed_load_adaptation_length_label.setVisible(idx == 0)
        self.sediment_tab.bed_load_adaptation_length_value.setVisible(idx == 0)
        self.sediment_tab.bed_load_adaptation_length_units.setVisible(idx == 0)
        self.sediment_tab.bed_load_adaptation_time_label.setVisible(idx == 1)
        self.sediment_tab.bed_load_adaptation_time_value.setVisible(idx == 1)
        self.sediment_tab.bed_load_adaptation_time_units.setVisible(idx == 1)
        self.sediment_tab.bed_load_adaptation_depth_label.setVisible(idx == 2)
        self.sediment_tab.bed_load_adaptation_depth.setVisible(idx == 2)

    def _set_suspended_load_method_visible(self, idx):
        """Sets options appropriate for the given suspended load adaptation method.

        Args:
            idx (int): The current index in the suspended load adaptation method combobox.
        """
        # 0 = Constant length
        # 1 = Constant time
        # 2 = Constant coefficient
        self.sediment_tab.suspended_load_adaptation_length_label.setVisible(idx == 0)
        self.sediment_tab.suspended_load_adaptation_length_value.setVisible(idx == 0)
        self.sediment_tab.suspended_load_adaptation_length_units.setVisible(idx == 0)
        self.sediment_tab.suspended_load_adaptation_time_label.setVisible(idx == 1)
        self.sediment_tab.suspended_load_adaptation_time_value.setVisible(idx == 1)
        self.sediment_tab.suspended_load_adaptation_time_units.setVisible(idx == 1)
        self.sediment_tab.suspended_load_adaptation_coefficient_label.setVisible(idx == 2)
        self.sediment_tab.suspended_load_adaptation_coefficient.setVisible(idx == 2)

    def _set_multiple_grain_sizes_visible(self, idx):
        """Sets options appropriate for the given method of dealing with multiple grain sizes.

        Args:
            idx (int): The current index in the multiple grain sizes combobox.
        """
        # 0 = Specify number of size classes only
        # 1 = Enter grain size values to use
        self.sediment_tab.number_of_size_classes_label.setVisible(idx == 0)
        self.sediment_tab.number_of_size_classes.setVisible(idx == 0)
        self.sediment_tab.note_on_number_of_size_classes_label.setVisible(idx == 0)
        self.grain_table.setVisible(idx == 1)

    def _set_bed_composition_visible(self, idx):
        """Sets options appropriate for the given bed composition input.

        Args:
            idx (int): The current index in the bed composition combobox.
        """
        # 0 = D50 sigma
        # 1 = D16 D50 D84
        # 2 = D35 D50 D90
        self.sediment_tab.sediment_standard_deviation.setVisible(idx == 0)
        self.sediment_tab.sediment_standard_deviation_label.setVisible(idx == 0)
        self.sediment_tab.d50.setVisible(idx in [0, 1, 2])
        self.sediment_tab.d50_label.setVisible(idx in [0, 1, 2])
        self.sediment_tab.d50_path.setVisible(idx in [0, 1, 2])
        self.sediment_tab.d16.setVisible(idx == 1)
        self.sediment_tab.d16_label.setVisible(idx == 1)
        self.sediment_tab.d16_path.setVisible(idx == 1)
        self.sediment_tab.d84.setVisible(idx == 1)
        self.sediment_tab.d84_label.setVisible(idx == 1)
        self.sediment_tab.d84_path.setVisible(idx == 1)
        self.sediment_tab.d35.setVisible(idx == 2)
        self.sediment_tab.d35_label.setVisible(idx == 2)
        self.sediment_tab.d35_path.setVisible(idx == 2)
        self.sediment_tab.d90.setVisible(idx == 2)
        self.sediment_tab.d90_label.setVisible(idx == 2)
        self.sediment_tab.d90_path.setVisible(idx == 2)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_general_data()
        self._save_flow_data()
        self._save_sediment_data()
        self._save_sal_temp_data()
        self._save_wave_data()
        self._save_wind_data()
        self._save_output_data()
        super().accept()
