"""A dialog for editing the attributes of a save point."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.cmsflow.gui.save_point_attributes_dlg_ui import Ui_SavePointAttributesDlg


class SavePointAttributesDlg(XmsDlg):
    """A dialog for viewing a single save point's data."""
    def __init__(self, save_point_data, multi_msg, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            save_point_data (SavePointsData): The save point data to view
            multi_msg (str): Warning/multi-select message, if any
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.cmsflow.gui.save_point_attributes_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/CMS-Flow/Save_Points'
        self.ui = Ui_SavePointAttributesDlg()
        self.ui.setupUi(self)
        self.save_point_data = save_point_data
        self.multi_msg = multi_msg
        self._add_multi_select_warning()
        self._load_data()
        self.ui.btn_box.helpRequested.connect(self.help_requested)

    def _add_multi_select_warning(self):
        """Add a label for the multi-select warning, if applicable."""
        if self.multi_msg:
            label = QLabel(self.multi_msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.layout().insertWidget(0, label)

    def _load_data(self):
        """Loads the data into the widgets."""
        if not self.save_point_data.name.size == 0:
            self.ui.edt_name.setText(self.save_point_data['name'].item())
            self.ui.check_hydro.setChecked(self.save_point_data['hydro'].item() != 0)
            self.ui.check_sediment.setChecked(self.save_point_data['sediment'].item() != 0)
            self.ui.check_salinity.setChecked(self.save_point_data['salinity'].item() != 0)
            self.ui.check_waves.setChecked(self.save_point_data['waves'].item() != 0)

    def _save_point_data(self):
        """Save data from the save point attributes."""
        self.save_point_data['name'] = self.ui.edt_name.text()
        self.save_point_data['hydro'] = 1 if self.ui.check_hydro.checkState() == Qt.Checked else 0
        self.save_point_data['sediment'] = 1 if self.ui.check_sediment.checkState() == Qt.Checked else 0
        self.save_point_data['salinity'] = 1 if self.ui.check_salinity.checkState() == Qt.Checked else 0
        self.save_point_data['waves'] = 1 if self.ui.check_waves.checkState() == Qt.Checked else 0

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_point_data()
        super().accept()
