"""This is a dialog for editing the properties of the save points coverage."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.cmsflow.gui.save_points_properties_dlg_ui import Ui_SavePointsPropertiesDlg
from xms.cmsflow.gui.unit_converter import UnitConverter


class SavePointsPropertiesDlg(XmsDlg):
    """A dialog for viewing save point properties data."""
    def __init__(self, save_points_data, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            save_points_data (SavePointsData): The save points data to view
            parent (Something derived from :obj:`QWidget`): The parent window

        """
        super().__init__(parent, 'xms.cmsflow.gui.save_points_properties_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/CMS-Flow/Save_Points'
        self.ui = Ui_SavePointsPropertiesDlg()
        self.ui.setupUi(self)
        self.save_points_data = save_points_data
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_validator.setDecimals(10)
        self.unit_dict = {'hydro': -1, 'sediment': -1, 'salinity': -1, 'wave': -1}
        self.units = [1.0, 60.0, 3600.0]
        self._load_data()
        self.ui.btn_box.helpRequested.connect(self.help_requested)
        self.ui.hydro_units.currentIndexChanged.connect(
            UnitConverter.make_change_units(self.ui.hydro_value, self.units, 'hydro', self.unit_dict)
        )
        self.ui.sediment_units.currentIndexChanged.connect(
            UnitConverter.make_change_units(self.ui.sediment_value, self.units, 'sediment', self.unit_dict)
        )
        self.ui.salinity_units.currentIndexChanged.connect(
            UnitConverter.make_change_units(self.ui.salinity_value, self.units, 'salinity', self.unit_dict)
        )
        self.ui.waves_units.currentIndexChanged.connect(
            UnitConverter.make_change_units(self.ui.waves_value, self.units, 'wave', self.unit_dict)
        )

    @staticmethod
    def _add_units(units_widget, unit_text):
        """Adds time units to the combo box."""
        old_idx = -1
        for idx, text in enumerate(['seconds', 'minutes', 'hours']):
            units_widget.addItem(text)
            if text == unit_text:
                units_widget.setCurrentIndex(idx)
                old_idx = idx
        return old_idx

    def _load_data(self):
        """Loads data into the widgets."""
        general_attrs = self.save_points_data.general.attrs
        self.ui.hydro_value.setText(str(general_attrs['HYDRO_VALUE']))
        self.unit_dict['hydro'] = self._add_units(self.ui.hydro_units, general_attrs['HYDRO_UNITS'])
        self.ui.sediment_value.setText(str(general_attrs['SEDIMENT_VALUE']))
        self.unit_dict['sediment'] = self._add_units(self.ui.sediment_units, general_attrs['SEDIMENT_UNITS'])
        self.ui.salinity_value.setText(str(general_attrs['SALINITY_VALUE']))
        self.unit_dict['salinity'] = self._add_units(self.ui.salinity_units, general_attrs['SALINITY_UNITS'])
        self.ui.waves_value.setText(str(general_attrs['WAVES_VALUE']))
        self.unit_dict['waves'] = self._add_units(self.ui.waves_units, general_attrs['WAVES_UNITS'])
        self.ui.hydro_value.setValidator(self.dbl_validator)
        self.ui.sediment_value.setValidator(self.dbl_validator)
        self.ui.salinity_value.setValidator(self.dbl_validator)
        self.ui.waves_value.setValidator(self.dbl_validator)

    def _save_points_data(self):
        """Save data from the save points properties."""
        general_attrs = self.save_points_data.general.attrs
        general_attrs['HYDRO_VALUE'] = self.ui.hydro_value.text()
        general_attrs['HYDRO_UNITS'] = self.ui.hydro_units.currentText()
        general_attrs['SEDIMENT_VALUE'] = self.ui.sediment_value.text()
        general_attrs['SEDIMENT_UNITS'] = self.ui.sediment_units.currentText()
        general_attrs['SALINITY_VALUE'] = self.ui.salinity_value.text()
        general_attrs['SALINITY_UNITS'] = self.ui.salinity_units.currentText()
        general_attrs['WAVES_VALUE'] = self.ui.waves_value.text()
        general_attrs['WAVES_UNITS'] = self.ui.waves_units.currentText()
        self.save_points_data.general.attrs = general_attrs

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_points_data()
        super().accept()
