"""This is a table widget for specifying sediment diameters, as well as other sediment properties."""

# 1. Standard Python modules
import typing

# 2. Third party modules
from PySide2.QtCore import QModelIndex, QRect, QSize, QSortFilterProxyModel, Qt
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import (
    QAbstractItemView, QApplication, QComboBox, QHBoxLayout, QLineEdit, QStyle, QStyledItemDelegate,
    QStyleOptionComboBox, QStyleOptionFrame, QStyleOptionSpinBox, QStyleOptionViewItem, QWidget
)

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.cmsflow.gui.cmsflow_table_widget import CmsflowTableWidget
from xms.cmsflow.gui.unit_converter import UnitConverter


class StretchHeaderRenameModel(QSortFilterProxyModel):
    """A model to rename header titles."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        self.column_names = column_names
        self.no_empty_column_names = [col for col in column_names if col != '']
        self.hide_columns = False
        super().__init__(parent)

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (int): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (int): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Horizontal:
            return self.no_empty_column_names[section]
        else:
            return super().headerData(section, orientation, role)

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Filters out columns from the view.

        Args:
            source_column (int): The source column index.
            source_parent (QModelIndex): The index in the source model.

        Returns:
            True if the column is to be shown in the view.
        """
        if self.hide_columns and source_column > 1:
            return False
        elif self.column_names[source_column]:
            return super().filterAcceptsColumn(source_column, source_parent)
        else:
            return False

    def data(self, index: QModelIndex, role: int = ...) -> typing.Any:
        """Gets the model data. This is an override of the function in QSortFilterProxyModel.

        Args:
            index (QModelIndex): The index that has the data.
            role (int): The role of the data.

        Returns:
            Gives the data for the index and role.
        """
        source_idx = self.mapToSource(index)
        if role == Qt.UserRole:
            return self.sourceModel().data(source_idx, Qt.DisplayRole)
        elif role == Qt.UserRole + 1:
            index = self.sourceModel().index(source_idx.row(), source_idx.column() + 1)
            return self.sourceModel().data(index, Qt.DisplayRole)
        return self.sourceModel().data(source_idx, role)

    def setData(self, index: QModelIndex, value: typing.Any, role: int = ...) -> bool:  # noqa: N802
        """Sets the model data. This is an override of the function in QSortFilterProxyModel.

        Args:
            index (QModelIndex): The index to set data on.
            value (Any): The value to set.
            role (int): The role to set the data on.
        """
        source_idx = self.mapToSource(index)
        if role == Qt.UserRole:
            return self.sourceModel().setData(source_idx, value, Qt.EditRole)
        elif role == Qt.UserRole + 1:
            index = self.sourceModel().index(source_idx.row(), source_idx.column() + 1)
            return self.sourceModel().setData(index, value, Qt.EditRole)
        return self.sourceModel().setData(source_idx, value, role)

    def flags(self, index: QModelIndex) -> Qt.ItemFlags:
        """Returns flags for the indexes of the filtered model.

        Args:
            index (QModelIndex): The index for which to get flags.
        """
        # diameter_col_idx = 0
        fall_velocity_method_col_idx = 1
        fall_velocity_col_idx = 2
        corey_shape_col_idx = 3
        critical_shear_method_col_idx = 4
        critical_shear_stress_col_idx = 5
        local_flags = super().flags(index)
        if index.column() in [fall_velocity_col_idx, corey_shape_col_idx]:
            cell = index.model().index(index.row(), fall_velocity_method_col_idx, index.parent())
            current_method = cell.data(Qt.EditRole)

            if index.column() == fall_velocity_col_idx:
                if current_method == 'User specified':
                    local_flags = local_flags | Qt.ItemIsEnabled
                else:
                    local_flags = local_flags & ~Qt.ItemIsEnabled
            else:
                if current_method == 'Wu and Wang (2006)':
                    local_flags = local_flags | Qt.ItemIsEnabled
                else:
                    local_flags = local_flags & ~Qt.ItemIsEnabled
        elif index.column() == critical_shear_stress_col_idx:
            cell = index.model().index(index.row(), critical_shear_method_col_idx, index.parent())
            current_shear_method = cell.data(Qt.EditRole)
            if current_shear_method == 'User specified':
                local_flags = local_flags | Qt.ItemIsEnabled
            else:
                local_flags = local_flags & ~Qt.ItemIsEnabled
        return local_flags


class ValueAndUnitDelegate(QStyledItemDelegate):
    """This class changes a cell to display and edit as an edit field and a combo box of units.

    Changing the units will automatically convert the value.
    """
    def __init__(self, units, unit_names, parent):
        """Initializes the class.

        Args:
            units (list): A list of floating point unit scaling factors.
            unit_names (list): A list of unit names. Parallel to units.
            parent (QObject): The parent object.
        """
        super().__init__(parent)
        self._dbl_validator = QxDoubleValidator(parent=self)
        self._dbl_validator.setBottom(0.0)
        self._dbl_validator.setDecimals(10)
        self._unit_names = unit_names
        self._units = units
        self._value_width_hint_ratio = 0.0
        self._row_height = 0.0
        self._unit_dict = {0: -1}
        self._resetting = False

    def paint(self, painter, option, index):
        """Paints the edit field and combobox.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        """
        if not index.isValid():
            return
        opt_units, opt_value = self._get_half_size_options(option)
        dummy = QLineEdit()
        value_text = index.data(Qt.UserRole)
        dummy.setText(value_text)
        edit_opt = QStyleOptionFrame()
        edit_opt.initFrom(dummy)
        edit_opt.text = value_text
        edit_opt.rect = opt_value.rect
        edit_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            edit_opt.state |= QStyle.State_Enabled
        edit_opt.editable = False

        QApplication.style().drawPrimitive(QStyle.PE_PanelLineEdit, edit_opt, painter)
        QApplication.style().drawPrimitive(QStyle.PE_FrameLineEdit, edit_opt, painter)
        margin = 6  # Qt uses 6 as a margin by default
        edit_opt.rect.setTop(edit_opt.rect.top() + margin)
        edit_opt.rect.setLeft(edit_opt.rect.left() + margin)
        painter.drawText(QApplication.style().subElementRect(QStyle.SE_LineEditContents, edit_opt), value_text)

        current_text = index.data(Qt.UserRole + 1)
        cbx_opt = QStyleOptionComboBox()
        cbx_opt.currentText = current_text
        cbx_opt.rect = opt_units.rect
        cbx_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            cbx_opt.state |= QStyle.State_Enabled
        cbx_opt.editable = False

        QApplication.style().drawComplexControl(QStyle.CC_ComboBox, cbx_opt, painter)
        QApplication.style().drawControl(QStyle.CE_ComboBoxLabel, cbx_opt, painter)

    def sizeHint(self, option, index):  # noqa: N802
        """Help keep the size adjusted for custom painted combobox.

        Args:
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            (QSize): An appropriate size hint

        """
        hint = super().sizeHint(option, index)
        unit_hint = QSize(hint)
        value_hint = QSize(hint)
        fm = QFontMetrics(option.font)
        cb_opt = QStyleOptionComboBox()
        cb_opt.rect = option.rect
        cb_opt.state = option.state | QStyle.State_Enabled

        for opt in self._unit_names:
            unit_hint = unit_hint.expandedTo(
                QApplication.style().sizeFromContents(
                    QStyle.CT_ComboBox, cb_opt, QSize(fm.boundingRect(opt).width(), unit_hint.height())
                )
            )

        edit_opt = QStyleOptionSpinBox()
        edit_opt.rect = option.rect
        edit_opt.state = option.state | QStyle.State_Enabled
        value_hint = value_hint.expandedTo(
            QApplication.style().sizeFromContents(
                QStyle.CT_LineEdit, edit_opt,
                QSize(fm.boundingRect('1000000000.0000000001').width(), value_hint.height())
            )
        )

        hint = unit_hint
        if self._value_width_hint_ratio == 0.0:
            self._value_width_hint_ratio = float(value_hint.width()) / float(value_hint.width() + unit_hint.width())
        hint.setWidth(value_hint.width() + hint.width())
        self._row_height = hint.height()
        return hint

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates the combobox and populates it.

        Args:
            parent (QWidget): The parent.
            option (QStyleOptionViewItem): The option
            index (QModelIndex): The index

        Returns:
            QWidget
        """
        opt_units, opt_value = self._get_half_size_options(option)
        widget = QWidget(parent)
        widget.setLayout(QHBoxLayout(widget))
        widget.layout().setSpacing(0)
        widget.layout().setMargin(0)
        widget.setContentsMargins(0, 0, 0, 0)
        widget.setGeometry(option.rect)
        value_edit = QLineEdit(parent)
        units_edit = QComboBox(parent)
        value_edit.setMinimumHeight(option.rect.height())
        units_edit.setMinimumHeight(option.rect.height())
        units_edit.addItems(self._unit_names)
        value_edit.setValidator(self._dbl_validator)
        value_edit.setGeometry(opt_value.rect)
        units_edit.setGeometry(opt_units.rect)
        self._unit_dict[0] = -1
        value_edit.editingFinished.connect(lambda: self.commitData.emit(widget))
        units_edit.currentIndexChanged.connect(lambda: self._set_value_for_unit(widget))
        widget.layout().addWidget(value_edit)
        widget.layout().addWidget(units_edit)
        return widget

    def setEditorData(self, editor, index):  # noqa: N802
        """Sets the data to be displayed and edited by the editor from the data model item specified by the model index.

        Args:
            editor (QWidget): The editor.
            index (QModelIndex): The index.

        """
        if not editor:
            return
        # check if we are manually resetting data, if so, don't come in here
        if self._resetting:
            return
        line_edits = editor.findChildren(QLineEdit)
        combo_boxes = editor.findChildren(QComboBox)
        current_text = index.data(Qt.UserRole)
        line_edits[0].setText(current_text)

        units = combo_boxes[0]
        if not units:
            return
        current_text = index.data(Qt.UserRole + 1)
        cb_index = units.findText(current_text, Qt.MatchFixedString)  # Case insensitive
        if cb_index >= 0:
            units.setCurrentIndex(cb_index)
            self._unit_dict[0] = cb_index

    def setModelData(self, editor, model, index):  # noqa: N802
        """Gets data from the editor widget and stores it in the specified model at the item index.

        Args:
            editor (QWidget): The editor.
            model (QAbstractItemModel): The model.
            index (QModelIndex): The index

        """
        if not editor:
            return
        line_edits = editor.findChildren(QLineEdit)
        combo_boxes = editor.findChildren(QComboBox)
        model.setData(index, line_edits[0].text(), Qt.UserRole)

        cb_index = combo_boxes[0].currentIndex()
        # If it is valid, adjust the combobox
        if cb_index >= 0:
            model.setData(index, combo_boxes[0].currentText(), Qt.UserRole + 1)

    def _get_half_size_options(self, option):
        """Gets options with rectangles for the new locations."""
        opt_value = QStyleOptionViewItem(option)
        opt_units = QStyleOptionViewItem(option)
        value_size = option.rect.size()
        if self._value_width_hint_ratio == 0.0:
            half_width = value_size.width() / 2.0
        else:
            half_width = value_size.width() * self._value_width_hint_ratio
        value_size.setWidth(half_width)
        unit_size = QSize(value_size)
        unit_size.setWidth(option.rect.width() - value_size.width())
        rect1 = QRect(option.rect.topLeft(), value_size)
        unit_top_left = option.rect.topLeft()
        unit_top_left.setX(unit_top_left.x() + half_width)
        rect2 = QRect(unit_top_left, unit_size)
        opt_value.rect = rect1
        opt_units.rect = rect2
        return opt_units, opt_value

    def _set_value_for_unit(self, widget):
        """Sets the value for the new unit."""
        if self._resetting:
            return
        line_edits = widget.findChildren(QLineEdit)
        combo_boxes = widget.findChildren(QComboBox)
        current_idx = combo_boxes[0].currentIndex()
        if self._unit_dict[0] >= 0 and self._unit_dict[0] != current_idx:
            UnitConverter.make_change_units(line_edits[0], self._units, 0, self._unit_dict)(current_idx)
            self._unit_dict[0] = current_idx
        self._resetting = True
        self.commitData.emit(widget)
        self._resetting = False


class SparseComboboxDelegate(QxCbxDelegate):
    """This allows cells to display and edit as combo boxes.

    This class also allows for the list of options in the combo box to change.
    """
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)
        self._current_strings = []  # List of str. The currently available items in the combo box.

    def set_current_strings(self, strings):
        """Sets the strings that are currently available in the combo box.

        Args:
            strings (list of str): The currently available strings in the combo box.

        """
        self._current_strings = strings

    def get_current_strings(self):
        """Returns the list of currently available strings in the combo box.

        Returns:
            See description.
        """
        return self._current_strings

    def paint(self, painter, option, index):
        """Paints a combobox with the current selection on a cell.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        """
        if not index.isValid():
            return

        current_text = index.data(Qt.EditRole)
        cbx_opt = QStyleOptionComboBox()
        if current_text in self._current_strings:
            cbx_opt.currentText = current_text
        else:
            cbx_opt.currentText = ''
        cbx_opt.rect = option.rect
        cbx_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            cbx_opt.state |= QStyle.State_Enabled
        cbx_opt.editable = False

        QApplication.style().drawComplexControl(QStyle.CC_ComboBox, cbx_opt, painter)
        QApplication.style().drawControl(QStyle.CE_ComboBoxLabel, cbx_opt, painter)

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates the combobox and populates it.

        Args:
            parent (QWidget): The parent.
            option (QStyleOptionViewItem): The option
            index (QModelIndex): The index

        Returns:
            QWidget
        """
        self.cb = QComboBox(parent)
        self.cb.addItems(self._current_strings)
        self.cb.currentIndexChanged.connect(self.on_index_changed)
        return self.cb


class SedimentDiametersTableWidget(CmsflowTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
        """
        self.rename_model = None
        self.edit_delegate = None
        self.length_delegate = None
        self.velocity_delegate = None
        self.velocity_method_delegate = None
        self.shear_method_delegate = None
        self.dbl_validator = None
        default_values = {
            'diameter_value': 0.2,
            'diameter_units': 'mm',
            'fall_velocity_method': 'Soulsby (1997)',
            'fall_velocity_value': 0.0,
            'fall_velocity_units': 'm/s',
            'corey_shape_factor': 0.7,
            'critical_shear_method': 'Soulsby (1997)',
            'critical_shear_stress': 0.2
        }
        super().__init__(parent, data_frame, 0, default_values)

    def setup_ui(self):
        """Sets up the table."""
        column_names = [
            'Diameter', '', 'Fall Velocity Method', 'Fall Velocity', '', 'Corey Shape Factor', 'Critical Shear Method',
            'Critical Shear Stress (Pa)'
        ]
        self.rename_model = StretchHeaderRenameModel(column_names, self)
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        units = [1000000.0, 10000.0, 1000.0, 1.0, 304800.0, 25400.0]
        names = ['m', 'cm', 'mm', 'um', 'ft', 'in']
        self.length_delegate = ValueAndUnitDelegate(units, names, self)
        self.velocity_delegate = ValueAndUnitDelegate([1000.0, 10.0, 1.0, 304.8], ['m/s', 'cm/s', 'mm/s', 'ft/s'], self)
        self.velocity_method_delegate = QxCbxDelegate(self)
        self.velocity_method_delegate.set_strings(['Soulsby (1997)', 'Wu and Wang (2006)', 'User specified'])
        self.shear_method_delegate = SparseComboboxDelegate(self)
        self.shear_method_delegate.set_strings(
            ['Soulsby (1997)', 'van Rijn (2007)', 'Wu and Wang (1999)', 'User specified']
        )
        delegates = {
            0: self.length_delegate,
            1: self.velocity_method_delegate,
            2: self.velocity_delegate,
            3: self.edit_delegate,
            4: self.shear_method_delegate,
            5: self.edit_delegate
        }
        super()._setup_ui(delegates, False, False, self.rename_model, False)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def set_columns_visible(self, use_advanced_state):
        """Sets whether to show the advanced columns.

        Args:
           use_advanced_state (Qt.CheckState): The state of the show advanced sizes toggle.
        """
        self.rename_model.hide_columns = use_advanced_state != Qt.Checked
        self.rename_model.invalidate()

    def set_critical_shear_methods(self, transport_formula_text):
        """Sets the currently available critical shear methods.

        Args:
            transport_formula_text(str): The current transport formula used.
        """
        # 'Lund-CIRP', 'van Rijn', 'Soulsby-van Rijn', 'Watanabe'
        current_methods = []
        # not sure how we could get an empty string, but just in case
        if not transport_formula_text:
            current_methods = ['Soulsby (1997)', 'van Rijn (2007)', 'Wu and Wang (1999)', 'User specified']
            self.shear_method_delegate.set_current_strings(current_methods)
            return

        if transport_formula_text != 'van Rijn':
            current_methods.append('Soulsby (1997)')
        else:
            current_methods.append('van Rijn (2007)')

        if transport_formula_text not in ['van Rijn', 'Soulsby-van Rijn']:
            current_methods.append('Wu and Wang (1999)')
            current_methods.append('User specified')
        self.shear_method_delegate.set_current_strings(current_methods)
