"""This handles the hidden component side of merging multiple save points coverages together."""

# 1. Standard Python modules
import os
import traceback
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList

# 4. Local modules
from xms.cmsflow.components.id_files import SAVE_INITIAL_ATT_ID_FILE, SAVE_INITIAL_COMP_ID_FILE
from xms.cmsflow.data.save_points_data import SavePointsData


class SavePointsMerger:
    """Handles merging of Save Points coverages."""
    ATT_FILE_IDX = 0
    COMP_FILE_IDX = 1
    MAIN_FILE_IDX = 0
    ID_FILE_IDX = 1

    def __init__(self, comp):
        """Construct the merger.

        Args:
            comp (SavePointsComponent): The Save Points coverage component to merge data into

        """
        self._comp = comp
        self._point_att_ids = []
        self._point_comp_ids = []

    def _merge_cov_data(self, main_file, general_dset):
        """Copy over display options and coverage-level attributes to the destination component.

        Args:
            main_file (str): Filepath of the source SavePointsComponent main file
            general_dset (xarray.Dataset): The general Dataset of the source SavePointsData

        """
        # Replace the source's display list UUID.
        point_categories = CategoryDisplayOptionList()
        old_disp_opts = os.path.join(os.path.dirname(main_file), 'save_point_display_options.json')
        point_json_dict = read_display_options_from_json(old_disp_opts)
        point_categories.from_dict(point_json_dict)
        point_categories.uuid = str(uuid.uuid4())
        write_display_options_to_json(self._comp.disp_opts_file, point_categories)
        self._comp.data.info.attrs['point_display_uuid'] = point_categories.uuid
        # Set coverage property data.
        self._comp.data.general = general_dset

    def _merge_point_data(self, data, att_id_file, comp_id_file):
        """Copy over the point attributes of a coverage in the merge list.

        Args:
            data (SavePointsData): The data to merge
            att_id_file (str): File path to the source XMS feature id file
            comp_id_file (str): File path to the source component id file

        """
        old_to_new_station_point_ids = self._comp.data.concat_save_points(data)
        att_ids = read_display_option_ids(att_id_file)
        comp_ids = read_display_option_ids(comp_id_file)
        self._point_att_ids.extend(att_ids)
        for comp_id in comp_ids:
            self._point_comp_ids.append(old_to_new_station_point_ids[comp_id])

    def _update_id_files(self):
        """Store save point att ids and comp ids in a temp file so they can be processed later.

        Also writes the persistent display id referenced in the display options json file for the component.

        """
        point_att_file = os.path.join(os.path.dirname(self._comp.main_file), SAVE_INITIAL_ATT_ID_FILE)
        write_display_option_ids(point_att_file, self._point_att_ids)
        point_comp_file = os.path.join(os.path.dirname(self._comp.main_file), SAVE_INITIAL_COMP_ID_FILE)
        write_display_option_ids(point_comp_file, self._point_comp_ids)
        # Write comp ids to persistent display id file.
        write_display_option_ids(self._comp.point_comp_id_file, self._point_comp_ids)

    def merge(self, merge_list):
        """Merge Save Points coverage data into destination component.

        Args:
            merge_list (:obj:`list` of :obj:`tuple`): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        """
        messages = []
        copied_cov_atts = False
        for cov in merge_list:
            try:
                save_pts_data = SavePointsData(cov[self.MAIN_FILE_IDX])

                # Copy over display options and coverage-level data of the first coverage in merge list.
                if not copied_cov_atts:
                    self._merge_cov_data(cov[self.MAIN_FILE_IDX], save_pts_data.general)
                    copied_cov_atts = True

                id_files = cov[self.ID_FILE_IDX].get('POINT')
                if not id_files:
                    continue  # Empty coverage

                # Concatenate the save points data.
                self._merge_point_data(
                    data=save_pts_data,
                    att_id_file=id_files[self.ATT_FILE_IDX],
                    comp_id_file=id_files[self.COMP_FILE_IDX]
                )
            except Exception as ex:
                trace = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                messages.append(('ERROR', f'Could not merge Save Points coverage attributes: {trace}'))

        self._update_id_files()

        # Wipe the old coverage's UUID. Will Query for a the new coverage's UUID in get_initial_display_options.
        self._comp.data.info.attrs['cov_uuid'] = ''
        self._comp.data.commit()

        # Send back ActionRequest to Query for the new coverage's UUID and update id files.
        return messages, [self._comp.get_display_refresh_action()]
