"""StruturesComponent class. Data for structure Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import (write_display_option_ids, write_display_options_to_json)
from xms.core.filesystem import filesystem
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.cmswave.components.structures_component_display import StructuresComponentDisplay
from xms.cmswave.components.structures_coverage_component import duplicate_display_opts, StructuresCoverageComponent
from xms.cmswave.data.structures_data import StructuresData

STRUCT_INITIAL_ATT_ID_FILE = 'initial_struct_polys.ids'
STRUCT_INITIAL_COMP_ID_FILE = 'initial_struct_comp.ids'


class StructuresComponent(StructuresCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the structure coverage."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.is_sediment = False
        self.data = StructuresData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Structure List and Properties', 'open_structures_properties'),
        ]
        self.polygon_commands = [('Assign Structure', 'open_assign_poly_structures')]
        self.disp_opts_files = ['']
        # Copy default display options if needed
        display_helper = StructuresComponentDisplay(self)
        display_helper.ensure_display_options_exist()

    @classmethod
    def unique_name(cls) -> str:
        """
        The unique name for this component.

        XMS needs this to be unique for a model. By convention, we just make this equal to the class name.
        """
        return cls.__name__

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of 'DUPLICATE', 'PACKAGE', 'SAVE', 'SAVE_AS', 'LOCK'.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple(str, list, list)`):

                new_main_file (:obj:`str`): Path to the new component's mainfile

                messages (:obj:`list(tuple(str, str))`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            json_dict = duplicate_display_opts(new_path, os.path.basename(self.disp_opts_files[0]))
            data = StructuresData(new_main_file)
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.commit()

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            params (:obj:`list[dict]`): Generic map of parameters. Contains selection map and component id files.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        display_helper = StructuresComponentDisplay(self)
        return display_helper.initialize_display(query)

    def open_assign_poly_structures(self, query, params, win_cont):
        """Opens the Assign Structures dialog and saves component data state on OK.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            params (:obj:`list[dict]`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`QWidget`): The window container.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        display_helper = StructuresComponentDisplay(self)
        return display_helper.open_assign_poly_structures(query, params[0], win_cont)

    def open_structures_properties(self, query, params, win_cont):
        """Opens the Structure Properties dialog and saves component data state on OK.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            params (:obj:`list[dict]`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`QWidget`): The window container.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        display_helper = StructuresComponentDisplay(self)
        return display_helper.open_structures_properties(query, win_cont)

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list[tuple]`):

                tuple containing:

                    main_file (:obj:`str`): The absolute path to the main file of the old component this
                    component is being merged from.

                    id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                    Each value is a tuple that may have two absolute file paths or none. The first
                    file is for the ids in XMS on the coverage. The second file contains the ids the
                    old component used for those objects. Both id files should be equal in length.
                    This dictionary is only applicable if the component derives from
                    CoverageComponentBase.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        main_file_idx = 0
        # att_file = 0
        # comp_file = 1
        # id_file_idx = 1
        # all_att_ids = []
        # new_comp_ids = []
        for m in merge_list:
            struct = StructuresData(m[main_file_idx])
            self.data.add_structures(struct)
            # old_to_new_mat_ids = self.data.add_materials(mat)
            # if len(m[id_file_idx]['POLYGON']) < 1:
            #     continue
            # att_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][att_file])
            # comp_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][comp_file])
            # all_att_ids.extend(att_ids)
            # for _, comp_id in zip(att_ids, comp_ids):
            #     new_comp_ids.append(int(old_to_new_mat_ids[comp_id]))
        # save attids and compids to a temp file so they can be processed later
        # tmpfname = os.path.join(os.path.dirname(self.main_file), 'tmp_att_ids_comp_ids.txt')
        # with open(tmpfname, 'w') as file:
        #     for att_id, comp_id in zip(all_att_ids, new_comp_ids):
        #         file.write(f'{att_id} {comp_id}\n')
        display_helper = StructuresComponentDisplay(self)
        display_helper.update_display_options_file()
        all_struct_ids = list(self.data.structures.to_dataframe()['id'].astype(int))
        display_helper.update_display_id_files([], all_struct_ids)
        self.data.commit()

        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            comp_uuid=self.uuid,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def update_display_id_files(self, old_ids, new_ids):
        """Update the display files.

        Args:
            old_ids (:obj:`list`): list of ids before editing structures
            new_ids (:obj:`list`): list of current structure ids

        Returns:
            (:obj:`list`) : deleted ids
        """
        deleted_ids = old_ids
        path = os.path.dirname(self.main_file)
        for struct_id in new_ids:
            if struct_id >= 0:
                id_file = f'struct_{struct_id}.structid'
                filename = os.path.join(path, id_file)
                write_display_option_ids(filename, [struct_id])
            if struct_id in deleted_ids:
                deleted_ids.remove(struct_id)

        for struct_id in deleted_ids:
            id_file = f'struct_{struct_id}.structid'
            filename = os.path.join(path, id_file)
            filesystem.removefile(filename)
        return deleted_ids

    def update_display_options_file(self):
        """Update the XMS display options JSON file to match what we have in memory."""
        category_list = CategoryDisplayOptionList()
        category_list.target_type = TargetType.polygon
        category_list.uuid = str(self.data.info.attrs['display_uuid'])
        category_list.comp_uuid = self.uuid
        df = self.data.structures.to_dataframe()
        for struct_row in range(0, len(df.index)):
            category = CategoryDisplayOption()
            category.id = int(df.iloc[struct_row]['id'])
            category.description = df.iloc[struct_row]['name']
            category.file = f'struct_{category.id}.structid'
            category.options = PolygonOptions()
            style = df.iloc[struct_row]['color'].split()
            category.options.color = QColor(int(style[0]), int(style[1]), int(style[2]), 255)
            category.options.texture = int(style[3])
            # Make bathymetry modification structure the default category.
            if category.id == 0:  # Should always have 0 as "structure id"
                category.is_unassigned_category = True
            category_list.categories.append(category)
        write_display_options_to_json(self.disp_opts_files[0], category_list)
