"""A writer for CMS-Wave simulation files."""

# 1. Standard Python modules
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.data_objects.parameters import FilterLocation
from xms.snap.snap_point import SnapPoint

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const
from xms.cmswave.data.simulation_data import SimulationData
from xms.cmswave.dmi.xms_data import XmsData


# Adapted from the xml sim file written by ldabell
class SimWriter:
    """A class for writing out an CMS-Wave simulation file."""
    def __init__(self):
        """Constructor."""
        self._logger = logging.getLogger('xms.cmswave')
        self.ss = StringIO()
        self.simulation_name = ''
        self.grid_angle = 0.0  # initialized in write_grid_section(), don't use until after called
        self.case_times = []
        self.query = Query()
        self.data = None
        self.grid = None
        self.cogrid_file = ''
        self.proj = None
        self.snapper = SnapPoint()

    def _get_iprp(self):
        """Get iprp parameter based on values of 'Source terms' and 'Fast mode' options in interface.

        Returns:
            (:obj:`int`): The CMS-Wave value for iprp
        """
        #  0 == Wind and spectra (source terms and propagation), 1 == Spectra only
        # -2 == Fast mode using spectra only, -1 == Fast mode with spectra and wind
        iprp = 1 if self.data.info.attrs['source_terms'] == 'Propagation only' else 0
        if self.data.info.attrs['fastmode']:
            iprp = -2 if iprp == 1 else -1
        return iprp

    def write_opts_file(self, filename, version=2):
        """
        Writes out the depth file.

        Arguments:
            filename(:obj:`str`):  The file to write to.
            version(:obj:`int`):  The version of the file to output.

        Returns:
            (:obj:`bool`):  True if success, else False
        """
        self._logger.info('Writing opts file.')
        if version == 2:
            return self._write_opts_file_v2(filename)
        else:
            raise IOError("Invalid OPTS file version")

    def _write_opts_file_v2(self, filename):
        """
        Writes out the opts file.

        Arguments:
            filename(:obj:`str`):  The file to write to.

        Returns:
            (:obj:`bool`):  True if success, else False
        """
        self._logger.info('Writing opts file (v2).')
        ss = StringIO()

        if self.grid:
            self._logger.info('Initializing grid snapper.')
            self.snapper.set_grid(self.grid, True)

        # Write header, identifying the file as the new file format
        ss.write('CMS_WAVE_STD               2\n\n')

        # First section:  iprp, iview, isolv, icur, ibnd, ibf, iark, iarkr, iwet, akap, iproc

        choices = {0: 'WIND_AND_SPECTRA', 1: 'SPECTRA_ONLY', -1: 'FAST-MODE', -2: 'FAST-MODE_SPECTRA_ONLY'}
        ss.write(f"WV_PROPAGATION_TYPE        '{choices[self._get_iprp()]}'\n")
        iview = 0 if self.data.info.attrs['plane'] == const.CBX_TEXT_HALF_PLANE else 1 \
            if self.data.info.attrs['plane'] == const.CBX_TEXT_FULL_PLANE else 2
        choices = {0: 'HALF-PLANE', 1: 'FULL-PLANE', 2: 'FULL-PLANE_WITH_REVERSE_SPECTRA'}
        ss.write(f"WV_PLANE_DEFINITION        '{choices[iview]}'\n")
        isolv = 0 if self.data.info.attrs['matrix_solver'] == 'Gauss-Seidel' else 1
        choices = {0: 'GAUSS-SEIDEL', 1: 'ADI'}
        ss.write(f"WV_MATRIX_SOLVER           '{choices[isolv]}'\n")
        icur = 0 if self.data.info.attrs['current_interaction'] == const.TEXT_NONE else 1
        choices = {0: 'OFF', 1: 'MULTIPLE', 2: 'SINGLE'}
        ss.write(f"WV_CURRENT_TYPE            '{choices[icur]}'\n")
        # Don't use the existence of a linked nesting coverage in the logic for writing the WV_BOUNDARY_NESTING card.
        # nest_cov = self.query.item_with_uuid(self.data.info.attrs['nesting_uuid'])
        choices = {0: 'OFF', 1: 'AVERAGE_SPECTRA', 2: 'INVERSE_DISTANCE'}

        # If no linked nesting coverage set to "OFF", otherwise use the model control value.
        # ibnd = 0 if not nest_cov else 2 if self.data.info.attrs['interpolation'] == 'IDW' else 1
        ibnd = 2 if self.data.info.attrs['interpolation'] == 'IDW' else 1

        ss.write(f"WV_BOUNDARY_NESTING        '{choices[ibnd]}'\n")
        ibf = 0
        ibf = 1 if self.data.info.attrs['friction'] == const.TEXT_DARCY_CONSTANT else ibf
        ibf = 2 if self.data.info.attrs['friction'] == const.TEXT_DARCY_DATASET else ibf
        ibf = 3 if self.data.info.attrs['friction'] == const.TEXT_MANNINGS_CONSTANT else ibf
        ibf = 4 if self.data.info.attrs['friction'] == const.TEXT_MANNINGS_DATASET else ibf
        choices = {
            0: 'OFF',
            1: 'CONSTANT_DARCY_WEISBACH',
            2: 'VARIABLE_DARCY_WEISBACH',
            3: 'CONSTANT_MANNINGS',
            4: 'VARIABLE_MANNINGS'
        }
        ss.write(f"WV_BOTTOM_FRICTION         '{choices[ibf]}'\n")
        iark = 0 if self.data.info.attrs['forward_reflection'] == const.TEXT_NONE else 1
        choices = {0: 'OFF', 1: 'CONSTANT', 2: 'VARIABLE'}
        ss.write(f"WV_FWD_REFLECTION          '{choices[iark]}'\n")
        iarkr = 0 if self.data.info.attrs['backward_reflection'] == const.TEXT_NONE else 1
        choices = {0: 'OFF', 1: 'CONSTANT', 2: 'VARIABLE'}
        ss.write(f"WV_BWD_REFLECTION          '{choices[iarkr]}'\n")
        iwet = 1
        if self.data.info.attrs['wet_dry'] == 1:
            iwet = 0
            if self.data.info.attrs['sea_swell'] == 1:
                iwet = -1
        choices = {0: 'ON', 1: 'OFF', -1: 'ON_WITH_SEA-SWELL_FILES'}
        ss.write(f"WV_WETTING_DRYING          '{choices[iwet]}'\n")
        akap = self.data.info.attrs['diffraction_intensity']
        ss.write(f"WV_DIFFRACTION_INTENSITY   {akap:.3f}\n")
        iproc = self.data.info.attrs['num_threads']
        ss.write(f"WV_NUM_THREADS             {iproc}\n")
        ss.write('\n')

        # Next section:  nesting cells, observation cells
        self._logger.info('Writing nesting points.')
        nest_cov = self.query.item_with_uuid(self.data.info.attrs['nesting_uuid'])
        nest_points = None
        try:
            nest_points = nest_cov.get_points(FilterLocation.PT_LOC_DISJOINT) if nest_cov else []
            inest = len(nest_points)
        except Exception:
            inest = 0
        ss.write(f"WV_NESTING_CELLS           {inest}")
        if inest > 0 and self.grid:
            snapped_points = self.snapper.get_snapped_points(nest_points)
            for _, id in enumerate(snapped_points['id']):
                i, j = self.grid.get_cell_ij_from_index(id)
                ss.write(f' {i} {j}')
        ss.write('\n')

        self._logger.info('Writing observation points.')
        obs_cov = self.query.item_with_uuid(self.data.info.attrs['observation_uuid'])
        obs_points = None
        try:
            obs_points = obs_cov.get_points(FilterLocation.PT_LOC_DISJOINT) if obs_cov else []
            kout = len(obs_points)
        except Exception:
            kout = 0
        ss.write(f"WV_OBSERVATION_CELLS       {kout}")
        if kout > 0 and self.grid:
            snapped_points = self.snapper.get_snapped_points(obs_points)
            for _, idx in enumerate(snapped_points['id']):
                i, j = self.grid.get_cell_ij_from_index(idx)
                ss.write(f' {i} {j}')
        ss.write('\n')
        if self.data.info.attrs['limit_observation_output']:
            ss.write("WV_LIMIT_OBSERVATION_OUTPUT 'ON'\n")

        # Next section:  bottom friction, forward reflection, and backward reflection (only if constant)
        bf = None
        if self.data.info.attrs['friction'] == const.TEXT_DARCY_CONSTANT:
            bf = self.data.info.attrs['darcy']
        elif self.data.info.attrs['friction'] == const.TEXT_MANNINGS_CONSTANT:
            bf = self.data.info.attrs['manning']
        ark = self.data.info.attrs['forward_reflection_const']
        arkr = self.data.info.attrs['backward_reflection_const']
        if bf:
            ss.write(f"WV_BOTTOM_FRICTION_COEFF   {bf:.4f}\n")
        if iark == 1:
            ss.write(f"WV_FWD_REFLECTION_COEFF    {ark:.4f}\n")
        if iarkr == 1:
            ss.write(f"WV_BWD_REFLECTION_COEFF    {arkr:.4f}\n")
        ss.write('\n')

        # Next section:  Both iwvbk and gamma if using B&J 1978
        iwvbk = 0
        iwvbk = 1 if self.data.info.attrs['wave_breaking_formula'] == 'Extended Miche' else iwvbk
        iwvbk = 2 if self.data.info.attrs['wave_breaking_formula'] == 'Battjes and Janssen 1978' else iwvbk
        iwvbk = 3 if self.data.info.attrs['wave_breaking_formula'] == 'Chawla and Kirby' else iwvbk
        iwvbk = 4 if self.data.info.attrs['wave_breaking_formula'] == 'Battjes and Janssen 2007' else iwvbk
        iwvbk = 5 if self.data.info.attrs['wave_breaking_formula'] == 'Miche original' else iwvbk
        iwvbk = 6 if self.data.info.attrs['wave_breaking_formula'] == 'Lifting breaking' else iwvbk
        choices = {
            0: 'EXT_GODA',
            1: 'EXT_MICHE',
            2: 'BATTJES_JANSSEN_78',
            3: 'CHAWLA_KIRBY',
            4: 'BATTJES_JANSSEN_07',
            5: 'MICHE_ORIGINAL',
            6: 'LIFTING_BREAKING'
        }
        ss.write(f"WV_BREAKING_FORMULA        '{choices[iwvbk]}'\n")
        if iwvbk == 2:
            bj78 = self.data.info.attrs['gamma_bj78']
            ss.write(f"WV_SET_GAMMA_BJ78          {bj78:.4f}\n")
        ss.write('\n')

        # Next section:  igrav, imud, nonln, iroll, irunup, iwind
        igrav = self.data.info.attrs['infragravity']
        choices = {0: 'OFF', 1: 'ON'}
        ss.write(f"WV_ENABLE_INFRAGRAVITY     '{choices[igrav]}'\n")
        imud = 1 if self.data.info.attrs['muddy_bed'] == const.TEXT_NONE else 0
        choices = {0: 'ON', 1: 'OFF'}
        ss.write(f"WV_ENABLE_MUDDY_BED        '{choices[imud]}'\n")
        nonln = self.data.info.attrs['nonlinear_wave']
        choices = {0: 'OFF', 1: 'ON'}
        ss.write(f"WV_ENABLE_NONLINEAR_WAVES  '{choices[nonln]}'\n")
        choices = {0: 'OFF', 1: '25_PERCENT', 2: '50_PERCENT', 3: '75_PERCENT', 4: '100_PERCENT'}
        iroll = self.data.info.attrs['roller']
        ss.write(f"WV_ROLLER_EFFECT           '{choices[iroll]}'\n")
        irunup = self.data.info.attrs['runup']
        choices = {0: 'OFF', 1: 'REL_TO_ABS_DATUM', 2: 'REL_TO_UPDATED_MWL'}
        ss.write(f"WV_ENABLE_RUNUP            '{choices[irunup]}'\n")
        iwnd = 1 if self.data.info.attrs['source_terms'] == 'Propagation only' else 0
        choices = {0: 'ON', 1: 'OFF'}
        iwnd = choices[iwnd]
        if iwnd == 'ON' and self.data.info.attrs['limit_wave_inflation']:
            # If wind is on and the limit wave inflation option is enabled, change the card.
            iwnd = 'ON-LIMIT_WAVE_INFLATION'
        ss.write(f"WV_ENABLE_WIND             '{iwnd}'\n")

        # Next section:  ibreak, irs, ixmdf
        ibreak = 0 if self.data.info.attrs['breaking_type'] == const.TEXT_NONE else 1 \
            if self.data.info.attrs['breaking_type'] == const.TEXT_DISSIPATION_INDICES else 2
        choices = {0: 'OFF', 1: 'BREAKING_INDICES', 2: 'DISSIPATION_VALUES'}
        ss.write(f"WV_BREAKING_OUTPUT         '{choices[ibreak]}'\n")
        irs = self.data.info.attrs['rad_stress']
        choices = {0: 'OFF', 1: 'RAD_STRESS_FILE', 2: 'STRESS_SETUP_FILES'}
        ss.write(f"WV_RAD_STRESS_OUTPUT       '{choices[irs]}'\n")
        ixmdf = 'OFF'
        ss.write(f"WV_OUTPUT_XMDF             '{ixmdf}'\n")

        # Added in 13.4 for the 'Wind Only' option instead of 'None' for Source term
        ival = 1 if self.data.info.attrs['boundary_source'] == 'Wind Only' else 0
        choices = {0: 'OFF', 1: 'ON'}
        ss.write(f"WV_WIND_ONLY               '{choices[ival]}'\n")
        ss.write('\n')

        # Final section:  closing end parameters
        ss.write('END_PARAMETERS\n')

        # flush to file
        self._logger.info('Flushing to disk.')
        out = open(filename, "w")
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()
        return True

    def write(self):
        """Writes all data to file.

        Returns:
            (:obj:`bool`): False on error.
        """
        xms_data = XmsData(self.query)
        # Get the simulation tree item
        sim_uuid = self.query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, sim_uuid)
        # get simulation name - will need it to build filenames
        self.simulation_name = sim_item.name
        sim_filename = self.simulation_name + '.sim'
        self._logger.info(f'Writing simulation file to {sim_filename}')
        # Get the simulation's hidden component data.
        sim_comp = self.query.item_with_uuid(sim_uuid, model_name='CMS-Wave', unique_name='Sim_Component')
        self.data = SimulationData(sim_comp.main_file)
        # Get the grid
        self._logger.info('Retrieving domain grid from SMS...')
        do_grid = xms_data.do_ugrid
        if not do_grid:
            self._logger.error('No grid found. A Cartesian grid must be linked to the simulation. Aborting export.')
            return False
        self.cogrid_file = do_grid.cogrid_file
        self.grid = read_grid_from_file(do_grid.cogrid_file)
        self.grid_angle = self.grid.angle  # store grid angle so it can be used to translate angles
        self.proj = do_grid.projection

        # get case times - will need them in various places
        self.case_times = self.data.case_times['Time']

        # file header:  Origin & grid orientation
        self.ss.write(f'CMS-WAVE {self.grid.origin[0]:>14.4f} {self.grid.origin[1]:>14.4f} {self.grid_angle:>14.4f}\n')

        # Write out all of the lines below, even if they aren't used.  CMS-WAVE will ignore if unused.
        self._logger.info('Writing .sim file...')

        # DEP: Input file containing depth values (Required)
        filename = f'{self.simulation_name}.dep'
        self._logger.info(f'Writing .dep file to {filename}')
        self.ss.write(f'DEP       {filename}\n')

        # OPTS: Input file containing model parameters (Required)
        opts_file_version = 2
        filename = f'{self.simulation_name}.std'
        self._logger.info(f'Writing .std file to {filename}')
        self.write_opts_file(filename, version=opts_file_version)
        self.ss.write(f'OPTS      {filename}\n')

        # CURR:  Optional input file contianing current values (vx and vy) for each cell
        self.ss.write(f'CURR      {self.simulation_name}.cur\n')

        # SPEC:  Input file containing energy density spectra (written from spectral_coverage_writer.py)
        self.ss.write(f'SPEC      {self.simulation_name}-side1.eng\n')

        # SPEC2:  Input file containing energy density spectra on side 3 (written from spectral_coverage_writer.py)
        self.ss.write(f'SPEC2     {self.simulation_name}-side3.eng\n')

        # WAVE:  Output file for spatial wave conditions (height, direction) for each cell (Required)
        self.ss.write(f'WAVE      {self.simulation_name}.wav\n')

        # OBSE:  Output file to save full spectra at specified monitoring locations
        self.ss.write(f'OBSE      {self.simulation_name}.obs\n')

        # NEST:  Output file to save full spectra at specified nesting cells
        self.ss.write(f'NEST      {self.simulation_name}.nst\n')

        # BREAK:  Output file to save wave breaking indicies or energy disipations at each cell
        self.ss.write(f'BREAK     {self.simulation_name}.brk\n')

        # SPGEN:  Spectral parameter file
        self.ss.write(f'SPGEN     {self.simulation_name}.txt\n')

        # RADS:  Output file to save wave radiation stress gradients at each cell
        self.ss.write(f'RADS      {self.simulation_name}.rad\n')

        # STRUCT:  Input file containing structure flags for each cell
        self.ss.write(f'STRUCT    {self.simulation_name}.struct\n')

        # FRIC:  Input file containing Manning's N value for each cell (DEP format)
        self.ss.write(f'FRIC      {self.simulation_name}.fric\n')

        # FREF:  Input file containing forward reflection value for each cell (DEP format)
        self.ss.write(f'FREF      {self.simulation_name}.fref\n')

        # BREF:  Input file containing backward reflection value for each cell (DEP format)
        self.ss.write(f'BREF      {self.simulation_name}.bref\n')

        # ETA:  Optional input file containing water levels for each cell (DEP format)
        self.ss.write(f'ETA       {self.simulation_name}.eta\n')

        # MUD:  Input file containing mud (absorption) coefficient value for each cell (DEP format)
        self.ss.write(f'MUD       {self.simulation_name}.mud\n')

        # WIND:  Input file containing wind components value for each cell (CUR format)
        self.ss.write(f'WIND      {self.simulation_name}.wind\n')

        # SELHTS:  Output file where CMS-Wave will write bulk wave parameters for each monitoring cell
        self.ss.write(f'SELHTS    {self.simulation_name}.out\n')

        # flush to file
        out = open(sim_filename, 'w')
        self.ss.seek(0)
        shutil.copyfileobj(self.ss, out)
        out.close()

        return True
