"""Creates an Materials and Sediment Materials coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.data_objects.parameters import Component

# 4. Local modules
from xms.cmswave.components import structures_component as stc
from xms.cmswave.components.structures_component import StructuresComponent


class StructuresComponentBuilder:
    """Builds hidden component for a CMS-Wave structures coverage."""
    def __init__(self, comp_dir, struct_data, poly_assignments, new_comp_uuid=None, display_atts=None):
        """Construct the builder.

        Args:
            comp_dir (:obj:`str`): Path to the directory to put component data. Should be XMS temp component directory
                if not testing.
            struct_data (StructuresData): Structure ID to structure name
            poly_assignments (:obj:`dict`): Structure ID to list of polygon IDs with that structure assignment
            new_comp_uuid (:obj:`str`): UUID to use for the new component (useful for testing). Randomized if not
                provided.
            display_atts (:obj:`dict`): Structure ID to display option string ('r g b texture'). If not provided,
                will be randomly generated.
        """
        self._comp_dir = comp_dir
        self._comp_uuid = new_comp_uuid if new_comp_uuid else str(uuid.uuid4())
        self._struct_data = struct_data
        self._polys = poly_assignments

    def _write_component_id_files(self, comp_dir):
        """Write XMS feature id and component id files so we can initialize map component ids in SMS.

        Args:
            comp_dir (:obj:`str`): Path to the component data directory (folder containing the component main file)
        """
        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        att_ids = []
        comp_ids = []
        for struct_id, poly_ids in self._polys.items():
            if struct_id < 0:  # pragma no cover
                continue  # Don't need to write out unassigned bathymetry modification polys since it's the default cat.
            non_default_poly_ids = [poly_id for poly_id in poly_ids if poly_id > 0]
            att_ids.extend(non_default_poly_ids)
            comp_ids.extend([struct_id for _ in range(len(non_default_poly_ids))])
        id_file = os.path.join(comp_dir, stc.STRUCT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(comp_dir, stc.STRUCT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)

    def build_structures_component(self, cov_uuid):
        """Create CMS-Wave structure coverage's hidden component and data_object.

        Args:
            cov_uuid (:obj:`str`): UUID of the Structure coverage to build component for

        Returns:
            (:obj:`data_objects.parameters.Component`): data_object for the new StructuresComponent to send back to SMS.
        """
        # Create a new UUID and folder for the component data
        struct_comp_dir = os.path.join(self._comp_dir, self._comp_uuid)
        os.makedirs(struct_comp_dir, exist_ok=True)
        struct_main_file = os.path.join(struct_comp_dir, 'struct_comp.nc')

        # Create the data_object Component to send back to SMS
        do_comp = Component(
            comp_uuid=self._comp_uuid,
            unique_name='StructuresComponent',
            model_name='CMS-Wave',
            main_file=struct_main_file
        )

        struct_comp = StructuresComponent(struct_main_file)
        # struct_data, sorted_struct_ids = self._build_structures_data()
        struct_comp.data.structures = self._struct_data.structures
        struct_comp.data.info.attrs['cov_uuid'] = cov_uuid
        struct_comp.cov_uuid = cov_uuid

        struct_comp.data.commit()
        struct_ids = self._struct_data.structures.id.data.tolist()
        struct_comp.update_display_id_files([], struct_ids)
        struct_comp.update_display_options_file()

        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        self._write_component_id_files(struct_comp_dir)

        return do_comp
