"""Base class for XMS DMI model runner components."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules
from xms.guipy.data.category_display_option_list import CategoryDisplayOption, CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules


class RunBase:
    """Base class for XMS DMI model runner components."""
    def __init__(self):
        """Construct the base class.

        self.plots is the plot parameters name to a display option list of individual plot data for the run
        and a context: {str: (xms.guipy.category_display_option_list.CategoryDisplayOptionList, xms.api.dmi.Context)}
        """
        self._plots = {}
        self.color_idx = 0
        # For randomizing plot colors. Based on matplotlib's tab20 color map:
        # https://matplotlib.org/stable/users/explain/colors/colormaps.html#colormaps
        # yapf: disable
        rgbs = [
            # darker colors
            (31, 119, 180), (255, 127, 14), (44, 160, 44), (214, 39, 40), (148, 103, 189), (140, 86, 75),
            (227, 119, 194), (127, 127, 127), (188, 189, 34), (23, 190, 207),
            # lighter colors
            (174, 199, 232), (255, 187, 120), (152, 223, 138), (255, 152, 150), (197, 176, 213), (196, 156, 148),
            (247, 182, 210), (199, 199, 199), (219, 219, 141), (158, 218, 229)
        ]
        # yapf: enable
        self.next_color = [QColor(rgb[0], rgb[1], rgb[2]) for rgb in rgbs]

    @property
    def plots(self):
        """Property to retrieve the plot options."""
        return self._plots

    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.


        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ExecutableCommand`):
                The executable objects to run and the action requests that go with it.
        """
        return []

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): The solution load ActionRequests for the simulation
        """
        return []

    def add_plot_data(self, plot_parameter, plot_data_name, on=True, display_options=None, context=None):
        """Add a plot data with defaulted display options and Context.

        Args:
            plot_parameter (str): XML definition plot parameter name
            plot_data_name (str): XML definition plot data name
            on (bool): True if the plot is being displayed
            display_options (list): The LineOptions for the plot data
            context (xms.api.dmi.Context): Query Context for the plot data
        """
        if plot_parameter not in self.plots:
            self._plots[plot_parameter] = (CategoryDisplayOptionList(), [])
            self._plots[plot_parameter][0].target_type = TargetType.arc
        option = CategoryDisplayOption()
        option.description = plot_data_name
        option.on = on
        if display_options:
            option.options = display_options
        else:
            option.options = LineOptions()
            option.options.color = self.next_color[self.color_idx]
            self.color_idx += 1
            if self.color_idx == len(self.next_color):
                self.color_idx = 0
        self.plots[plot_parameter][0].categories.append(option)
        if context:
            stick_context = context.get_context_from_place_marks()
            # Unwrap the pure Python Context
            self._plots[plot_parameter][1].append(stick_context._instance)
        else:
            self._plots[plot_parameter][1].append(None)
