"""Module for the ModelRunManager class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ModelRunManager']

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ExecutableCommand, Query
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.data_objects.parameters import Simulation
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.components.bases.run_base import RunBase as OldRunBase
from xms.components.dmi.xms_data import XmsData


class ModelRunManager(OldRunBase):
    """
    Base class for a simple model run manager.

    The model run manager's main job is to tell XMS where the model runner and model tracker scripts are at. It also has
    easy access to the simulation's model control via `self.data` and can use that to send extra information to the
    runner and progress scripts that they might find useful (e.g. the model runner might like to know where the user
    said the model executable is at, and the progress tracker might like to know how many time steps there are).

    The base class suggests connections to plots and importing the solution, but those aren't implemented yet.
    """
    def __init__(self):
        """Initialize the manager."""
        super().__init__()
        self.runner: str = ''
        self.runner_args: list[str] = []
        self.tracker: str = ''
        self.tracker_args: list[str] = []
        self.data: Optional[XmsData] = None
        self.reported_error: bool = False

    def get_executables(self, do_sim: Simulation, query: Query, file_location: str) -> list[ExecutableCommand]:
        """
        Called by XMS when it wants to know the executables used to run the model.

        Derived classes should probably leave this alone and override `self.find_model_run_script` instead.

        Args:
            do_sim: The data_objects simulation that is being run. This implementation ignores it in favor of wrapping
                `query` in an `XmsData`.
            query: Low-level inter-process communication object.
            file_location: Where the model native files for the simulation should be at. This will be the directory
                containing the files, but may not actually exist if the user tries to run before exporting or deleted
                the files between exporting and running.
        """
        self.data = XmsData(query)

        try:
            self.find_model_run_script(file_location)
        except Exception as exc:
            self.debug_log(exc)
            return []

        if not self.runner or not self.tracker:
            if not self.reported_error:
                self.debug_log('ModelRunManager failed to find model runner or progress tracker.')
            return []

        display_name = f'Running {self.data.model_name}'
        tracker = relative_to_xml(self.tracker)

        cmd = ExecutableCommand(
            executable=self.runner,
            model=self.data.model_name,
            display_name=display_name,
            run_weight=100,
            executable_is_script=True,
            progress_script=tracker,
        )

        for arg in self.runner_args:
            cmd.add_commandline_arg(arg)

        for arg in self.tracker_args:
            cmd.add_progress_arg(arg)

        return [cmd]

    def find_model_run_script(self, file_location: str):
        """
        Find the model run and progress script information.

        Args:
            file_location: Where the model native files for the simulation should be at.
        """
        # file_location is where the files *should* be at, but they might not *actually* be there. This typically
        # happens when the model is run before exporting.

        # This method has access to self.data, which can be used to get the simulation data if needed.

        # This method should generally:
        # - Use self.find_model_executable to make sure the executable exists, and self.report_error if it doesn't.
        # - Check that the model's control file exists, and use self.report_error to tell the user if not.
        # - Import the entry point modules for the runner and progress scripts
        # - - Import should be at the top, just mentioned here since it's where people are likely to find it
        # - Put those modules' file paths into self.runner and self.tracker
        # - Add any other arguments to self.runner_args and self.tracker_args that the scripts need
        # - - E.g. some models only write their stdout to a file under file_location, which means a progress script that
        #     wants to look at that output to decide how far along the model is might need file_location. Other models
        #     have multiple executables, and the run script might need additional data to decide which one to pick. That
        #     information may be more easily accessed here where you can get at your simulation data than in the runner
        #     script where you'd have to parse a control file to find it (assuming it's even there in the first place).
        #
        #     While not strictly needed, adding the executable's path to self.runner_args may be worthwhile to do here
        #     since it saves reimplementing self.find_model_executable in your runner script.
        raise AssertionError('ModelRunManager script finder did not override find_script.')

    def report_error(self, problem: str):
        """
        Inform the user that some error occurred.

        Args:
            problem: Message for XMS to display to the user.
        """
        XmEnv.report_error(problem)
        self.reported_error = True

    def debug_log(self, problem: str | Exception):
        """
        Write a message to python_debug.log and inform the user that an unexpected error occurred.

        This could maybe be useful for writing debug messages, but it's mainly meant as a last-chance exception handler.

        Args:
            problem: A message or exception indicating what went wrong. If an exception, a stack trace will be logged.
        """
        self.report_error('An unexpected error occurred. Please contact Aquaveo Tech Support.')
        XmEnv.report_error(problem, XmEnv.xms_environ_debug_file())

    def find_model_executable(self, name: str) -> Optional[Path]:
        """
        Find a model's executable.

        Assumes self.data is initialized.

        Args:
            name: Name of the executable to find. This should match the `text` attribute on an `<executable>` tag in
                the model's XML.

        Returns:
            Absolute path to the executable if found, or None if not. The path might not be found if the user neglected
            to set it and there was no default.
        """
        settings = SettingsManager(python_path=False)
        package = 'File_Location_Preferences'
        model = self.data.model_name
        key = f'{model} - {name}'
        model_path = str(settings.get_setting(package, key, ''))
        if model_path:
            return Path(model_path)
        else:
            return None


def relative_to_xml(path: str) -> str:
    """Make a path to a script relative to the path to its .xml file."""
    path = Path(path)

    base = path
    while base.parent and base.parent.name and base.parent.name != 'xms':
        base = base.parent

    relative = path.relative_to(base)
    return str(relative)
