"""Utilities for tests."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['module_directory', 'running_multiple_tests', 'empty_main_file', 'empty_temp_directory', 'test_directory']

# 1. Standard Python modules
import contextlib
import os
from pathlib import Path
import shutil
from unittest.mock import patch

# 2. Third party modules
import pytest

# 3. Aquaveo modules
from xms.constraint import read_grid_from_file
from xms.coverage.file_io.map_file_reader import read as read_coverage_from_file
from xms.data_objects.parameters import Coverage
from xms.guipy.testing.testing_tools import read_tree_from_file
from xms.testing.fixtures import empty_temp_directory as testing_empty_temp_directory  # noqa: F401  False positive

# 4. Local modules


@pytest.fixture
def module_directory(request) -> Path:
    """
    Get the module-level directory for the test files for this test.

    This has been designed to also work in other packages.
    """
    here: Path = request.path
    while here.name != 'tests' and here.parent != here:
        here = here.parent

    sub_path = request.path.relative_to(here).with_suffix('')
    parts = [part.replace('test_', '') for part in sub_path.parts]
    directory = here.joinpath('files', *parts)
    directory.mkdir(exist_ok=True, parents=True)
    return directory


@pytest.fixture
def test_directory(module_directory, request) -> Path:
    """
    Get a directory for this test.

    If your test function can be imported like `from tests.test_package.test_module import test_function`
    then this fixture will give you the path `./files/package/module/function` relative to this file.

    This has been designed to also work in other packages.
    """
    directory = module_directory / request.node.originalname
    directory.mkdir(exist_ok=True)
    return directory


@pytest.fixture
def empty_temp_directory(test_directory):
    """
    Get a temp directory for the test.

    The directory will be located inside the test's directory, and be named `temp`.
    """
    temp_directory = test_directory / 'temp'
    with contextlib.suppress(FileNotFoundError):
        shutil.rmtree(temp_directory)
    os.makedirs(temp_directory)
    return temp_directory


@pytest.fixture
def empty_work_directory(empty_temp_directory):
    """Same as empty_temp_directory, but changes the current directory to it and restores it afterward."""
    original_directory = os.getcwd()
    os.chdir(empty_temp_directory)
    yield empty_temp_directory
    os.chdir(original_directory)


@pytest.fixture
def initialized_temp_directory(test_directory):
    """
    A temp directory whose contents are initialized from another directory.

    The temp directory will be in the test's directory, and named `temp`. It will be cleaned out and reinitialized with
    the contents of the directory named `init`, which is also in the test's directory.
    """
    target = test_directory / 'temp'
    if target.is_file():
        target.unlink()
    elif target.is_dir():
        shutil.rmtree(target)

    shutil.copytree(test_directory / 'init', target)
    return target


@pytest.fixture
def initialized_work_directory(initialized_temp_directory):
    """Same as initialized_temp_directory, except the current directory is also changed to it."""
    original_directory = os.getcwd()
    os.chdir(initialized_temp_directory)
    yield initialized_temp_directory
    os.chdir(original_directory)


@pytest.fixture
def patched_xms_temp_directory(empty_temp_directory):
    """
    Patch the XMS temp directory to be inside the test directory.

    The value provided to the test is the same as from the empty_temp_directory fixture.
    """
    with patch('xms.components.component_builders.main_file_maker.XmEnv.xms_environ_temp_directory') as temp_directory:
        temp_directory.return_value = empty_temp_directory
        yield empty_temp_directory


@pytest.fixture
def patched_new_component_uuid():
    """
    Patch the UUID that new components are assigned when they aren't provided a main-file.

    The value provided to the test is the UUID that the component will be assigned.
    """
    uuid = '12345678-90ab-cdef-1234-567890abcdef'
    with patch('xms.components.component_builders.main_file_maker.uuid.uuid4') as uuid_generator:
        uuid_generator.return_value = uuid
        yield uuid


@pytest.fixture
def empty_main_file(test_directory: Path) -> str:
    """Fixture to get an empty mainfile."""
    os.makedirs(test_directory, exist_ok=True)
    path = test_directory / 'main_file.nc'
    with contextlib.suppress(FileNotFoundError):
        os.remove(path)
    return str(path)


@pytest.fixture
def patched_component_uuids(testing_empty_temp_directory):  # noqa: F811  False positive
    last_uuid = 0

    def uuid_generator():
        nonlocal last_uuid
        hex_str = hex(last_uuid)
        no_prefix = hex_str.removeprefix('0x')
        uuid_str = f'{no_prefix:>032}'
        last_uuid += 1
        return uuid_str

    with patch('xms.components.component_builders.main_file_maker.XmEnv.xms_environ_temp_directory') as temp_directory:
        with patch('xms.components.component_builders.main_file_maker.uuid.uuid4') as uuid4:
            temp_directory.return_value = testing_empty_temp_directory
            uuid4.side_effect = uuid_generator
            yield testing_empty_temp_directory


@pytest.fixture
def running_multiple_tests(request) -> bool:
    """
    Check whether there are multiple tests in the current batch of tests.

    This is mainly used as a heuristic for dialog tests to decide whether they should leave the dialog open after the
    test finishes. If multiple tests are running, it's likely either the CI or a mass test to see what fails, and
    leaving dialogs open would be irritating. If only a single test is running though, it's probably being run manually
    because it failed and needs to be debugged, in which case leaving the dialog open is more likely to be appreciated.

    This used to just check whether we were running Tox, but my workflow involves running all the tests without Tox, so
    there were a lot of nuisance dialogs left open. This seems like a better heuristic.
    """
    # request.session.items is a list of all the tests that pytest decided should be run.
    return len(request.session.items) > 1


@pytest.fixture
def project_tree(test_directory):
    """Fixture to get a project tree for testing."""
    path = test_directory / 'tree.txt'
    node = read_tree_from_file(path)
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert node is not None
    return node


@pytest.fixture
def module_project_tree(module_directory):
    """Fixture to get a module-level project tree for testing."""
    path = module_directory / 'tree.txt'
    node = read_tree_from_file(path)
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert node is not None
    return node


@pytest.fixture
def test_grid(test_directory):
    """Fixture to get a UGrid from the file named grid.xmc inside the test's directory."""
    read_grid = read_grid_from_file(str(test_directory / 'grid.xmc'))
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert read_grid is not None
    return read_grid


@pytest.fixture
def module_grid(module_directory):
    """Fixture to get a UGrid from the file named grid.xmc inside the test module's directory."""
    read_grid = read_grid_from_file(str(module_directory / 'grid.xmc'))
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert read_grid is not None
    return read_grid


@pytest.fixture
def test_coverage(test_directory) -> Coverage:
    """A coverage loaded from a file named 'coverage.map' inside the test's directory."""
    [coverage], _lines, _series = read_coverage_from_file(str(test_directory / 'coverage.map'))
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert coverage is not None
    return coverage


@pytest.fixture
def module_coverage(module_directory) -> Coverage:
    """A coverage loaded from a file named 'coverage.map' inside the test module's directory."""
    [coverage], _lines, _series = read_coverage_from_file(str(module_directory / 'coverage.map'))
    # The above just returns None whenever it fails for any reason. Then it takes a while to debug the test and find out
    # it failed due to a typo in the file name. This assert moves the exception closer to the actual cause.
    assert coverage is not None
    return coverage
