from .. import _xmsconstraint


class DisuCalculator:
    """Calculator for MODFLOW DISU package data from a UGrid with vertically
    prismatic 3D cells."""
    def __init__(self, ugrid):
        self._instance = _xmsconstraint.modflow.MfDisuCalculator(
            ugrid._instance)
        self.ugrid = ugrid

    def calculate(self, tops, bottoms, centroids, grid_angle):
        """
        Calculate the values so they can be retrieved.
        Args:
            tops: The cell top elevations.
            bottoms: The cell bottom elevations.
            centroids: The cell centroids.
        """
        cell_count = self.ugrid.cell_count
        if len(tops) != cell_count:
            raise ValueError('Length of tops must equal UGrid cell count.')
        if len(bottoms) != cell_count:
            raise ValueError('Length of bottoms must equal UGrid cell count.')
        if len(centroids) != cell_count:
            raise ValueError('Length of centroids must equal UGrid cell '
                             'count.')
        self._instance.Calculate(tops, bottoms, centroids, grid_angle)

    def get_grid_type(self):
        """
        Get the grid type.
        Returns (int):
            The grid type (-1 stacked, 1 sub-layered, 0 no sub-layers)
        """
        return self._instance.GetGridType()

    def get_adjacent_counts(self):
        """
        Get the count of adjacent cells array (IAC).
        Returns:
            An IAC array containing a list of adjacent cell counts.
        """
        return self._instance.GetAdjacentCounts()

    def get_adjacent_cells(self):
        """
        Get the adjacent cells array (JA).
        Returns:
            The adjacent cells array. Array with negative index of cell
            followed by adjacent cell IDs for each cell.
        """
        return self._instance.GetAdjacentCells()

    def get_connection_direction(self):
        """
        Get each connection direction (IVC).
        Returns:
            A JA length array with the connection direction
            (1 vertical, 0 horizontal).
        """
        return self._instance.GetConnectionDirection()

    def get_plan_view_area(self):
        """
        Get the plan view area of each cell (AREA).
        Returns:
            Cell count length array of cell's plan view area.
        """
        return self._instance.GetPlanViewArea()

    def get_interface_length(self):
        """
        Get the length of each interface connection (CL12).
        Returns:
            A JA length array with the length of each cell to cell
            connection. Warning: not necessarily the exact distance.
            For horizontal connections the length is taken as the
            sum of the distances of the two centroids from the face
            plane connecting the two cells.
        """
        return self._instance.GetInterfaceLength()

    def get_interface_area(self):
        """
        Get the area of the face connecting the two cells (FAHL).
        Returns:
            A JA length array giving the area of the face connecting
            the two cells.
        """
        return self._instance.GetInterfaceArea()

    def get_interface_angle(self):
        """
        Get the interface angle of each connection.
        Returns:
            A JA length array giving the interface angle of each
            connection.
        """
        return self._instance.GetInterfaceAngle()
