from abc import ABC
from enum import IntEnum


class Numbering(IntEnum):
    """Enumeration for index ordering in rectilinear and quad tree grids."""
    kij = 0
    """Numbering goes along J columns first followed by I rows followed by Z
    layers."""
    kji = 1
    """Numbering goes along I rows first followed by J columns followed by Z
    layers."""


class Orientation(IntEnum):
    """Enumeration for describing the direction of rows (I), columns (J), and
    layers (K) of a rectilinear grid relative to the X, Y, and Z axis."""
    x_decrease = -1
    x_increase = 1
    y_decrease = -2
    y_increase = 2
    z_decrease = -3
    z_increase = 3

    @classmethod
    def is_x(cls, orientation):
        """Is orientation in X direction?
        Args:
            orientation: The orientation.
        Returns:
            True if orientation is in positive or negative X direction."""
        is_x = orientation == Orientation.x_increase
        is_x = is_x or orientation == Orientation.x_decrease
        return is_x

    @classmethod
    def is_y(cls, orientation):
        """Is orientation in Y direction?
        Args:
            orientation: The orientation.
        Returns:
            True if orientation is in positive or negative Y direction."""
        is_y = orientation == Orientation.y_increase
        is_y = is_y or orientation == Orientation.y_decrease
        return is_y

    @classmethod
    def is_z(cls, orientation):
        """Is orientation in Z direction?
        Args:
            orientation: The orientation.
        Returns:
            True if orientation is in positive or negative Z direction."""
        is_z = orientation == Orientation.z_increase
        is_z = is_z or orientation == Orientation.z_decrease
        return is_z


class RectilinearGeometry(ABC):
    """Abstract class with getters for rectilinear grid geometry."""

    def __init__(self, instance):
        """Only called by subclasses.
        Args:
            instance: The C++ wrapped instance."""
        self._instance = instance

    @property
    def is_2d_grid(self):
        """Determine if this is a 2D constraint.
        Returns:
            True if a 2D grid?"""
        return self._instance.Is2dGrid()

    @property
    def is_3d_grid(self):
        """Determine if this is a 3D constraint.
        Returns:
            True if a 3D grid?"""
        return self._instance.Is3dGrid()

    @property
    def origin(self):
        """The location of the bottom-left corner of the grid."""
        return self._instance.GetOrigin()

    @property
    def angle(self):
        """The angle of rotation about the bottom-left corner of the grid in
        degrees."""
        return self._instance.GetAngle()

    @property
    def numbering(self):
        """The numbering of the cells. Possible values: Numbering.kij (J
        changes first followed by I and then K) or Numbering.kji (I changes
        first)."""
        return Numbering(self._instance.GetNumbering())

    @property
    def orientation(self):
        """The xyz direction for the I, J and K axes. Tuple of Orientation enum
        values. The K value only needs to be specified for a 3D grid."""
        orientation = self._instance.GetOrientation()
        return tuple(Orientation(o) for o in orientation)

    @property
    def locations_x(self):
        """The X coordinate of the base grid edges. Given as a distance from
        the origin."""
        return self._instance.GetLocationsX()

    @property
    def locations_y(self):
        """The Y coordinate of the base grid edges. Given as a distance from
        the origin."""
        return self._instance.GetLocationsY()

    @property
    def locations_z(self):
        """The Z coordinate of the edges. Given as a distance from the
        origin."""
        return self._instance.GetLocationsZ()

    def get_base_cell_dimensions(self, index):
        """The X and Y lengths of the cell with the given index.
                Args:
                    index: The index of the cell.
                Returns:
                    The lengths of the cell's edges along the X and
                    Y axes. A tuple of X,Y lengths."""
        return self._instance.GetBaseCellDimensions(index)

    def get_cell_ij_from_index(self, index):
        """Gets the Cell IJ values from the index."""
        return self._instance.CellIJFromID(index + 1)

    def get_cell_ijk_from_index(self, index):
        """Gets the Cell IJK values from the index."""
        return self._instance.CellIjkFromId(index + 1)

    def get_cell_index_from_ij(self, i, j):
        """Gets the Cell index values from IJ."""
        return self._instance.CellIdFromIJ(i, j) - 1

    def get_cell_index_from_ijk(self, i, j, k):
        """Gets the Cell index values from IJK."""
        return self._instance.CellIdFromIjk(i, j, k) - 1
