"""Utility functions."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules

Extents = tuple[list[float, float, float], list[float, float, float]]


def read_coverage_name(h5_file):
    """Reads and returns the coverage name from the h5 file.

    Args:
        h5_file: h5 file object.

    Returns:
        (str): The coverage name.
    """
    cov_names = h5_file['Map Data'].keys()
    if not cov_names:
        return ''
    the_cov_name = None
    for c_name in cov_names:
        the_cov_name = c_name
        break
    return the_cov_name


def coverage_extents(coverage: Coverage) -> Extents:
    """Returns the min and max coverage extents.

    Args:
        coverage: The coverage.

    Returns:
        tuple[list[float, float, float], list[float, float, float]]: min, max. min will all be float('inf') and max
        will all be float('-inf') if coverage is None or has no data.
    """
    mn = [float('inf'), float('inf'), float('inf')]
    mx = [float('-inf'), float('-inf'), float('-inf')]
    if not coverage:
        return mn, mx

    points = coverage.get_points(FilterLocation.PT_LOC_ALL)
    for point in points:
        # Min
        if point.x < mn[0]:
            mn[0] = point.x
        if point.y < mn[1]:
            mn[1] = point.y
        if point.z < mn[2]:
            mn[2] = point.z
        # Max
        if point.x > mx[0]:
            mx[0] = point.x
        if point.y > mx[1]:
            mx[1] = point.y
        if point.z > mx[2]:
            mx[2] = point.z
    return mn, mx
