"""Code to store interface with the XMS 1D Hyd. cross section coverage."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DataDumpIOBase
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules


class CrossSection:
    """A cross section."""
    def __init__(self):
        """Initialize the class."""
        self.geometry_d = []
        self.geometry_x = []
        self.geometry_y = []
        self.geometry_z = []

        # line properties for the cross section
        self.line_f_values = []
        self.line_i_values = []
        self.line_froms = []
        self.line_tos = []
        self.line_types = []

        # point properties for the cross section
        self.point_measures = []
        self.point_types = []

        # not a class
        self.name = ''
        self.uuid = ''
        self.db_link = -1
        self.note = ''
        self.reach_name = ''
        self.station = 0.0
        self.topo = ''
        self.type = -1
        self.p_type = -1
        self.user_station = 0.0
        self.index = -1


class LineProperty:
    """Line properties."""
    def __init__(self, category, description, exclusive, f_default, line_id, name):
        """Initialize the class.

        Args:
            category (int): The index of the category.
            description (str): The description of the line property.
            exclusive (int): 1 if exclusive, 0 if not.
            f_default (float): The default f value.
            line_id (int): The id of the line.
            name (str): The name of the line property.
        """
        self.category = category
        self.description = description
        self.exclusive = exclusive
        self.f_default = f_default
        self.line_id = line_id
        self.name = name


class PointProperty:
    """Point properties."""
    def __init__(self, description, exclusive, point_id, name):
        """Initialize the class.

        Args:
            description (str): The description of the point property.
            exclusive (int): 1 if exclusive, 0 if not.
            point_id (int): The id of the point.
            name (str): The name of the point property.
        """
        self.description = description
        self.exclusive = exclusive
        self.point_id = point_id
        self.name = name


class Profile:
    """A single profile."""
    def __init__(self, topo_description, topo_name):
        """Initialize the class.

        Args:
            topo_description (str): The topology description.
            topo_name (str): The topology name.
        """
        self.topo_description = topo_description
        self.topo_name = topo_name


class Material:
    """A single material."""
    def __init__(self, enum_value, material_id, material_name):
        """Initialize the class.

        Args:
            enum_value (:obj:`list` of int): The enum value per line.
            material_id (:obj:`list` of int): The material id per line.
            material_name (:obj:`list` of str): The material name per line.
        """
        self.enum_value = enum_value
        self.material_id = material_id
        self.material_name = material_name


class CrossSectionDB:
    """A cross section database."""
    def __init__(self):
        """Constructor for a cross section database."""
        self.profiles = []
        self.line_properties = []
        self.point_properties = []
        self.materials = []
        self.cross_sections = {}  # Key is UUID of the cross-section
        self.name = ''
        self.uuid = ''


class ArcCrossSection:
    """Arc cross section reference."""
    def __init__(self):
        """Constructor for an arc cross section."""
        self.arc_id = -1
        self.reach_name = ''
        self.intersect_x = 0.0
        self.intersect_y = 0.0
        self.intersect_z = 0.0
        self.cross_section_index = -1
        self.reach_id = -1
        self.computed_station = 0.0
        self.user_station = 0.0
        self.flags = -1
        self.cross_section_uuid = ''


class CrossSectionCoverage(DataDumpIOBase):
    """Class to interface with the XMS 1D Hyd. cross section coverage."""
    def __init__(self, file_name=''):
        """Construct the cross section coverage.

        Args:
            file_name(str): Path to the H5 dump file
        """
        super().__init__()
        super().SetSelf(self)
        self._cov = None
        self._has_read = False
        self._file_name = file_name
        self._databases = []
        self._arc_attributes = {}

    @property
    def m_cov(self):
        """Get the cross section coverage geometry.

        Returns:
            xms.data_objects.parameters.Coverage: The cross section coverage geometry
        """
        if self._has_read is False and self._file_name:
            self.ReadDump(self._file_name)
        return self._cov

    @m_cov.setter
    def m_cov(self, val):
        """Set the cross section coverage geometry.

        Args:
            val (xms.data_objects.parameters.Coverage): The cross section coverage geometry
        """
        self._cov = val

    @property
    def databases(self):
        """Get the cross section coverage databases.

        Returns:
            list: The cross section coverage databases
        """
        if self._has_read is False and self._file_name:
            self.ReadDump(self._file_name)
        return self._databases

    @databases.setter
    def databases(self, val):
        """Set the cross section coverage databases.

        Args:
            val (list): The cross section coverage databases
        """
        self._databases = val

    @property
    def arc_attributes(self):
        """Get the cross section coverage arc attributes.

        Returns:
            dict: The arc id to the cross section coverage arc attributes
        """
        if self._has_read is False and self._file_name:
            self.ReadDump(self._file_name)
        return self._arc_attributes

    @arc_attributes.setter
    def arc_attributes(self, val):
        """Set the cross section coverage arc attributes.

        Args:
            val (dict): The arc id to the cross section coverage arc attributes
        """
        self._arc_attributes = val

    def ReadDump(self, file_name):  # noqa: N802
        """Read cross section coverage geometry and attributes from an H5 file written by XMS.

        Args:
            file_name (str): Path to the file to read
        """
        self._has_read = True
        f = h5py.File(file_name, 'r')
        cov_names = f['Map Data'].keys()
        if not cov_names:
            f.close()
            return
        the_cov_name = None
        for c_name in cov_names:
            the_cov_name = c_name
            break

        attributes_group = f['Map Data/' + the_cov_name + '/Attributes']
        database_group = f['Map Data/' + the_cov_name + '/Database']
        for db in database_group:
            self._read_db(database_group[db])
        self._read_attributes(attributes_group)
        f.close()

        # read geometry
        self._cov = Coverage(file_name, '/Map Data/' + the_cov_name)
        self._cov.get_points(FilterLocation.LOC_NONE)  # force geometry to load from H5

    def _read_db(self, db_group):
        """Read a cross-section database.

        Args:
            db_group (h5py.Group): The cross section database to read.
        """
        db = CrossSectionDB()
        db.name = db_group['DB_Name'][0]
        db.uuid = db_group['GUID'][0].decode('UTF-8')
        num_cs = int(db_group['XSECS/XSECS'][0])
        geom_group = db_group['CrossSectionGeometry']
        line_group = db_group['LineProperties']
        point_group = db_group['PointProperties']
        xsec_group = db_group['XSECS']
        user_station_dset = db_group['USERSTATION']
        for cs_idx in range(num_cs):
            cs = CrossSection()
            self._read_cross_section_geometry(cs, cs_idx, geom_group)
            self._read_cross_section_line(cs, cs_idx, line_group)
            self._read_cross_section_point(cs, cs_idx, point_group)
            self._read_xsec(cs, cs_idx, xsec_group)
            cs.user_station = float(user_station_dset[cs_idx])
            cs.index = cs_idx
            db.cross_sections[cs.uuid] = cs

        num_lines = line_group['Line Prop. Number'][0]
        self._read_general_line_properties(db, line_group)
        self._read_general_point_properties(db, point_group)
        self._read_profiles(db, db_group)
        self._read_materials(db, line_group, num_lines)
        self.databases.append(db)

    @staticmethod
    def _read_cross_section_geometry(cs, cs_idx, geom_group):
        """Read the geometry for a cross section.

        Args:
            cs (CrossSection): The cross section being read.
            cs_idx (int): The index of the cross section being read.
            geom_group (h5py.Group): The geometry group.
        """
        geom = geom_group[f'CrossSectionGeometry{cs_idx}']
        cs.geometry_d = geom['Geometry D'][:]
        cs.geometry_x = geom['Geometry X'][:]
        cs.geometry_y = geom['Geometry Y'][:]
        cs.geometry_z = geom['Geometry Z'][:]

    @staticmethod
    def _read_cross_section_line(cs, cs_idx, line_group):
        """Read the line properties for a cross section.

        Args:
            cs (CrossSection): The cross section being read.
            cs_idx (int): The index of the cross section being read.
            line_group (h5py.Group): The line properties group.
        """
        line = line_group[f'LineProperties{cs_idx}']
        cs.line_f_values = line['Line Prop. F-Value'][:]
        cs.line_froms = line['Line Prop. From'][:]
        cs.line_i_values = line['Line Prop. I-Value'][:]
        cs.line_tos = line['Line Prop. To'][:]
        cs.line_types = line['Line Prop. Type'][:]

    @staticmethod
    def _read_cross_section_point(cs, cs_idx, point_group):
        """Read the point properties for a cross section.

        Args:
            cs (CrossSection): The cross section being read.
            cs_idx (int): The index of the cross section being read.
            point_group (h5py.Group): The point properties group.
        """
        point = point_group[f'PointProperties{cs_idx}']
        cs.point_measures = point['Point Prop. Measure'][:]
        cs.point_types = point['Point Prop. Type'][:]

    @staticmethod
    def _read_xsec(cs, cs_idx, xsec_group):
        """Read the XSEC properties for a cross section.

        Args:
            cs (CrossSection): The cross section being read.
            cs_idx (int): The index of the cross section being read.
            xsec_group (h5py.Group): The XSEC group.
        """
        cs.name = xsec_group['CS Name'][cs_idx]
        cs.uuid = xsec_group['CS UUID'][cs_idx]
        cs.db_link = int(xsec_group['CSDBlink'][cs_idx])
        cs.note = xsec_group['Note'][cs_idx]
        cs.reach_name = xsec_group['Reach Name'][cs_idx]
        cs.station = float(xsec_group['Station'][cs_idx])
        cs.topo = xsec_group['Topo ID'][cs_idx]
        cs.type = int(xsec_group['Type'][cs_idx])
        cs.p_type = int(xsec_group['pType'][cs_idx])

    @staticmethod
    def _read_general_line_properties(db, line_group):
        """Read the general line properties.

        Args:
            db (CrossSectionDB): The cross section database being read.
            line_group (h5py.Group): The general line properties group.
        """
        # Read the general line properties
        line_cat_dset = line_group['Line Prop. Category'][:]
        line_desc_dset = line_group['Line Prop. Description'][:]
        line_exclusive_dset = line_group['Line Prop. Exclusive'][:]
        line_f_dset = line_group['Line Prop. F-Default'][:]
        line_id_dset = line_group['Line Prop. ID'][:]
        line_name_dset = line_group['Line Prop. Name'][:]
        for line_cat, line_desc, line_exclusive, line_f, line_id, line_name in \
                zip(line_cat_dset, line_desc_dset, line_exclusive_dset, line_f_dset, line_id_dset,
                    line_name_dset):
            line_prop = LineProperty(line_cat, line_desc, line_exclusive, line_f, line_id, line_name)
            db.line_properties.append(line_prop)

    @staticmethod
    def _read_general_point_properties(db, point_group):
        """Read the general point properties.

        Args:
            db (CrossSectionDB): The cross section database being read.
            point_group (h5py.Group): The general point properties group.
        """
        # Read the general point properties
        point_desc_dset = point_group['Point Prop. Desc'][:]
        point_exclusive_dset = point_group['Point Prop. Exclusive'][:]
        point_id_dset = point_group['Point Prop. ID'][:]
        point_name_dset = point_group['Point Prop. Name'][:]
        for point_desc, point_exclusive, point_id, point_name in zip(
            point_desc_dset, point_exclusive_dset, point_id_dset, point_name_dset
        ):
            point_prop = PointProperty(point_desc, point_exclusive, point_id, point_name)
            db.point_properties.append(point_prop)

    @staticmethod
    def _read_profiles(db, db_group):
        """Read the profiles.

        Args:
            db (CrossSectionDB): The cross section database being read.
            db_group (h5py.Group): The cross section database group.
        """
        # Read profiles
        topo_desc_dset = db_group['Profiles/Topo Desc'][:]
        topo_name_dset = db_group['Profiles/Topo Name'][:]
        for topt_desc, topo_name in zip(topo_desc_dset, topo_name_dset):
            profile = Profile(topt_desc, topo_name)
            db.profiles.append(profile)

    @staticmethod
    def _read_materials(db, line_group, num_lines):
        """Read the materials from line properties.

        Args:
            db (CrossSectionDB): The cross section database being read.
            line_group (h5py.Group): The general line properties group being read.
            num_lines (int): The number of line properties.
        """
        # Read materials
        for line_idx in range(num_lines):
            material_group = line_group[f'Material Enums{line_idx}']
            num_mat = material_group['Number of Enums'][0]
            if num_mat > 0:
                enum_dset = material_group['Line Prop. Enum'][:]
                mat_id_dset = material_group['Line Prop. ID'][:]
                mat_name_dset = material_group['Line Prop. Name'][:]
                material = Material(enum_dset, mat_id_dset, mat_name_dset)
            else:
                material = Material([], [], [])
            db.materials.append(material)

    def _read_attributes(self, attributes_group):
        """Read the attributes group.

        Args:
            attributes_group (h5py.Group): The group where the cross-section attributes are.
        """
        arc_id_dset = attributes_group['Arc_Id'][:]
        reach_name_dset = attributes_group['Name'][:]
        intersect_x_dset = attributes_group['Intersect_X'][:]
        intersect_y_dset = attributes_group['Intersect_Y'][:]
        intersect_z_dset = attributes_group['Intersect_Z'][:]
        cross_section_index_dset = attributes_group['Cross_Section_Index'][:]
        reach_id_dset = attributes_group['Reach_Id'][:]
        computed_station_dset = attributes_group['Computed_Station'][:]
        user_station_dset = attributes_group['User_Station'][:]
        flags_dset = attributes_group['Flags'][:]
        cross_section_uuid_dset = attributes_group['Cross_Section_UUID'][:]
        for idx in range(len(arc_id_dset)):
            arc_cs = ArcCrossSection()
            arc_cs.arc_id = arc_id_dset[idx]
            arc_cs.reach_name = reach_name_dset[idx].decode('utf-8')
            arc_cs.intersect_x = intersect_x_dset[idx]
            arc_cs.intersect_y = intersect_y_dset[idx]
            arc_cs.intersect_z = intersect_z_dset[idx]
            arc_cs.cross_section_index = cross_section_index_dset[idx]
            arc_cs.reach_id = reach_id_dset[idx]
            arc_cs.computed_station = computed_station_dset[idx]
            arc_cs.user_station = user_station_dset[idx]
            arc_cs.flags = flags_dset[idx]
            arc_cs.cross_section_uuid = cross_section_uuid_dset[idx].decode('utf-8')
            self._arc_attributes[arc_cs.arc_id] = arc_cs

    def Copy(self):  # noqa: N802
        """Return a reference to this object."""
        return self

    def GetDumpType(self):  # noqa: N802
        """Get the XMS coverage dump type."""
        return "xms.coverage.cross_section"

    def WriteDump(self, file_name):  # noqa: N802,C901
        """Write the wind coverage geometry and attributes to an H5 file XMS can read.

        Args:
            file_name (str): Path to the output H5 file
        """
        self._cov.write_h5(file_name)

        f = h5py.File(file_name, 'a')
        cov_group = f['Map Data/Coverage1']
        att_group = cov_group.create_group('Attributes')
        db_group = cov_group.create_group('Database')

        for idx, db in enumerate(self.databases):
            self._write_db(db_group, idx, db)
        self._write_attributes(att_group, self._arc_attributes)
        f.close()

    @staticmethod
    def _write_attributes(attributes_group, arc_attributes):
        """Write the arc attributes to h5.

        Args:
            attributes_group (h5py.Group): The group for all cross-section databases.
            arc_attributes (dict of int: ArcCrossSection): The cross section arc attribute values.
        """
        num_arcs = len(arc_attributes)
        max_reach_name_length = 0
        for atts in arc_attributes.values():
            max_reach_name_length = max(max_reach_name_length, len(atts.reach_name))
        arc_id_dset = attributes_group.create_dataset('Arc_Id', (num_arcs, ), 'i')
        reach_name_dset = attributes_group.create_dataset('Name', (num_arcs, ), f'S{max_reach_name_length + 1}')
        intersect_x_dset = attributes_group.create_dataset('Intersect_X', (num_arcs, ), 'float64')
        intersect_y_dset = attributes_group.create_dataset('Intersect_Y', (num_arcs, ), 'float64')
        intersect_z_dset = attributes_group.create_dataset('Intersect_Z', (num_arcs, ), 'float64')
        cross_section_index_dset = attributes_group.create_dataset('Cross_Section_Index', (num_arcs, ), 'i')
        reach_id_dset = attributes_group.create_dataset('Reach_Id', (num_arcs, ), 'i')
        computed_station_dset = attributes_group.create_dataset('Computed_Station', (num_arcs, ), 'float64')
        user_station_dset = attributes_group.create_dataset('User_Station', (num_arcs, ), 'float64')
        flags_dset = attributes_group.create_dataset('Flags', (num_arcs, ), 'i')
        cross_section_uuid_dset = attributes_group.create_dataset('Cross_Section_UUID', (num_arcs, ), 'S37')
        for arc_idx, atts in enumerate(arc_attributes.values()):
            arc_id_dset[arc_idx] = atts.arc_id
            reach_name_dset[arc_idx] = atts.reach_name
            intersect_x_dset[arc_idx] = atts.intersect_x
            intersect_y_dset[arc_idx] = atts.intersect_y
            intersect_z_dset[arc_idx] = atts.intersect_z
            cross_section_index_dset[arc_idx] = atts.cross_section_index
            reach_id_dset[arc_idx] = atts.reach_id
            computed_station_dset[arc_idx] = atts.computed_station
            user_station_dset[arc_idx] = atts.user_station
            flags_dset[arc_idx] = atts.flags
            cross_section_uuid_dset[arc_idx] = atts.cross_section_uuid

    @staticmethod
    def _write_db(all_db_group, idx, db):
        """Write a cross section database to H5.

        Args:
            all_db_group (h5py.Group): The group for all cross-section databases.
            idx (int): The index of the current database being written.
            db (CrossSectionDB): The cross section database values.
        """
        db_group = all_db_group.create_group(f'Cross Section{idx}')
        db_name = db_group.create_dataset('DB_Name', (1, ), f'S{len(db.name) + 1}')
        db_uuid = db_group.create_dataset('GUID', (1, ), 'S37')
        db_name[0] = db.name
        db_uuid[0] = db.uuid
        num_cs = len(db.cross_sections)
        cs_list = [None for _ in range(num_cs)]
        for cs in db.cross_sections.values():
            cs_list[cs.index] = cs
        user_station = db_group.create_dataset('USERSTATION', (num_cs, ), 'float64')
        for cs_idx, cs in enumerate(cs_list):
            user_station[cs_idx] = cs.user_station

        CrossSectionCoverage._write_cross_section_geometry(db_group, cs_list)
        CrossSectionCoverage._write_line_properties(db_group, db, cs_list)
        CrossSectionCoverage._write_point_properties(db_group, db, cs_list)

        # Create the profile group
        CrossSectionCoverage._write_profiles(db_group, db)

        CrossSectionCoverage._write_xsecs(db_group, cs_list, num_cs)

    @staticmethod
    def _write_cross_section_geometry(db_group, cs_list):
        """Write the cross section geometry group.

        Args:
            db_group (h5py.Group): The group for this cross-section database.
            cs_list (:obj:`list` of CrossSection): The cross sections in index order.
        """
        geom_group = db_group.create_group('CrossSectionGeometry')
        for cs_idx, cs in enumerate(cs_list):
            cs_group = geom_group.create_group(f'CrossSectionGeometry{cs_idx}')
            num_geom = len(cs.geometry_d)
            d_dset = cs_group.create_dataset('Geometry D', (num_geom, ), 'float64')
            x_dset = cs_group.create_dataset('Geometry X', (num_geom, ), 'float64')
            y_dset = cs_group.create_dataset('Geometry Y', (num_geom, ), 'float64')
            z_dset = cs_group.create_dataset('Geometry Z', (num_geom, ), 'float64')
            for dset_idx in range(num_geom):
                d_dset[dset_idx] = cs.geometry_d[dset_idx]
                x_dset[dset_idx] = cs.geometry_x[dset_idx]
                y_dset[dset_idx] = cs.geometry_y[dset_idx]
                z_dset[dset_idx] = cs.geometry_z[dset_idx]

    @staticmethod
    def _write_line_properties(db_group, db, cs_list):
        """Write the line properties group.

        Args:
            db_group (h5py.Group): The group for this cross-section database.
            db (CrossSectionDB): The cross section database.
            cs_list (:obj:`list` of CrossSection): The cross sections in index order.
        """
        num_lines = len(db.line_properties)
        line_group = db_group.create_group('LineProperties')
        max_desc_length = 0
        max_name_length = 0
        for line in db.line_properties:
            max_desc_length = max(max_desc_length, len(line.description))
            max_name_length = max(max_name_length, len(line.name))
        cat_dset = line_group.create_dataset('Line Prop. Category', (num_lines, ), 'i')
        desc_dset = line_group.create_dataset('Line Prop. Description', (num_lines, ), f'S{max_desc_length + 1}')
        excu_dset = line_group.create_dataset('Line Prop. Exclusive', (num_lines, ), 'i')
        f_dset = line_group.create_dataset('Line Prop. F-Default', (num_lines, ), 'float64')
        id_dset = line_group.create_dataset('Line Prop. ID', (num_lines, ), 'i')
        name_dset = line_group.create_dataset('Line Prop. Name', (num_lines, ), f'S{max_name_length + 1}')
        num_dset = line_group.create_dataset('Line Prop. Number', (1, ), 'i')
        for line_idx, line in enumerate(db.line_properties):
            cat_dset[line_idx] = line.category
            desc_dset[line_idx] = line.description
            excu_dset[line_idx] = line.exclusive
            f_dset[line_idx] = line.f_default
            id_dset[line_idx] = line.line_id
            name_dset[line_idx] = line.name
        num_dset[0] = num_lines
        for cs_idx, cs in enumerate(cs_list):
            cs_line_group = line_group.create_group(f'LineProperties{cs_idx}')
            num_cs_lines = len(cs.line_f_values)
            cs_f_dset = cs_line_group.create_dataset('Line Prop. F-Value', (num_cs_lines, ), 'float64')
            cs_from_dset = cs_line_group.create_dataset('Line Prop. From', (num_cs_lines, ), 'float64')
            cs_i_dset = cs_line_group.create_dataset('Line Prop. I-Value', (num_cs_lines, ), 'i')
            cs_to_dset = cs_line_group.create_dataset('Line Prop. To', (num_cs_lines, ), 'float64')
            cs_type_dset = cs_line_group.create_dataset('Line Prop. Type', (num_cs_lines, ), 'i')
            for cs_line_idx in range(len(cs.line_f_values)):
                cs_f_dset[cs_line_idx] = cs.line_f_values[cs_line_idx]
                cs_from_dset[cs_line_idx] = cs.line_froms[cs_line_idx]
                cs_i_dset[cs_line_idx] = cs.line_i_values[cs_line_idx]
                cs_to_dset[cs_line_idx] = cs.line_tos[cs_line_idx]
                cs_type_dset[cs_line_idx] = cs.line_types[cs_line_idx]

        CrossSectionCoverage._write_materials(db, line_group)

    @staticmethod
    def _write_materials(db, line_group):
        """Write the materials.

        Args:
            db (CrossSectionDB): The cross section database.
            line_group (h5py.Group): The line properties group.
        """
        # Write materials
        for mat_idx, material in enumerate(db.materials):
            mat_group = line_group.create_group(f'Material Enums{mat_idx}')
            num_enum_dset = mat_group.create_dataset('Number of Enums', (1, ), 'i')
            num_enums = len(material.enum_value)
            num_enum_dset[0] = num_enums
            if num_enums > 0:
                max_mat_name_length = 0
                for mat_name in material.material_name:
                    max_mat_name_length = max(max_mat_name_length, len(mat_name))
                enum_dset = mat_group.create_dataset('Line Prop. Enum', (num_enums, ), 'i')
                mat_id_dset = mat_group.create_dataset('Line Prop. ID', (num_enums, ), 'i')
                mat_name_dset = mat_group.create_dataset(
                    'Line Prop. Name', (num_enums, ), f'S{max_mat_name_length + 1}'
                )
                for mat_enum_idx in range(num_enums):
                    enum_dset[mat_enum_idx] = material.enum_value[mat_enum_idx]
                    mat_id_dset[mat_enum_idx] = material.material_id[mat_enum_idx]
                    mat_name_dset[mat_enum_idx] = material.material_name[mat_enum_idx]

    @staticmethod
    def _write_point_properties(db_group, db, cs_list):
        """Write the point properties group.

        Args:
            db_group (h5py.Group): The group for this cross-section database.
            db (CrossSectionDB): The cross section database.
            cs_list (:obj:`list` of CrossSection): The cross sections in index order.
        """
        num_points = len(db.point_properties)
        point_group = db_group.create_group('PointProperties')
        max_desc_length = 0
        max_name_length = 0
        for point in db.point_properties:
            max_desc_length = max(max_desc_length, len(point.description))
            max_name_length = max(max_name_length, len(point.name))
        desc_dset = point_group.create_dataset('Point Prop. Desc', (num_points, ), f'S{max_desc_length + 1}')
        exclu_dset = point_group.create_dataset('Point Prop. Exclusive', (num_points, ), 'i')
        id_dset = point_group.create_dataset('Point Prop. ID', (num_points, ), 'i')
        name_dset = point_group.create_dataset('Point Prop. Name', (num_points, ), f'S{max_name_length + 1}')
        for point_idx, point in enumerate(db.point_properties):
            desc_dset[point_idx] = point.description
            exclu_dset[point_idx] = point.exclusive
            id_dset[point_idx] = point.point_id
            name_dset[point_idx] = point.name
        for cs_idx, cs in enumerate(cs_list):
            cs_point_group = point_group.create_group(f'PointProperties{cs_idx}')
            num_cs_points = len(cs.point_measures)
            cs_measure_dset = cs_point_group.create_dataset('Point Prop. Measure', (num_cs_points, ), 'float64')
            cs_type_dset = cs_point_group.create_dataset('Point Prop. Type', (num_cs_points, ), 'i')
            for cs_point_idx in range(len(cs.point_measures)):
                cs_measure_dset[cs_point_idx] = cs.point_measures[cs_point_idx]
                cs_type_dset[cs_point_idx] = cs.point_types[cs_point_idx]

    @staticmethod
    def _write_profiles(db_group, db):
        """Write the profile group.

        Args:
            db_group (h5py.Group): The group for this cross-section database.
            db (CrossSectionDB): The cross section database.
        """
        profile_group = db_group.create_group('Profiles')
        num_profiles = len(db.profiles)
        max_topo_desc_size = 0
        max_topo_name_size = 0
        for profile in db.profiles:
            max_topo_desc_size = max(max_topo_desc_size, len(profile.topo_description))
            max_topo_name_size = max(max_topo_name_size, len(profile.topo_name))
        topo_desc = profile_group.create_dataset('Topo Desc', (num_profiles, ), f'S{max_topo_desc_size + 1}')
        topo_name = profile_group.create_dataset('Topo Name', (num_profiles, ), f'S{max_topo_name_size + 1}')
        for idx, profile in enumerate(db.profiles):
            topo_desc[idx] = profile.topo_description
            topo_name[idx] = profile.topo_name

    @staticmethod
    def _write_xsecs(db_group, cs_list, num_cs):
        """Write the XSECS section.

        Args:
            db_group (h5py.Group): The group for this cross-section database.
            cs_list (:obj:`list` of CrossSection): The cross sections in index order.
            num_cs (int): The length of cs_list.
        """
        xsec_group = db_group.create_group('XSECS')
        xsec_dset = xsec_group.create_dataset('XSECS', (1, ), 'i')
        xsec_dset[0] = num_cs
        max_name_size = 0
        max_note_size = 0
        max_reach_name_size = 0
        max_topo_size = 0
        for cs in cs_list:
            max_name_size = max(max_name_size, len(cs.name))
            max_note_size = max(max_note_size, len(cs.note))
            max_reach_name_size = max(max_reach_name_size, len(cs.reach_name))
            max_topo_size = max(max_topo_size, len(cs.topo))
        name_dset = xsec_group.create_dataset('CS Name', (num_cs, ), f'S{max_name_size + 1}')
        uuid_dset = xsec_group.create_dataset('CS UUID', (num_cs, ), 'S37')
        db_link_dset = xsec_group.create_dataset('CSDBlink', (num_cs, ), 'i')
        note_dset = xsec_group.create_dataset('Note', (num_cs, ), f'S{max_note_size + 1}')
        reach_name_dset = xsec_group.create_dataset('Reach Name', (num_cs, ), f'S{max_reach_name_size + 1}')
        station_dset = xsec_group.create_dataset('Station', (num_cs, ), 'float64')
        topo_dset = xsec_group.create_dataset('Topo ID', (num_cs, ), f'S{max_topo_size + 1}')
        type_dset = xsec_group.create_dataset('Type', (num_cs, ), 'i')
        p_type_dset = xsec_group.create_dataset('pType', (num_cs, ), 'i')
        for cs_idx, cs in enumerate(cs_list):
            name_dset[cs_idx] = cs.name
            uuid_dset[cs_idx] = cs.uuid
            db_link_dset[cs_idx] = cs.db_link
            note_dset[cs_idx] = cs.note
            reach_name_dset[cs_idx] = cs.reach_name
            station_dset[cs_idx] = cs.station
            topo_dset[cs_idx] = cs.topo
            type_dset[cs_idx] = cs.type
            p_type_dset[cs_idx] = cs.p_type


def ReadDumpWithObject(file_name):  # noqa: N802
    """Read a cross section coverage dump file.

    Args:
        file_name (str): Filepath to the dumped coverage to read

    Returns:
        CrossSectionCoverage: The loaded cross section coverage
    """
    wind_dump = CrossSectionCoverage(file_name)
    return wind_dump
