"""Reads an XMS .map file."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon

# 4. Local modules
from xms.coverage.xy.xy_series import XySeries


def read(file_path: str | Path) -> tuple[list[Coverage], dict, dict[int, XySeries]]:
    """Reads the .map file and returns a tuple: list of coverages, dict of extra non-geom lines, dict of XySeries.

    Args:
        file_path: .map file path.

    Returns:
        See description.
    """
    reader = MapFileReader(file_path)
    return reader.read()


class MapFileReader:
    """Reads an XMS .map file."""
    def __init__(self, file_path: str | Path) -> None:
        """Initializer.

        Args:
            file_path: .map file path.
        """
        self._file_path = file_path

    def read(self) -> tuple[list[Coverage], dict, dict[int, XySeries]]:
        """Reads the .map file and returns a tuple: list of coverages, dict of extra non-geom lines, dict of XySeries.

        Returns:
            See description.
        """
        point, arc, polygon = None, None, None  # Current object
        points: dict[int, Point] = {}
        arcs: dict[int, Arc] = {}
        polygons: dict[int, Polygon] = {}
        coverages: list[Coverage] = []
        extras = {}  # coverage uuid -> dict of non-geometry text for the coverage and its objects
        coverage_object_extras: list[str] = []  # non-geometry text found that pertains to the coverage object
        feature_object_extras: list[str] = []  # non-geometry text found when reading points, arcs, polygons
        first_polygon = True
        xy_series_dict: dict[int, XySeries] = {}

        with (open(self._file_path, 'r') as file):
            for i, line in enumerate(file):
                line = line.rstrip('\n')
                card, card_value = _get_card_and_value(line)
                match card:
                    case 'BEGCOV':
                        coverage = Coverage()
                        coverage_extras = {'coverage': [], 'points': {}, 'arcs': {}, 'polygons': {}}
                    case 'COVGUID':
                        coverage.uuid = card_value.strip('"')
                    case 'COVNAME':
                        coverage.name = card_value.strip('"')
                    case 'POINT':
                        point = Point()
                    case 'NODE':
                        point = Point()
                    case 'ARC':
                        arc = Arc()
                    case 'NODES':  # Arc nodes
                        node_id1, node_id2 = map(int, card_value.split())
                        arc.start_node, arc.end_node = points[node_id1], points[node_id2]
                    case 'ARCVERTICES':
                        nvert = int(card_value)
                        vertices = []
                        for _ in range(nvert):
                            line = next(file)
                            x, y, z = map(float, line.split())
                            vertices.append(Point(x, y, z))
                        arc.vertices = vertices
                    case 'POLYGON':
                        if first_polygon:  # Skip the first polygon - the universal polygon
                            first_polygon = False
                            while card != 'END':
                                line = next(file)
                                card, _ = _get_card_and_value(line.rstrip('\n'))
                        else:
                            polygon = Polygon()
                    case 'ARCS':  # polygon outer arcs
                        narcs = int(card_value)
                        poly_arcs = []
                        for _ in range(narcs):
                            line = next(file)
                            arc_id = int(line.split(' ', 1)[0])
                            poly_arcs.append(arcs[arc_id])
                        polygon.set_arcs(poly_arcs)
                    case 'HARCS':  # polygon interior arcs
                        narcs = int(card_value)
                        harcs = []
                        for _ in range(narcs):
                            line = next(file)
                            arc_id = int(line.split(' ', 1)[0])
                            harcs.append(arcs[arc_id])
                        # There can be multiple 'HARCS' per polygon. Append this list of arcs to the existing list
                        interior_arcs = polygon.interior_arcs
                        interior_arcs.append(harcs)
                        polygon.set_interior_arcs(interior_arcs)
                    case 'ID':
                        if point:
                            point.id = int(card_value)
                            points[point.id] = point
                        elif arc:
                            arc.id = int(card_value)
                            arcs[arc.id] = arc
                        elif polygon:
                            polygon.id = int(card_value)
                            polygons[polygon.id] = polygon
                    case 'XY':
                        point.x, point.y, point.z = map(float, card_value.split())
                    case 'END':
                        feature_type = 'points' if point else 'arcs' if arc else 'polygons'
                        object_id = point.id if point else arc.id if arc else polygon.id
                        coverage_extras[feature_type][object_id] = feature_object_extras
                        feature_object_extras = []
                        point, arc, polygon = None, None, None
                    case 'ENDCOV':
                        # Extras
                        coverage_extras['coverage'] = coverage_object_extras
                        extras[coverage.uuid] = coverage_extras
                        coverage_extras = {}
                        coverage_object_extras = []

                        # Build coverage
                        coverage.set_points(list(points.values()))
                        coverage.arcs = list(arcs.values())
                        coverage.polygons = list(polygons.values())
                        coverage.complete()
                        coverages.append(coverage)
                        coverage = None

                        # Reset
                        first_polygon = True
                        points, arcs, polygons = {}, {}, {}
                    case 'XYS':
                        xy_series = XySeries.from_file(file, line, i + 1)
                        xy_series_dict[xy_series.series_id] = xy_series
                    case _:
                        # Save whatever we read but don't recognize
                        if point or arc or polygon:
                            feature_object_extras.append(line)
                        else:
                            coverage_object_extras.append(line)

        return coverages, extras, xy_series_dict


def _get_card_and_value(line: str) -> tuple[str, str]:
    """Given a line from the file, returns a tuple of: card - the first word, value - everything else.

    Args:
        line: A line from a file.

    Returns:
        See description.
    """
    card_and_value = line.rstrip('\n').split(' ', 1)  # Split line into 2: card, everything else
    return card_and_value[0], card_and_value[1].strip() if len(card_and_value) > 1 else ''
