"""Mapping utilities."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.grid.geometry import geometry

# 4. Local modules

XM_NONE = -1


def extents_2d(point_list):
    """Returns the minimum xy and maximum xy (bounding rectangle) of the list of points.

    Args:
         point_list (list[tuple[float, float]]): List of points.

    Returns:
        tuple[float, float], tuple[float, float]: Min xy, max xy.
    """
    if point_list is None or len(point_list) == 0:
        return None, None
    # Not sure if this is faster or just iterating through the list once
    min_x = min([point[0] for point in point_list])
    min_y = min([point[1] for point in point_list])
    max_x = max([point[0] for point in point_list])
    max_y = max([point[1] for point in point_list])
    return (min_x, min_y), (max_x, max_y)


def point_in_rectangle(min_xy, max_xy, xy):
    """Returns True if the point xy is inside the rectangle formed by min_x, min_y...

    Args:
        min_xy (tuple[float, float]): Minimum rectangle xy.
        max_xy (tuple[float, float]): Maximum rectangle x.
        xy (tuple[float, float]): The point in question.
    """
    return not (xy[0] < min_xy[0] or xy[1] < min_xy[1] or xy[0] > max_xy[0] or xy[1] > max_xy[1])


def point_in_polygon_2d(polygon, point, strictly_in: bool = False):
    """Wraps geometry.point_in_polygon_2d for convenience.

     geometry.point_in_polygon_2d() doesn't want last point repeated, and returns 1 for in, 0 for on.

    Args:
        polygon (list[tuple[float, float, float]]): The polygon.
        point (tuple[float, float, float]): The point.
        strictly_in (bool): If True, returns False if the point is on the polygon.
    """
    if strictly_in:
        return geometry.point_in_polygon_2d(polygon[0:-1], point) > 0
    else:
        return geometry.point_in_polygon_2d(polygon[0:-1], point) >= 0


def distance_along_arc(arc_locations, location, normalized):
    """Returns the distance along the arc to the location, maybe normalized, or -1.0 if point not on arc.

    Args:
        arc_locations: list-like collection of xyz points.
        location: tuple-like xyz point in question.
        normalized: If True, distance is returned as a decimal between 0.0 and 1.0.

    Returns:
        float: See description.
    """
    tol = 0.0001  # needs to be smarter
    length = 0.0  # Length to the point along the arc
    arc_length = 0.0  # Arc length so far, and eventually the total arc length if normalizing
    last_i = 0
    for i in range(1, len(arc_locations)):
        if geometry.on_line_and_between_endpoints_2d(arc_locations[i - 1], arc_locations[i], location, tol):
            length = arc_length + geometry.distance_2d(arc_locations[i - 1], location)
            arc_length += geometry.distance_2d(arc_locations[i - 1], arc_locations[i])
            last_i = i
            break
        else:
            arc_length += geometry.distance_2d(arc_locations[i - 1], arc_locations[i])
    else:
        return XM_NONE  # Didn't find it

    if normalized:
        # Get the rest of the arc length
        if last_i < len(arc_locations) - 1:
            for i in range(last_i + 1, len(arc_locations)):
                arc_length += geometry.distance_2d(arc_locations[i - 1], arc_locations[i])
        return length / arc_length
    return length


def point_on_polyline_2d(point, polyline_points, tolerance):
    """Returns True if the point is on the polyline (and between the endpoints).

    Args:
        point (tuple[float, float]): The point xy.
        polyline_points (list[tuple[float, float]]): The points defining the polyline.
        tolerance (float): Tolerance used.

    Returns:
        (bool): True or False
    """
    for i in range(1, len(polyline_points)):
        if geometry.on_line_and_between_endpoints_2d(polyline_points[i - 1], polyline_points[i], point, tolerance):
            return True
    return False


def average_point_2d(points):
    """Returns a xy tuple that is the average of the points in xy.

    Args:
        points (list[tuple(float, float)]): List of points.

    Returns:
        tuple(float, float): The average point xy.
    """
    x = 0
    y = 0
    for point in points:
        x += point[0]
        y += point[1]
    return x / len(points), y / len(points)
