"""Time utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import logging

# 2. Third party modules
from dateutil import parser

# 3. Aquaveo modules

# 4. Local modules

# Constants
SECONDS_PER_DAY = 86400
SECONDS_PER_HOUR = 3600
SECONDS_PER_MINUTE = 60
MINUTES_PER_DAY = 1440
HOURS_PER_DAY = 24
DAYS_PER_YEAR = 365.2425
SECONDS_PER_YEAR = DAYS_PER_YEAR * SECONDS_PER_DAY


def datetime_from_arbitrary_string(time_string):
    """Tries to parse a date/time string with unknown format into a Python datetime object.

    See https://dateutil.readthedocs.io/en/stable/parser.html

    Args:
        time_string (str): The time string.

    Returns:
        (datetime): The date/time.
    """
    if not time_string:
        return None
    try:
        return parser.parse(time_string)
    except Exception as error:
        logger = logging.getLogger('xms.coverage')
        logger.error(str(error))
        return None


def is_number(s):
    """Returns True if the string s is a number, otherwise returns False.

        From https://stackoverflow.com/questions/354038

    Args:
        s (str): The string.

    Returns:
        (bool): True if the string is a number, otherwise False.
    """
    if s is None:
        return False
    try:
        float(s)
        return True
    except ValueError:
        return False


def time_duration(start_date_time, end_date_time, time_units):
    """Returns the difference between start_date_time and end_date_time in the specified time units.

    Args:
        start_date_time (datetime):
        end_date_time (datetime):
        time_units (str): Time units: 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'

    Returns:
        (float): See description.
    """
    if time_units not in ['SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS']:
        raise RuntimeError('Invalid time units in data_util.time_duration')

    # Create a python timedelta object by subtracting the dates
    delta = end_date_time - start_date_time

    # Get the days and seconds from the timedelta (which are the only thing it stores)
    days = delta.days
    seconds = delta.seconds

    float_value = -999.0
    if time_units == 'SECONDS':
        float_value = (days * SECONDS_PER_DAY) + seconds
    elif time_units == 'MINUTES':
        float_value = (days * MINUTES_PER_DAY) + (seconds / SECONDS_PER_MINUTE)
    elif time_units == 'HOURS':
        float_value = (days * HOURS_PER_DAY) + (seconds / SECONDS_PER_HOUR)
    elif time_units == 'DAYS':
        float_value = days + (seconds / SECONDS_PER_DAY)
    elif time_units == 'YEARS':
        float_value = (days / DAYS_PER_YEAR) + (seconds / SECONDS_PER_YEAR)
    return float_value


def time_value_from_string(word, date_times_to_floats, start_date_time, time_units):
    """Returns the time value, which can be either a float or a date/time string, as a float.

    Args:
        word (str): Time (x) string from xy series file.
        date_times_to_floats (bool): If true and start_date_time and time_units are given, converts date/time
         strings to relative offsets from the start_date_time.
        start_date_time (datetime|None): Starting date/time
        time_units (str|None): Time units: 'UNKNOWN', 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'

    Returns:
        tuple[float or datetime]: Time value.
    """
    time_value = None
    if is_number(word):
        time_value = float(word)
    else:
        try:
            if date_times_to_floats:
                start_is_date_time = start_date_time and isinstance(start_date_time, datetime)
                if start_is_date_time and time_units and time_units != 'UNKNOWN':
                    date_time = datetime_from_arbitrary_string(word)
                    time_value = time_duration(start_date_time, date_time, time_units)
            else:
                time_value = datetime_from_arbitrary_string(word)
        except Exception:
            pass  # We do nothing here - caller should check return value and so something if the want
    return time_value
