"""Code to store interface with the SMS wind coverage."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from enum import IntEnum

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DataDumpIOBase
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules


class AutoNumber(IntEnum):
    """IntEnum that auto numbers itself."""
    def __new__(cls):
        """Construct the enum."""
        value = len(cls.__members__)  # note no + 1
        obj = int.__new__(cls)
        obj._value_ = value
        return obj


class DEV_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for supported storm levels of development."""
    DEV_DISTURBANCE = ()
    DEV_TROPICAL_DEPRESSION = ()
    DEV_TROPICAL_STORM = ()
    DEV_TYPHOON = ()
    DEV_SUPER_TYPHOON = ()
    DEV_TROPICAL_CYCLONE = ()
    DEV_HURRICANE = ()
    DEV_SUBTROPICAL_DEPRESSION = ()
    DEV_SUBTROPICAL_STORM = ()
    DEV_EXTRATROPICAL_SYSTEMS = ()
    DEV_INLAND = ()
    DEV_DISSIPATING = ()
    DEV_LOW = ()
    DEV_TROPICAL_WAVE = ()
    DEV_EXTRAPOLATED = ()
    DEV_UNKNOWN = ()
    DEV_END = ()


class RADIUS_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for supported wind radius codes."""
    RADIUS_FULL_CIRCLE = ()
    RADIUS_NORTH_SEMICIRCLE = ()
    RADIUS_NORTHEAST_SEMICIRCLE = ()
    RADIUS_EAST_SEMICIRCLE = ()
    RADIUS_SOUTHEAST_SEMICIRCLE = ()
    RADIUS_SOUTH_SEMICIRCLE = ()
    RADIUS_SOUTHWEST_SEMICIRCLE = ()
    RADIUS_WEST_SEMICIRCLE = ()
    RADIUS_NORTHWEST_SEMICIRCLE = ()
    RADIUS_NORTH_QUAD = ()
    RADIUS_NORTHEAST_QUAD = ()
    RADIUS_EAST_QUAD = ()
    RADIUS_SOUTHEAST_QUAD = ()
    RADIUS_SOUTH_QUAD = ()
    RADIUS_SOUTHWEST_QUAD = ()
    RADIUS_WEST_QUAD = ()
    RADIUS_NORTHWEST_QUAD = ()
    RADIUS_END = ()


class DEPTH_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for supported system depth types."""
    SYSDEPTH_DEEP = ()
    SYSDEPTH_MEDIUM = ()
    SYSDEPTH_SHALLOW = ()
    SYSDEPTH_UNKNOWN = ()
    SYSDEPTH_END = ()


class WIND_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for wind intensity at a node."""
    _order_ = 'WIND_34 WIND_50 WIND_64 WIND_100 WIND_END'
    WIND_34 = ()
    WIND_50 = ()
    WIND_64 = ()
    WIND_100 = ()
    WIND_END = ()


class QUADRANT_enum(AutoNumber):  # noqa: N801
    """Enum for isotachs of a wind intensity at a node."""
    _order_ = 'QUADRANT_1 QUADRANT_2 QUADRANT_3 QUADRANT_4 QUADRANT_END'
    QUADRANT_1 = ()
    QUADRANT_2 = ()
    QUADRANT_3 = ()
    QUADRANT_4 = ()
    QUADRANT_END = ()


class SUBREGION_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for defined sub-regions."""
    SUBREGION_ARABIAN_SEA = ()
    SUBREGION_BAY_OF_BENGAL = ()
    SUBREGION_CENTRAL_PACIFIC = ()
    SUBREGION_EASTERN_PACIFIC = ()
    SUBREGION_ATLANTIC = ()
    SUBREGION_SOUTH_PACIFIC = ()
    SUBREGION_SOUTH_INDIAN_OCEAN = ()
    SUBREGION_WESTERN_PACIFIC = ()
    SUBREGION_END = ()


class BASIN_CARDS_enum(AutoNumber):  # noqa: N801
    """Enum for defined basins."""
    BASIN_NORTH_PACIFIC = ()
    BASIN_NORTH_INDIAN_OCEAN = ()
    BASIN_SOUTHERN_HEMISPHERE = ()
    BASIN_CENTRAL_PACIFIC = ()
    BASIN_EASTERN_PACIFIC = ()
    BASIN_ATLANTIC = ()
    BASIN_END = ()


class WindModel_enum(AutoNumber):  # noqa: N801
    """Enum for supported wind model types."""
    WM_HOLLAND_SYMMETRIC = ()
    WM_HOLLAND_ASYMMETRIC = ()
    WM_HOLLAND_SIMPLIFIED_ASYMMETRIC = ()


class WindNodeIsotach:
    """Class to represent a wind intensity isotach at a node."""
    def __init__(self):
        """Construct the wind intensity isotach."""
        self.m_radius = 0
        self.m_rMax = 0.0
        self.m_on = False
        self.m_hollandB = 0.0
        self.m_vMax = 0.0


class WindRadiiData:
    """Class to store the isotach data of a wind intensity at a node."""
    def __init__(self):
        """Construct the wind radii data."""
        self.m_data = [
            WindNodeIsotach(), WindNodeIsotach(),
            WindNodeIsotach(), WindNodeIsotach()
        ]  # same value as QUADRANT_END
        self.m_defined = False


class WindNode:
    """Class to hold the attributes of a wind coverage node."""
    def __init__(self):
        """Construct the wind node."""
        self.m_id = 0
        self.m_date = 0.0  # Julian double
        self.m_iTechNumOrMinutes = 0
        self.m_sTech = ""
        self.m_iVMax = 0
        self.m_iMinSeaLevelPressure = 1
        self.m_eLevelOfDevelopment = DEV_CARDS_enum.DEV_TROPICAL_DEPRESSION
        self.m_eWindRadiusCode = RADIUS_CARDS_enum.RADIUS_FULL_CIRCLE
        self.m_radii = {
            WIND_CARDS_enum.WIND_34.value: WindRadiiData(),
            WIND_CARDS_enum.WIND_50.value: WindRadiiData(),
            WIND_CARDS_enum.WIND_64.value: WindRadiiData(),
            WIND_CARDS_enum.WIND_100.value: WindRadiiData()
        }
        self.m_iPressure = 1013
        self.m_iPressureRadius = 0
        self.m_iMaxWindRadius = 0
        self.m_iGusts = 0
        self.m_iEyeDiameter = 0
        self.m_iMaxSeas = 0
        self.m_sInitials = ""
        self.m_iStormDirection = 0
        self.m_iStormSpeed = 0
        self.m_sStormName = "Unnamed"
        self.m_eDepth = DEPTH_CARDS_enum.SYSDEPTH_DEEP
        self.m_iWaveHeight = 0
        self.m_eSeasRadiusCode = RADIUS_CARDS_enum.RADIUS_NORTHEAST_QUAD
        self.m_iSeas1 = 0
        self.m_iSeas2 = 0
        self.m_iSeas3 = 0
        self.m_iSeas4 = 0
        self.m_hollandB = 0.0
        # Simplified asymmetric (TUFLOWFV)
        # I know it is inconsistent, but I can't do 'm_' in Python anymore. Someday we will clean up this code or
        # delete it.
        self.rhoa = 0.0  # Air density [RHOA]
        self.km = 0.0  # Km (mean near surface wind speed / gradient level wind speed) [KM]
        self.deltafm = 0  # Asymmetry factor [DELTAFM]
        self.wbgx = 0.0  # Background wind X component [WBGX]
        self.wbgy = 0.0  # Background wind X component [WBGY]


class WindCoverage(DataDumpIOBase):
    """Python interface for the SMS Map module Wind Coverage data."""
    def __init__(self, file_name=''):
        """Construct the wind coverage."""
        super().__init__()
        super().SetSelf(self)
        self.__m_cov = None
        self.m_nodeWind = []
        self.m_hasRead = False
        self.m_fileName = file_name
        self.m_windModel = WindModel_enum.WM_HOLLAND_SYMMETRIC
        self.m_basin = BASIN_CARDS_enum.BASIN_ATLANTIC
        self.m_subregion = SUBREGION_CARDS_enum.SUBREGION_ATLANTIC
        self.m_cycloneNumber = 1

    @property
    def m_cov(self):
        """Get the wind coverage geometry.

        Returns:
            xms.data_objects.parameters.Coverage: The wind coverage geometry
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_cov

    @m_cov.setter
    def m_cov(self, val):
        """Set the wind coverage geometry.

        Args:
            val (xms.data_objects.parameters.Coverage): The wind coverage geometry
        """
        self.__m_cov = val

    @property
    def m_nodeWind(self):  # noqa: N802
        """Get the wind coverage nodal attributes.

        Returns:
            list of WindNode: The wind coverage nodal attributes
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_nodeWind

    @m_nodeWind.setter
    def m_nodeWind(self, val):  # noqa: N802
        """Set the wind coverage nodal attributes.

        Args:
            val (list of WindNode): The wind coverage nodal attributes
        """
        self.__m_nodeWind = val

    @property
    def m_windModel(self):  # noqa: N802
        """Get the wind coverage model attribute.

        Returns:
            WindModel_enum: The wind coverage model attribute
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_windModel

    @m_windModel.setter
    def m_windModel(self, val):  # noqa: N802
        """Set the wind coverage model attribute.

        Args:
            val (WindModel_enum): The wind coverage model attribute
        """
        self.__m_windModel = val

    @property
    def m_basin(self):
        """Get the wind coverage basin attribute.

        Returns:
            BASIN_CARDS_enum: The wind coverage basin attribute
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_basin

    @m_basin.setter
    def m_basin(self, val):
        """Set the wind coverage basin attribute.

        Args:
            val (BASIN_CARDS_enum): The wind coverage basin attribute
        """
        self.__m_basin = val

    @property
    def m_subregion(self):
        """Get the wind coverage sub-region attribute.

        Returns:
            SUBREGION_CARDS_enum: The wind coverage sub-region attribute
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_subregion

    @m_subregion.setter
    def m_subregion(self, val):
        """Set the wind coverage sub-region attribute.

        Args:
            val (SUBREGION_CARDS_enum): The wind coverage sub-region attribute
        """
        self.__m_subregion = val

    @property
    def m_cycloneNumber(self):  # noqa: N802
        """Get the wind coverage cyclone number attribute.

        Returns:
            int: The wind coverage cyclone number attribute
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_cycloneNumber

    @m_cycloneNumber.setter
    def m_cycloneNumber(self, val):  # noqa: N802
        """Set the wind coverage cyclone number attribute.

        Args:
            val (int): The wind coverage cyclone number attribute
        """
        if val < 1:
            self.__m_cycloneNumber = 1
        elif val:
            self.__m_cycloneNumber = val

    def WriteDump(self, file_name):  # noqa: N802,C901
        """Write the wind coverage geometry and attributes to an H5 file XMS can read.

        Args:
            file_name (str): Path to the output H5 file
        """
        self.m_cov.write_h5(file_name)

        f = h5py.File(file_name, 'a')
        cov_group = f['Map Data/Coverage1']
        atts = cov_group.create_group("Attributes")

        wind_model = atts.create_dataset("WindModel", (1, ), 'i')
        basin = atts.create_dataset("Basin", (1, ), 'i')
        sub_region = atts.create_dataset("Subregion", (1, ), 'i')
        cyc_num = atts.create_dataset("CycloneNumber", (1, ), 'i')
        # This is dumb. We didn't know how to write Python when this piece of garbage was created. It will break
        # to many things for us to really fix it right now, so we patch.
        if isinstance(self.__m_windModel, WindModel_enum):
            wind_model[0] = self.__m_windModel.value
            is_symm = self.__m_windModel.value == WindModel_enum.WM_HOLLAND_SYMMETRIC.value
            is_asymm = self.__m_windModel.value == WindModel_enum.WM_HOLLAND_ASYMMETRIC.value
            is_simple_asymm = self.__m_windModel.value == WindModel_enum.WM_HOLLAND_SIMPLIFIED_ASYMMETRIC.value
        else:
            wind_model[0] = self.__m_windModel
            model_enum = WindModel_enum(self.__m_windModel)
            is_symm = model_enum.value == WindModel_enum.WM_HOLLAND_SYMMETRIC.value
            is_asymm = model_enum.value == WindModel_enum.WM_HOLLAND_ASYMMETRIC.value
            is_simple_asymm = model_enum.value == WindModel_enum.WM_HOLLAND_SIMPLIFIED_ASYMMETRIC.value
        if isinstance(self.__m_basin, BASIN_CARDS_enum):
            basin[0] = self.__m_basin.value
        else:
            basin[0] = self.__m_basin
        if isinstance(self.__m_subregion, SUBREGION_CARDS_enum):
            sub_region[0] = self.__m_subregion.value
        else:
            sub_region[0] = self.__m_subregion
        cyc_num[0] = self.__m_cycloneNumber

        nodes = atts.create_group("Nodes")
        node_length = len(self.__m_nodeWind)

        node_id = nodes.create_dataset("Id", (node_length, ), 'i')
        node_date = nodes.create_dataset("Date", (node_length, ), 'float64')
        node_tech_num = nodes.create_dataset("TechNumOrMinutes", (node_length, ), 'i')
        node_tech = nodes.create_dataset("Tech", (node_length, ), 'S5')
        node_v_max = nodes.create_dataset("VMax", (node_length, ), 'i')
        node_min_sea = nodes.create_dataset("MinSeaLevelPressure", (node_length, ), 'i')
        node_level = nodes.create_dataset("LevelOfDevelopment", (node_length, ), 'i')
        node_wind_radius_code = nodes.create_dataset("WindRadiusCode", (node_length, ), 'i')
        node_pressure = nodes.create_dataset("Pressure", (node_length, ), 'i')
        node_pressure_radius = nodes.create_dataset("PressureRadius", (node_length, ), 'i')
        node_max_wind = nodes.create_dataset("MaxWindRadius", (node_length, ), 'i')
        node_gusts = nodes.create_dataset("Gusts", (node_length, ), 'i')
        node_eye_diameter = nodes.create_dataset("EyeDiameter", (node_length, ), 'i')
        node_max_sea = nodes.create_dataset("MaxSeas", (node_length, ), 'i')
        node_initials = nodes.create_dataset("Initials", (node_length, ), 'S4')
        node_storm_dir = nodes.create_dataset("StormDirection", (node_length, ), 'i')
        node_storm_speed = nodes.create_dataset("StormSpeed", (node_length, ), 'i')
        node_storm_name = nodes.create_dataset("StormName", (node_length, ), 'S11')
        if is_symm:
            node_depth = nodes.create_dataset("Depth", (node_length, ), 'i')  # only sym
            node_wave_height = nodes.create_dataset("WaveHeight", (node_length, ), 'i')  # only sym
            node_sea_radius_code = nodes.create_dataset("SeasRadiusCode", (node_length, ), 'i')  # only sym
            node_sea_1 = nodes.create_dataset("Seas1", (node_length, ), 'i')  # only sym
            node_sea_2 = nodes.create_dataset("Seas2", (node_length, ), 'i')  # only sym
            node_sea_3 = nodes.create_dataset("Seas3", (node_length, ), 'i')  # only sym
            node_sea_4 = nodes.create_dataset("Seas4", (node_length, ), 'i')  # only sym
        else:
            node_holland_b = nodes.create_dataset("HollandB", (node_length, ), 'float64')  # only asym
            if is_simple_asymm:
                node_rhoa = nodes.create_dataset('Rhoa', (node_length, ), 'float64')  # only simple asym
                node_km = nodes.create_dataset('Km', (node_length, ), 'float64')  # only simple asym
                node_deltafm = nodes.create_dataset('DeltaFm', (node_length, ), 'i')  # only simple asym
                node_wbgx = nodes.create_dataset('Wbgx', (node_length, ), 'float64')  # only simple asym
                node_wbgy = nodes.create_dataset('Wbgy', (node_length, ), 'float64')  # only simple asym

        isotach_defined_dset = []
        rad_dset = []
        rad_on_dset = []
        rad_max_dset = []
        rad_holland_b_dset = []
        rad_v_max_dset = []
        for rad_group in ["Radii34", "Radii50", "Radii64", "Radii100"]:
            rad = nodes.create_group(rad_group)
            isotach_defined_dset.append(rad.create_dataset('IsotachDefined', (node_length, ), 'i'))
            for quad in ["Radius1", "Radius2", "Radius3", "Radius4"]:
                rad_quad = rad.create_group(quad)
                rad_dset.append(rad_quad.create_dataset("Radius", (node_length, ), 'i'))
                if is_asymm:
                    rad_on_dset.append(rad_quad.create_dataset("On", (node_length, ), 'i'))
                    rad_max_dset.append(rad_quad.create_dataset("RMax", (node_length, ), 'f'))
                    rad_holland_b_dset.append(rad_quad.create_dataset("HollandB_Quadrant", (node_length, ), 'f'))
                    rad_v_max_dset.append(rad_quad.create_dataset("VMax_Quadrant", (node_length, ), 'f'))

        for i in range(0, node_length):
            node_id[i] = self.__m_nodeWind[i].m_id
            node_date[i] = self.__m_nodeWind[i].m_date
            node_tech_num[i] = self.__m_nodeWind[i].m_iTechNumOrMinutes
            node_tech[i] = bytes(self.__m_nodeWind[i].m_sTech, 'utf-8')
            node_v_max[i] = self.__m_nodeWind[i].m_iVMax
            node_min_sea[i] = self.__m_nodeWind[i].m_iMinSeaLevelPressure
            # This is dumb. We didn't know how to write Python when this piece of garbage was created. It will break
            # to many things for us to really fix it right now, so we patch.
            if isinstance(self.__m_nodeWind[i].m_eLevelOfDevelopment, DEV_CARDS_enum):
                node_level[i] = self.__m_nodeWind[i].m_eLevelOfDevelopment.value
            else:
                node_level[i] = self.__m_nodeWind[i].m_eLevelOfDevelopment
            if isinstance(self.__m_nodeWind[i].m_eWindRadiusCode, RADIUS_CARDS_enum):
                node_wind_radius_code[i] = self.__m_nodeWind[i].m_eWindRadiusCode.value
            else:
                node_wind_radius_code[i] = self.__m_nodeWind[i].m_eWindRadiusCode
            node_pressure[i] = self.__m_nodeWind[i].m_iPressure
            node_pressure_radius[i] = self.__m_nodeWind[i].m_iPressureRadius
            node_max_wind[i] = self.__m_nodeWind[i].m_iMaxWindRadius
            node_gusts[i] = self.__m_nodeWind[i].m_iGusts
            node_eye_diameter[i] = self.__m_nodeWind[i].m_iEyeDiameter
            node_max_sea[i] = self.__m_nodeWind[i].m_iMaxSeas
            node_initials[i] = bytes(self.__m_nodeWind[i].m_sInitials, 'utf-8')
            node_storm_dir[i] = self.__m_nodeWind[i].m_iStormDirection
            node_storm_speed[i] = self.__m_nodeWind[i].m_iStormSpeed
            node_storm_name[i] = bytes(self.__m_nodeWind[i].m_sStormName, 'utf-8')
            j = 0
            for w in range(WIND_CARDS_enum.WIND_34.value, WIND_CARDS_enum.WIND_END.value):
                isotach_defined_dset[w][i] = 1 if self.__m_nodeWind[i].m_radii[w].m_defined else 0
                for q in range(QUADRANT_enum.QUADRANT_1.value, QUADRANT_enum.QUADRANT_END.value):
                    rad_dset[j][i] = self.__m_nodeWind[i].m_radii[w].m_data[q].m_radius
                    if is_asymm:
                        rad_max_dset[j][i] = self.__m_nodeWind[i].m_radii[w].m_data[q].m_rMax
                        rad_holland_b_dset[j][i] = self.__m_nodeWind[i].m_radii[w].m_data[q].m_hollandB
                        rad_v_max_dset[j][i] = self.__m_nodeWind[i].m_radii[w].m_data[q].m_vMax
                        if self.__m_nodeWind[i].m_radii[w].m_data[q].m_on:
                            rad_on_dset[j][i] = 1
                        else:
                            rad_on_dset[j][i] = 0
                    j = j + 1
            if is_symm:
                node_depth[i] = self.__m_nodeWind[i].m_eDepth.value
                node_wave_height[i] = self.__m_nodeWind[i].m_iWaveHeight
                node_sea_radius_code[i] = self.__m_nodeWind[i].m_eSeasRadiusCode.value
                node_sea_1[i] = self.__m_nodeWind[i].m_iSeas1
                node_sea_2[i] = self.__m_nodeWind[i].m_iSeas2
                node_sea_3[i] = self.__m_nodeWind[i].m_iSeas3
                node_sea_4[i] = self.__m_nodeWind[i].m_iSeas4
            else:
                node_holland_b[i] = self.__m_nodeWind[i].m_hollandB
                if is_simple_asymm:
                    node_rhoa[i] = self.__m_nodeWind[i].rhoa
                    node_km[i] = self.__m_nodeWind[i].km
                    node_deltafm[i] = self.__m_nodeWind[i].deltafm
                    node_wbgx[i] = self.__m_nodeWind[i].wbgx
                    node_wbgy[i] = self.__m_nodeWind[i].wbgy
        f.close()

    def ReadDump(self, file_name):  # noqa: N802,C901
        """Read wind coverage geometry and attributes from an H5 file written by XMS.

        Args:
            file_name (str): Path to the file to read
        """
        self.m_hasRead = True
        # read activity
        f = h5py.File(file_name, 'r')
        cov_names = f['Map Data'].keys()
        if not cov_names:
            f.close()
            return
        the_cov_name = None
        for c_name in cov_names:
            the_cov_name = c_name
            break
        att_group = f['Map Data/Coverage1/Attributes']
        node_group = f['Map Data/Coverage1/Attributes/Nodes']

        wind_model = att_group['WindModel']
        basin = att_group['Basin']
        sub_region = att_group['Subregion']
        cyc_num = att_group['CycloneNumber']
        self.__m_windModel = int(wind_model[0])
        self.__m_basin = int(basin[0])
        self.__m_subregion = int(sub_region[0])
        self.__m_cycloneNumber = cyc_num[0]

        node_id = node_group['Id']
        node_date = node_group['Date']
        node_tech_num = node_group['TechNumOrMinutes']
        node_tech = node_group['Tech']
        node_v_max = node_group['VMax']
        node_min_sea = node_group['MinSeaLevelPressure']
        node_level = node_group['LevelOfDevelopment']
        node_wind_radius_code = node_group['WindRadiusCode']
        node_pressure = node_group['Pressure']
        node_pressure_radius = node_group['PressureRadius']
        node_max_wind = node_group['MaxWindRadius']
        node_gusts = node_group['Gusts']
        node_eye_diameter = node_group['EyeDiameter']
        node_max_sea = node_group['MaxSeas']
        node_initials = node_group['Initials']
        node_storm_dir = node_group['StormDirection']
        node_storm_speed = node_group['StormSpeed']
        node_storm_name = node_group['StormName']

        # radii
        iso34_defined = node_group['Radii34/IsotachDefined']
        radius34_1 = node_group['Radii34/Radius1']
        radius34_2 = node_group['Radii34/Radius2']
        radius34_3 = node_group['Radii34/Radius3']
        radius34_4 = node_group['Radii34/Radius4']

        iso50_defined = node_group['Radii50/IsotachDefined']
        radius50_1 = node_group['Radii50/Radius1']
        radius50_2 = node_group['Radii50/Radius2']
        radius50_3 = node_group['Radii50/Radius3']
        radius50_4 = node_group['Radii50/Radius4']

        iso64_defined = node_group['Radii64/IsotachDefined']
        radius64_1 = node_group['Radii64/Radius1']
        radius64_2 = node_group['Radii64/Radius2']
        radius64_3 = node_group['Radii64/Radius3']
        radius64_4 = node_group['Radii64/Radius4']

        iso100_defined = node_group['Radii100/IsotachDefined']
        radius100_1 = node_group['Radii100/Radius1']
        radius100_2 = node_group['Radii100/Radius2']
        radius100_3 = node_group['Radii100/Radius3']
        radius100_4 = node_group['Radii100/Radius4']

        isotachs_defined = {
            WIND_CARDS_enum.WIND_34.value: iso34_defined,
            WIND_CARDS_enum.WIND_50.value: iso50_defined,
            WIND_CARDS_enum.WIND_64.value: iso64_defined,
            WIND_CARDS_enum.WIND_100.value: iso100_defined,
        }

        wind_card_idx = 0
        quad_idx = 1
        r_group_idx = 2
        rad_group_list = [
            (WIND_CARDS_enum.WIND_34.value, QUADRANT_enum.QUADRANT_1.value, radius34_1),
            (WIND_CARDS_enum.WIND_34.value, QUADRANT_enum.QUADRANT_2.value, radius34_2),
            (WIND_CARDS_enum.WIND_34.value, QUADRANT_enum.QUADRANT_3.value, radius34_3),
            (WIND_CARDS_enum.WIND_34.value, QUADRANT_enum.QUADRANT_4.value, radius34_4),
            (WIND_CARDS_enum.WIND_50.value, QUADRANT_enum.QUADRANT_1.value, radius50_1),
            (WIND_CARDS_enum.WIND_50.value, QUADRANT_enum.QUADRANT_2.value, radius50_2),
            (WIND_CARDS_enum.WIND_50.value, QUADRANT_enum.QUADRANT_3.value, radius50_3),
            (WIND_CARDS_enum.WIND_50.value, QUADRANT_enum.QUADRANT_4.value, radius50_4),
            (WIND_CARDS_enum.WIND_64.value, QUADRANT_enum.QUADRANT_1.value, radius64_1),
            (WIND_CARDS_enum.WIND_64.value, QUADRANT_enum.QUADRANT_2.value, radius64_2),
            (WIND_CARDS_enum.WIND_64.value, QUADRANT_enum.QUADRANT_3.value, radius64_3),
            (WIND_CARDS_enum.WIND_64.value, QUADRANT_enum.QUADRANT_4.value, radius64_4),
            (WIND_CARDS_enum.WIND_100.value, QUADRANT_enum.QUADRANT_1.value, radius100_1),
            (WIND_CARDS_enum.WIND_100.value, QUADRANT_enum.QUADRANT_2.value, radius100_2),
            (WIND_CARDS_enum.WIND_100.value, QUADRANT_enum.QUADRANT_3.value, radius100_3),
            (WIND_CARDS_enum.WIND_100.value, QUADRANT_enum.QUADRANT_4.value, radius100_4)
        ]
        rad_dset_list = []
        rad_dset_on_list = []
        rad_dset_r_max_list = []
        rad_dset_holland_b_list = []
        rad_dset_v_max_list = []
        for r_group in rad_group_list:
            rad_dset_list.append((r_group[wind_card_idx], r_group[quad_idx], r_group[r_group_idx]['Radius']))

        # dependant on wind model
        node_depth = None
        node_wave_height = None
        node_sea_radius_code = None
        node_sea_1 = None
        node_sea_2 = None
        node_sea_3 = None
        node_sea_4 = None
        node_holland_b = None
        node_rhoa = None
        node_km = None
        node_deltafm = None
        node_wbgx = None
        node_wbgy = None
        if self.__m_windModel == WindModel_enum.WM_HOLLAND_SYMMETRIC.value:
            node_depth = node_group['Depth']  # only sym
            node_wave_height = node_group['WaveHeight']  # only sym
            node_sea_radius_code = node_group['SeasRadiusCode']  # only sym
            node_sea_1 = node_group['Seas1']  # only sym
            node_sea_2 = node_group['Seas2']  # only sym
            node_sea_3 = node_group['Seas3']  # only sym
            node_sea_4 = node_group['Seas4']  # only sym
        else:
            node_holland_b = node_group['HollandB']  # only asym
            if self.__m_windModel == WindModel_enum.WM_HOLLAND_SIMPLIFIED_ASYMMETRIC.value:
                node_rhoa = node_group['Rhoa']
                node_km = node_group['Km']
                node_deltafm = node_group['DeltaFm']
                node_wbgx = node_group['Wbgx']
                node_wbgy = node_group['Wbgy']
            else:
                for r_group in rad_group_list:
                    rad_dset_on_list.append((r_group[wind_card_idx], r_group[quad_idx], r_group[r_group_idx]['On']))
                    rad_dset_r_max_list.append(
                        (r_group[wind_card_idx], r_group[quad_idx], r_group[r_group_idx]['RMax'])
                    )
                    if r_group[r_group_idx].__contains__("HollandB_Quadrant"):
                        rad_dset_holland_b_list.append(
                            (r_group[wind_card_idx], r_group[quad_idx], r_group[r_group_idx]['HollandB_Quadrant'])
                        )
                    if r_group[r_group_idx].__contains__("VMax_Quadrant"):
                        rad_dset_v_max_list.append(
                            (r_group[wind_card_idx], r_group[quad_idx], r_group[r_group_idx]['VMax_Quadrant'])
                        )

        node_length = node_id.len()
        for i in range(node_length):
            the_node = WindNode()
            the_node.m_id = node_id[i]
            the_node.m_date = node_date[i]
            the_node.m_iTechNumOrMinutes = node_tech_num[i]
            the_node.m_sTech = node_tech[i].decode('UTF-8')
            the_node.m_iVMax = node_v_max[i]
            the_node.m_iMinSeaLevelPressure = node_min_sea[i]
            the_node.m_eLevelOfDevelopment = int(node_level[i])
            the_node.m_eWindRadiusCode = int(node_wind_radius_code[i])
            the_node.m_iPressure = node_pressure[i]
            the_node.m_iPressureRadius = node_pressure_radius[i]
            the_node.m_iMaxWindRadius = node_max_wind[i]
            the_node.m_iGusts = node_gusts[i]
            the_node.m_iEyeDiameter = node_eye_diameter[i]
            the_node.m_iMaxSeas = node_max_sea[i]
            the_node.m_sInitials = node_initials[i].decode('UTF-8')
            the_node.m_iStormDirection = node_storm_dir[i]
            the_node.m_iStormSpeed = node_storm_speed[i]
            the_node.m_sStormName = node_storm_name[i].decode('UTF-8')
            for rad in rad_dset_list:
                the_node.m_radii[rad[wind_card_idx]].m_data[rad[quad_idx]].m_radius = rad[r_group_idx][i]
            if self.__m_windModel == WindModel_enum.WM_HOLLAND_SYMMETRIC.value:
                the_node.m_eDepth = node_depth[i]
                the_node.m_iWaveHeight = node_wave_height[i]
                the_node.m_eSeasRadiusCode = node_sea_radius_code[i]
                the_node.m_iSeas1 = node_sea_1[i]
                the_node.m_iSeas2 = node_sea_2[i]
                the_node.m_iSeas3 = node_sea_3[i]
                the_node.m_iSeas4 = node_sea_4[i]
            else:
                the_node.m_hollandB = node_holland_b[i]
                if self.__m_windModel == WindModel_enum.WM_HOLLAND_SIMPLIFIED_ASYMMETRIC.value:
                    the_node.rhoa = node_rhoa[i]
                    the_node.km = node_km[i]
                    the_node.deltafm = node_deltafm[i]
                    the_node.wbgx = node_wbgx[i]
                    the_node.wbgy = node_wbgy[i]
                else:
                    for wind_type, defined_dset in isotachs_defined.items():
                        the_node.m_radii[wind_type].m_defined = True if defined_dset[i] else False
                    for rad_on, rad_max in zip(rad_dset_on_list, rad_dset_r_max_list):
                        the_node.m_radii[rad_on[wind_card_idx]].m_data[rad_on[quad_idx]].m_on = rad_on[r_group_idx][i]
                        the_node.m_radii[rad_max[wind_card_idx]].m_data[rad_max[quad_idx]].m_rMax = \
                            rad_max[r_group_idx][i]
                    for rad_h, rad_v in zip(rad_dset_holland_b_list, rad_dset_v_max_list):
                        the_node.m_radii[rad_h[wind_card_idx]].m_data[rad_h[quad_idx]].m_hollandB = \
                            rad_h[r_group_idx][i]
                        the_node.m_radii[rad_v[wind_card_idx]].m_data[rad_v[quad_idx]].m_vMax = rad_v[r_group_idx][i]
            self.__m_nodeWind.append(the_node)

        f.close()
        # read geometry
        self.m_cov = Coverage(file_name, "/Map Data/" + the_cov_name)
        self.m_cov.get_points(FilterLocation.LOC_NONE)  # force geometry to load from H5

    def Copy(self):  # noqa: N802
        """Return a reference to this object."""
        return self

    def GetDumpType(self):  # noqa: N802
        """Get the XMS coverage dump type."""
        return "xms.coverage.windCoverage"


def ReadDumpWithObject(file_name):  # noqa: N802
    """Read a wind coverage dump file.

    Args:
        file_name (str): Filepath to the dumped coverage to read

    Returns:
        WindCoverage: The loaded wind coverage
    """
    wind_dump = WindCoverage(file_name)
    return wind_dump
