"""XySeries class."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime, timedelta
import logging
import math
import shlex
import sys
from typing import Sequence

# 2. Third party modules
from h5py import Group

# 3. Aquaveo modules

# 4. Local modules
from xms.coverage import time_util

# Constants
ID_WORD = 1
COUNT_WORD = 2

X_VALS = 'XVals'
Y_VALS = 'YVals'
NAME = 'Name'
Id = 'Id'
DATE_TIME = 'DateTime'
X_TITLE = 'XTitle'
Y_TITLE = 'YTitle'

XYS = 'XYS'


class XySeries:
    """Class to store xy values."""
    def __init__(
        self,
        x: Sequence[int | float | datetime] | None = None,
        y: Sequence[int | float | datetime] | None = None,
        name: str = '',
        series_id: int = 1,
        use_dates_times: bool = False,
        x_title: str = '',
        y_title: str = ''
    ):
        """Initializes the class.

        Args:
            x: X values.
            y: Y values.
            name: The name.
            series_id: The ID number.
            use_dates_times: Flag indicating x values should be treated as dates/times (even if they're floats).
            x_title: Title of the x values.
            y_title: Title of the y values.
        """
        if x is not None and y is not None and len(x) != len(y):
            raise RuntimeError('XySeries x and y lists must be the same length.')
        self._x = x  # Can be any list-like type
        self._y = y  # Can be any list-like type
        self._name = name
        self._series_id = series_id  # Because 'id' is already taken by python
        self._use_dates_times = use_dates_times
        self._x_title = x_title
        self._y_title = y_title

    @property
    def name(self) -> str:
        """Returns the name."""
        return self._name

    @name.setter
    def name(self, value: str):
        """Sets the name."""
        self._name = value

    @property
    def series_id(self) -> int:
        """Returns the _series_id."""
        return self._series_id

    @series_id.setter
    def series_id(self, value: int):
        """Sets the _series_id."""
        self._series_id = value

    @property
    def x(self) -> list[int | float]:
        """Returns the list of x."""
        return self._x

    @x.setter
    def x(self, value: list[int | float]):
        """Sets the list of x."""
        self._x = value

    @property
    def y(self) -> list[int | float]:
        """Returns the list of y."""
        return self._y

    @y.setter
    def y(self, value: list[int | float]):
        """Sets the list of y."""
        self._y = value

    @property
    def use_dates_times(self) -> bool:
        """Returns _use_dates_times."""
        return self._use_dates_times

    @use_dates_times.setter
    def use_dates_times(self, value: bool):
        """Sets _use_dates_times."""
        self._use_dates_times = value

    @property
    def x_title(self) -> str:
        """Returns _x_title."""
        return self._x_title

    @x_title.setter
    def x_title(self, value: str):
        """Sets _x_title."""
        self._x_title = value

    @property
    def y_title(self) -> str:
        """Returns _y_title."""
        return self._y_title

    @y_title.setter
    def y_title(self, value: str):
        """Sets _y_title."""
        self._y_title = value

    def count(self) -> int:
        """Returns the number of x (assumes y is same len as x)."""
        return len(self._x)

    def _compare_lists(self, list1, list2):
        """Returns True if the lists are equal element-wise."""
        if len(list1) != len(list2):
            return False

        for item1, item2 in zip(list1, list2):
            if isinstance(item1, datetime):
                if item1 != item2:
                    return False
            elif not math.isclose(item1, item2):
                return False
        return True

    def __eq__(self, other: 'XySeries'):
        """Returns True if x, y, and _use_dates_times of the XySeries are equal (doesn't compare name or ID)."""
        if not isinstance(other, XySeries):
            # don't attempt to compare against unrelated types
            return NotImplemented

        # Don't compare names and IDs, just x, y and _use_dates_times
        return (
            self._compare_lists(self.x, other.x) and self._compare_lists(self.y, other.y) and  # noqa: W504
            self._use_dates_times == other._use_dates_times
        )  # noqa: W503 (line break)

    def __repr__(self):
        """Unambiguous representation of this XySeries object."""
        rep = (
            f'XySeries(x={repr(self.x)}, y={repr(self.y)}, name={repr(self._name)},'
            f' series_id={repr(self._series_id)}, use_dates_times={repr(self._use_dates_times)},'
            f' x_title={repr(self._x_title)}, y_title={repr(self._y_title)})'
        )
        return rep

    def write(self, file, card: str = 'XYS', xy1_format: bool = False) -> None:
        """Writes the series to the file which must already be open.

        Args:
            file: The file.
            card: 'XYS', 'XY1' etc.
            xy1_format: True to write all the extra numbers that the XY1 format used (regardless of card).
        """
        if xy1_format:
            file.write(f'{card} {self.series_id} {self.count()} 0 0 0 0.0 "{self.name}"\n')
        else:
            file.write(f'{card} {self.series_id} {self.count()} "{self.name}"\n')

        if self.count() == 0:
            return

        # Get x list as either numbers or date strings
        xs = self.x if not self._use_dates_times else [x.strftime('%Y-%m-%d %H:%M:%S') for x in self.x]
        if self._use_dates_times:
            for x, y in zip(xs, self.y):
                file.write(f'"{x}" {y}\n')
        else:
            for x, y in zip(xs, self.y):
                file.write(f'{x} {y}\n')

    @staticmethod
    def from_file(
        file,
        line: str,
        line_number: int,
        date_times_to_floats: bool = False,
        start_date_time: datetime | None = None,
        time_units: str = None
    ):
        """Returns a XySeries read from file which must already be open.

        Args:
            file: The file we are reading from.
            line: Line from the file containing the 'XYS' or 'XY1' card.
            line_number: The line number in the file for line (used with error messages).
            date_times_to_floats: If true and start_date_time and time_units are given, converts date/time
             strings to relative offsets from the start_date_time.
            start_date_time: Starting date/time
            time_units: Time units: 'UNKNOWN', 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'
        """
        # Read id, count, and name
        words = shlex.split(line, posix="win" not in sys.platform)  # Use shlex to handle quoted strings
        word_count = len(words)
        id_ = int(words[ID_WORD]) if word_count > ID_WORD else -1
        count = int(words[COUNT_WORD]) if word_count > COUNT_WORD else 0
        name_word_index = word_count - 1  # The name is always the last thing
        # Strip all types of quotes we might find
        name = words[name_word_index].strip('"\'‘’“”') if word_count > COUNT_WORD + 1 else ''

        # Read xy values
        x_list = []
        y_list = []
        for _ in range(count):
            line = next(file)
            line_number += 1
            words = shlex.split(line, posix="win" not in sys.platform)  # Use shlex to handle quoted strings
            if len(words) > 1:
                x_word = words[0].strip('"\'‘’“”')
                time_value = time_util.time_value_from_string(x_word, date_times_to_floats, start_date_time, time_units)
                if time_value is None:
                    logger = logging.getLogger('xms.coverage')
                    logger.error(f'Could not read time value "{x_word}" on line {line_number}.')
                x_list.append(time_value)
                y_list.append(float(words[1]))

        xy_series = XySeries(x=x_list, y=y_list, name=name, series_id=id_)
        return xy_series

    @staticmethod
    def from_h5(curve_group: Group):
        """Reads the xy series from the h5 file and stores it in the dict.

        Args:
            curve_group: H5 group for the xy series.
        """
        x_vals = curve_group[X_VALS][:]
        y_vals = curve_group[Y_VALS][:]
        name = curve_group[NAME][0].astype(str)
        id_ = curve_group[Id][0]
        use_dates_times = curve_group[DATE_TIME][0]
        x_title = curve_group[X_TITLE][0].astype(str) if X_TITLE in curve_group else ''
        y_title = curve_group[Y_TITLE][0].astype(str) if Y_TITLE in curve_group else ''
        if use_dates_times:
            x_vals = _mfc_date_numbers_to_datetimes(x_vals)
        return XySeries(
            x=x_vals,
            y=y_vals,
            name=name,
            series_id=id_,
            use_dates_times=use_dates_times,
            x_title=x_title,
            y_title=y_title
        )

    def to_h5(self, curve_group: Group):
        """Writes the xy series to the h5 file.

        Args:
            curve_group: H5 group for the xy series.
        """
        xs = _datetimes_to_mfc_date_numbers(self._x) if self._use_dates_times else self.x
        curve_group.create_dataset(X_VALS, (len(xs), ), data=xs)
        curve_group.create_dataset(Y_VALS, (len(self.y), ), data=self.y)
        curve_group.create_dataset(NAME, (1, ), dtype=f'S{len(self.name) + 1}', data=self.name)
        curve_group.create_dataset(Id, (1, ), dtype='i', data=self.series_id)
        curve_group.create_dataset(DATE_TIME, (1, ), dtype='i', data=self.use_dates_times)
        curve_group.create_dataset(X_TITLE, (1, ), dtype=f'S{len(self.x_title) + 1}', data=self.x_title)
        curve_group.create_dataset(Y_TITLE, (1, ), dtype=f'S{len(self.y_title) + 1}', data=self.y_title)


def _mfc_date_numbers_to_datetimes(x_vals: list[float]) -> list[datetime]:
    """Opposite of _datetimes_to_mfc_date_numbers."""
    base_date = datetime(year=1899, month=12, day=30, hour=0)
    new_x = []
    for x in x_vals:
        delta = timedelta(days=x)
        new_date = base_date + delta
        new_x.append(new_date)
    return new_x


def _datetimes_to_mfc_date_numbers(x_vals: list[datetime]) -> list[float]:
    """Converts the x datetime values to an MFC DATE object number.

    The DATE type is implemented using an 8-byte floating-point number. Days are represented by whole number
    increments starting with 30 December 1899, midnight as time zero. Hour values are expressed as the absolute
    value of the fractional part of the number.
    """
    base_date = datetime(year=1899, month=12, day=30, hour=0)
    return [(x - base_date).total_seconds() / 86400 for x in x_vals]
