"""Module for exporting an RSM simulation with feedback."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.cstorm.dmi.xms_data import XmsData
from xms.cstorm.file_io import util
from xms.cstorm.file_io.control_writer import ControlWriter

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting an RSM simulation in a worker thread."""
    processing_finished = Signal()

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = util.get_logger()
        self._xms_data = XmsData()

    def _export_simulation(self):
        """Export a simulation."""
        self._logger.info('Establishing communication with SMS.')
        writer = ControlWriter(self._xms_data)
        writer.write()

    def _do_work(self):
        """Thread worker method."""
        try:
            self._export_simulation()
        except Exception:
            self._logger.exception('Error exporting simulation.')


def export_simulation_with_feedback():
    """Export a simulation with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()
    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    worker = ExportSimulationRunner(parent)
    # from xms.guipy import debug_pause
    # debug_pause()
    # worker.do_work()
    # return

    display_text = {
        'title': 'CSTORM Export Simulation',
        'working_prompt': 'Exporting CSTORM simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.cstorm', worker=worker, parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if parent is not None:
            # Bring top-level Python window to foreground. Do this before send() so we immediately give focus
            # back to SMS.
            win_gui.raise_main_xms_window(main_hwnd)
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
