"""CSTORM control file reader."""
# 1. Standard python modules
import os
import uuid

# 2. Third party modules
from PySide2.QtCore import QDate, QDateTime, QTime

# 3. Aquaveo modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort15_reader import Fort15Reader
from xms.core.filesystem import filesystem as xfs
from xms.data_objects.parameters import Component, Simulation
from xms.stwave.file_io.sim_reader import SimReader

# 4. Local modules
from xms.cstorm.data.sim_data import SimData
from xms.cstorm.file_io import util

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


def datetime_to_qdatetime(dt_literal):
    """Convert a Python datetime object to a QDateTime.

    Args:
        dt_literal (:obj:`datetime.datetime`): Datetime to convert

    Returns:
        (:obj:`QDateTime`): See description
    """
    return QDateTime(
        QDate(dt_literal.year, dt_literal.month, dt_literal.day),
        QTime(dt_literal.hour, dt_literal.minute, dt_literal.second)
    )


# Cards that we care about
ADCIRC_CARD = 'adcgrid'
STWAVE_CARD = 'simfile'
WSID_CARD = 'wsid'
CONFIG_CARDS = {
    ADCIRC_CARD,
    STWAVE_CARD,
    WSID_CARD,
}
MIN_LEN_CARD_SPLIT = 2


class ControlReader:
    """Reader class for the CSTORM control file."""
    def __init__(self, filename, query):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the CSTORM simulation file to read
            query (:obj:`Query`): XMS interprocess communicator
        """
        self._filename = filename
        self._query = query
        self._logger = util.get_logger()
        self._lineno = 1
        self._adcirc_ctrl = ''
        self._stwave_sims = []
        self._data = {}  # Model control data
        self._sim_uuid = str(uuid.uuid4())  # For linking child simulations
        self._simulation = None
        self._sim_data = None
        self._comp_dir = ''

    def _fill_model_control_data(self):
        """Populate the model control values from inputs."""
        gm = self._sim_data.generic_model
        gp = gm.global_parameters
        grp = gp.group('model_control')
        for k, v in self._data.items():
            pk = k.split()
            # Handle cards that are on the same line as the namelist name
            if pk and len(pk) > 1:
                k = pk[1].strip()
            # TODO: This only works for comboboxes because that is all we have right now.
            opts = grp.parameter(k).options
            grp.parameter(k).value = opts[int(float(v)) - 1]
        self._sim_data.info.attrs['OPTIONS'] = gp.extract_values()
        self._sim_data.commit()

    def _find_adcirc_ctl(self, grd_file):
        """Find the ADCIRC control file.

        Args:
            grd_file (:obj:`str`): Absolute path to the ADCIRC grid file
        """
        # First check if there is a control file (fort.15) in the same directory with the same basename.
        ctl_file = os.path.join(os.path.dirname(grd_file), f'{os.path.splitext(grd_file)[0]}.ctl')
        if os.path.isfile(ctl_file):
            self._adcirc_ctrl = ctl_file
        else:
            # Now check if there is the ADCIRC hardcoded control file.
            ctl_file = os.path.join(os.path.dirname(grd_file), 'fort.15')
            if os.path.isfile(ctl_file):
                self._adcirc_ctrl = ctl_file
            else:
                self._logger.error(f'Unable to find ADCIRC control file referenced on line {self._lineno}')
        self._alt_filenames = {14: grd_file}

    def _resolve_input_filename(self, card, value):
        """Get an absolute path to an input file referenced in the CSTORM simulation file.

        Args:
            card (:obj:`str`): The card from the sim file
            value (:obj:`str`): The filepath to resolve
        """
        abs_path = xfs.resolve_relative_path(os.path.dirname(self._filename), value)
        if not os.path.isfile(abs_path):
            self._logger.error(f'Unable to find input file {abs_path} referenced on line {self._lineno}')
        if ADCIRC_CARD in card:
            self._find_adcirc_ctl(abs_path)
        else:  # card == STWAVE_CARD
            self._stwave_sims.append(abs_path)

    def _parse_line(self, line):
        """Parse a line from the CSTORM simulation file.

        Args:
            line (:obj:`str`): The unparsed line from the file
        """
        # Skip comment and empty lines
        if line.startswith('#') or not line:
            return
        # Split on the card/value separator
        line = line.split('=')
        if len(line) < MIN_LEN_CARD_SPLIT:
            return
        # Check if this is a card we care about
        card = line[0].strip().lower()
        if not any(cc in card for cc in CONFIG_CARDS):
            return
        # Get read of quote and comma characters from the end of the value
        value = line[1].strip().strip('",')
        # Get an absolute path to the ADCIRC/STWAVE input file
        if any(cc in card for cc in [ADCIRC_CARD, STWAVE_CARD]):
            self._resolve_input_filename(card, value)
        else:
            self._data[card] = value

    def _read_config(self):
        """Reads the config file for the CSTORM model."""
        self._logger.info('$XMS_BOLD$Parsing CSTORM config file...')
        with open(self._filename, 'r') as f:
            lines = f.readlines()
        for line in lines:
            self._parse_line(line.strip())
            self._lineno += 1

    def _initialize_sim(self):
        """Creates a default CSTORM simulation."""
        # Create the CSTORM simulation data_object
        self._simulation = Simulation(name=os.path.basename(self._filename), model='CSTORM', sim_uuid=self._sim_uuid)
        # Create the simulation's hidden component
        self._comp_dir = os.path.join(self._query.xms_temp_directory, 'Components')
        comp_uuid = str(uuid.uuid4())
        comp_dir = os.path.join(self._comp_dir, comp_uuid)
        os.makedirs(comp_dir, exist_ok=True)
        sim_mainfile = os.path.join(comp_dir, 'sim_comp.nc')
        self._sim_data = SimData(sim_mainfile)  # This will create a default model control
        comp = Component(name='hidden', comp_uuid=comp_uuid, main_file=sim_mainfile)
        comp.set_unique_name_and_model_name('SimComponent', 'CSTORM')
        self._query.add_simulation(self._simulation, [comp])

    def _get_adcirc_global_time(self):
        """Get the global time for the ADCIRC simulation.

        Returns:
            (:obj:`QDateTime`): See description
        """
        global_time = self._query.global_time
        if global_time is not None:
            global_time = datetime_to_qdatetime(global_time)
        else:
            global_time = QDateTime.currentDateTime()
        return global_time

    def read(self):
        """Writes the CSTORM input files."""
        self._initialize_sim()
        self._read_config()
        self._fill_model_control_data()
        # Read in as much data as we can even if we couldn't find the ADCIRC simulation.
        if os.path.isfile(self._adcirc_ctrl):
            XmLog().instance.set_logger(self._logger)
            self._logger.info('$XMS_BOLD$Reading ADCIRC model...')
            reader = Fort15Reader(
                filename=self._adcirc_ctrl,
                query=self._query,
                comp_dir=self._comp_dir,
                global_time=self._get_adcirc_global_time(),
                sim_name=os.path.basename(self._adcirc_ctrl)
            )
            reader.read()
            reader.add_built_data(send=False)
            self._query.link_item(self._sim_uuid, reader.sim_uuid)
        # Now read all the linked STWAVE simulations
        for i, sim_file in enumerate(self._stwave_sims):
            self._logger.info(f'$XMS_BOLD$Reading STWAVE model {i + 1} of {len(self._stwave_sims)}...')
            if not os.path.isfile(sim_file):
                self._logger.info(f'Uable to find STWAVE simulation file: {sim_file}')
                continue
            reader = SimReader(sim_file, self._query)
            reader._logger = self._logger
            reader.read()
            self._query.link_item(self._sim_uuid, reader.sim_uuid)
