"""CSTORM control file writer."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.cstorm.file_io import util
from xms.cstorm.file_io.adcirc_writer import AdcircWriter
from xms.cstorm.file_io.stwave_writer import StwaveWriter

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class ControlWriter:
    """Writer class for the RSM control file."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._filename = 'mf_config.in'
        self._logger = util.get_logger()
        self._xms_data = xms_data
        sd = self._xms_data.sim_data
        gp = sd.generic_model.global_parameters
        self._grp = gp.group('model_control')
        self._data = {'simfile': [], 'stwprocs': [], 'st_times': []}
        self._adc_writer = AdcircWriter(self._logger, self._xms_data, self._data)
        self._stwave_writer = StwaveWriter(self._logger, self._xms_data, self._data)

    def write(self):
        """Writes the CSTORM input files."""
        self._get_data()
        self._adc_writer.write(self._data['wsid'])
        self._stwave_writer.write()
        self._write_config()

    def _get_output_str(self, var):
        """Gets the output string for a variable.

        Args:
            var (:obj:`str`): name of a variable

        Returns:
            (:obj:`str`): the output string
        """
        rval = f'  {var} = '
        if var in self._data:
            rval = rval + f'{self._data[var]}'
        rval = rval + '\n'
        return rval

    def _get_data(self):
        """Get the data to write to the config file."""
        self._data['template'] = self._grp.parameter('template').value
        opts = self._grp.parameter('wsid').options
        self._data['wsid'] = opts.index(self._grp.parameter('wsid').value) + 1
        self._data['geo_and_stpl_coord'] = '.true.'
        if self._xms_data.query.display_projection.coordinate_system != 'GEOGRAPHIC':
            self._data['geo_and_stpl_coord'] = '.false.'

    def _write_config(self):
        """Writes the config file for the CSTORM model."""
        with open('mf_config.in', 'w') as f:
            f.write('#Adcirc Grid\n')
            f.write('&adc_def\n')
            f.write(self._get_output_str('adcgrid'))
            f.write(self._get_output_str('adcprocs'))
            f.write(self._get_output_str('writerprocs'))
            if self._data['template']:  # We are writing a template config file
                self._data['adcstart'] = '%ADCSTART%'
                self._data['adcfinish'] = '%ADCFINISH%'

            f.write(self._get_output_str('adcstart'))
            f.write(self._get_output_str('adcfinish'))
            f.write('/\n')
            f.write('# Wave Service\n')
            f.write('&service\n')
            f.write(self._get_output_str('wsid'))
            f.write(self._get_output_str('stwgrids'))
            if self._data['template']:  # We are writing a template config file
                self._data['stwstart'] = '%STWSTART%'
                self._data['stwfinish'] = '%STWFINISH%'
                self._data['stwtiminc'] = '%STWTIMINC%'

            f.write(self._get_output_str('stwstart'))
            f.write(self._get_output_str('stwfinish'))
            f.write(self._get_output_str('stwtiminc'))
            f.write(self._get_output_str('geo_and_stpl_coord'))
            f.write('/\n')
            for idx in range(len(self._data['simfile'])):
                simfile = self._data['simfile'][idx]
                stwprocs = self._data['stwprocs'][idx]
                f.write(f'# Stwave Grid {idx + 1}\n')
                f.write('&stw_def\n')
                f.write(f'  simfile = "{simfile}"\n')
                f.write(f'  stwprocs = {stwprocs}\n')
                f.write('/\n')
