"""Python wrapping for xms._data_objects.parameters.Polygon."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import Polygon as CPolygon
from xms.data_objects.parameters.spatial.Arc import Arc as PyArc
from xms.data_objects.parameters.spatial.Point import Point as PyPoint


class Polygon:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Polygon objects."""
    def __init__(self, feature_id=None, instance=None):
        """Construct the wrapper.

        Args:
            feature_id (int): Polygon's XMS feature id
            instance (CPolygon): The C++ Polygon object to wrap
        """
        if instance:
            self._instance = instance
        else:
            self._instance = CPolygon()

        if feature_id is not None:
            self._instance.SetId(feature_id)

    @property
    def id(self):
        """Returns the polygon's XMS feature id."""
        return self._instance.GetId()

    @id.setter
    def id(self, poly_id):
        """Sets the polygon's XMS feature id.

        Args:
            poly_id (int): Id to assign the polygon
        """
        self._instance.SetId(poly_id)

    @property
    def arcs(self):
        """Returns the polygon's outer arcs."""
        return [PyArc(instance=arc) for arc in self._instance.GetArcs()]

    @property
    def arc_directions(self):
        """Returns the directions of the polygon's outer arcs."""
        return self._instance.GetArcDirections()

    @property
    def point_ids(self):
        """Returns the polygon's point IDs as a list of integers."""
        return self._instance.GetPointIds()

    @property
    def centroid(self):
        """Returns the polygon's centroid as a xms.data_objects.parameters.Point."""
        centroid = self._instance.GetCentroid()
        if centroid is not None:
            # Create a pure Python wrapper for the centroid if there is one defined.
            centroid = PyPoint(instance=centroid)
        return centroid

    @centroid.setter
    def centroid(self, point):
        """Sets the polygon's centroid.

        Args:
            point (PyPoint): The location of the centroid
        """
        self._instance.SetCentroid(point._instance)

    @property
    def interior_arcs(self):
        """Returns the polygon's interior arcs."""
        return [[PyArc(instance=arc) for arc in hole] for hole in self._instance.GetInteriorArcs()]

    @property
    def interior_arc_directions(self):
        """Returns the directions of the polygon's outer arcs."""
        return self._instance.GetInteriorArcDirections()

    def set_arcs(self, outer_arcs, directions=None):
        """Sets the polygon's outer arcs.

        Args:
            outer_arcs (list of PyArc): The polygon outer arcs
            directions (list of bool): If provided, must be parallel with arcs. True if polygon is on the left of the
                arc.
        """
        carcs = [arc._instance for arc in outer_arcs]
        self._instance.SetArcs(carcs, directions if directions else [])

    def get_points(self, location):
        """Returns the polygon's points at a specified location.

        Args:
            location (xms.data_objects.parameters.FilterLocation): The point locations to retrieve. Valid values:
                FilterLocation.PT_LOC_CORNER - arc start/end nodes (includes inside and outside arcs)
                FilterLocation.PT_LOC_MID - arc vertices (includes inside and outside arcs)
                FilterLocation.PT_LOC_CENTER - polygon centroid (may or may not be defined)
                FilterLocation.PT_LOC_ALL - all points of the polygon

                Note that the FilterLocation enum supports bitwise OR operations.
        Returns:
            list of xms.data_objects.parameters.Point: The polygon points at the specified location.
        """
        return [PyPoint(instance=pt) for pt in self._instance.GetPoints(location)]

    def set_points(self, points, vertice_count=0, last_is_repeat=False):
        """Sets the polygon's points.

        Args:
            points (list of PyPoint): The polygon points
            vertice_count (int): number of vertices
            last_is_repeat (bool): True if the start node is repeated
        """
        cpoints = [pt._instance for pt in points]
        self._instance.SetPoints(cpoints, vertice_count, last_is_repeat)

    def set_interior_arcs(self, interior_arcs, directions=None):
        """Sets the polygon's interior arcs.

        Args:
            interior_arcs (list of list of PyArc): The polygon interior arcs
            directions (list of list of bool): parallel with interior arcs; True if polygon is on the left of the arc
        """
        carcs = [[arc._instance for arc in hole] for hole in interior_arcs]
        self._instance.SetInteriorArcs(carcs, directions if directions else [])
