"""Table view widget for the dredging computation table."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import QAbstractButton, QAbstractItemView, QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts
from xms.ewn.gui.computation_table_filter_model import ComputationTableFilterModel
from xms.ewn.gui.computation_table_model import ComputationTableModel
from xms.ewn.gui.dredge_format_edit_field_delegate import EditFieldDelegate
from xms.ewn.gui.int_cbx_delegate import IntQxCbxDelegate
from xms.ewn.tools.dredge_calculator import rename_columns_for_io


def rename_columns_for_gui(df):
    """Rename the column names so they look pretty in the table header.

    Args:
        df (:obj:`pandas.DataFrame`): The DataFrame to rename columns of

    Returns:
        (:obj:`pandas.DataFrame`): See description
    """
    return df.rename(
        columns={
            'polygon_name': 'Name',
            'sediment_type': 'Type',
            'priority': 'Priority',
            'priority_percent': 'Priority %',
            'cut_fill_type': 'Cut/Fill Type',
            'volume_value': 'Value',
            'total_volume': 'Total Volume\nBased on Slope',
            'required_volume': 'Required\nVolume',
            'available_volume': 'Available\nVolume',
            'cut_volume': 'Cut\nVolume',
            'fill_volume': 'Fill\nVolume',
        }
    )


class ComputationTableWidget(BasicTableWidget):
    """Computation table widget class."""
    def __init__(self, parent, data_frame, comp_ids):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
            comp_ids (:obj:`dict`): Mapping of component id to list of feature ids with that
                component id (should be 1-1)
        """
        super().__init__(parent)
        self._comp_ids = comp_ids
        data_frame = self._rename_columns_and_rows_for_gui(data_frame)
        self.model = ComputationTableModel(data_frame, self)
        self.model.show_nan_as_blank = True
        self._filter_model = ComputationTableFilterModel(self)
        self._dbl_delegate = None
        self._percent_delegate = None
        self._sediment_type_delegate = None
        self._priority_delegate = None
        self._cut_fill_delegate = None
        self._dredge_result_delegate = None
        self.setup_ui()

    def _set_priority_strings(self):
        """Set the priority combobox strings based on the number of available type polygons."""
        type_counts = self.model.data_frame['Type'].value_counts().astype(np.int32)
        num_bins = type_counts.get(consts.SEDIMENT_TYPE_AVAILABLE_CUT, 0)
        num_bins += type_counts.get(consts.SEDIMENT_TYPE_AVAILABLE_FILL, 0)
        self._priority_delegate.set_strings([str(idx + 1) for idx in range(num_bins)])

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        # Double edit field delegates
        dbl_validator = QxDoubleValidator(parent=self)
        self._dbl_delegate = EditFieldValidator(dbl_validator, self)
        percent_validator = QxDoubleValidator(parent=self)
        percent_validator.setRange(0.0, 100.0)
        self._percent_delegate = EditFieldValidator(percent_validator, self)
        # Combobox delegates
        self._sediment_type_delegate = IntQxCbxDelegate(self)
        self._sediment_type_delegate.set_strings(list(consts.EWN_POLYGON_SEDIMENT_TYPES.values()))
        self._priority_delegate = IntQxCbxDelegate(self)
        self._set_priority_strings()
        self._cut_fill_delegate = IntQxCbxDelegate(self)
        self._cut_fill_delegate.set_strings(list(consts.EWN_CUT_FILL_TYPE.values()))
        self._dredge_result_delegate = EditFieldDelegate(self)
        delegates = {
            consts.COL_SEDIMENT_TYPE: self._sediment_type_delegate,
            consts.COL_PRIORITY: self._priority_delegate,
            consts.COL_CUT_FILL_TYPE: self._cut_fill_delegate,
            consts.COL_VOLUME_VALUE: self._dbl_delegate,
            consts.COL_PRIORITY_PERCENT: self._percent_delegate,
            consts.COL_TOTAL_VOLUME: self._dredge_result_delegate,
            consts.COL_REQUIRED_VOLUME: self._dredge_result_delegate,
            consts.COL_AVAILABLE_VOLUME: self._dredge_result_delegate,
            consts.COL_CUT_VOLUME: self._dredge_result_delegate,
            consts.COL_FILL_VOLUME: self._dredge_result_delegate,
        }
        super()._setup_ui(
            column_delegates=delegates, stretch_last_section=False, fixed_size=True, filter_model=self._filter_model
        )

        # Update the priority combobox options as the sediment type options change.
        self.model.dataChanged.connect(self.on_data_changed)

        # Edit like a spreadsheet
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

        # Set text for the vertical header column. Use the XMS feature polygon id.
        corner_btn = self.table_view.findChild(QAbstractButton)
        if corner_btn is not None:
            # Create a text label for the vertical header. This could just be a non-editable column in the table
            # I suppose, but this is just how the data gets represented using default base class behavior.
            layout = QVBoxLayout(corner_btn)
            layout.setContentsMargins(0, 0, 0, 0)
            header_lbl = QLabel('ID')
            header_lbl.setContentsMargins(0, 0, 0, 0)
            header_lbl.setAlignment(Qt.AlignCenter)
            layout.addWidget(header_lbl, alignment=Qt.AlignCenter)
            # Expand the vertical column so header text is visible.
            font_metrics = QFontMetrics(header_lbl.font())
            self.table_view.verticalHeader().setMinimumWidth(font_metrics.horizontalAdvance(header_lbl.text()) + 5)
            self.table_view.verticalHeader().setDefaultAlignment(Qt.AlignCenter)

    def _rename_columns_and_rows_for_gui(self, df):
        """Rename the column names so they look pretty in the table header.

        Args:
            df (:obj:`pandas.DataFrame`): The DataFrame to rename columns of

        Returns:
            (:obj:`pandas.DataFrame`): See description
        """
        data_frame = rename_columns_for_gui(df)
        new_index = pd.Series([xms_id for comp_id in data_frame.index for xms_id in self._comp_ids[comp_id]])
        data_frame.set_index(new_index, inplace=True)
        return data_frame

    def table_data_to_xarray(self):
        """Convert table DataFrame to xarray.Dataset.

        Returns:
            (:obj:`xarray.Dataset`): See description
        """
        data_frame = rename_columns_for_io(self.model.data_frame)
        comp_ids = [
            comp_id for xms_id in data_frame.index for comp_id in self._comp_ids if xms_id in self._comp_ids[comp_id]
        ]
        new_index = pd.Series(comp_ids)
        data_frame.set_index(new_index, inplace=True)
        dset = data_frame.to_xarray()
        # Rename dimensions and coordinates after pandas <-> xarray conversions.
        if 'index' in dset.dims:
            dset = dset.rename_dims({'index': 'comp_id'})
        if 'index' in dset.coords:
            dset = dset.rename({'index': 'comp_id'})
        return dset

    def sort_by_priority(self):
        """Sort available types by priority."""
        # Sort the rows in the table by priority
        self.model.sort(consts.COL_PRIORITY, Qt.AscendingOrder)

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (:obj:`QModelIndex`): Top left index.
            bottom_right_index (:obj:`QModelIndex`): Bottom right index. Same as top_left_index.
                QxPandasTableModel only supports single cell editing.
        """
        if not top_left_index.isValid():
            return

        col = top_left_index.column()
        if col == consts.COL_SEDIMENT_TYPE:
            model = top_left_index.model()
            # Update the priority strings combobox options.
            self._set_priority_strings()
            # Set priority group to nan if no longer applicable.
            type_idx = model.index(top_left_index.row(), consts.COL_SEDIMENT_TYPE)
            sediment_type = float(type_idx.data(Qt.EditRole))
            priority_idx = model.index(top_left_index.row(), consts.COL_PRIORITY)
            if sediment_type < consts.SEDIMENT_TYPE_AVAILABLE_CUT:
                # Make sure this row is not sorted with available rows
                model.setData(priority_idx, np.nan, Qt.EditRole)
            else:
                try:
                    _ = float(priority_idx.data(Qt.EditRole))
                except Exception:
                    # Ensure there is a valid priority grouping if now available.
                    model.setData(priority_idx, 0, Qt.EditRole)
