"""A dialog for EWN dredging calculator."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.file_selector_dialogs import get_save_filename
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts
from xms.ewn.gui.computation_table_widget import ComputationTableWidget
from xms.ewn.gui.dredge_calculator_dialog_ui import Ui_DredgeCalculatorDialog
from xms.ewn.tools.dredge_calculator import SedimentVolumeManagementComputations


class DredgeCalculatorDialog(XmsDlg):
    """A dialog for EWN dredging calculator."""
    def __init__(self, ewn_comp, pe_tree, query, projection, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            ewn_comp (:obj:`EwnCoverageComponent`): The EWN component that trigger the tool
            pe_tree (:obj:`TreeNode`): The XMS project explorer tree
            query (:obj:`Query`): XMS interprocess communication object
            projection (:obj:`data_objects.parameters.Projection`): The current display projection
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.ewn.gui.dredge_calculator_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Sediment_Volume_Management'
        self._comp = ewn_comp
        self._pe_tree = pe_tree
        self._query = query
        self._computation_table = None
        self._selected_geom_uuid = ''

        self._ugrid = None
        self._geom_type = None
        self._cov_geom = None
        self.sed_comps = None

        self._dbl_validator = None
        self._num_corrector = None

        self._rectilinear_data = None

        self.ui = Ui_DredgeCalculatorDialog()
        self.ui.setupUi(self)
        self._setup_ui(projection)

    def _setup_ui(self, projection):
        """Add the widgets.

        Args:
            projection (:obj:`data_objects.parameters.Projection`): The current display projection
        """
        # Display the vertical units in the groupbox text
        vert_units = 'm' if projection.vertical_units == 'METERS' else 'ft'
        self.ui.grp_available_volume.setTitle(f'Computation (volume units in {vert_units}^3)')

        self._setup_computation_table()
        self._load_coverage_data()
        self._connect_slots()

    def _setup_computation_table(self):
        """Create a the computation table widget and populate with a DataFrame."""
        # Get a dataframe of all the polygons where dredging is enabled.
        dredge_dset = self._comp.data.polygons.where(
            self._comp.data.polygons.sediment_type != consts.SEDIMENT_TYPE_NONE, drop=True
        )
        computation_df = dredge_dset.to_dataframe()
        comp_ids = []
        if self._comp.data.info.attrs['cov_uuid'] in self._comp.comp_to_xms:
            comp_ids = self._comp.comp_to_xms[self._comp.data.info.attrs['cov_uuid']][TargetType.polygon]
        self._computation_table = ComputationTableWidget(self, computation_df, comp_ids)
        self.ui.vert_layout_available.insertWidget(0, self._computation_table)

    def _load_coverage_data(self):
        """Load coverage level attributes into appropriate widgets."""
        # Load the coverage level data.
        data = self._comp.data
        target_geom_path = tree_util.build_tree_path(self._pe_tree, data.sediment.attrs['target_geom'])
        self.ui.lbl_selected_geom.setText(target_geom_path if target_geom_path else '(none selected)')
        self._selected_geom_uuid = data.sediment.attrs['target_geom']
        self.ui.edt_max_slope.setText(str(data.sediment.attrs['maximum_slope']))
        output_geom_name = data.sediment.attrs['output_geom_name']
        self.ui.edt_output_geom_name.setText(output_geom_name if output_geom_name else 'EWN Dredging Output')

        # Add a double validator to maximum slope edit field.
        float_validator = QxDoubleValidator(parent=self)
        self.ui.edt_max_slope.setValidator(float_validator)

        # Make computed value edit fields read only.
        self.ui.edt_sum_cut.setReadOnly(True)
        self.ui.edt_sum_fill.setReadOnly(True)
        self.ui.edt_net_cut.setReadOnly(True)
        self.ui.edt_net_fill.setReadOnly(True)

        # Load previous computed values.
        total_cut = data.sediment.attrs['total_cut']
        total_fill = data.sediment.attrs['total_fill']
        self.set_total_and_sum_edit_fields(total_cut, total_fill)

        # Disable the compute and generate geometry buttons if no target geom.
        self.ui.edt_output_geom_name.setEnabled(False)
        if not target_geom_path:
            self.ui.btn_compute.setEnabled(False)
            self.ui.tog_generate_geom.setEnabled(False)

    def _connect_slots(self):
        """Connect slots to widget signals."""
        self.ui.btn_target_geom.clicked.connect(self.on_select_geometry)
        self.ui.btn_sort.clicked.connect(self.on_sort_normalize)
        self.ui.btn_compute.clicked.connect(self.on_compute)
        self.ui.btn_save_to_file.clicked.connect(self.on_save_to_file)
        self.ui.tog_generate_geom.toggled.connect(self.on_tog_generate_geometry)

    def _ensure_coverage_geom_exists(self):
        """Query for the EWN coverage geometry, if needed."""
        if self._cov_geom is not None:
            return
        cov_uuid = self._comp.data.info.attrs['cov_uuid']
        self._cov_geom = self._query.item_with_uuid(cov_uuid)

    def _ensure_input_geom_exists(self):
        """Query for the input geometry, if needed."""
        if self._ugrid is not None:
            return
        # Store the tree type of the geometry
        tree_node = tree_util.find_tree_node_by_uuid(self._pe_tree, self._selected_geom_uuid)
        self._geom_type = tree_node.item_typename
        do_ugrid = self._query.item_with_uuid(self._selected_geom_uuid)
        co_grid = read_grid_from_file(do_ugrid.cogrid_file)
        self._ugrid = co_grid.ugrid

        # Store the QuadTree cells if input is a QuadTree
        if self._geom_type == 'TI_QUADTREE':
            self._rectilinear_data = co_grid

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Generate output geometry and send back to XMS if toggled enabled."""
        if self.ui.tog_generate_geom.isEnabled() and self.ui.tog_generate_geom.isChecked():
            # Generate geometry and close dialog if no errors.
            if self.compute(True):
                super().accept()
        else:  # Option to generate geometry is disabled, accept dialog.
            super().accept()

    def on_select_geometry(self):
        """Slot called when the target geometry selector button is clicked."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select EWN Geometry',
            target_type='',
            pe_tree=self._pe_tree,
            previous_selection=self._selected_geom_uuid,
            parent=self,
            selectable_xms_types=['TI_MESH2D', 'TI_QUADTREE', 'TI_UGRID_SMS']
        )

        if selector_dlg.exec():
            self._selected_geom_uuid = selector_dlg.get_selected_item_uuid()
            if self._selected_geom_uuid:
                self.ui.lbl_selected_geom.setText(tree_util.build_tree_path(self._pe_tree, self._selected_geom_uuid))
                self.ui.edt_output_geom_name.setEnabled(self.ui.tog_generate_geom.isChecked())
                self.ui.btn_compute.setEnabled(True)
                self.ui.tog_generate_geom.setEnabled(True)
            else:  # No target geometry selected, clear and disable the initial dataset selection
                self.ui.lbl_selected_geom.setText('(none selected)')
                self.ui.edt_output_geom_name.setEnabled(False)
                self.ui.btn_compute.setEnabled(False)
                self.ui.tog_generate_geom.setEnabled(False)
                self._ugrid = None
                self._geom_type = None
                self._rectilinear_data = None

    def on_sort_normalize(self):
        """Slot called when the sort/normalize button is clicked."""
        self._ensure_coverage_geom_exists()
        self.sed_comps = SedimentVolumeManagementComputations(
            comp_data=self._computation_table.model.data_frame, cover=self._cov_geom, parent=self
        )
        self.sed_comps.sort_normalize()
        self.sed_comps.save_poly_atts()
        self._computation_table.sort_by_priority()

    def on_compute(self):
        """Slot to call compute method without generating a new geometry."""
        self.compute()

    def compute(self, modify_mesh=False):
        """Slot called when the compute button is clicked.

        Returns:
            (:obj:`bool`): Success

        """
        self._ensure_coverage_geom_exists()
        self._ensure_input_geom_exists()
        max_slope = float(self.ui.edt_max_slope.text())
        self.sed_comps = SedimentVolumeManagementComputations(
            comp_data=self._computation_table.model.data_frame,
            cover=self._cov_geom,
            ugrid=self._ugrid,
            max_slope=max_slope,
            parent=self
        )
        self.sed_comps.rectilinear_data = self._rectilinear_data
        geom_type = self._geom_type if modify_mesh else None
        geom_name = self.ui.edt_output_geom_name.text()
        result = self.sed_comps.compute_data(geom_type=geom_type, geom_name=geom_name)
        self._computation_table.sort_by_priority()
        self.set_total_and_sum_edit_fields(self.sed_comps.sum_of_cut, self.sed_comps.sum_of_fill)
        return result

    def set_total_and_sum_edit_fields(self, total_cut, total_fill):
        """Update total and net edit field widgets.

        Args:
            total_cut (:obj:`float`): Cut volume sum
            total_fill (:obj:`float`): Fill volume sum
        """
        net_cut = 0.0
        net_fill = 0.0
        if total_cut > total_fill:
            net_cut = total_cut - total_fill
        else:
            net_fill = total_fill - total_cut
        self.ui.edt_sum_cut.setText(f'({total_cut:.2f})')
        self.ui.edt_sum_fill.setText(f'{total_fill:.2f}')
        self.ui.edt_net_cut.setText(f'({net_cut:.2f})')
        self.ui.edt_net_fill.setText(f'{net_fill:.2f}')

    def on_save_to_file(self):
        """Slot called when the save to file button is clicked."""
        text_filter = 'Text files (*.txt)'
        all_filters = f'{text_filter};;All files (*.*)'
        filename = get_save_filename(self, text_filter, all_filters)
        if filename:
            if self.compute(False):
                self.sed_comps.save_to_file(filename)

    def on_tog_generate_geometry(self, toggled):
        """Slot called when the generate geometry button is clicked.

        Args:
            toggled (:obj:`bool`): True if the checkbox
        """
        self.ui.edt_output_geom_name.setEnabled(toggled)

    def dialog_data_to_xarray(self):
        """Retrieve the updated polygon attributes from the computation table and the coverage level attributes.

        Returns:
            (:obj:`tuple(xarray.Dataset, dict)`:
                polygon attributes dataset (:obj:`xarray.Dataset`), coverage level attrs (:obj:`dict`)
        """
        attrs = {
            'target_geom': self._selected_geom_uuid,
            'maximum_slope': float(self.ui.edt_max_slope.text()),
            'output_geom_name': self.ui.edt_output_geom_name.text(),
            'total_cut': float(self.ui.edt_sum_cut.text().strip("()")),
            'total_fill': float(self.ui.edt_sum_fill.text())
        }
        return self._computation_table.table_data_to_xarray(), attrs
