"""Qt delegate for displaying a combobox."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QApplication, QStyle, QStyleOptionComboBox

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules


class IntQxCbxDelegate(QxCbxDelegate):
    """Store integer combobox options instead of text."""
    def __init__(self, parent):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Qt parent
        """
        super().__init__(parent)

    def paint(self, painter, option, index):
        """Override the paint event.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return

        try:
            current_idx = int(float(index.data(Qt.EditRole)))
            current_text = self._strings[current_idx]
        except Exception:
            current_text = ''
        cbx_opt = QStyleOptionComboBox()
        cbx_opt.currentText = current_text
        cbx_opt.rect = option.rect
        cbx_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            cbx_opt.state |= QStyle.State_Enabled
        cbx_opt.editable = False

        QApplication.style().drawComplexControl(QStyle.CC_ComboBox, cbx_opt, painter)
        QApplication.style().drawControl(QStyle.CE_ComboBoxLabel, cbx_opt, painter)

    def setEditorData(self, editor, index):  # noqa: N802
        """Sets the data to be displayed and edited by the editor from the data model item specified by the model index.

        Args:
            editor (:obj:`QWidget`): The editor.
            index (:obj:`QModelIndex`): The index.
        """
        cb = editor
        if not cb:
            return
        try:
            current_idx = int(float(index.data(Qt.EditRole)))
        except Exception:
            current_idx = 0
        if current_idx >= 0:
            cb.blockSignals(True)
            cb.setCurrentIndex(current_idx)
            cb.blockSignals(False)
            self.commitData.emit(self.cb)
            self.closeEditor.emit(self.cb)
        cb.showPopup()

    def setModelData(self, editor, model, index):  # noqa: N802
        """Gets data from the editor widget and stores it in the specified model at the item index.

        Args:
            editor (:obj:`QWidget`): The editor.
            model (:obj:`QAbstractItemModel`): The model.
            index (:obj:`QModelIndex`): The index
        """
        cb = editor
        if not cb:
            return

        cb_index = cb.currentIndex()
        # If it is valid, adjust the combobox
        if cb_index >= 0:
            model.setData(index, cb_index, Qt.EditRole)
            self.state_changed.emit(index)
