"""Class for performing dredging calculations."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math
import os
import sys
import uuid

# 2. Third party modules
import numpy
from PySide2.QtWidgets import QMessageBox

# 3. Aquaveo modules
from xms.constraint.quadtree_grid_builder import QuadtreeGridBuilder
from xms.constraint.ugrid_builder import UGridBuilder
from xms.data_objects.parameters import UGrid as DoUGrid
from xms.grid.geometry import geometry
from xms.grid.ugrid import UGrid as XmUGrid

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as const
from xms.ewn.tools.runners.runner_util import get_polygon_outside_points_ccw


def rename_columns_for_io(data_frame):
    """Rename DataFrame columns to their original file text.

    Args:
        data_frame (:obj:`pandas.DataFrame`): data frame of the polygon dredge data

    Returns:
        (:obj:`pandas.DataFrame`): See description
    """
    return data_frame.rename(
        columns={
            'Name': 'polygon_name',
            'Type': 'sediment_type',
            'Priority': 'priority',
            'Priority %': 'priority_percent',
            'Cut/Fill Type': 'cut_fill_type',
            'Value': 'volume_value',
            'Total Volume\nBased on Slope': 'total_volume',
            'Required\nVolume': 'required_volume',
            'Available\nVolume': 'available_volume',
            'Cut\nVolume': 'cut_volume',
            'Fill\nVolume': 'fill_volume',
        }
    )


class SedimentVolumeManagementComputations:
    """Class for computing sediment volume management (Dredging)."""
    def __init__(self, comp_data, cover=None, ugrid=None, max_slope=0.0, parent=None):
        """Constructor.

        Args:
            comp_data (:obj:`pandas.DataFrame`): The component polygon attributes as a DataFrame (from the dialog)
            cover (:obj:`data_objects.parameters.Coverage`): The EWN coverage geometry
            ugrid (:obj:`xms.grid.ugrid.UGrid`): 2d Unstructured grid
            max_slope (:obj:`float`): The maximum slope coverage attribute
            parent (:obj:`QObject`): The Qt dialog, if there is one
        """
        self._comp_data = comp_data
        self._cover = cover
        self._ugrid = ugrid
        self._polygons = {poly.id: get_polygon_outside_points_ccw(poly) for poly in self._cover.polygons}
        self._parent = parent

        self._max_slope = max_slope

        self._specified_cut_atts = None
        self._specified_fill_atts = None
        self._available_cut_atts = None
        self._available_fill_atts = None

        self._elevations = []
        self._area = []
        self._cell_indices = []
        self._cell_centers = []
        self._cell_points = []
        self._elevations_with_slope = []

        self.sum_of_cut = 0.0
        self.sum_of_fill = 0.0

        self.rectilinear_data = None  # Set if input geometry is a QuadTree
        self.output_geom = None

        self._logger = logging.getLogger('xms.ewn')

        self.output_file = ''

    def get_number_of_polygons(self, gather):
        """Get the number of EWN polygons that have dredging enabled.

        Args:
            gather (:obj:`bool`): True if we need to get polygons attributes (only the first call)

        Returns:
            (:obj:`int`): See description
        """
        num_polys = 0
        if gather:
            self._gather_all_polys()

        num_polys += len(self._specified_cut_atts)
        num_polys += len(self._specified_fill_atts)
        num_polys += len(self._available_cut_atts)
        num_polys += len(self._available_fill_atts)
        return num_polys

    def save_poly_atts(self):
        """Save changes tp polygon attributes."""
        loc = self._comp_data.loc
        if not self._specified_cut_atts.empty:
            loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_SPECIFY_CUT] = self._specified_cut_atts
        if not self._specified_fill_atts.empty:
            loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_SPECIFY_FILL] = self._specified_fill_atts
        if not self._available_cut_atts.empty:
            loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_AVAILABLE_CUT] = self._available_cut_atts
        if not self._available_fill_atts.empty:
            loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_AVAILABLE_FILL] = self._available_fill_atts

    def _gather_all_polys(self):
        """Extracts polygon attributes for polygons that have dredging enabled."""
        # clear computed data
        self._comp_data.iloc[:, const.COL_TOTAL_VOLUME] = 0.0
        self._comp_data.iloc[:, const.COL_REQUIRED_VOLUME] = 0.0
        self._comp_data.iloc[:, const.COL_AVAILABLE_VOLUME] = 0.0
        self._comp_data.iloc[:, const.COL_CUT_VOLUME] = 0.0
        self._comp_data.iloc[:, const.COL_FILL_VOLUME] = 0.0

        self._specified_cut_atts = self._comp_data.loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_SPECIFY_CUT]
        self._specified_fill_atts = self._comp_data.loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_SPECIFY_FILL]
        self._available_cut_atts = self._comp_data.loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_AVAILABLE_CUT]
        self._available_fill_atts = self._comp_data.loc[self._comp_data['Type'] == const.SEDIMENT_TYPE_AVAILABLE_FILL]

    def _build_geometry(self, geom_type, geom_name):
        """Build a new geometry with the adjusted elevations from the dredging computation.

        Args:
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS
            geom_name (:obj:`str`): Name to give the new geometry
        """
        if geom_type in ['TI_MESH2D', 'TI_UGRID_SMS']:  # Mesh2D or UGrid
            xmugrid = XmUGrid(self._ugrid.locations, self._ugrid.cellstream)
            co_builder = UGridBuilder()
            if geom_type == 'TI_MESH2D':
                co_builder.set_is_2d()
            else:
                co_builder.set_unconstrained()
            co_builder.set_ugrid(xmugrid)
        elif geom_type == 'TI_QUADTREE':
            co_builder = QuadtreeGridBuilder()
            co_builder.is_2d_grid = True
            co_builder.origin = self.rectilinear_data.origin
            co_builder.angle = self.rectilinear_data.angle
            co_builder.numbering = self.rectilinear_data.numbering
            co_builder.orientation = self.rectilinear_data.orientation
            co_builder.locations_x = self.rectilinear_data.locations_x
            co_builder.locations_y = self.rectilinear_data.locations_y
            co_builder.quad_grid_cells = self.rectilinear_data.quad_grid_cells
        else:  # pragma: nocover
            return  # How did you get here?

        cogrid = co_builder.build_grid()
        cogrid.cell_elevations = self._elevations

        ugrid_uuid = str(uuid.uuid4())
        temp_dir = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', os.getcwd())
        constraint_file = self.output_file if self.output_file else os.path.join(temp_dir, f'{ugrid_uuid}.xmc')
        cogrid.write_to_file(constraint_file, True)
        geom_name = geom_name if geom_name else 'EWN Dredging Output'
        self.output_geom = DoUGrid(
            cogrid_file=constraint_file, uuid=ugrid_uuid, name=geom_name, projection=self._cover.projection
        )

    def save_to_file(self, filename):
        """Save polygon attributes to a csv file.

        Args:
            filename (:obj:`str`): Path to the output file
        """
        net_cut = 0.0
        net_fill = 0.0
        if self.sum_of_cut > self.sum_of_fill:
            net_cut = self.sum_of_cut - self.sum_of_fill
        else:
            net_fill = self.sum_of_fill - self.sum_of_cut
        df = rename_columns_for_io(self._comp_data)
        df.to_csv(filename, sep=',', na_rep='0', index_label='ID')
        with open(filename, "a") as f:
            f.write(f'\nSum of cut:,({self.sum_of_cut:.2f}),Sum of fill:,{self.sum_of_fill:.2f}')
            f.write(f'\nNet of cut:,({net_cut:.2f}),Net of fill:,{net_fill:.2f}')

    @staticmethod
    def _normalize(poly_atts, num_priorities):
        """Adjust polygon available priorities based on defined priorities of all polygons.

        Args:
            poly_atts (:obj:`pandas.DataFrame`): The polygon attributes
            num_priorities (:obj:`int`): The number of used priority bins
        """
        percentages_per_priority = [0.0] * num_priorities
        for index, row in poly_atts.iterrows():
            priority = int(row['Priority'])
            # if the priority is invalid, assign it to have the first priority
            if (priority < 0) or (priority >= num_priorities):
                priority = 0
                poly_atts.at[index, 'Priority'] = priority
            percentages_per_priority[priority] += row['Priority %']
        for index, row in poly_atts.iterrows():
            priority_percent = percentages_per_priority[int(row['Priority'])]
            poly_atts.at[index, 'Priority %'] = row['Priority %'] * 100.0 / priority_percent

    def sort_normalize(self):
        """Sort and normalize the available cut/fill polygons."""
        num_polys = self.get_number_of_polygons(True)
        SedimentVolumeManagementComputations._normalize(self._available_fill_atts, num_polys)
        SedimentVolumeManagementComputations._normalize(self._available_cut_atts, num_polys)

    def perform_available_cut_fill(self, total_net_volume, cell_indices, elevations, elevations_limits, geom_type, cut):
        """Perform the computations of available cut or fill.

        Args:
            total_net_volume (:obj:`float`): Total net volume for the EWN polygon
            cell_indices (:obj:`list[list]`): Cell indices of grid cells that intersect with a EWN polygon.
                Inner list for each polygon
            elevations (:obj:`list[float]`): Cell elevations of the grid
            elevations_limits (list[float]`): Maximum/minimum elevation after cut or fill
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS
            cut (:obj:`bool`): True if cut, False if Fill

        Returns:
            (:obj:`float`): The total volume cut or filled using all the available cut/fill polygons
        """
        total_cut_fill_volume = 0.0
        cut_fill = total_net_volume
        priority = 0
        num_polys = self.get_number_of_polygons(False)

        if len(cell_indices) > 0:
            if cut:
                atts = self._available_cut_atts
            else:
                atts = self._available_fill_atts

            # go through each priority and distribute 'cutfill'
            while ((cut and cut_fill > 0.0) or (not cut and cut_fill < 0.0)) and priority < num_polys:
                storage_for_priority = 0.0
                poly_index = 0
                # Gather available storage for current priority
                cur_priority_atts = {}
                cur_cell_indices = []
                cur_priority_percent = []
                for index, row in atts.iterrows():
                    if int(row['Priority']) == priority:
                        storage_for_priority += row['Available\nVolume']
                        cur_priority_atts[index] = row
                        cur_priority_percent.append(row['Priority %'])
                        cur_cell_indices.append(cell_indices[poly_index])
                    poly_index += 1
                # if we have more available than cut/filled, change it to how much volume we will move
                if storage_for_priority > abs(cut_fill):
                    storage_for_priority = abs(cut_fill)
                loop_counter = 0
                volume_remaining = storage_for_priority
                while volume_remaining > 0.0 and loop_counter < 50:
                    # temporarily normalize the priority percentage
                    percentages_for_priority = 0.0
                    for percent in cur_priority_percent:
                        percentages_for_priority += percent
                    if percentages_for_priority <= 0.0:
                        loop_counter = 50  # We have filled all the polygons in this priority
                        continue
                    for idx in range(len(cur_priority_percent)):
                        cur_priority_percent[idx] *= 100.0 / percentages_for_priority

                    volume_to_use_this_round = volume_remaining
                    # Now apply the cut/fill according to the percent allocated per priority
                    for idx, (index, att) in enumerate(cur_priority_atts.items()):
                        if att['Priority'] == priority:
                            # Determine the assigned volume according to percent; adjust for previously cut filled polys
                            volume = cur_priority_percent[idx] / 100.0 * volume_to_use_this_round
                            if att['Type'] in [const.SEDIMENT_TYPE_SPECIFY_CUT, const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                                total_moved = self._available_cut_atts.at[index, 'Cut\nVolume']
                            else:
                                total_moved = self._available_fill_atts.at[index, 'Fill\nVolume']
                            available_volume = att['Available\nVolume'] - total_moved
                            # Check if we have cut/filled this poly
                            if available_volume > 0.0:
                                # check if we are going to overflow this poly
                                if volume > available_volume:
                                    volume = available_volume
                                    # This poly is full
                                    cur_priority_percent[idx] = 0.0
                                # distribute the 'volume' on this poly
                                volume_with_sign = volume
                                if cut:
                                    volume_with_sign = -volume

                                cut_fill_type = att['Cut/Fill Type']
                                poly_cells = cur_cell_indices[idx]
                                if cut_fill_type in [const.CUT_FILL_TYPE_CONSTANT, const.CUT_FILL_TYPE_TOTAL_VOL_ELEV]:
                                    total_moved += SedimentVolumeManagementComputations.fill_volume_by_elevation(
                                        att, poly_cells, elevations, elevations_limits, self._area, volume_with_sign,
                                        geom_type
                                    )
                                else:
                                    total_moved += SedimentVolumeManagementComputations.fill_volume_by_thickness(
                                        att, poly_cells, elevations, elevations_limits, self._area, volume_with_sign,
                                        geom_type
                                    )

                                if att['Type'] in [const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                                    self._available_cut_atts.at[index, 'Cut\nVolume'] = total_moved
                                else:
                                    self._available_fill_atts.at[index, 'Fill\nVolume'] = total_moved

                                volume_remaining -= volume
                    loop_counter += 1
                total_cut_fill_volume += storage_for_priority
                if cut:
                    cut_fill -= storage_for_priority
                else:
                    cut_fill += storage_for_priority
                priority += 1
        return total_cut_fill_volume

    def compute_max_elevations_based_on_slope(self, cell_indices, elevations, elevations_with_slope, cut):
        """Compute cell elevations based on intersecting EWN polygons.

        Args:
            cell_indices (:obj:`list[list]`): Cell indices of grid cells that intersect with a EWN polygon.
                Inner list for each polygon.
            elevations (:obj:`list[float]`): Cell elevations of the grid
            elevations_with_slope (:obj:`list[float]`): Current adjusted cell elevations
            cut (:obj:`bool`): True if cut, False if Fill

        Returns:
            (:obj:`bool`): True if the polygon elevation was adjusted
        """
        change_in_poly = True
        tol = 0.001
        max_slope = self._max_slope
        count = 0

        while change_in_poly and count < 100:
            change_in_poly = False
            for poly_cell in cell_indices:
                for cell_index in poly_cell:
                    neighbors = self._ugrid.get_cell_adjacent_cells(cell_index)
                    num_edges = self._ugrid.get_cell_edge_count(cell_index)
                    boundary_cell = False
                    for edge_index in range(0, num_edges):
                        if self._ugrid.get_cell_2d_edge_adjacent_cell(cell_index, edge_index) < 0:
                            boundary_cell = True
                            break
                    if not boundary_cell:
                        if cut:
                            # Go through each neighbor and determine the max elevation can cut to
                            new_elev = -sys.float_info.max
                            for neighbor in neighbors:
                                if neighbor >= 0:
                                    distance = geometry.distance_2d(
                                        self._cell_centers[cell_index], self._cell_centers[neighbor]
                                    )
                                    # This is a cut (relative from neighbor's position)
                                    cur_elevation = elevations_with_slope[neighbor] - distance / max_slope
                                    # Determine the highest elevation of the neighbors (the governing elevation)
                                    if cur_elevation > new_elev:
                                        new_elev = cur_elevation
                            if new_elev > -sys.float_info.max and new_elev + tol < elevations_with_slope[cell_index]:
                                change_in_poly = True
                                elevations_with_slope[cell_index] = new_elev
                        else:  # Fill
                            # Go through each neighbor and determine the max elevation can fill to
                            new_elev = sys.float_info.max
                            for neighbor in neighbors:
                                if neighbor >= 0:
                                    distance = geometry.distance_2d(
                                        self._cell_centers[cell_index], self._cell_centers[neighbor]
                                    )
                                    # This is a Fill
                                    cur_elevation = elevations_with_slope[neighbor] + distance / max_slope
                                    # Check that it is below the elevation there currently, but not below
                                    # the currently computed elevation (we need the slope that works with all neighbors)
                                    if elevations[cell_index] < cur_elevation < new_elev:
                                        new_elev = cur_elevation
                            if new_elev < sys.float_info.max and new_elev - tol > elevations_with_slope[cell_index]:
                                change_in_poly = True
                                elevations_with_slope[cell_index] = new_elev
            count += 1
        return change_in_poly

    def determine_available_storage_in_polys(self, cell_indices, elevations, elevations_with_slope, atts):
        """Update total available volume for EWN available fill polygons.

        Args:
            cell_indices (:obj:`list[list]`): Cell indices of grid cells that intersect with a EWN polygon.
                Inner list for each polygon.
            elevations (:obj:`list[float]`): Cell elevations of the grid
            elevations_with_slope (:obj:`list[float]`): Current adjusted cell elevations
            atts (:obj:`pandas.DataFrame`): The polygon attribute data
        """
        att_index = 0
        for poly_cells in cell_indices:
            available_volume = 0.0
            for cell_index in poly_cells:
                elevation = elevations_with_slope[cell_index] - elevations[cell_index]
                available_volume += elevation * self._area[cell_index]

            atts.iat[att_index, const.COL_TOTAL_VOLUME] = (abs(available_volume))
            att_index += 1

    def compute_data(self, geom_type, geom_name):
        """Top level method to trigger dredging computations.

        Args:
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS. None if not generating geometry.
            geom_name (:obj:`str`): Name to give generated geometry, if applicable

        Returns:
            (:obj:`bool`): Successful
        """
        total_cut_volume = 0.0
        total_fill_volume = 0.0

        # Gather Polys
        self.sort_normalize()

        # Ensure non-zero slope
        if self._max_slope == 0.0:  # pragma: nocover
            msgbox = QMessageBox(
                QMessageBox.Critical, 'SMS', 'Enter a non-zero maximum slope.', QMessageBox.Ok, self._parent
            )
            msgbox.exec()
            return False

        # elevations and volumes of the UGrid/quad elements
        if self._ugrid.cell_count <= 0:  # pragma: nocover
            msgbox = QMessageBox(
                QMessageBox.Critical, 'SMS', 'It is required that the geometry have cells.', QMessageBox.Ok,
                self._parent
            )
            msgbox.exec()
            return False
        self._elevations = []
        self._area = []
        self._cell_indices = []
        self._cell_centers = []
        self._cell_points = []
        for cell_index in range(self._ugrid.cell_count):
            success, cell_centroid = self._ugrid.get_cell_centroid(cell_index)
            if not success:  # pragma: nocover
                msgbox = QMessageBox(
                    QMessageBox.Critical, 'SMS', f'Unable to compute the centroid of cell: {cell_index + 1}',
                    QMessageBox.Ok, self._parent
                )
                msgbox.exec()
                return False
            else:
                self._cell_centers.append(cell_centroid)
                self._cell_points.append(self._ugrid.get_cell_locations(cell_index))
                avg_elev = 0.0
                for pt in self._cell_points[-1]:
                    avg_elev += pt[2]
                avg_elev /= len(self._cell_points[-1])
                self._elevations.append(avg_elev)
                self._area.append(abs(geometry.polygon_area_2d(self._cell_points[-1])))

        # Determine available storage based solely on slope
        self._elevations_with_slope = self._elevations.copy()
        specified_cut_cell_indices = []
        specified_fill_cell_indices = []
        available_cut_cell_indices = []
        available_fill_cell_indices = []
        all_cell_indexes = []

        # Gather cell indices first
        for index in self._specified_cut_atts.index:
            cell_indices = self.get_cells_for_polygon(index)
            specified_cut_cell_indices.append(cell_indices)
            all_cell_indexes.extend(cell_indices)

        for index in self._specified_fill_atts.index:
            cell_indices = self.get_cells_for_polygon(index)
            specified_fill_cell_indices.append(cell_indices)
            all_cell_indexes.extend(cell_indices)

        for index in self._available_cut_atts.index:
            cell_indices = self.get_cells_for_polygon(index)
            available_cut_cell_indices.append(cell_indices)
            all_cell_indexes.extend(cell_indices)

        for index in self._available_fill_atts.index:
            cell_indices = self.get_cells_for_polygon(index)
            available_fill_cell_indices.append(cell_indices)
            all_cell_indexes.extend(cell_indices)

        # Check that each cell only belongs to one polygon
        temp = set(all_cell_indexes)
        if len(temp) != len(all_cell_indexes):  # pragma: nocover
            msgbox = QMessageBox(QMessageBox.Critical, 'SMS', 'Polygons cannot overlap!', QMessageBox.Ok, self._parent)
            msgbox.exec()
            return False

        # Create elevation list that shows the maximum cut and fill available for the polygons
        count = 0
        change_in_elevations = True
        while change_in_elevations and count < 100:
            change_in_elevations = False
            # Specified Cuts
            if self.compute_max_elevations_based_on_slope(
                specified_cut_cell_indices, self._elevations, self._elevations_with_slope, True
            ):
                change_in_elevations = True
            # Specified Fills
            if self.compute_max_elevations_based_on_slope(
                specified_fill_cell_indices, self._elevations, self._elevations_with_slope, False
            ):
                change_in_elevations = True
            # Available Cuts
            if self.compute_max_elevations_based_on_slope(
                available_cut_cell_indices, self._elevations, self._elevations_with_slope, True
            ):
                change_in_elevations = True
            # Available Fills
            if self.compute_max_elevations_based_on_slope(
                available_fill_cell_indices, self._elevations, self._elevations_with_slope, False
            ):
                change_in_elevations = True
            count += 1

        # Determine available Storage
        self.determine_available_storage_in_polys(
            specified_cut_cell_indices, self._elevations, self._elevations_with_slope, self._specified_cut_atts
        )
        self.determine_available_storage_in_polys(
            specified_fill_cell_indices, self._elevations, self._elevations_with_slope, self._specified_fill_atts
        )
        self.determine_available_storage_in_polys(
            available_cut_cell_indices, self._elevations, self._elevations_with_slope, self._available_cut_atts
        )
        self.determine_available_storage_in_polys(
            available_fill_cell_indices, self._elevations, self._elevations_with_slope, self._available_fill_atts
        )

        # Compute/perform Specified Cuts
        poly_index = 0
        for index, row in self._specified_cut_atts.iterrows():
            # Determine the volume cut and apply the cut if modify
            volume_change = self.determine_volume_change(
                row, specified_cut_cell_indices[poly_index], self._elevations, self._elevations_with_slope, geom_type
            )
            total_cut_volume += volume_change
            self._specified_cut_atts.at[index, 'Cut\nVolume'] = volume_change
            self._specified_cut_atts.at[index, 'Required\nVolume'] = volume_change
            poly_index += 1

        poly_index = 0
        for index, row in self._specified_fill_atts.iterrows():
            # Determine the volume cut and apply the cut if modify
            volume_change = self.determine_volume_change(
                row, specified_fill_cell_indices[poly_index], self._elevations, self._elevations_with_slope, geom_type
            )
            total_fill_volume += volume_change
            self._specified_fill_atts.at[index, 'Fill\nVolume'] = volume_change
            self._specified_fill_atts.at[index, 'Required\nVolume'] = volume_change
            poly_index += 1

        # Determine Available Storage
        poly_index = 0
        for index, row in self._available_cut_atts.iterrows():
            # Determine the volume cut
            volume_change = self.determine_volume_change(
                row, available_cut_cell_indices[poly_index], self._elevations, self._elevations_with_slope, None
            )
            # Reset the total volume moved so it can be computed below
            self._available_cut_atts.at[index, 'Cut\nVolume'] = 0.0
            self._available_cut_atts.at[index, 'Available\nVolume'] = volume_change
            poly_index += 1

        poly_index = 0
        for index, row in self._available_fill_atts.iterrows():
            # Determine the volume filled
            volume_change = self.determine_volume_change(
                row, available_fill_cell_indices[poly_index], self._elevations, self._elevations_with_slope, None
            )
            # Reset the total volume moved so it can be computed below
            self._available_fill_atts.at[index, 'Fill\nVolume'] = 0.0
            self._available_fill_atts.at[index, 'Available\nVolume'] = volume_change
            poly_index += 1

        total_net_volume = total_fill_volume - total_cut_volume
        # compute/perform Available Cuts/Fills (and apply surplus from Specified)
        if total_net_volume > 0.0:
            # if total_net_volume is positive, we have filled more volume than cut, so now we cut
            total_cut_volume += self.perform_available_cut_fill(
                total_net_volume, available_cut_cell_indices, self._elevations, self._elevations_with_slope, geom_type,
                True
            )
        elif total_net_volume < 0.0:
            # if total_net_volume is negative, we have cut more volume than filled, so now we fill
            total_fill_volume += self.perform_available_cut_fill(
                total_net_volume, available_fill_cell_indices, self._elevations, self._elevations_with_slope, geom_type,
                False
            )

        self.sum_of_cut = total_cut_volume
        self.sum_of_fill = total_fill_volume

        self.save_poly_atts()

        if geom_type is not None:
            # generate new geometry
            self._build_geometry(geom_type, geom_name)

        return True

    def get_cells_for_polygon(self, poly_id):
        """Get the 0-based cell indices of grid cells that intersect an EWN polygon.

        Args:
            poly_id (:obj:`int`): Feature polygon id of the EWN polygon

        Returns:
            (:obj:`list[int]`): See description
        """
        cell_indices = []
        # Get Polygon points - fill: outside_poly_pointer
        for index, pt in enumerate(self._cell_centers):
            if geometry.point_in_polygon_2d(self._polygons[poly_id], pt) >= 0:
                cell_indices.append(index)
        return cell_indices

    def determine_volume_change(self, poly_atts, cell_indices, elevations, elevations_limits, geom_type):
        """Compute the change in volume for EWN polygons.

        Args:
            poly_atts (:obj:`pandas.DataFrame`): The polygon attributes
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            (:obj:`float`): See description
        """
        polygon_volume_change = 0.0
        value = poly_atts['Value']
        poly_type = poly_atts['Type']

        if len(cell_indices) > 0:
            if poly_atts['Cut/Fill Type'] == const.CUT_FILL_TYPE_CONSTANT:
                if poly_type in [const.SEDIMENT_TYPE_SPECIFY_CUT, const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                    polygon_volume_change = SedimentVolumeManagementComputations.cut_constant_elevation(
                        cell_indices, elevations, elevations_limits, self._area, value, geom_type
                    )
                elif poly_type in [const.SEDIMENT_TYPE_SPECIFY_FILL, const.SEDIMENT_TYPE_AVAILABLE_FILL]:
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_constant_elevation(
                        cell_indices, elevations, elevations_limits, self._area, value, geom_type
                    )
            elif poly_atts['Cut/Fill Type'] == const.CUT_FILL_TYPE_REL_THICK:
                if poly_type in [const.SEDIMENT_TYPE_SPECIFY_CUT, const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                    # Can fill volume with negative volume (means to cut)
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_relative_thickness(
                        cell_indices, elevations, elevations_limits, self._area, -value, geom_type
                    )
                elif poly_type in [const.SEDIMENT_TYPE_SPECIFY_FILL, const.SEDIMENT_TYPE_AVAILABLE_FILL]:
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_relative_thickness(
                        cell_indices, elevations, elevations_limits, self._area, value, geom_type
                    )
            elif poly_atts['Cut/Fill Type'] == const.CUT_FILL_TYPE_TOTAL_VOL_ELEV:
                if poly_type in [const.SEDIMENT_TYPE_SPECIFY_CUT, const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                    # Can fill cvolume with negative volume (means to cut)
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_volume_by_elevation(
                        poly_atts, cell_indices, elevations, elevations_limits, self._area, -value, geom_type
                    )
                elif poly_type in [const.SEDIMENT_TYPE_SPECIFY_FILL, const.SEDIMENT_TYPE_AVAILABLE_FILL]:
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_volume_by_elevation(
                        poly_atts, cell_indices, elevations, elevations_limits, self._area, value, geom_type
                    )
            elif poly_atts['Cut/Fill Type'] == const.CUT_FILL_TYPE_TOTAL_VOL_THICK:
                # assume relative thickness
                if poly_type in [const.SEDIMENT_TYPE_SPECIFY_CUT, const.SEDIMENT_TYPE_AVAILABLE_CUT]:
                    # Can fill volume with negative volume (means to cut)
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_volume_by_thickness(
                        poly_atts, cell_indices, elevations, elevations_limits, self._area, -value, geom_type
                    )
                elif poly_type in [const.SEDIMENT_TYPE_SPECIFY_FILL, const.SEDIMENT_TYPE_AVAILABLE_FILL]:
                    polygon_volume_change = SedimentVolumeManagementComputations.fill_volume_by_thickness(
                        poly_atts, cell_indices, elevations, elevations_limits, self._area, value, geom_type
                    )
        return abs(polygon_volume_change)

    @staticmethod
    def cut_constant_elevation(cell_indices, elevations, elevations_limits, area, elevation, geom_type):
        """Adjust elevations by constant cut elevation.

        Args:
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            area (:obj:`list[float]`): The grid cell areas
            elevation (:obj:`float`): The constant elevation
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            (:obj:`float`): The change in volume
        """
        polygon_volume_change = 0.0
        for cell_index in cell_indices:
            elev = elevation
            # Check that we can cut according to the elevation and that the slope allows for a cut
            if elevations[cell_index] > elev and elevations_limits[cell_index] < elevations[cell_index]:
                # Check that the new elevation does not exceed the slope limit cut
                if elevations_limits[cell_index] > elev:
                    elev = elevations_limits[cell_index]
                polygon_volume_change += area[cell_index] * (elevations[cell_index] - elev)
                if geom_type is not None:
                    elevations[cell_index] = elev
        return polygon_volume_change

    @staticmethod
    def fill_constant_elevation(cell_indices, elevations, elevations_limits, area, elevation, geom_type):
        """Adjust elevations by constant fill elevation.

        Args:
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            area (:obj:`list[float]`): The grid cell areas
            elevation (:obj:`float`): The constant elevation
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            (:obj:`float`): The change in volume
        """
        polygon_volume_change = 0.0
        for cell_index in cell_indices:
            elev = elevation
            # Check that we can fill according to the elevation and that the slope allows for a cut
            if elevations[cell_index] < elev and elevations_limits[cell_index] > elevations[cell_index]:
                # Check that the new elevation does not exceed the slope limit fill
                if elevations_limits[cell_index] < elev:
                    elev = elevations_limits[cell_index]
                polygon_volume_change += area[cell_index] * (elev - elevations[cell_index])
                if geom_type is not None:
                    elevations[cell_index] = elev
        return polygon_volume_change

    @staticmethod
    def fill_relative_thickness(cell_indices, elevations, elevations_limits, area, thickness, geom_type):
        """Adjust elevations by filling by relative thickness.

        Args:
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            area (:obj:`list[float]`): The grid cell areas
            thickness (:obj:`float`): The target change in thickness
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            (:obj:`float`): The change in volume
        """
        polygon_volume_change = 0.0
        for cell_index in cell_indices:
            thick = thickness
            if thick < 0.0:
                if elevations_limits[cell_index] - elevations[cell_index] > thick:
                    thick = elevations_limits[cell_index] - elevations[cell_index]
                if thick > 0.0:
                    thick = 0.0
            else:
                if elevations_limits[cell_index] - elevations[cell_index] < thick:
                    thick = elevations_limits[cell_index] - elevations[cell_index]
                if thick < 0.0:
                    thick = 0.0
            polygon_volume_change += area[cell_index] * thick
            if geom_type is not None:
                elevations[cell_index] += thick
        return polygon_volume_change

    @staticmethod
    def fill_volume_by_thickness(poly_atts, cell_indices, elevations, elevations_limits, area, volume, geom_type):
        """Adjust elevations by filling by thickness.

        Args:
            poly_atts (:obj:`pandas.DataFrame`): The polygon attributes
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            area (:obj:`list[float]`): The grid cell areas
            volume (:obj:`float`): The target volume
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            (:obj:`float`): The change in volume
        """
        cut = volume < 0.0
        total_area = 0.0

        for cell_index in cell_indices:
            total_area += area[cell_index]
        target_volume = volume

        total_available_volume = poly_atts['Total Volume\nBased on Slope']
        if cut:
            total_available_volume = -total_available_volume

        if abs(target_volume) > abs(total_available_volume):
            target_volume = total_available_volume

        # Because we do not know how the slopes will affect our volume,
        # it is not a straight forward computation, but needs iteration:
        new_thickness = target_volume / total_area
        thickness = [0.0, new_thickness, new_thickness * 5.0]
        volumes = [
            0.0,
            SedimentVolumeManagementComputations.fill_relative_thickness(
                cell_indices, elevations, elevations_limits, area, new_thickness, None
            )
        ]
        tol = 1.0e-2
        max_iter = 100
        cur_iter = 0
        last_volume = 0.0
        while not math.isclose(target_volume, last_volume, abs_tol=tol) and cur_iter < max_iter:
            volumes.append(
                SedimentVolumeManagementComputations.fill_relative_thickness(
                    cell_indices, elevations, elevations_limits, area, thickness[-1], None
                )
            )
            last_volume = volumes[-1]
            # Figure out a new elevation to try
            thickness.sort()
            volumes.sort()
            new_thickness = numpy.interp(target_volume, volumes, thickness)
            thickness.append(new_thickness)
            cur_iter += 1
        return abs(
            SedimentVolumeManagementComputations.fill_relative_thickness(
                cell_indices, elevations, elevations_limits, area, thickness[-1], geom_type
            )
        )

    @staticmethod
    def fill_volume_by_elevation(poly_atts, cell_indices, elevations, elevations_limits, area, volume, geom_type):
        """Adjust elevations by filling by elevation.

        Args:
            poly_atts (:obj:`pandas.DataFrame`): The polygon attributes
            cell_indices (:obj:`list[int]`): Grid cells intersected by the EWN polygon
            elevations (:obj:`list[float]`): The grid elevations
            elevations_limits (:obj:`list[float]`): The current adjusted grid elevations
            area (:obj:`list[float]`): The grid cell areas
            volume (:obj:`float`): The target volume
            geom_type (:obj:`str`): One of TI_MESH2D, TI_QUADTREE, or TI_UGRID_SMS

        Returns:
            float: The change in volume
        """
        cut = volume < 0.0
        total_area = 0.0
        average_elevation = 0.0
        max_elevation = -sys.float_info.max
        min_elevation = sys.float_info.max
        target_volume = volume
        total_available_volume = poly_atts['Total Volume\nBased on Slope']
        if cut:
            total_available_volume = -total_available_volume

        if abs(target_volume) > abs(total_available_volume):
            target_volume = total_available_volume
        for cell_index in cell_indices:
            total_area += area[cell_index]
            average_elevation += elevations[cell_index]
            max_elevation = max(max_elevation, elevations[cell_index])
            min_elevation = min(min_elevation, elevations[cell_index])
        average_elevation /= len(cell_indices)

        new_thickness = abs(target_volume / total_area)
        elevations_list = [
            max_elevation if cut else max_elevation + new_thickness * 3.0,
            min_elevation - new_thickness * 3.0 if cut else min_elevation
        ]
        volumes_list = []
        if cut:
            volumes_list.append(
                -SedimentVolumeManagementComputations.
                cut_constant_elevation(cell_indices, elevations, elevations_limits, area, elevations_list[0], None)
            )
            volumes_list.append(
                -SedimentVolumeManagementComputations.
                cut_constant_elevation(cell_indices, elevations, elevations_limits, area, elevations_list[-1], None)
            )
        else:
            volumes_list.append(
                SedimentVolumeManagementComputations.fill_constant_elevation(
                    cell_indices, elevations, elevations_limits, area, elevations_list[0], None
                )
            )
            volumes_list.append(
                SedimentVolumeManagementComputations.fill_constant_elevation(
                    cell_indices, elevations, elevations_limits, area, elevations_list[-1], None
                )
            )

        elevations_list.append(average_elevation - new_thickness if cut else average_elevation + new_thickness)

        tol = 1.0e-2
        max_iter = 100
        cur_iter = 0
        last_volume = 0.0
        while not math.isclose(last_volume, target_volume, abs_tol=tol) and cur_iter < max_iter:
            if cut:
                volumes_list.append(
                    -SedimentVolumeManagementComputations.cut_constant_elevation(
                        cell_indices, elevations, elevations_limits, area, elevations_list[-1], None
                    )
                )
            else:
                volumes_list.append(
                    SedimentVolumeManagementComputations.fill_constant_elevation(
                        cell_indices, elevations, elevations_limits, area, elevations_list[-1], None
                    )
                )
            last_volume = volumes_list[-1]
            # Figure out a new elevation to try
            elevations_list.sort()
            volumes_list.sort()
            new_elevation = numpy.interp(target_volume, volumes_list, elevations_list)
            elevations_list.append(new_elevation)
            cur_iter += 1

        if cut:
            result = SedimentVolumeManagementComputations.cut_constant_elevation(
                cell_indices, elevations, elevations_limits, area, elevations_list[-1], geom_type
            )
        else:
            result = SedimentVolumeManagementComputations.fill_constant_elevation(
                cell_indices, elevations, elevations_limits, area, elevations_list[-1], geom_type
            )
        return result
