"""Class for running the insert ewn feature command in a feedback dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.constraint import read_grid_from_file
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.ewn.tools.ugrid_subset import UGridSubset


class EwnPreviewRunner(QThread):
    """Class for running the insert ewn feature command in a feedback dialog."""
    processing_finished = Signal()

    def __init__(self, parent, ewn_process=None):
        """Constructor for class.

        Args:
            parent (:obj:`AssignEwnPolygonDialog`): parent dialog
            ewn_process (:obj:`EwnProcess`): The EWN process
        """
        super().__init__()
        self._parent = parent
        self._logger = logging.getLogger('xms.ewn')
        while len(self._logger.handlers) > 0:
            self._logger.removeHandler(self._logger.handlers[0])
        self._logger.level = logging.DEBUG
        self._ewn_process = ewn_process

    def run(self):
        """Inserts ewn features into an existing mesh."""
        try:
            if self._ewn_process is None:
                self._logger.info('Retreiving geometry from SMS.')
                data = self._parent._dlg_data
                source_geom_uuid = data['geom_uuid']
                query = data['query']
                do_ugrid = query.item_with_uuid(source_geom_uuid)
                co_grid = read_grid_from_file(do_ugrid.cogrid_file)
                self._parent.cogrid = co_grid
                self._parent._ugrids[source_geom_uuid] = co_grid.ugrid
                self._parent._subsets[source_geom_uuid] = UGridSubset(co_grid.ugrid)
                self._parent._subsets[source_geom_uuid].set_logger(self._logger)
                self._parent._subsets[source_geom_uuid].calc_cell_boxes()
            else:
                self._ewn_process.insert_feature()
        except:  # noqa
            if self._ewn_process is None:
                self._logger.exception('Error processing geometry.')
            else:
                self._logger.exception('Error inserting EWN feature.')
        finally:
            self.processing_finished.emit()


def preview_get_geometry_feedback(parent):
    """Run the Insert EWN Features command with a feedback dialog.

    Args:
        parent (:obj:`AssignEwnPolygonDialog`): The Qt parent window container

    Returns:
        (:obj:`tuple(bool, InsertEwnFeatureRunner)`):
            False if an error level message was logged during the operation,
            the runner object for the operation (contains output data)
    """
    LogEchoQSignalStream.reset_flags()
    worker = EwnPreviewRunner(parent)
    error_str = 'Error(s) encountered processing geometry. Review log output for more details.'
    warning_str = 'Warning(s) encountered processing geometry. Review log output for more details.'
    display_text = {
        'title': 'Processing Geometry',
        'working_prompt': 'Retreiving geometry from SMS. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully processed geometry.',
        'note': '',
        'auto_load': 'Close this dialog automatically when insertion of features is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.setModal(True)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker


def preview_ewn_insert_feedback(parent, ewn_process):
    """Run the Insert EWN Features command with a feedback dialog.

    Args:
        parent (:obj:`AssignEwnPolygonDialog`): The Qt parent window container
        ewn_process (:obj:`EwnProcess`): The EWN process

    Returns:
        (:obj:`tuple (bool, InsertEwnFeatureRunner)`):
        False if an error level message was logged during the operation,
        the runner object for the operation (contains output data)
    """
    LogEchoQSignalStream.reset_flags()
    worker = EwnPreviewRunner(parent, ewn_process)
    error_str = 'Error(s) encountered inserting features. Review log output for more details.'
    warning_str = 'Warning(s) encountered inserting features. Review log output for more details.'
    display_text = {
        'title': 'Insert EWN Feature',
        'working_prompt': 'Inserting feature into mesh/ugrid. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully processed geometry.',
        'note': '',
        'auto_load': 'Close this dialog automatically when insertion of features is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.setModal(True)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker
