"""Module for running the quadtree command in a feedback dialog."""

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.ewn.dmi.tool_runner_input_queries import ToolRunnerInputQueries
from xms.ewn.tools.refine_quadtree import QuadTreeRefiner


class RefineQuadTreeRunner(QThread):
    """Worker thread running the refine quadtree command."""
    processing_finished = Signal()

    def __init__(self, query, inputs):
        """Constructor for class.

        Args:
            query (:obj:`xms.api.dmi.Query`): query for xms communication
            inputs (:obj:`dict`): User input data from dialog
        """
        super().__init__()
        self._inputs = inputs
        self._logger = logging.getLogger('xms.ewn')
        self.tool = None
        self._queries = ToolRunnerInputQueries(query)

        self._ugrid = None
        self._ewn_coverage = None
        self.projection = None

    def run(self):
        """Refine a quadtree."""
        try:
            self._retrieve_xms_data()
            self._run_tool()
        except:  # noqa
            self._logger.exception('Error updating ADH friction.')
        finally:
            self.processing_finished.emit()

    def _run_tool(self):
        """Runs the refine quadtree process."""
        self.tool = QuadTreeRefiner(self._ugrid, self._ewn_coverage)
        self.tool.refine_quadtree()

    def _retrieve_xms_data(self):
        self._ugrid = self._queries.get_ugrid(self._inputs['ugrid_uuid'], return_co_grid=True)
        self._ewn_coverage = self._queries.get_coverage_data([self._inputs['ewn_uuid']], coverage_type='EWN')
        self.projection = self._queries.projection


def refine_quadtree_with_feedback(inputs, parent, query):
    """Run the Refine QuadTree command with a feedback dialog.

    Args:
        inputs (:obj:`dict`): User input data from dialog
        parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container
        query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

    Returns:
        (:obj:`tuple (bool, QuadTreeRefiner)`):

            False if an error level message was logged during the operation,
            the generator object for the operation (contains output data)
    """
    worker = RefineQuadTreeRunner(query, inputs)
    error_str = 'Error(s) encountered refining QuadTree Review log output for more details.'
    warning_str = 'Warning(s) encountered refining QuadTree. Review log output for more details.'
    display_text = {
        'title': 'Refining QuadTree',
        'working_prompt': 'Refining QuadTree. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully refined QuadTree.',
        'note': '',
        'auto_load': 'Close this dialog when operation completes.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker
