"""Class for running EWN coverage tools."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
import xms.ewn.components.ewn_component_display as ecd
from xms.ewn.dmi import ewn_component_queries as ecq
from xms.ewn.tools.runners.generate_roughness_dataset_runner import generate_roughness_dataset_with_feedback
from xms.ewn.tools.runners.insert_ewn_feature_runner import insert_ewn_features_with_feedback
from xms.ewn.tools.runners.insert_levee_runner import insert_levee_with_feedback
from xms.ewn.tools.runners.refine_quadtree_runner import refine_quadtree_with_feedback
from xms.ewn.tools.runners.update_adh_friction_runner import update_adh_friction_with_feedback


class ToolRunner:
    """Class for running EWN coverage tools."""
    INSERT_FEATURES_TOOL = 0
    GENERATE_ROUGHNESS_TOOL = 1
    INSERT_LEVEE_TOOL = 2
    UPDATE_ADH_FRICTION_TOOL = 3
    REFINE_QUADTREE = 4

    def __init__(self, ewn_comp, query, parent):
        """Constructor.

        Args:
            ewn_comp (:obj:`EwnCoverageComponent`): The component to run tools for
            query (:obj:`Query`): XMS interprocess communication object
            parent (:obj:`QtWidget`): Parent Qt dialog
        """
        self._comp = ewn_comp
        self._query = query
        self._parent = parent
        self._display_helper = None
        self._inputs = None
        self._errors = []

    def _get_geometry_pe_tree(self):
        """Get the geometry selector project explorer tree.

        Returns:
            (:obj:`bool`): True on success, False if error or no 2D Mesh/UGrid
        """
        self._display_helper = ecd.EwnCoverageDisplay(self._comp)
        problem = self._display_helper.get_project_explorer_for_command(self._query)
        if problem:  # No 2D Mesh/UGrid in project explorer tree or error getting the tree.
            self._errors.append(('WARNING', problem))
            return False
        return True

    def _select_inputs(self, tool_id):
        """Run an EWN coverage tool.

        Args:
            tool_id (:obj:`int`): One of INSERT_FEATURES_TOOL, GENERATE_ROUGHNESS_TOOL

        Returns:
            (:obj:`bool`): False if the user cancels
        """
        if tool_id == self.INSERT_FEATURES_TOOL:
            # Have the user select a target 2D Mesh or UGrid and input EWN Feature coverages.
            self._inputs = self._display_helper.select_insert_features_inputs(self._parent)
        elif tool_id == self.INSERT_LEVEE_TOOL:
            self._inputs = self._display_helper.select_insert_levee_inputs(self._parent, self._query)
            if self._inputs:
                self._inputs['coverage'] = self._comp.cov_uuid
        elif tool_id == self.GENERATE_ROUGHNESS_TOOL:
            self._inputs = self._display_helper.select_generate_roughness_inputs(self._parent, self._query)
        elif tool_id == self.UPDATE_ADH_FRICTION_TOOL:
            self._inputs = self._display_helper.select_update_adh_friction_inputs(self._parent, self._query)
        elif tool_id == self.REFINE_QUADTREE:
            self._inputs = self._display_helper.select_refine_quadtree_inputs(self._parent, self._query)

        if not self._inputs:
            return False  # User canceled, exit silently.
        return True

    def _run_insert_features_tool(self):
        """Run the Insert EWN Features tool."""
        # Run the Insert EWN Features operation with a feedback dialog.
        error, worker = insert_ewn_features_with_feedback(
            self._inputs['target_geometry'], self._inputs['lock_dataset_uuid'], self._inputs['ewn_coverages'],
            self._inputs['bias'], self._query, self._parent
        )
        if not error:  # Only send if no error.
            if worker.tool.non_intersecting_transects:  # Add a coverage containing the non-intersecting transects.
                ecq.add_non_intersecting_transects_coverage(
                    self._query, worker.tool.non_intersecting_transects, worker.tool.error_cov_name
                )
            else:  # Add the successfully generated output UGrid
                if worker.tool._is_cartesian:
                    ecq.add_output_cogrid(self._query, worker.tool.out_ugrid, projection=worker.projection)
                else:
                    ecq.add_output_ugrid(self._query, worker.tool.out_ugrid, projection=worker.projection)

    def _run_insert_levee_tool(self):
        """Run the insert Levee tool."""
        error, worker = insert_levee_with_feedback(
            self._inputs['ugrid'], self._inputs['adcirc_coverage'], self._inputs['coverage'], self._inputs['bias'],
            self._query, self._parent
        )
        if not error:  # Only send if no error.
            ecq.add_output_ugrid(self._query, worker.tool.out_ugrid, projection=worker.projection)
            ecq.add_adcirc_bc_coverage(self._query, worker.out_bc_coverage)

    def _run_generate_roughness_tool(self):
        """Run the Generate Roughness Data Set tool."""
        # Run the Generate Roughness Data Set operation with a feedback dialog.
        error, tool = generate_roughness_dataset_with_feedback(self._inputs, self._query, self._parent)
        if not error:  # Only send if no error.
            self._query.add_dataset(tool.output_dset)

    def _run_update_adh_roughness_tool(self):
        error, tool = update_adh_friction_with_feedback(self._inputs, self._parent, self._query)
        if not error:
            ecq.add_adh_materials_coverage(self._query, tool.out_adh_mat_coverage)

    def _run_refine_quadtree_tool(self):
        error, worker = refine_quadtree_with_feedback(self._inputs, self._parent, self._query)
        if not error:
            ecq.add_output_cogrid(self._query, worker.tool.out_ugrid, projection=worker.projection)

    def run_tool(self, tool_id):
        """Run an EWN coverage tool.

        Args:
            tool_id (:obj:`int`):
                One of INSERT_FEATURES_TOOL, GENERATE_ROUGHNESS_TOOL, CHANNEL_INSERT_TOOL

        Returns:
            (:obj:`list`):
                Error messages to send back to XMS. Structured as required by xms.components
        """
        if not self._get_geometry_pe_tree():
            return self._errors  # No valid geometry

        if not self._select_inputs(tool_id):
            return self._errors  # Should be empty. User canceled, exit silently.

        if tool_id == self.INSERT_FEATURES_TOOL:
            self._run_insert_features_tool()
        elif tool_id == self.INSERT_LEVEE_TOOL:
            self._run_insert_levee_tool()
        elif tool_id == self.GENERATE_ROUGHNESS_TOOL:
            self._run_generate_roughness_tool()
        elif tool_id == self.UPDATE_ADH_FRICTION_TOOL:
            self._run_update_adh_roughness_tool()
        elif tool_id == self.REFINE_QUADTREE:
            self._run_refine_quadtree_tool()

        return self._errors
