"""Module for running the update adh friction command in a feedback dialog."""

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.ewn.dmi.tool_runner_input_queries import ToolRunnerInputQueries
from xms.ewn.tools.update_adh_friction import AdhFrictionUpdater


class UpdateAdhFrictionRunner(QThread):
    """Worker thread running the update adh friction command."""
    processing_finished = Signal()

    def __init__(self, query, inputs):
        """Constructor for class.

        Args:
            query (:obj:`xms.api.dmi.Query`): query for xms communication
            inputs (:obj:`dict`): User input data from dialog
        """
        super().__init__()
        self._inputs = inputs
        self._logger = logging.getLogger('xms.ewn')
        self.tool = None
        self._queries = ToolRunnerInputQueries(query)

        # XMS Data
        self._material_coverage_data = None
        self._ewn_coverage_data = None
        self._out_adh_mat_coverage = None

        # Rename or move?
        self._new_mat_comp = None

    def run(self):
        """Updates ADH friction from an ewn coverage."""
        try:
            self._retrieve_xms_data()
            self._run_tool()
        except:  # noqa
            self._logger.exception('Error updating ADH friction.')
        finally:
            self.processing_finished.emit()

    def _run_tool(self):
        """Runs the update adh friction process."""
        self.tool = AdhFrictionUpdater(
            target_cov=self._material_coverage_data,
            source_cov=self._ewn_coverage_data,
        )
        self.tool.update_adh_friction()

    def _retrieve_xms_data(self):
        self._material_coverage_data = self._queries.get_coverage_data(
            [self._inputs['target_uuid']], coverage_type='ADH_MAT'
        )
        self._ewn_coverage_data = self._queries.get_coverage_data([self._inputs['ewn_uuid']], coverage_type='EWN')


def update_adh_friction_with_feedback(inputs, parent, query):
    """Run the Update Adh Friction command with a feedback dialog.

    Args:
        inputs (:obj:`dict`): User input data from dialog
        parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container
        query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

    Returns:
        (:obj:`tuple (bool, AdhFrictionGenerator)`):
            False if an error level message was logged during the operation,
            the generator object for the operation (contains output data)
    """
    worker = UpdateAdhFrictionRunner(query, inputs)
    error_str = 'Error(s) encountered updating ADH friction. Review log output for more details.'
    warning_str = 'Warning(s) encountered updating ADH friction. Review log output for more details.'
    display_text = {
        'title': 'Updating ADH Friction',
        'working_prompt': 'Updating ADH friction. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully updated ADH friction.',
        'note': '',
        'auto_load': 'Close this dialog when operation completes.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker.tool
