#pragma once

#ifndef _GSSHA80_HPP_
#define _GSSHA80_HPP_

//<! For GSSHA version 8.0.16

//<! define symbols to import from the GSSHA library
#define DllImport __declspec(dllimport)

//<! C++ standard library includes used below 
#include <string>
#include <vector>

//<! gssha_bmi inherits from the standard BMI definition. It needs to be included.
#include "bmi.hxx"

//<! put the GSSHA library classes into the gssha namespace
namespace gssha {

    //<! GSSHA basic type definitions
    typedef int GBOOL;
    typedef long GINT;
    typedef double GDOUBLE;
    typedef double GFLOAT;
    typedef char  GCHAR;
    typedef unsigned long GUINT;

#ifndef _MPI
#define MPI_Comm GINT
#endif

    //<! Forward typedef GSSHA Class definitions
    typedef class mpi_setup mpi_setup;
    typedef class replace_io replace_io;
    typedef class replace_io* replace_io_ptr;
    typedef class observed_data observed_data;
    typedef class run_results run_results;
    typedef class gssha_sim gssha_sim;
    typedef class main_var_struct* main_var_struct_ptr;
    typedef class gssha_bmi gssha_bmi;

 
    extern "C" {

        // Function Headers
        // 
        //<! NextGen Framework Functions
        DllImport gssha_bmi* bmi_model_create();
        DllImport void bmi_model_destroy(gssha_bmi* ptr);

        //<! Standard create/destroy functions
        DllImport gssha_sim* model_create();
        DllImport void model_destroy(gssha_sim* ptr);

        //<! Run Results create/destroy functions
        DllImport run_results* run_results_create();
        DllImport void run_results_destroy(run_results* rr);

        //<! MPI Setup create/destroy functions
        DllImport mpi_setup* mpi_setup_create();
        DllImport void mpi_setup_destroy(mpi_setup* ms);

        //<! observed_data create/destroy functions
        DllImport observed_data* observed_data_create();
        DllImport void observed_data_destroy(observed_data* ms);

        //<! functions to work with GSSHA dates
        DllImport GDOUBLE greg_2_jul(GINT  year, GINT  mon, GINT  day, GINT  h, GINT  mi, GDOUBLE  se);
        DllImport void calc_date(GDOUBLE  jd, GINT* y, GINT* m, GINT* d, GINT* h, GINT* mi, GDOUBLE* sec);
        DllImport GINT dayofweek(GDOUBLE  j);
        DllImport GINT dayofyear(GDOUBLE j);


        //<! Just run GSSHA...
        DllImport GINT main_gssha8(const GCHAR* prj_name, gssha::run_results* calib_data, const char input[]);

        //<! Run GSSHA once in calibration mode
        DllImport double main_calib(char* replace_file_name, char* prjname,
            double* xpar, observed_data* obs, GINT npar,
            char input[], GINT runIDNum, mpi_setup* mpidata);


    }

    //<! Partial definitions of GSSHA classes (public portion)

    class DllImport run_results {
    public:
        run_results();
        ~run_results();
        GINT num_data_points;
        GINT num_locations;

        double* com_peaks;
        double* com_volumes;

        double* qp_loc;
        double* vol_loc;

    };

    //<! holds observed data for use by the run_results container
    class DllImport observed_data
    {
    public:
        observed_data();
        ~observed_data();

        GINT read_observed_weighted_peakvol(char* infile);
        double compute_weighted_hmle(double* sim_peaks, double* sim_vols, GINT myid);
        double compute_weighted_rmse(double* sim_peaks, double* sim_vols, GINT myid);
        double compute_weighted_logerr(double* sim_peaks, double* sim_vols, GINT myid);
        double compute_cost(double* sim_peaks, double* sim_vols, GINT myid);
        double get_max_peak(double* sim_peaks);
        double get_max_vol(double* sim_vols);
        void output_results();
        double compute_results(double* sim_peaks, double* sim_vols, GINT myid, GINT num_param, double* x);
        void print_results(GINT my_id, GINT my_partition, GINT my_partition_id,
            GINT through_partition_comm, GINT npartitions, GINT total_all_runs,
            GINT total_partition_runs, replace_io_ptr rio);
        void print_comparison(double* sim_peaks, double* sim_vols, GINT myid);

        void compute_unweighted_results(double* sim_peaks, double* sim_vols, GINT myid);

        double get_hmle();
        double get_rmse();
        double get_cost();
        void reset_log();
    };


    class gssha_sim {
    public:
        DllImport gssha_sim(gssha::mpi_setup* mpidataptr);
        DllImport gssha_sim();
        DllImport gssha_sim(gssha::gssha_sim& gs);
        DllImport ~gssha_sim();

        // primary gssha routines
        DllImport int  main_gssha(const char* prj_name, gssha::run_results* calib_data, const char input[]);
        DllImport void  main_gssha_init(const char* prj_name, gssha::run_results* calib_data, const char input[]);
        DllImport void  main_gssha_run_one_step();
        DllImport void  main_gssha_run_standard();
        DllImport void  main_gssha_finalize();

        // these routines handle the try/throw/catch error handling
        DllImport bool  init_gssha(const char* prj_name, gssha::run_results* calib_data, const char input[]);
        DllImport bool  finalize_gssha();

        DllImport std::string version();

        DllImport double  main_calib(char* replace_file_name, char* prjname, double* xpar, gssha::observed_data* obs, int npar, const char input[], int runIDNum);

    };

    //<! The class that wraps the gssha_sim class with the bmi definition
    class DllImport gssha_bmi : public bmi::Bmi {

         //<! Basic Model Interface (BMI) methods
    public:
        gssha_bmi();
        gssha_bmi(gssha::gssha_bmi& gb); // copy constructor
        gssha_bmi(gssha::mpi_setup* mpienvironment);
        ~gssha_bmi();

        //<! Model control functions.
        virtual void Initialize(std::string config_file);
        virtual void Update();
        virtual void UpdateUntil(double time);
        virtual void Finalize();

        //<! Model information functions.
        virtual std::string GetComponentName();
        virtual int GetInputItemCount();
        virtual int GetOutputItemCount();
        virtual std::vector<std::string> GetInputVarNames();
        virtual std::vector<std::string> GetOutputVarNames();

        //<! Variable information functions
        virtual int GetVarGrid(std::string name);
        virtual std::string GetVarType(std::string name);
        virtual std::string GetVarUnits(std::string name);
        virtual int GetVarItemsize(std::string name);
        virtual int GetVarNbytes(std::string name);
        virtual std::string GetVarLocation(std::string name);

        virtual double GetCurrentTime();
        virtual double GetStartTime();
        virtual double GetEndTime();
        virtual std::string GetTimeUnits();
        virtual double GetTimeStep();

        //<! Variable getters
        virtual void GetValue(std::string name, void* dest);
        virtual void* GetValuePtr(std::string name);
        virtual void GetValueAtIndices(std::string name, void* dest, int* inds, int count);

        //<! Variable setters
        virtual void SetValue(std::string name, void* src);
        virtual void SetValueAtIndices(std::string name, int* inds, int count, void* src);

        //<! Grid information functions
        virtual int GetGridRank(const int grid);
        virtual int GetGridSize(const int grid);
        virtual std::string GetGridType(const int grid);

        virtual void GetGridShape(const int grid, int* shape);
        virtual void GetGridSpacing(const int grid, double* spacing);
        virtual void GetGridOrigin(const int grid, double* origin);

        virtual void GetGridX(const int grid, double* x);
        virtual void GetGridY(const int grid, double* y);
        virtual void GetGridZ(const int grid, double* z);

        virtual int GetGridNodeCount(const int grid);
        virtual int GetGridEdgeCount(const int grid);
        virtual int GetGridFaceCount(const int grid);

        virtual void GetGridEdgeNodes(const int grid, int* edge_nodes);
        virtual void GetGridFaceEdges(const int grid, int* face_edges);
        virtual void GetGridFaceNodes(const int grid, int* face_nodes);
        virtual void GetGridNodesPerFace(const int grid, int* nodes_per_face);
    };

    //<! this class is used to set up parallel processing capabilities. Contrary to its name, it does OpenMP and MPI. On PCs, it currently only does OpenMP.
    class DllImport mpi_setup {
    public:
        mpi_setup();
        mpi_setup(gssha::mpi_setup& ms);
        ~mpi_setup();

        void init_mask(GINT GNUM, GINT rows, GINT cols, GINT** mask);

        int GetMyId();
        int GetNPes();
        int GetMyPartitionRank();
        int GetNPartitions();
        int GetMyPartition();

        int SetupPartitions(GINT num_p_per_model);

        void merge_in_x(double* dataset);
        void merge_in_y(double* dataset);

        void communicate_across_partition(GINT total_vals, double* vals);
        double* communicate_through_partitions(GINT total_vals, double* vals);

        int get_through_partition_comm();
        int get_npartitions();

    };


}


#endif