"""SimComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util, TreeNode
from xms.data_objects.parameters import Component
from xms.gmi.component_bases.sim_component_base import SimComponentBase
from xms.gmi.data.generic_model import Parameter, Section
from xms.gmi.gui.dataset_callback import DatasetRequest

# 4. Local modules
from xms.funwave.data.model import get_model
from xms.funwave.dmi.xms_data import XmsData

SIM_DATA_MAINFILE = 'sim_comp.nc'


def get_component_data_object(main_file, comp_uuid, unique_name, name=''):
    """Create a data_object Component to send back to SMS to be built.

    Args:
        main_file (:obj:`str`): Path to the component main file
        comp_uuid: (:obj:`str`): UUID of the component
        unique_name (:obj:`str`): XML component unique name
        name (:obj:`str`): Tree item name of the component

    Returns:
        (:obj:`Component`): data_object for the new component
    """
    return Component(
        name=name,
        comp_uuid=comp_uuid,
        main_file=main_file,
        model_name='FUNWAVE',
        unique_name=unique_name,
        locked=False
    )


class SimComponent(SimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the FUNWAVE model simulation."""

    def __init__(self, main_file):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)

        self._section_dialog_keyword_args['enable_unchecked_groups'] = True
        self._section_dialog_keyword_args['hide_checkboxes'] = True

    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return get_model().global_parameters

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> str | TreeNode:  # pragma: no cover
        """
        Handle a request for information when picking a dataset. This is an override from xms.gmi.
        """
        if request == DatasetRequest.GetTree:
            xms_data = XmsData(query=self._query)
            grid_ptr_node = tree_util.descendants_of_type(tree_root=xms_data.sim_item, xms_types=['TI_CGRID2D_PTR'],
                                                          allow_pointers=True, only_first=True)
            if grid_ptr_node is None:
                return None
            ugrid_uuid = grid_ptr_node.uuid
            grid_node = tree_util.find_tree_node_by_uuid(self._query.copy_project_tree(), ugrid_uuid)
            filtered_grid_tree = tree_util.trim_tree_to_items_of_type(grid_node, xms_types=['TI_SFUNC'])
            return filtered_grid_tree
        return super()._dataset_callback(request, parameter)
