"""Simulation model control variables."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import logging
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data.generic_model import Parameter, Type

# 4. Local modules
from xms.funwave.data.wavemaker import WAVE_MAKER_OPTIONS


LARGE = 999999.0
UNSPECIFIED = '-- Unspecified --'


def get_model() -> GenericModel:
    """Create the hard bottom data definition."""
    gm = GenericModel(exclusive_point_conditions=True)

    # Coverage output points
    pp = gm.point_parameters
    output = pp.add_group(group_name='output_point', label='Output Point', is_default=True)
    output.add_text('name', 'Name')

    # Project
    proj = gm.global_parameters.add_group('Project', 'Project', is_active=True)
    proj.add_text('TITLE', 'Project title (TITLE)', '')
    proj.add_integer('PX', 'Number of processors to use in X (PX)', 1, low=0)
    proj.add_integer('PY', 'Number of processors to use in Y (PY)', 1, low=0)

    # Time
    time = gm.global_parameters.add_group('Time', 'Time')
    time.add_float('TOTAL_TIME', 'Simulation time in seconds (TOTAL_TIME)', 0.0, low=0.0)
    time.add_float('PLOT_INTV', 'Output interval in seconds (PLOT_INTV)', 1.0, low=0.0)
    time.add_float('SCREEN_INTV', 'Time interval of screen print (SCREEN_INTV)', 1.0, low=0.0)
    time.add_float('PLOT_INTV_STATION', 'Time interval of gauge output (PLOT_INTV_STATION)', 1.0, low=0.0)
    time.add_float('PLOT_START_TIME', 'Start time for output of field results (PLOT_START_TIME)', 0.0, low=0.0)
    time.add_float('T_INTV_mean', 'Mean interval (T_INTV_mean)', 999999.0)
    time.add_float('STEADY_TIME', 'Steady time (STEADY_TIME)', LARGE)
    method = time.add_boolean('USE_DT_fixed', 'Use fixed DT (USE_DT_fixed)', default=False)
    flags = {True: True, False: False}
    time.add_float('DT_fixed', 'Time step (s) (DT_fixed)', 0.0, low=0.0).add_dependency(method, flags)

    # Bathymetry
    bathymetry = gm.global_parameters.add_group('Bathymetry', 'Bathymetry')
    method = bathymetry.add_option('DEPTH_TYPE', 'Grid size(n) in y direction (DEPTH_TYPE)', 'DATA',
                                   ['DATA', 'FLAT', 'SLOPE'], 'Depth input type')
    flags = {'DATA': True, 'FLAT': False, 'SLOPE': False}
    # bathymetry.add_dataset('DEPTH_FILE', 'Depth (DEPTH_FILE)', 'Depth file').add_dependency(method, flags)
    flags = {'DATA': False, 'FLAT': True, 'SLOPE': True}
    bathymetry.add_float('DEPTH_FLAT', 'Water depth of flat bottom (DEPTH_FLAT)', 10.0).add_dependency(method, flags)
    flags = {'DATA': False, 'FLAT': False, 'SLOPE': True}
    bathymetry.add_float('SLP', 'Slope (SLP)', 0.1).add_dependency(method, flags)
    bathymetry.add_float('Xslp', 'Starting x (m) of a slope (Xslp)', 0.0).add_dependency(method, flags)
    bathymetry.add_float('WaterLevel', 'Added water level (WaterLevel)', 0.0)

    # Physics
    physics = gm.global_parameters.add_group('Physics', 'Physics')
    method = physics.add_boolean('DISPERSION', 'Include disperion terms (DISPERSION)', default=True)
    flags = {True: True, False: False}
    method2 = physics.add_option('Dispersion model', 'Dispersion model', 'Fully nonlinear equations',
                                 ['Fully nonlinear equations', 'Nwogu equations', 'Linear shallow water equations',
                                  'Specify Gammma (advanced)'])
    method2.add_dependency(method, flags)
    flags = {'Fully nonlinear equations': False, 'Nwogu equations': False, 'Linear shallow water equations': False,
             'Specify Gammma (advanced)': True}
    physics.add_float('Gamma1', 'Gamma1', 1.0, low=0.0, high=1.0).add_dependency(method2, flags)
    physics.add_float('Gamma2', 'Gamma2', 1.0, low=0.0, high=1.0).add_dependency(method2, flags)
    physics.add_float('Gamma3', 'Gamma3', 1.0, low=0.0, high=1.0).add_dependency(method2, flags)
    physics.add_float('Beta_ref', 'Parameter \ud835\udefd defined for the reference level (Beta_ref)', -0.531)
    breaking_options = ['Eddy-Viscosity breaking used in numerical scheme (VISCOSITY_BREAKING=T & SHOW_BREAKING=T)',
                        'Shock-Capturing breaking used in numerical scheme (VISCOSITY_BREAKING=F & SHOW_BREAKING=F)']
    method = physics.add_option('VISCOSITY_BREAKING', 'Breaking option (VISCOSITY_BREAKING & SHOW_BREAKING)',
                                breaking_options[0], breaking_options)
    flags = {breaking_options[0]: True,
             breaking_options[1]: False}
    method2 = physics.add_boolean('Specify Cbrk (advanced)', 'Specify Cbrk (Advanced)', default=False)
    method2.add_dependency(method, flags)
    flags = {True: True, False: False}
    physics.add_float('Cbrk1', 'Cbrk1', 0.45).add_dependency(method2, flags)
    physics.add_float('Cbrk2', 'Cbrk2', 0.35).add_dependency(method2, flags)
    flags = {breaking_options[0]: False,
             breaking_options[1]: True}
    physics.add_float('SWE_ETA_DEP', 'Shock-capturing height/depth ratio (SWE_ETA_DEP)', 0.8).add_dependency(method,
                                                                                                             flags)
    physics.add_boolean('ROLLER_EFFECT', 'Enable roller effects (ROLLER_EFFECT)', default=False)
    # Friction (under Physics group)
    # friction = physics.add_group('Friction', 'Friction')
    method = physics.add_boolean('FRICTION_MATRIX', 'Friction option (FRICTION_MATRIX)', default=False)
    flags = {True: True, False: False}
    physics.add_dataset('FRICTION_FILE', 'Friction (FRICTION_FILE)', 'Friction dataset').add_dependency(method, flags)
    flags2 = {True: False, False: True}
    physics.add_float('Cd', 'Fixed bottom friction coefficient (Cd)', 0.0).add_dependency(method, flags2)
    physics.add_float('WAVEMAKER_Cbrk', 'Breaking parameter inside wavemaker (WAVEMAKER_Cbrk)', 1.0)

    # Numerical parameters
    numerical = gm.global_parameters.add_group('Numerical parameters', 'Numerical parameters')
    numerical.add_float('CFL', 'CFL number (CFL)', 0.5, low=0.5)
    numerical.add_float('FroudeCap', 'Cap for Froude number in velocity calculation (FroudeCap)', 3.0, low=1.0,
                        high=10.0)
    numerical.add_float('MinDepth', 'Minimum water depth (m) for wetting and drying scheme (MinDepth)', 0.1)
    # MinDepthFrc is for legacy reasons; do not show user, only put it in the file, with the same values as MinDepth.
    # numerical.add_float('MinDepthFrc', 'Merge to MinDepth for Version 3.1 or higher (MinDepthFrc)', 0.1)
    method = numerical.add_boolean('BATHY_CORRECTION', 'Bathymetry correction (BATHY_CORRECTION)', default=False)
    flags = {True: True, False: False}
    numerical.add_float('SmoothBelowDepth', 'Smoothing start depth (m) (SmoothBelowDepth)', -999999.0).add_dependency(
        method, flags)
    numerical.add_float('SlopeCap', 'Smooth slope cap (SlopeCap)', 1.0).add_dependency(method, flags)
    method = numerical.add_boolean('Numerical_advanced', 'Enable advanced numerical options', default=False)
    flags = {True: True, False: False}
    numerical.add_option('Time_Scheme', 'Time stepping option (Time_Scheme)', 'Runge_Kutta',
                         ['Runge_Kutta', 'Predictor_Corrector']
                         ).add_dependency(method, flags)
    numerical.add_option('CONSTRUCTION', 'Construction method (CONSTRUCTION)', 'HLL', ['HLL', 'HLLC']
                         ).add_dependency(method, flags)
    numerical.add_option('HIGH_ORDER', 'Spatial scheme option (HIGH_ORDER)', 'FOURTH', ['SECOND', 'THIRD', 'FOURTH']
                         ).add_dependency(method, flags)

    # Wave-maker
    wave_maker = gm.global_parameters.add_group('Wave-maker', 'Wave-maker')
    wm_options = list(WAVE_MAKER_OPTIONS.values())
    default = wm_options[8]
    method = wave_maker.add_option('WAVEMAKER', 'Wave-maker type (WAVEMAKER)', default, wm_options)

    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: True, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_option('WAVE_DATA_TYPE', 'Type of wave data needed (WAVE_DATA_TYPE)', 'DATA',
                          ['DATA', 'TMA2D', 'JON2D', 'JON1D']).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: True, wm_options[2]: True, wm_options[3]: True,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('AMP', 'Amplitude (m) of initial η (AMP)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: True, wm_options[3]: True,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('DEP', 'Water depth at wavemaker location (DEP)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: True,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('LAGTIME', 'Time lag (s) (LAGTIME)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: True, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('XWAVEMAKER', 'x (m) coordinate for WAVEMAKER = INI\\_SOL. (XWAVEMAKER)', 0.0, low=0.0
                         ).add_dependency(method, flags)
    flags = {wm_options[0]: True, wm_options[1]: True, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Xc', 'X (m) coordinate of the center of a rectangular hump (Xc)', 0.0, low=0.0
                         ).add_dependency(method, flags)
    flags = {wm_options[0]: True, wm_options[1]: True, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Yc', 'Y (m) coordinate of the center of a rectangular hump (Yc)', 0.0, low=0.0
                         ).add_dependency(method, flags)
    flags = {wm_options[0]: True, wm_options[1]: True, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('WID', 'Width (m) of a rectangular hump (WID)', 0.0, low=0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: True, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Time_ramp', 'Time ramp (s) (Time_ramp)', 0.0, low=0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Delta_WK', 'Width parameter δ (Delta_WK)', 0.5, low=0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: True, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('DEP_WK', 'Water depth (m) (DEP_WK)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: True, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Xc_WK', 'X coordinate (m) (Xc_WK)', 0.0, low=0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: True, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Yc_WK', 'Y coordinate (m) (Yc_WK)', 0.0, low=0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: True, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Ywidth_WK', 'Width (m) in y direction (Ywidth_WK)', 999999.0, low=0.0).add_dependency(
        method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: True, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Tperiod', 'Period (s) of regular wave (Tperiod)', 0.0, low=0.0).add_dependency(
        method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: True, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('AMP_WK', 'Amplitude (m) of regular wave (AMP_WK)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: True, wm_options[9]: False, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Theta_WK', 'Direction (degrees) of regular wave (Theta_WK)', 0.0).add_dependency(
        method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_integer('Nfreq', 'Number of frequency components (Nfreq)', 45, low=0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_integer('Ntheta', 'Number of direction components (Ntheta)', 24, low=0).add_dependency(
        method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('FreqPeak', 'Peak frequency (1/s) (FreqPeak)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('FreqMin', 'Low frequency cutoff (1/s) (FreqMin)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('FreqMax', 'High frequency cutoff (1/s) (FreqMax)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Hmo', 'Hmo (m) (Hmo)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: True, wm_options[6]: True, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('GammaTMA', 'TMA parameter \ud835\udefe (GammaTMA)', 3.3).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: True,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('ThetaPeak', 'Peak direction (degrees) (ThetaPeak)', 0.0).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: False, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('Sigma_Theta', 'Parameter of directional spectrum (Sigma_Theta)', 10.0).add_dependency(
        method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: True, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_boolean('EqualEnergy', 'Enable frequency splitting based on Equal-Energy (EqualEnergy)',
                           default=False).add_dependency(method, flags)
    flags = {wm_options[0]: False, wm_options[1]: False, wm_options[2]: False, wm_options[3]: False,
             wm_options[4]: False, wm_options[5]: False, wm_options[6]: False, wm_options[7]: False,
             wm_options[8]: False, wm_options[9]: False, wm_options[10]: True, wm_options[11]: False,
             wm_options[12]: False, wm_options[13]: False}
    wave_maker.add_float('alpha_c', 'Percentage of coherence (alpha_c)', 0.0).add_dependency(method, flags)

    # WAVE_MAKER_OPTIONS = {
    #   0  'INI_REC': 'Initial rectangular hump (INI_REC)',
    #   1  'INI_GAUS': 'Initial Gaussian hump (INI_GAUS)',
    #   2  'INI_SOL': 'Initial solitary wave propagation in +x direction (INI_SOL)',
    #   3  'LEF_SOL': 'Left boundary solitary (LEF_SOL)',
    #   4  'LEFT_BC_IRR': 'Wavemaker in the left boundary (LEFT_BC_IRR)',
    #   5  'TMA_1D': 'TMA 1D spectrum wavemaker (TMA_1D)',
    #   6  'JON_1D': 'JONSWAP 1D spectrum wavemaker (internal) (JON_1D)',
    #   7  'JON_2D': 'JONSWAP spectrum wavemaker (internal) (JON_2D)',
    #   8  'WK_REG': 'Wei and Kirby 1999 internal wave maker (WK_REG)',
    #   9  'WK_IRR': 'Wei and Kirby 1999 TMA spectrum wavemaker (internal) (WK_IRR)',
    #  10  'WK_NEW_IRR': 'Wei and Kirby directional wavemaker (WK_NEW_IRR)',
    #  11  'WK_TIME_SERIES': 'FFT of a time series (WK_TIME_SERIES)',
    #  12  'WK_DATA2D': '2D directional spectrum data specified in WaveCompFile (WK_DATA2D)',
    #  13  'WK_NEW_DATA2D': 'Wei and Kirby wave data (WK_NEW_DATA2D)',
    # }

    # Initial Conditions
    initial_conditions = gm.global_parameters.add_group('Initial conditions', 'Initial conditions')
    flags = {True: True, False: False}
    method = initial_conditions.add_boolean('INI_UVZ', 'Use hot start (INI_UVZ)', default=False)
    initial_conditions.add_dataset('ETA_FILE', 'Name of file for initial \ud835\udf02 (ETA_FILE)',
                                   'ETA_FILE').add_dependency(method, flags)
    initial_conditions.add_dataset('U_FILE', 'Name of file for initial u (U_FILE)', 'U_FILE').add_dependency(method,
                                                                                                             flags)
    initial_conditions.add_dataset('V_FILE', 'Name of file for initial v (V_FILE)', 'V_FILE').add_dependency(method,
                                                                                                             flags)
    # initial_conditions.add_input_file('MASK_FILE', 'Name of file for initial MASK (MASK_FILE)', '',
    #                                   'Text files (*.txt);;Any file (*.*)').add_dependency(method, flags)

    # Boundary conditions
    boundary_conditions = gm.global_parameters.add_group('Boundary conditions', 'Boundary conditions')
    boundary_conditions.add_boolean('PERIODIC', 'Enable periodic boundary condition in the y-direction (PERIODIC)',
                                    default=False)
    boundary_conditions.add_boolean('TIDAL_BC_ABS', 'Enable tidal absorbing boundary conditions (TIDAL_BC_ABS)',
                                    default=False)
    boundary_conditions.add_boolean('TIDAL_BC_GEN_ABS',
                                    'Enable combined tidal and absorbing-generating boundary conditions '
                                    '(TIDAL_BC_GEN_ABS)', default=False)
    method = boundary_conditions.add_option('TideBcType', 'Tide BC Data type (TideBcType)', UNSPECIFIED,
                                            [UNSPECIFIED, 'CONSTANT', 'DATA'])
    flags = {UNSPECIFIED: False, 'CONSTANT': True, 'DATA': False}
    flags2 = {True: True, False: False}
    method2 = boundary_conditions.add_boolean('TideWest', 'Enable tide WEST boundary (TideWest)', default=False)
    method2.add_dependency(method, flags)
    boundary_conditions.add_float('TideWest_ETA', 'Constant ETA value at the WEST boundary (TideWest_ETA)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideWest_U', 'Constant U value at the WEST boundary (TideWest_U)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideWest_V', 'Constant V value at the WEST boundary (TideWest_V)', 0.0
                                  ).add_dependency(method2, flags2)
    method2 = boundary_conditions.add_boolean('TideEast', 'Enable tide EAST boundary (TideEast)', default=False)
    method2.add_dependency(method, flags)
    boundary_conditions.add_float('TideEast_ETA', 'Constant ETA value at the EAST boundary (TideEast_ETA)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideEast_U', 'Constant U value at the EAST boundary (TideEast_U)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideEast_V', 'Constant v value at the EAST boundary (TideEast_V)', 0.0
                                  ).add_dependency(method2, flags2)
    method2 = boundary_conditions.add_boolean('TideSouth', 'Enable tide SOUTH boundary (TideSouth)', default=False)
    method2.add_dependency(method, flags)
    boundary_conditions.add_float('TideSouth_ETA', 'Constant ETA value at the SOUTH boundary (TideSouth_ETA)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideSouth_U', 'Constant U value at the SOUTH boundary (TideSouth_U)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideSouth_V', 'Constant V value at the SOUTH boundary (TideSouth_V)', 0.0
                                  ).add_dependency(method2, flags2)
    method2 = boundary_conditions.add_boolean('TideNorth', 'Enable tide NORTH boundary (TideNorth)', default=False)
    method2.add_dependency(method, flags)
    boundary_conditions.add_float('TideNorth_ETA', 'Constant ETA value at the NORTH boundary (TideNorth_ETA)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideNorth_U', 'Constant U value at the NORTH boundary (TideNorth_U)', 0.0
                                  ).add_dependency(method2, flags2)
    boundary_conditions.add_float('TideNorth_V', 'Constant V value at the NORTH boundary (TideNorth_V)', 0.0
                                  ).add_dependency(method2, flags2)

    # Sponge layer
    sponge_layer = gm.global_parameters.add_group('Sponge layer', 'Sponge layer')
    method = sponge_layer.add_boolean('DIRECT_SPONGE', 'Enable L-D type sponge (DIRECT_SPONGE)', default=False)
    flags = {True: True, False: False}
    sponge_layer.add_float('R_sponge', 'Decay rate (R_sponge)', 0.85).add_dependency(method, flags)
    sponge_layer.add_float('A_sponge', 'Maximum damping magnitude (A_sponge)', 5.0).add_dependency(method, flags)
    method = sponge_layer.add_boolean('FRICTION_SPONGE', 'Enable friction type sponge (FRICTION_SPONGE)', default=False)
    sponge_layer.add_float('CDsponge', 'Maximum Cd (CDsponge)', 5.0).add_dependency(method, flags)
    method = sponge_layer.add_boolean('DIFFUSION_SPONGE', 'Enable diffusion type sponge (DIFFUSION_SPONGE)',
                                      default=False)
    sponge_layer.add_float('Csp', 'The maximum diffusion coefficient for diffusion type sponge (Csp)', 0.1
                           ).add_dependency(method, flags)
    sponge_layer.add_float('Sponge_west_width', 'Width (m) of sponge layer at west boundary (Sponge_west_width)', 0.0)
    sponge_layer.add_float('Sponge_east_width', 'Width (m) of sponge layer at east boundary (Sponge_east_width)', 0.0)
    sponge_layer.add_float('Sponge_south_width', 'Width (m) of sponge layer at south boundary (Sponge_south_width)',
                           0.0)
    sponge_layer.add_float('Sponge_north_width', 'Width (m) of sponge layer at north boundary (Sponge_north_width)',
                           0.0)

    # Breakwater and Obstacles
    bwater_and_obstacles = gm.global_parameters.add_group('Breakwater and Obstacles', 'Breakwater and Obstacles')
    bwater_and_obstacles.add_dataset('OBSTACLE_FILE', 'Obstacle file (OBSTACLE_FILE)', 'Obstacle file')
    bwater_and_obstacles.add_dataset('BREAKWATER_FILE', 'Breakwater file (BREAKWATER_FILE)', 'Breakwater file')
    bwater_and_obstacles.add_float('BreakWaterAbsorbCoef', 'Value of maximum Cd (BreakWaterAbsorbCoef)', 10.0)

    # Output
    output = gm.global_parameters.add_group('Output', 'Output')
    output.add_option('FIELD_IO_TYPE', 'Output format (FIELD_IO_TYPE)', 'ASCII', ['ASCII', 'BINARY'])
    output.add_boolean('Radiation', 'Enable output of radiation stress (Radiation)', default=False)
    # output.add_boolean('U', 'Enable U output (U)', default=True)
    # output.add_boolean('V', 'Enable V output (V)', default=True)
    output.add_boolean('ETA', 'Enable ETA output (ETA)', default=True)
    output.add_boolean('Velocity', 'Enable Velocity output (U and V)', default=True)
    # output.add_boolean('Umean', 'Enable output of Umean (Umean)', default=False)
    # output.add_boolean('Vmean', 'Enable output of Vmean (Vmean)', default=False)
    output.add_boolean('Average velocity', 'Enable average velocity output (Umean and Vmean)', default=False)
    output.add_boolean('ETAmean', 'Enable output of ETAmean (ETAmean)', default=False)
    output.add_boolean('OUT_NU', 'Enable output of breaking location (OUT_NU)', default=False)
    output.add_boolean('WaveHeight', 'Enable output of wave height, Hsig, Hrms, Havg (WaveHeight)', default=False)
    output.add_boolean('MASK', 'Enable output of wetting-drying MASK (MASK)', default=True)
    method = output.add_boolean('Output_advanced', 'Enable advanced output options', default=False)
    flags = {True: True, False: False}
    output.add_integer('OUTPUT_RES', 'Output data resolution (every _ points) (OUTPUT_RES)', 1).add_dependency(
        method, flags)
    output.add_boolean('DEPTH_OUT', 'Enable output of depth (DEPTH_OUT)', default=True).add_dependency(method, flags)
    output.add_boolean('ETAscreen', 'Enable ETAscreen output (ETAscreen)', default=False).add_dependency(method, flags)
    output.add_boolean('MASK9', 'Enable output of MASK9 (MASK9)', default=False).add_dependency(method, flags)

    # output.add_boolean('SourceX', 'Enable output of source terms in x direction (SourceX)', default=False
    #                    ).add_dependency(method, flags)
    # output.add_boolean('SourceY', 'Enable output of source terms in y direction (SourceY)', default=False
    #                    ).add_dependency(method, flags)
    # output.add_boolean('P', 'Enable output of momentum flux in x direction (P)', default=False).add_dependency(
    #     method, flags)
    # output.add_boolean('Q', 'Enable output of momentum flux in y direction (Q)', default=False).add_dependency(
    #     method, flags)
    # output.add_boolean('Fx', 'Enable output of numerical flux F in x direction (Fx)', default=False).add_dependency(
    #     method, flags)
    # output.add_boolean('Fy', 'Enable output of numerical flux F in y direction (Fy)', default=False).add_dependency(
    #     method, flags)
    # output.add_boolean('Gx', 'Enable output of numerical flux G in x direction (Gx)', default=False).add_dependency(
    #     method, flags)
    # output.add_boolean('Gy', 'Enable output of numerical flux G in y direction (Gy)', default=False).add_dependency(
    #     method, flags)
    output.add_boolean('AGE', 'Enable output of breaking age (AGE)', default=False).add_dependency(method, flags)
    output.add_boolean('Hmax', 'Enable output of recorded maximum surface elevation (Hmax)', default=False
                       ).add_dependency(method, flags)
    output.add_boolean('Hmin', 'Enable output of recorded minimum surface elevation (Hmin)', default=False
                       ).add_dependency(method, flags)
    output.add_boolean('Umax', 'Enable output of recorded maximum velocity (Umax)', default=False).add_dependency(
        method, flags)
    output.add_boolean('VORmax', 'Enable output of recorded maximum vorticity (VORmax)', default=False
                       ).add_dependency(method, flags)
    output.add_boolean('MFmax', 'Enable output of recorded maximum momentum flux (MFmax)', default=False
                       ).add_dependency(method, flags)
    output.add_boolean('OUT_Time', 'Enable output of recorded tsunami arrival time (OUT_Time)', default=False
                       ).add_dependency(method, flags)
    # output.add_boolean('SXL', 'Enable output of SXL (SXL)', default=False).add_dependency(method, flags)
    # output.add_boolean('SXR', 'Enable output of SXR (SXR)', default=False).add_dependency(method, flags)
    # output.add_boolean('SYL', 'Enable output of SYL (SYL)', default=False).add_dependency(method, flags)
    # output.add_boolean('SYR', 'Enable output of SYR (SYR)', default=False).add_dependency(method, flags)
    output.add_boolean('OUT_METEO', 'Enable output of pressure field (OUT_METEO)', default=False).add_dependency(
        method, flags)
    output.add_boolean('ROLLER', 'Enable output of roller-induced flux (ROLLER)', default=False).add_dependency(
        method, flags)
    output.add_boolean('UNDERTOW', 'Enable output of undertow (UNDERTOW)', default=False).add_dependency(method, flags)
    output.add_boolean('TMP', 'Enable output of TMP (TMP)', default=False).add_dependency(method, flags)

    # # params.add_text('unimplemented', 'Unimplemented TODO: WaveCompFile types have not been '
    # #                 'implemented', '')  # Internal ?

    return gm


def variables_by_name_to_dict() -> dict:
    """Convert the data values to a dictionary.

    Returns:
        The dictionary of values.
    """
    values = {}
    parameters = get_variables_by_name()
    for name, parameter in parameters.items():
        values = _value_to_dict(values, parameter, name)

    return values


def _value_to_dict(values, parameter, name) -> dict:
    """Convert the data values to a dictionary.

    Returns:
        The dictionary of values.
    """
    if name == 'unimplemented':
        return values
    if name == 'COMPLEXITY':
        return values
    if name == 'WAVEMAKER':
        value = find_key_from_value(parameter.value)
        values['WAVEMAKER'] = value
        return values

    values[name] = parameter.value

    return values


def find_key_from_value(value: str) -> str:
    """
    Find the key corresponding to the given value in the WAVE_MAKER_OPTIONS dictionary.

    Args:
        value (str): The value to search for.

    Returns:
        str: The key corresponding to the value, or None if not found.
    """
    for key, val in WAVE_MAKER_OPTIONS.items():
        if val == value:
            return key
    return None


def get_variables_by_name() -> dict:
    """Get dictionary of variables.

    Returns:
        Dictionary of variables with key as value name.
    """
    variables_by_name = {}
    for group_names in get_model().global_parameters.group_names:
        group = get_model().global_parameters.group(group_names)
        for parameter_name in group.parameter_names:
            parameter = group.parameter(parameter_name)
            # name = variable['name']
            values = copy.deepcopy(parameter)
            # values.pop('name')
            variables_by_name[parameter.parameter_name] = values
    if 'VISCOSITY_BREAKING' in variables_by_name:
        v_breaking = True
        show_breaking = True
        if variables_by_name['VISCOSITY_BREAKING'].value != \
                'Eddy-Viscosity breaking used in numerical scheme (VISCOSITY_BREAKING=T & SHOW_BREAKING=T)':
            v_breaking = True
            show_breaking = True
        variables_by_name['VISCOSITY_BREAKING'] = Parameter(
            '', '', 'VISCOSITY_BREAKING', Parameter.create(Type.BOOLEAN, 'VISCOSITY_BREAKING', v_breaking))
        variables_by_name['SHOW_BREAKING'] = Parameter(
            '', '', 'SHOW_BREAKING', Parameter.create(Type.BOOLEAN, 'SHOW_BREAKING', show_breaking))

    if 'Dispersion model' in variables_by_name:
        if variables_by_name['Dispersion model'].value == 'Fully nonlinear equations':
            variables_by_name['Gamma1'].value = 1.0
            variables_by_name['Gamma2'].value = 1.0
            variables_by_name['Gamma3'].value = 1.0
        elif variables_by_name['Dispersion model'].value == 'Nwogu equations':
            variables_by_name['Gamma1'].value = 1.0
            variables_by_name['Gamma2'].value = 0.0
            variables_by_name['Gamma3'].value = 1.0
        elif variables_by_name['Dispersion model'].value == 'Linear shallow water equations':
            variables_by_name['Gamma1'].value = 0.0
            variables_by_name['Gamma2'].value = 0.0
            variables_by_name['Gamma3'].value = 0.0

    variables_by_name['DX'] = Parameter('', '', 'DX', Parameter.create(Type.FLOAT, 'DX', 0.0))
    variables_by_name['DY'] = Parameter('', '', 'DY', Parameter.create(Type.FLOAT, 'DY', 0.0))
    variables_by_name['Mglob'] = Parameter('', '', 'Mglob', Parameter.create(Type.INTEGER, 'Mglob', 0))
    variables_by_name['Nglob'] = Parameter('', '', 'Nglob', Parameter.create(Type.INTEGER, 'Nglob', 0))
    variables_by_name['MinDepthFrc'] = Parameter('', '', 'MinDepthFrc', Parameter.create(Type.FLOAT, 'MinDepthFrc',
                                                                                         0.1))
    variables_by_name['RESULT_FOLDER'] = Parameter('', '', 'RESULT_FOLDER', Parameter.create(
        Type.TEXT, 'RESULT_FOLDER', './output/'))
    variables_by_name['NumberStations'] = Parameter('', '', 'NumberStations', Parameter.create(
        Type.INTEGER, 'NumberStations', 0))
    variables_by_name['STATIONS_FILE'] = Parameter('', '', 'STATIONS_FILE', Parameter.create(
        Type.TEXT, 'STATIONS_FILE', ''))

    return variables_by_name


def variable_is_internal(name: str) -> bool:
    """Determine if a variable is internal.

    Args:
        name: The name of the variable.
        info: The parameter information.

    Returns:
        True if the variable is internal.
    """
    if name in ['COMPLEXITY', 'USE_DT_fixed', 'TideWest', 'TideEast', 'TideSouth', 'TideNorth',
                'unimplemented']:
        return True
    return False


def variable_is_enabled(name: str, info: Parameter, values: dict, ) -> bool:
    """Determine if a variable is enabled.

    Returns:
        True if the variable is enabled.
    """
    if name in ['Cbrk1', 'Cbrk2', 'Gamma1', 'Gamma2', 'Gamma3']:
        return True
    if info.dependency_flags is None:
        return True
    parent = info.parent[-1]
    if parent in values:
        parent_value = values[parent]
        if parent in ['VISCOSITY_BREAKING', 'SHOW_BREAKING']:
            return parent_value
        if parent in ['WAVEMAKER'] and parent_value in WAVE_MAKER_OPTIONS:
            parent_value = WAVE_MAKER_OPTIONS[parent_value]
        if parent_value in info.dependency_flags:
            return info.dependency_flags[parent_value]

    if __debug__:
        raise ValueError(f'Unable to determine if Variable {name} is enabled')

    return False


def variable_file_value(name: str, value: any, info: Parameter) -> str:
    """Get the variable value to write to a file.

    Args:
        name: The name of the variable.
        value: The value of the variable.

    Returns:
        The value to write to input.funwave file.
    """
    file_value = ''
    if info is not None:
        if info.parameter_type == Type.BOOLEAN:
            file_value = 'T' if value else 'F'
        elif info.parameter_type == Type.DATASET:
            file_value = f'input/{name.lower()}.txt'
        elif info.parameter_type == Type.INPUT_FILE or info.parameter_type == Type.OUTPUT_FILE and value:
            project_folder = Path(os.environ.get('XMS_PYTHON_APP_PROJECT_PATH')).parent
            file_path = Path(value)
            input_txt_folder = os.getcwd()
            if not file_path.is_absolute():
                file_path = project_folder / file_path
            if file_path.is_relative_to(project_folder):
                file_path = Path(os.path.relpath(file_path, start=input_txt_folder))
            file_value = file_path.as_posix()
        else:
            file_value = str(value)
    return file_value


def read_value(name: str, value: str, variable_type: Type) -> any:
    """Read value from string.

    Args:
        name: The variable name.
        value: The string value read from the file.
        variable_type: The variable type.

    Returns:
        The variable value.
    """
    result = value
    match variable_type:
        case list():
            pass
        case Type.FLOAT:
            result = float(value)
        case Type.INTEGER:
            result = int(value)
        case Type.TEXT:
            pass
        case Type.BOOLEAN:
            if value in ['T', 't', 'True', 'true', 'TRUE', '1']:
                result = True
            elif value in ['F', 'f', 'False', 'false', 'FALSE', '0']:
                result = False
            else:
                raise ValueError()
        case Type.INPUT_FILE | Type.DATASET:
            file = Path(os.getcwd()) / value
            if file.is_file():
                result = file.as_posix()
            else:
                logging.getLogger('xms.funwave').warning(f'Missing file: {name} = {value}')
                result = ''
        case Type.INPUT_FOLDER:
            pass
        case 'grid.m_size':
            result = int(value)
        case 'grid.n_size':
            result = int(value)
        case 'grid.x_size':
            result = float(value)
        case 'grid.y_size':
            result = float(value)
        case 'number_stations':
            result = int(value)
        case _:
            pass
    return result


UNIMPLEMENTED_VALUES = {
    ('WAVEMAKER', 'LEFT_BC_IRR'),
    ('WAVEMAKER', 'WK_TIME_SERIES'),
    ('WAVEMAKER', 'WK_DATA2D'),
    ('WAVEMAKER', 'WK_NEW_DATA2D')
}


def is_unimplemented(variable: str, value: any) -> bool:
    """Is a variable with a value unimplemented?

    Args:
        variable: The variable.
        value: The variable value.

    Returns:
        True if unimplemented.
    """
    unimplemented = (variable, value) in UNIMPLEMENTED_VALUES
    return unimplemented
