"""Utility methods used by the GUI modules."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.funwave.data.model import get_variables_by_name, is_unimplemented, read_value
from xms.funwave.file_io.io_util import use_cwd


def read_input_txt_file(file: Path) -> dict:
    """Read an input.funwave file.

    Args:
        file(:obj:`Path`): The input file.

    Returns:
        Dictionary of the input values.
    """
    values = read_input_txt_values(file)
    depth_file = None
    if 'DEPTH_FILE' in values:
        depth_file = values['DEPTH_FILE']
    values = get_valid_input_txt_values(values, file)
    return values, depth_file


def read_input_txt_values(file: Path) -> dict:
    """Read an input.funwave file.

    Args:
        file(:obj:`Path`): The input file.

    Returns:
        (:obj:`dict`):Dictionary of the input values.
    """
    white_space = ' \t\n\r'
    input = {}
    with open(file, 'r') as f:
        with use_cwd(file.parent):
            for line in f:
                line = line.strip(white_space)
                first_comment_char = line.find('!')
                if first_comment_char >= 0:
                    line = line[:first_comment_char]
                line = line.strip(white_space)
                equal_char = line.find('=')
                if equal_char >= 0:
                    name = line[:equal_char]
                    name = name.strip(white_space)
                    value = line[equal_char + 1:]
                    value = value.strip(white_space)
                    input[name] = value
    return input


def get_valid_input_txt_values(values: dict, file: Path) -> dict:
    """Check validity of values read from an input.funwave file.

    Args:
        values(:obj:`dict`): The input values.
        file(:obj:`Path`): The input file.

    Returns:
        (:obj:`dict`):Dictionary of the input values.
    """
    logger = logging.getLogger('xms.funwave')
    variables = get_variables_by_name()
    input = {}

    if 'U' in values or 'V' in values:
        if 'U' in values:
            value = values['U']
            values['Velocity'] = value
            values.pop('U')
        if 'V' in values:
            value = values['V']
            values['Velocity'] = value
            values.pop('V')
    if 'Umean' in values or 'Vmean' in values:
        if 'Umean' in values:
            value = values['Umean']
            values['Average velocity'] = value
            values.pop('Umean')
        if 'Vmean' in values:
            value = values['Vmean']
            values['Average velocity'] = value
            values.pop('Vmean')

    with use_cwd(file.parent):
        for name, value in values.items():
            variable_info = variables.get(name)
            if variable_info is None:
                if name != 'DEPTH_FILE':
                    logger.warning(f'Unsupported variable: {name}')
                continue
            value = read_value(name, value, variable_info.parameter_type)
            if is_unimplemented(name, value):
                logger.warning(f'Unimplemented variable value: {name} = {value}')
            input[name] = value
    return input
