"""Utility methods used by the GUI modules."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.funwave.data.model import (get_variables_by_name, UNSPECIFIED, variable_file_value,
                                    variable_is_enabled, variable_is_internal)


def write_input_txt_file(file: Path, values: dict, valid_grid: bool):
    """Write an input.funwave file.

    Args:
        file(:obj:`Path`): The input file.
        values(:obj:`dict`): The values to write.

    Returns:
        (:obj:`dict`): Dictionary of the input values.
    """
    if 'VISCOSITY_BREAKING' in values:
        values['SHOW_BREAKING'] = values['VISCOSITY_BREAKING']
    if 'MinDepth' in values and 'MinDepthFrc' not in values:
        values['MinDepthFrc'] = values['MinDepth']
    write_velocity = False
    output_velocity = 'F'
    write_ave_velocity = False
    output_ave_velocity = 'F'
    if 'Velocity' in values:
        write_velocity = True
        if values['Velocity']:
            output_velocity = 'T'
        values.pop('Velocity')
    if 'Average velocity' in values:
        write_ave_velocity = True
        if values['Average velocity']:
            output_ave_velocity = 'T'
        values.pop('Average velocity')

    values_to_skip = ['Specify Cbrk (advanced)', 'Dispersion model']
    with open(file, 'w') as f:
        variables_by_name = get_variables_by_name()
        for name, info in variables_by_name.items():
            if name in values and name not in values_to_skip:
                enabled = variable_is_enabled(name, info, values)
                has_value = values[name] != '' and values[name] != UNSPECIFIED
                internal = variable_is_internal(name)
                if enabled and has_value and not internal:
                    value = variable_file_value(name, values[name], info)
                    f.write(f'{name} = {value}\n')
        if write_velocity:
            f.write(f'U = {output_velocity}\n')
            f.write(f'V = {output_velocity}\n')
        if write_ave_velocity:
            f.write(f'Umean = {output_ave_velocity}\n')
            f.write(f'Vmean = {output_ave_velocity}\n')
        if valid_grid:
            f.write('DEPTH_FILE = input/depth_file.txt')
