"""StationComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType


# 4. Local modules
from xms.gencade.components.gencade_component import GenCadeComponent
from xms.gencade.data.points_data import PointsData
from xms.gencade.gui.point_atts_dlg import PointAttsDlg


class PointsComponent(GenCadeComponent):
    """A Dynamic Model Interface (DMI) component base for the ADCIRC model."""

    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file.strip('"\''))
        self.data = PointsData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.class_name = 'PointsComponent'
        self.module_name = 'xms.gencade.components.points_component'
        self.point_commands = [
            ('Assign Point Attributes...', 'open_point_attributes'),
        ]
        comp_dir = os.path.dirname(self.main_file)
        self.point_comp_id_file = os.path.join(comp_dir, 'point.display_ids')
        # Copy default display options if needed
        self.disp_opts_file = os.path.join('point_display_options.json')
        if not os.path.exists(self.disp_opts_file):
            # Read the default point display options, and save ourselves a copy with a randomized UUID.
            point_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            point_default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)),
                                              'gui', 'resources', 'default_data',
                                              'point_default_display_options.json')
            point_json_dict = read_display_options_from_json(point_default_file)
            point_categories.from_dict(point_json_dict)
            point_categories.comp_uuid = self.uuid
            write_display_options_to_json(self.disp_opts_file, point_categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['point_display_uuid'] = point_categories.uuid
            self.data.commit()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_options_action()]
        return messages, action_requests

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed, and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            # Reset the component's coverage UUID if a duplicate event
            new_data = PointsData(new_main_file)
            new_data.info.attrs['cov_uuid'] = ''
            new_data.commit()

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Set component ids in XMS after a model-native import.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        if not self.cov_uuid:  # Store the UUID of the owning coverage
            self.cov_uuid = query.parent_item_uuid()
            self.data.info.attrs['cov_uuid'] = self.cov_uuid

        if params and 'POINT_ATT_IDS' in params[0]:  # Came from a merge, initialize feature point component ids.
            # Update the point component ids in XMS.
            point_att_file = params[0]['POINT_ATT_IDS']
            point_comp_file = params[0]['POINT_COMP_IDS']
            point_att_ids = read_display_option_ids(point_att_file)
            point_comp_ids = read_display_option_ids(point_comp_file)
            for att_id, comp_id in zip(point_att_ids, point_comp_ids):
                self.update_component_id(TargetType.point, att_id, comp_id)
            io_util.removefile(point_att_file)
            io_util.removefile(point_comp_file)
        self.data.commit()
        return [], []

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list[tuple]`): tuple containing:

                main_file (:obj:`str`): The absolute path to the main file of the old component this
                component is being merged from.

                id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                Each value is a tuple that may have two absolute file paths or none. The first
                file is for the ids in XMS on the coverage. The second file contains the ids the
                old component used for those objects. Both id files should be equal in length.
                This dictionary is only applicable if the component derives from
                CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        att_file = 0
        comp_file = 1
        main_file_idx = 0
        id_file_idx = 1
        point_att_ids = []
        point_comp_ids = []
        for cov in merge_list:
            # Build up list of XMS att id to component id in the new merged coverage.
            point_data = PointsData(cov[main_file_idx])

            # Concatenate the pipe points
            old_to_new_point_ids = self.data.concat_points(point_data)
            att_ids = read_display_option_ids(cov[id_file_idx]['POINT'][att_file])
            comp_ids = read_display_option_ids(cov[id_file_idx]['POINT'][comp_file])
            point_att_ids.extend(att_ids)
            for _, comp_id in zip(att_ids, comp_ids):
                point_comp_ids.append(old_to_new_point_ids[comp_id])

        # Save point attids and compids to a temp file, so they can be processed later
        point_att_file = os.path.join(os.path.dirname(self.main_file), 'tmp_point_att_ids.txt')
        write_display_option_ids(point_att_file, point_att_ids)
        point_comp_file = os.path.join(os.path.dirname(self.main_file), 'tmp_point_comp_ids.txt')
        write_display_option_ids(point_comp_file, point_comp_ids)

        # Wipe the old coverage's UUID, so we query for the new one.
        self.data.info.attrs['cov_uuid'] = ''
        self.data.commit()

        # Send back ActionRequest to update id files.
        parameters = {'POINT_ATT_IDS': point_att_file, 'POINT_COMP_IDS': point_comp_file}
        action = ActionRequest(main_file=self.main_file, modality='NO_DIALOG', class_name=self.class_name,
                               module_name=self.module_name, method_name='get_initial_display_options',
                               comp_uuid=self.uuid, parameters=parameters)
        return [], [action]

    def open_point_attributes(self, query, params, win_cont):
        """Opens the Open_Point_Attributes dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:'dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        prj_horiz_units = query.display_projection.horizontal_units.lower()
        comp_id = self._unpack_xms_data(params[0], TargetType.point)
        incremented_id = False
        if comp_id not in self.data.points.comp_id:
            comp_id = self.data.add_point_atts()
            incremented_id = True  # We just got a new component id, so don't duplicate on OK
        single_point = self.data.points.loc[dict(comp_id=[comp_id])]
        dlg = PointAttsDlg(win_cont, single_point, self.data, prj_horiz_units, self.dlg_message)
        if dlg.exec_():
            if incremented_id:  # Was previously unassigned, so we already got a new component id. Just update it.
                self.data.update_point(comp_id, dlg.point_data)
            else:  # Always generate a new component id, old one might be shared if assigned during a multi-select.
                comp_id = self.data.add_point_atts(dlg.point_data)
            self._append_comp_id_to_file(self. point_comp_id_file, comp_id)
            self.data.commit()
            for pt_id in self.selected_att_ids:
                self.update_component_id(TargetType.point, pt_id, comp_id)
            # Send back updated display lists to XMS after ActionRequest
            self.display_option_list.append(
                # Update the point display list only.
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid)
            )
        return [], []
