"""The Structure properties dialog."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QDateTime, Qt
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.gencade.components.mapped_grid_component import MappedGridComponent
from xms.gencade.data.sim_data import LATERAL_BC_TYPES, LATERAL_DISP_TYPES, MONTE_CARLO_PROB_TYPES, WAVE_COMPONENTS
from xms.gencade.gui.model_control_dlg_ui import Ui_ModelControlDlg
from xms.gencade.gui.print_table_widget import PrintTableWidget
from xms.gencade.gui.water_level_table_widget import WaterLevelTableWidget


class ModelControlDlg(XmsDlg):
    """A dialog for setting model control parameters."""

    def __init__(self, sim_data, units, parent=None):
        """Initializes the dialog, sets up the ui.

        Args:
            sim_data (:obj:`SimData`): data for the dialog
            units (:obj:'str'): Units as determined by a query in the component.
            parent (:obj:`QWidget`): Parent window
        """
        super().__init__(parent, 'xms.gencade.gui.model_control_dlg')
        self._help_url = 'https://cirpwiki.info/wiki/GenCade_2.0_Model_Control_Dialog'
        self._units = 'Imperial' if 'feet' in units else 'Metric'
        self.sim_data = sim_data
        self.sim_data.info.attrs['units'] = self._units
        self.setWindowTitle('GenCade Model Control')
        self._query = Query()
        self._theta = None
        self._num_cells = None
        self._grid_len = None

        # Commonly used edit field validators
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.pos_validator = QxDoubleValidator(parent=self, bottom=0.0)
        self.int_validator = QxIntValidator(parent=self, bottom=0)
        self.pct_validator = QxDoubleValidator(parent=self, bottom=0.0, top=100.0)

        self.ui = Ui_ModelControlDlg()
        self.ui.setupUi(self)
        self.ui.units_label.setText(f'Units for this projection are {self._units}.')
        self.ui.units_label.setStyleSheet('color: red;')
        self.ui.user_defined_label.setStyleSheet('color: red;')
        self.ui.smooth_note_label.setStyleSheet('color: red;')

        self._get_simulation()
        self._get_sim_grid()

        grid_exists = True if self._theta is not None else False
        self.ui.grid_angle.setText(str(self._theta)) if grid_exists else self.ui.grid_angle.setText(str('n/a'))
        self.ui.num_cells.setText(str(self._num_cells)) if grid_exists else self.ui.num_cells.setText(str('n/a'))
        self.ui.grid_len.setText(str(self._grid_len)) if grid_exists else self.ui.grid_len.setText(str('n/a'))

        self._add_validators()
        self.print_table = self._add_print_table(self.sim_data.print_table.to_dataframe(), 'print_table')
        self.wl_table = self._add_wl_table(self.sim_data.wl_table.to_dataframe(), 'wl_table')
        self._add_cbx_options()
        self._add_button_box()
        self.ui.user_defined_label.setVisible(False)
        self._load_data()
        # self._setup_mc_connection()

    def _get_simulation(self):
        sim_uuid = self._query.parent_item_uuid()
        self._sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)

    def _get_sim_grid(self):
        mapped_grid_item = tree_util.descendants_of_type(self._sim_item, only_first=True,
                                                         unique_name='MappedGridComponent')
        if mapped_grid_item is None:
            # log error
            return

        self._mapped_grid = MappedGridComponent(mapped_grid_item.main_file)
        self._theta = round(self._mapped_grid.data.info.attrs["theta"].item(), 2)
        self._num_cells = self._mapped_grid.data.info.attrs["num_cells"].item()
        self._grid_len = round(sum(self._mapped_grid.data.locations.dx.values), 2)

    def _add_print_table(self, df, name):
        """Adds print table to model control dialog.

        Args:
            df (:obj:`pandas.Dataframe`): The dataframe of the table data.
            name (:obj:`str`): The object name for the new table.
        """
        group = self.ui.print_dates_group
        table = PrintTableWidget(group, df)
        table.setObjectName(name)
        table.setMinimumHeight(125)
        group.setLayout(QVBoxLayout())
        group.layout().addWidget(table)
        return table

    def _add_wl_table(self, df, name):
        """Adds water level table to model control dialog.

        Args:
            df (:obj:`pandas.Dataframe`): The dataframe of the table data.
            name (:obj:`str`): The object name for the new table.
        """
        group = self.ui.enable_wl_group
        table = WaterLevelTableWidget(group, df)
        table.setObjectName(name)
        table.setMinimumHeight(125)
        group.setLayout(QVBoxLayout())
        group.layout().addWidget(table)
        return table

    def _load_data(self):
        """Loads all the data for each of the menu tabs."""
        self._load_model_setup()
        self._load_beach_setup()
        self._load_seaward_bc()

        self._load_lateral_bc()  # load data before making connections
        self._setup_lateral_connections()

        self._load_adaptive_time_steps()
        self._load_cross_shore()
        self._load_monte_carlo()
        self._set_mc_connections()  # load data before making connections
        self._load_water_level()

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def _add_cbx_options(self):
        """Setup combo box options on all combo boxes."""
        self.ui.wave_components_cbx.addItems(WAVE_COMPONENTS)
        self.ui.left_bc_type_cbx.addItems(LATERAL_BC_TYPES)
        self.ui.left_disp_type_cbx.addItems(LATERAL_DISP_TYPES)
        self.ui.rt_bc_type_cbx.addItems(LATERAL_BC_TYPES)
        self.ui.rt_disp_type_cbx.addItems(LATERAL_DISP_TYPES)
        self.ui.prob_function_cbx.addItems(MONTE_CARLO_PROB_TYPES)

    def _add_validators(self):
        """Add validators for all edit fields."""
        self._add_model_setup_validators()
        self._add_beach_setup_validators()
        self._add_seaward_validators()
        self._add_lateral_validators()
        self._add_cross_shore_validators()
        self._add_monte_carlo_validators()

    def _add_model_setup_validators(self):
        """Adds validators to the field widgets for the 'Model Setup' tab."""
        self.ui.time_step.setValidator(self.pos_validator)
        self.ui.recording.setValidator(self.pos_validator)

    def _add_beach_setup_validators(self):
        """Adds validators to the field widgets for the 'Beach Setup' tab."""
        self.ui.eff_grain_size.setValidator(self.pos_validator)
        self.ui.avg_berm_ht.setValidator(self.pos_validator)
        self.ui.closure_depth.setValidator(self.pos_validator)
        self.ui.K1.setValidator(self.pos_validator)
        self.ui.K2.setValidator(self.pos_validator)
        self.ui.KTIDE.setValidator(self.pos_validator)
        self.ui.regional_slc.setValidator(self.dbl_validator)
        self.ui.regional_subsidence.setValidator(self.dbl_validator)
        self.ui.beach_slope.setValidator(self.dbl_validator)

    def _add_seaward_validators(self):
        """Adds validators to the field widgets for the 'Seaward BC' tab."""
        self.ui.ht_amp_factor.setValidator(self.pos_validator)
        self.ui.angle_amp_factor.setValidator(self.pos_validator)
        self.ui.angle_offset.setValidator(self.dbl_validator)
        self.ui.num_cells_smoothing.setValidator(self.int_validator)

    def _add_lateral_validators(self):
        """Adds validators to the field widgets for the 'Lateral BC' tab."""
        self.ui.left_length_to_tip.setValidator(self.pos_validator)
        self.ui.left_shoreline_disp_distance.setValidator(self.dbl_validator)
        self.ui.rt_length_to_tip.setValidator(self.pos_validator)
        self.ui.rt_shoreline_disp_distance.setValidator(self.dbl_validator)

    def _add_adaptive_time_step_validators(self):
        """Adds validators to the field widgets for the 'Adaptive Time Steps' tab."""
        self.ui.threshold_min.setValidator(self.pos_validator)
        self.ui.threshold_max.setValidator(self.pos_validator)
        self.ui.days_stable.setValidator(self.int_validator)  # integer
        self.ui.stability_min.setValidator(self.pos_validator)
        self.ui.stability_max.setValidator(self.pos_validator)

    def _add_cross_shore_validators(self):
        """Adds validators to the field widgets for the 'Cross Shore' tab."""
        self.ui.cross_shore_scale.setValidator(self.dbl_validator)
        self.ui.onshore_trans_rate.setValidator(self.dbl_validator)

    def _add_monte_carlo_validators(self):
        """Adds validators to the field widgets for the 'Monte Carlo' tab."""
        self.ui.num_mc_sims.setValidator(self.int_validator)
        self.ui.AL_mean.setValidator(self.dbl_validator)
        self.ui.AL_sigma.setValidator(self.dbl_validator)
        self.ui.AL_max.setValidator(self.dbl_validator)
        self.ui.AAO_val.setValidator(self.dbl_validator)
        self.ui.BBO_val.setValidator(self.dbl_validator)
        self.ui.FKO_val.setValidator(self.dbl_validator)
        self.ui.HCUT_val.setValidator(self.dbl_validator)
        self.ui.H_min.setValidator(self.dbl_validator)
        self.ui.H_max.setValidator(self.dbl_validator)
        self.ui.H_mean.setValidator(self.dbl_validator)
        self.ui.wave_interval.setValidator(self.dbl_validator)
        self.ui.std_percent.setValidator(self.pct_validator)  # percent

    def _load_model_setup(self):
        """Populate the widgets for model setup tab."""
        model_attrs = self.sim_data.model.attrs
        self.ui.title.setText(str(model_attrs['title']))
        date_time_text = model_attrs['start_date']
        if date_time_text:
            date_time = QDateTime.fromString(date_time_text, Qt.ISODate)
            self.ui.start_dateTime_edit.setDateTime(date_time)
        date_time_text = model_attrs['end_date']
        if date_time_text:
            date_time = QDateTime.fromString(date_time_text, Qt.ISODate)
            self.ui.end_dateTime_edit.setDateTime(date_time)
        self.ui.time_step.setText(str(model_attrs['time_step']))
        self.ui.recording.setText(str(model_attrs['recording']))
        self.ui.full_print_cbx.setChecked(True if model_attrs['full_print'] == 1 else False)

    def _save_model_setup(self):
        """Save data from the model setup attributes."""
        self.sim_data.model.attrs['title'] = self.ui.title.text()
        self.sim_data.model.attrs['start_date'] = self.ui.start_dateTime_edit.dateTime().toString(Qt.ISODate)
        self.sim_data.model.attrs['end_date'] = self.ui.end_dateTime_edit.dateTime().toString(Qt.ISODate)
        self.sim_data.model.attrs['time_step'] = float(self.ui.time_step.text())
        self.sim_data.model.attrs['recording'] = float(self.ui.recording.text())
        self.sim_data.model.attrs['full_print'] = 1 if self.ui.full_print_cbx.isChecked() else 0
        self.sim_data.print_table = self.print_table.model.data_frame.to_xarray()

    def _load_beach_setup(self):
        """Populate the widgets for beach setup tab."""
        beach_attrs = self.sim_data.beach.attrs
        self.ui.eff_grain_size.setText(str(beach_attrs['eff_grain_size']))
        self.ui.avg_berm_ht.setText(str(beach_attrs['avg_berm_ht']))
        self.ui.closure_depth.setText(str(beach_attrs['closure_depth']))
        self.ui.regional_slc.setText(str(beach_attrs['regional_slc']))
        self.ui.regional_subsidence.setText(str(beach_attrs['regional_subsidence']))
        self.ui.beach_slope.setText(str(beach_attrs['beach_slope']))
        self.ui.K1.setText(str(beach_attrs['K1']))
        self.ui.K2.setText(str(beach_attrs['K2']))
        self.ui.KTIDE.setText(str(beach_attrs['KTIDE']))

    def _save_beach_setup(self):
        """Save data from the beach setup attributes."""
        self.sim_data.beach.attrs['eff_grain_size'] = float(self.ui.eff_grain_size.text())
        self.sim_data.beach.attrs['avg_berm_ht'] = float(self.ui.avg_berm_ht.text())
        self.sim_data.beach.attrs['closure_depth'] = float(self.ui.closure_depth.text())
        self.sim_data.beach.attrs['regional_slc'] = float(self.ui.regional_slc.text())
        self.sim_data.beach.attrs['regional_subsidence'] = float(self.ui.regional_subsidence.text())
        self.sim_data.beach.attrs['beach_slope'] = float(self.ui.beach_slope.text())
        self.sim_data.beach.attrs['K1'] = float(self.ui.K1.text())
        self.sim_data.beach.attrs['K2'] = float(self.ui.K2.text())
        self.sim_data.beach.attrs['KTIDE'] = float(self.ui.KTIDE.text())

    def _load_seaward_bc(self):
        """Populate the widgets for seaward bc tab."""
        seaward_attrs = self.sim_data.seaward.attrs
        self.ui.ht_amp_factor.setText(str(seaward_attrs['ht_amp_factor']))
        self.ui.angle_amp_factor.setText(str(seaward_attrs['ang_amp_factor']))
        self.ui.angle_offset.setText(str(seaward_attrs['ang_offset']))
        self.ui.wave_components_cbx.setCurrentIndex(
            WAVE_COMPONENTS.index(seaward_attrs['wave_components']))
        self.ui.num_cells_smoothing.setText(str(seaward_attrs['num_cells_smoothing']))

    def _save_seaward_bc(self):
        """Save data from the seaward bc attributes."""
        self.sim_data.seaward.attrs['ht_amp_factor'] = float(self.ui.ht_amp_factor.text())
        self.sim_data.seaward.attrs['ang_amp_factor'] = float(self.ui.angle_amp_factor.text())
        self.sim_data.seaward.attrs['ang_offset'] = float(self.ui.angle_offset.text())
        self.sim_data.seaward.attrs['wave_components'] = self.ui.wave_components_cbx.currentText()
        self.sim_data.seaward.attrs['num_cells_smoothing'] = int(self.ui.num_cells_smoothing.text())

    def _setup_lateral_connections(self):
        """Set up signal/slot connections in the lateral bc tab."""
        self.ui.left_bc_type_cbx.currentIndexChanged[int].connect(self.ui.left_lateral_stack.setCurrentIndex)
        self.ui.rt_bc_type_cbx.currentIndexChanged[int].connect(self.ui.rt_lateral_stack.setCurrentIndex)

    def _load_lateral_bc(self):
        """Populate the widgets for lateral bc tab."""
        lateral_attrs = self.sim_data.lateral.attrs
        # set current page on the stack, enabled/disabled states, etc...
        self.ui.left_bc_type_cbx.setCurrentIndex(LATERAL_BC_TYPES.index(lateral_attrs['left_bc_type']))
        self.ui.left_lateral_stack.setCurrentIndex(self.ui.left_bc_type_cbx.currentIndex())
        self.ui.rt_bc_type_cbx.setCurrentIndex(LATERAL_BC_TYPES.index(lateral_attrs['rt_bc_type']))
        self.ui.rt_lateral_stack.setCurrentIndex(self.ui.rt_bc_type_cbx.currentIndex())
        # populate left side values
        self.ui.left_length_to_tip.setText(str(lateral_attrs['left_length_to_tip']))
        self.ui.left_shoreline_disp_distance.setText(str(lateral_attrs['left_shoreline_disp_distance']))
        self.ui.left_disp_type_cbx.setCurrentIndex(
            LATERAL_DISP_TYPES.index(lateral_attrs['left_disp_type']))
        # populate right side values
        self.ui.rt_length_to_tip.setText(str(lateral_attrs['rt_length_to_tip']))
        self.ui.rt_shoreline_disp_distance.setText(str(lateral_attrs['rt_shoreline_disp_distance']))
        self.ui.rt_disp_type_cbx.setCurrentIndex(
            LATERAL_DISP_TYPES.index(lateral_attrs['rt_disp_type']))

    def _save_lateral_bc(self):
        """Save data from the lateral bc attributes."""
        self.sim_data.lateral.attrs['left_bc_type'] = self.ui.left_bc_type_cbx.currentText()
        self.sim_data.lateral.attrs['left_length_to_tip'] = float(self.ui.left_length_to_tip.text())
        self.sim_data.lateral.attrs['left_shoreline_disp_distance'] = float(self.ui.left_shoreline_disp_distance.text())
        self.sim_data.lateral.attrs['left_disp_type'] = self.ui.left_disp_type_cbx.currentText()
        self.sim_data.lateral.attrs['rt_bc_type'] = self.ui.rt_bc_type_cbx.currentText()
        self.sim_data.lateral.attrs['rt_length_to_tip'] = float(self.ui.rt_length_to_tip.text())
        self.sim_data.lateral.attrs['rt_shoreline_disp_distance'] = float(self.ui.rt_shoreline_disp_distance.text())
        self.sim_data.lateral.attrs['rt_disp_type'] = self.ui.rt_disp_type_cbx.currentText()

    def _load_adaptive_time_steps(self):
        """Populate the widgets for adaptive time_steps tab."""
        adaptive_time_attrs = self.sim_data.adaptive.attrs
        self.ui.enable_adaptive_group.setChecked(adaptive_time_attrs['use_adaptive_ts'] != 0)
        self.ui.enable_adaptive_group.toggled.connect(self._on_toggle_adaptive_group)
        # setChecked does not emit the toggled signal, so do it manually.
        self._on_toggle_adaptive_group(adaptive_time_attrs['use_adaptive_ts'] != 0)
        self.ui.threshold_min.setText(str(adaptive_time_attrs['threshold_min']))
        self.ui.threshold_max.setText(str(adaptive_time_attrs['threshold_max']))
        self.ui.days_stable.setText(str(adaptive_time_attrs['days_stable']))
        self.ui.stability_min.setText(str(adaptive_time_attrs['stability_min']))
        self.ui.stability_max.setText(str(adaptive_time_attrs['stability_max']))

    def _save_adaptive_time_steps(self):
        """Save data from the adaptive time_steps attributes."""
        self.sim_data.adaptive.attrs['use_adaptive_ts'] = 1 if self.ui.enable_adaptive_group.isChecked() else 0
        self.sim_data.adaptive.attrs['threshold_min'] = float(self.ui.threshold_min.text())
        self.sim_data.adaptive.attrs['threshold_max'] = float(self.ui.threshold_max.text())
        self.sim_data.adaptive.attrs['days_stable'] = int(self.ui.days_stable.text())
        self.sim_data.adaptive.attrs['stability_min'] = float(self.ui.stability_min.text())
        self.sim_data.adaptive.attrs['stability_max'] = float(self.ui.stability_min.text())

    def _on_toggle_adaptive_group(self, toggled):
        """Slot for when the adaptive timesteps group is toggled.

        Args:
            toggled (:obj:`bool`): True if the groupbox is enabled.
        """
        self.ui.threshold_min.setEnabled(toggled)
        self.ui.threshold_min_label.setEnabled(toggled)
        self.ui.threshold_max.setEnabled(toggled)
        self.ui.threshold_max_label.setEnabled(toggled)
        self.ui.days_stable.setEnabled(toggled)
        self.ui.days_stable_label.setEnabled(toggled)
        self.ui.stability_min.setEnabled(toggled)
        self.ui.stability_min_label.setEnabled(toggled)
        self.ui.stability_max.setEnabled(toggled)
        self.ui.stability_max_label.setEnabled(toggled)

    def _load_cross_shore(self):
        """Populate the widgets for cross shore tab."""
        cross_shore_attrs = self.sim_data.cross_shore.attrs
        self.ui.enable_cross_shore_group.setChecked(cross_shore_attrs['use_cross_shore'] != 0)
        self.ui.enable_cross_shore_group.toggled.connect(self._on_toggle_cross_shore)
        # setChecked does not emit the toggled signal, so do it manually.
        self._on_toggle_cross_shore(cross_shore_attrs['use_cross_shore'] != 0)
        self.ui.cross_shore_scale.setText(str(cross_shore_attrs['cross_shore_scale']))
        self.ui.use_var_berm_ht_chk.setChecked(cross_shore_attrs['use_var_berm_ht'] != 0)
        self.ui.use_var_slope_chk.setChecked(cross_shore_attrs['use_var_slope'] != 0)
        self.ui.onshore_trans_rate.setText(str(cross_shore_attrs['onshore_trans_rate']))

    def _save_cross_shore(self):
        """Save data from the cross shore attributes."""
        self.sim_data.cross_shore.attrs['use_cross_shore'] = \
            1 if self.ui.enable_cross_shore_group.isChecked() else 0
        self.sim_data.cross_shore.attrs['cross_shore_scale'] = float(self.ui.cross_shore_scale.text())
        self.sim_data.cross_shore.attrs['use_var_berm_ht'] =\
            1 if self.ui.use_var_berm_ht_chk.checkState() == Qt.Checked else 0
        self.sim_data.cross_shore.attrs['use_var_slope'] =\
            1 if self.ui.use_var_slope_chk.checkState() == Qt.Checked else 0
        self.sim_data.cross_shore.attrs['onshore_trans_rate'] = float(self.ui.onshore_trans_rate.text())

    def _on_toggle_cross_shore(self, toggled):
        """Slot for when the cross shore group is toggled.

        Args:
            toggled (:obj:`bool`): True if the groupbox is enabled.
        """
        self.ui.cross_shore_scale.setEnabled(toggled)
        self.ui.use_var_berm_ht_chk.setEnabled(toggled)
        self.ui.use_var_slope_chk.setEnabled(toggled)
        self.ui.use_beach_fill_chk.setEnabled(toggled)
        self.ui.std_percent_label.setEnabled(toggled)
        self.ui.std_percent.setEnabled(toggled)

    def _on_toggle_mc_group(self, toggled):
        """Slot for when the monte carlo group is toggled.

        Args:
            toggled (bool): True if the groupbox is enabled.
        """
        self.ui.use_beach_fill_chk.setEnabled(toggled)

    def _on_toggle_beach_fill(self, toggled):
        """Slot for when the beach fill checkbox is toggled.

        Args:
            toggled (bool): True if the groupbox is enabled.
        """
        self.ui.std_percent_label.setEnabled(toggled)
        self.ui.std_percent.setEnabled(toggled)

    def _set_mc_connections(self):
        """Set up signal/slots for the monte carlo."""
        self.ui.prob_function_cbx.currentIndexChanged[int].connect(self._set_probability_options_visible)
        # TODO - There is an issue where the Use Beach Fill box enabled and value filled, then model control is closed.
        # When the user comes back into model control, the boxes are back disabled again.

    def _load_water_level(self):
        """Populate the widgets for Water Levels tab."""
        # There is only one variable in this group. I combined into the 'model' attrs group instead of a new one.
        self.ui.enable_wl_group.setChecked(self.sim_data.model.attrs['enable_water_level'] != 0)
        pass

    def _save_water_level(self):
        """Save data from the Water Levels attributes."""
        self.sim_data.model.attrs['enable_water_level'] = \
            1 if self.ui.enable_wl_group.isChecked() else 0
        self.sim_data.wl_table = self.wl_table.model.data_frame.to_xarray()
        pass

    def _load_monte_carlo(self):
        """Populate the widgets for Monte Carlo tab."""
        monte_carlo_attrs = self.sim_data.monte_carlo.attrs
        self.ui.enable_mc_group.setChecked(monte_carlo_attrs['use_monte_carlo'] != 0)
        self.ui.enable_mc_group.toggled.connect(self._on_toggle_mc_group)
        self.ui.prob_function_cbx.setCurrentIndex(MONTE_CARLO_PROB_TYPES.index(monte_carlo_attrs['prob_function']))
        self.ui.AL_mean.setText(str(monte_carlo_attrs['AL_mean']))
        self.ui.AL_sigma.setText(str(monte_carlo_attrs['AL_sigma']))
        self.ui.AL_max.setText(str(monte_carlo_attrs['AL_max']))
        self.ui.num_mc_sims.setText(str(monte_carlo_attrs['num_mc_sims']))
        self.ui.AAO_val.setText(str(monte_carlo_attrs['AAO_val']))
        self.ui.BBO_val.setText(str(monte_carlo_attrs['BBO_val']))
        self.ui.FKO_val.setText(str(monte_carlo_attrs['FKO_val']))
        self.ui.HCUT_val.setText(str(monte_carlo_attrs['HCUT_val']))
        self.ui.H_min.setText(str(monte_carlo_attrs['H_min']))
        self.ui.H_max.setText(str(monte_carlo_attrs['H_max']))
        self.ui.H_mean.setText(str(monte_carlo_attrs['H_mean']))
        self.ui.wave_interval.setText(str(monte_carlo_attrs['wave_interval']))
        self.ui.use_beach_fill_chk.setChecked(monte_carlo_attrs['use_beach_fill'] != 0)
        self.ui.use_beach_fill_chk.toggled.connect(self._on_toggle_beach_fill)
        self.ui.std_percent.setText(str(monte_carlo_attrs['std_percent']))
        self._set_probability_options_visible(self.ui.prob_function_cbx.currentIndex())

    def _set_probability_options_visible(self, idx):
        """Sets options appropriate for the given probability function selected.

        Args:
            idx (int): Current index of the probability function selected.
        """
        # 1 = Rayleigh+Weibull
        # 2 = User Defined
        self.ui.AAO_val_label.setVisible(idx > 0)
        self.ui.AAO_val.setVisible(idx > 0)
        self.ui.BBO_val_label.setVisible(idx > 0)
        self.ui.BBO_val.setVisible(idx > 0)
        self.ui.FKO_val_label.setVisible(idx > 0)
        self.ui.FKO_val.setVisible(idx > 0)
        self.ui.user_defined_label.setVisible(idx == 2)

    def _save_monte_carlo(self):
        """Save data from the Monte Carlo attributes."""
        self.sim_data.monte_carlo.attrs['use_monte_carlo'] = \
            1 if self.ui.enable_mc_group.isChecked() else 0
        self.sim_data.monte_carlo.attrs['prob_function'] = self.ui.prob_function_cbx.currentText()
        self.sim_data.monte_carlo.attrs['AL_mean'] = float(self.ui.AL_mean.text())
        self.sim_data.monte_carlo.attrs['AL_sigma'] = float(self.ui.AL_sigma.text())
        self.sim_data.monte_carlo.attrs['AL_max'] = float(self.ui.AL_max.text())
        self.sim_data.monte_carlo.attrs['num_mc_sims'] = float(self.ui.num_mc_sims.text())

        self.sim_data.monte_carlo.attrs['AA0_val'] = float(self.ui.AAO_val.text())
        self.sim_data.monte_carlo.attrs['BB0_val'] = float(self.ui.BBO_val.text())
        self.sim_data.monte_carlo.attrs['FKO_val'] = float(self.ui.FKO_val.text())
        self.sim_data.monte_carlo.attrs['HCUT_val'] = float(self.ui.HCUT_val.text())
        self.sim_data.monte_carlo.attrs['H_min'] = float(self.ui.H_min.text())
        self.sim_data.monte_carlo.attrs['H_max'] = float(self.ui.H_max.text())
        self.sim_data.monte_carlo.attrs['H_mean'] = float(self.ui.H_mean.text())
        self.sim_data.monte_carlo.attrs['wave_interval'] = float(self.ui.wave_interval.text())
        self.sim_data.monte_carlo.attrs['use_beach_fill'] =\
            1 if self.ui.use_beach_fill_chk.isChecked() else 0
        self.sim_data.monte_carlo.attrs['std_percent'] = float(self.ui.std_percent.text())

    def _setup_mc_connection(self):
        """Set up signal/slot connection for the Probability function."""
        self.ui.prob_function_cbx.currentIndexChanged[int].connect(self._set_mc_visibility)

    def _set_mc_visibility(self):
        """Enable/Disable visibility of the user_defined_label."""
        state = True if self.ui.enable_mc_group.isChecked and self.ui.prob_function_cbx.currentIndex == 2 else False
        self.ui.user_defined_label.setVisible(state)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_model_setup()
        self._save_beach_setup()
        self._save_seaward_bc()
        self._save_lateral_bc()
        self._save_adaptive_time_steps()
        self._save_cross_shore()
        self._save_monte_carlo()
        self._save_water_level()
        super().accept()
