"""A dialog for editing the point attributes."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.gencade.data.points_data import POINT_TYPES
from xms.gencade.gui.point_atts_ui import Ui_PointAttributes
from xms.gencade.gui.tide_table_widget import TideTableWidget
from xms.gencade.gui.wave_table_widget import WaveTableWidget


class PointAttsDlg(XmsDlg):
    """A dialog for viewing a single point's attribute data."""
    def __init__(self, win_cont, point_data, all_data, units, multi_msg):
        """Initializes the class, sets up the ui.

        Args:
            win_cont(:obj:`QWidget`): Parent dialog
            point_data(:obj:`xarray.Dataset`): Dataset containing the station parameters
            all_data(:obj:`StructData`): The point data for all points.
            multi_msg(:obj:`str`): Warning/multi-select message, if any
            units (:obj:'str'): Units as determined by a query in the component.
        """
        super().__init__(win_cont, 'xms.gencade.gui.point_atts_dlg')
        self._help_url = 'https://cirpwiki.info/wiki/GenCade_2.0_Point_Dialog'
        self._units = 'Imperial' if 'feet' in units else 'Metric'
        self.point_data = point_data
        self.all_data = all_data
        self._multi_msg = multi_msg

        # Set up the dialog
        self.setWindowTitle('GenCade Point Attributes')
        self.ui = Ui_PointAttributes()
        self.ui.setupUi(self)
        self.ui.units_label.setText(f'Units for this projection are {self._units}.')
        self.ui.units_label.setStyleSheet('color: red;')

        # Commonly used edit field validators
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.int_validator = QxIntValidator(parent=self)

        self._add_multi_select_warning()
        self._add_tables()
        self.ui.point_type.addItems(POINT_TYPES)
        self._add_validators()
        self._load_point_data()
        self._setup_connections()

    def _add_tables(self):
        """Adds tables to dialog for wave and tide gages."""
        table_id = int(self.point_data['wave_table'].item())
        if table_id <= 0:
            table_id = self.point_data.comp_id
            self.point_data['wave_table'] = table_id
        wave_table = self.all_data.wave_table_from_id(table_id)
        self.wave_table = WaveTableWidget(self.ui.wave_group, wave_table.to_dataframe())
        self.wave_table.setObjectName('wave_table')
        self.wave_table.setMinimumHeight(205)  # min height from the XML
        self.ui.wave_group.layout().insertWidget(0, self.wave_table)

        table_id = int(self.point_data['tide_table'].item())
        if table_id <= 0:
            table_id = self.point_data.comp_id
            self.point_data['tide_table'] = table_id
        tide_table = self.all_data.tide_table_from_id(table_id)
        self.tide_table = TideTableWidget(self.ui.tide_group, tide_table.to_dataframe())
        self.tide_table.setObjectName('tide_table')
        self.tide_table.setMinimumHeight(205)  # min height from the XML
        self.ui.tide_group.layout().insertWidget(0, self.tide_table)

    def _setup_connections(self):
        """Set up signal/slot connections for the point attributes."""
        self.ui.chk_k1_modify.stateChanged[int].connect(self.ui.edt_k1_modify.setEnabled)
        self.ui.chk_k2_modify.stateChanged[int].connect(self.ui.edt_k2_modify.setEnabled)
        self.ui.chk_ang_amplify.stateChanged[int].connect(self.ui.edt_ang_amplify.setEnabled)
        self.ui.chk_ang_adjust.stateChanged[int].connect(self.ui.edt_ang_adjust.setEnabled)
        self.ui.chk_hgt_amplify.stateChanged[int].connect(self.ui.edt_hgt_amplify.setEnabled)
        self.ui.chk_berm_height.stateChanged[int].connect(self.ui.edt_berm_height.setEnabled)
        self.ui.chk_depth_closure.stateChanged[int].connect(self.ui.edt_depth_closure.setEnabled)
        # self.ui.wave_options.pressed()
        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.point_type.currentIndexChanged[int].connect(self._set_point_stack)

    def _add_validators(self):
        """Add validators for all edit fields."""
        self.ui.edt_k1_modify.setValidator(self.dbl_validator)
        self.ui.edt_k2_modify.setValidator(self.dbl_validator)
        self.ui.edt_ang_amplify.setValidator(self.dbl_validator)
        self.ui.edt_ang_adjust.setValidator(self.dbl_validator)
        self.ui.edt_hgt_amplify.setValidator(self.dbl_validator)
        self.ui.edt_berm_height.setValidator(self.dbl_validator)
        self.ui.edt_depth_closure.setValidator(self.dbl_validator)
        self.ui.wave_depth.setValidator(self.dbl_validator)

    def _add_multi_select_warning(self):
        """Add a label for the multi-select warning, if applicable."""
        if self._multi_msg:
            label = QLabel(self.multi_msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.layout().insertWidget(0, label)

    def _set_point_stack(self):
        """Set point stack to the correct page depending on index.
        """
        idx = self.ui.point_type.currentIndex()
        self.ui.point_stack.setCurrentIndex(idx)

    def _load_point_data(self):
        """Loads the point data into the widgets."""
        point_data = self.point_data
        # Set proper stack page
        self.ui.point_type.setCurrentIndex(POINT_TYPES.index(
            point_data['point_type'].item()))
        self._set_point_stack()
        # Set check marks first
        self.ui.chk_k1_modify.setChecked(point_data['set_k1'].item() != 0)
        self.ui.chk_k2_modify.setChecked(point_data['set_k2'].item() != 0)
        self.ui.chk_ang_amplify.setChecked(point_data['set_ang_amplify'].item() != 0)
        self.ui.chk_ang_adjust.setChecked(point_data['set_ang_adjust'].item() != 0)
        self.ui.chk_hgt_amplify.setChecked(point_data['set_hgt_amplify'].item() != 0)
        self.ui.chk_berm_height.setChecked(point_data['set_berm_height'].item() != 0)
        self.ui.chk_depth_closure.setChecked(point_data['set_depth_closure'].item() != 0)
        # If checked, enable fields
        self.ui.edt_k1_modify.setEnabled(self.ui.chk_k1_modify.checkState())
        self.ui.edt_k2_modify.setEnabled(self.ui.chk_k2_modify.checkState())
        self.ui.edt_ang_amplify.setEnabled(self.ui.chk_ang_amplify.checkState())
        self.ui.edt_ang_adjust.setEnabled(self.ui.chk_ang_adjust.checkState())
        self.ui.edt_hgt_amplify.setEnabled(self.ui.chk_hgt_amplify.checkState())
        self.ui.edt_berm_height.setEnabled(self.ui.chk_berm_height.checkState())
        self.ui.edt_depth_closure.setEnabled(self.ui.chk_depth_closure.checkState())
        # Get values for fields
        self.ui.edt_k1_modify.setText(str(point_data['k1_val'].item()))
        self.ui.edt_k2_modify.setText(str(point_data['k2_val'].item()))
        self.ui.edt_ang_amplify.setText(str(point_data['ang_amplify_val'].item()))
        self.ui.edt_ang_adjust.setText(str(point_data['ang_adjust_val'].item()))
        self.ui.edt_hgt_amplify.setText(str(point_data['hgt_amplify_val'].item()))
        self.ui.edt_berm_height.setText(str(point_data['berm_height_val'].item()))
        self.ui.edt_depth_closure.setText(str(point_data['depth_closure_val'].item()))
        self.ui.wave_depth.setText(str(point_data['wave_depth'].item()))

    def _save_point_data(self):
        """Save data from the point attributes."""
        self.point_data['set_k1'] = 1 if self.ui.chk_k1_modify.checkState() == Qt.Checked else 0
        self.point_data['set_k2'] = 1 if self.ui.chk_k2_modify.checkState() == Qt.Checked else 0
        self.point_data['set_ang_amplify'] = 1 if self.ui.chk_ang_amplify.checkState() == Qt.Checked else 0
        self.point_data['set_ang_adjust'] = 1 if self.ui.chk_ang_adjust.checkState() == Qt.Checked else 0
        self.point_data['set_hgt_amplify'] = 1 if self.ui.chk_hgt_amplify.checkState() == Qt.Checked else 0
        self.point_data['set_berm_height'] = 1 if self.ui.chk_berm_height.checkState() == Qt.Checked else 0
        self.point_data['set_depth_closure'] = 1 if self.ui.chk_depth_closure.checkState() == Qt.Checked else 0
        self.point_data['k1_val'] = float(self.ui.edt_k1_modify.text())
        self.point_data['k2_val'] = float(self.ui.edt_k2_modify.text())
        self.point_data['ang_amplify_val'] = float(self.ui.edt_ang_amplify.text())
        self.point_data['ang_adjust_val'] = float(self.ui.edt_ang_adjust.text())
        self.point_data['hgt_amplify_val'] = float(self.ui.edt_hgt_amplify.text())
        self.point_data['berm_height_val'] = float(self.ui.edt_berm_height.text())
        self.point_data['depth_closure_val'] = float(self.ui.edt_depth_closure.text())
        self.point_data['wave_depth'] = float(self.ui.wave_depth.text())
        self.point_data['point_type'] = self.ui.point_type.currentText()
        self.all_data.set_wave_table(int(self.point_data['wave_table'].item()),
                                     self.wave_table.model.data_frame.to_xarray())
        self.all_data.set_tide_table(int(self.point_data['tide_table'].item()),
                                     self.tide_table.model.data_frame.to_xarray())

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_point_data()
        super().accept()
